#include "sigc++/dispatch.h"

namespace SigC
{

Dispatcher::Handler::~Handler()
{
}

Dispatcher::~Dispatcher()
{
}

StandardDispatcher::StandardDispatcher ()
{
  FD_ZERO(&rd_fds_);
  FD_ZERO(&wr_fds_);
  FD_ZERO(&ex_fds_);
}

StandardDispatcher::TimerEvent::TimerEvent(StandardDispatcher *_disp,
                                           Handler *_cb,
                                           unsigned long tmout)
    : disp(_disp), cb(_cb)
{
  expiration.tv_sec = 0;
  expiration.tv_usec = 0;
  
  gettimeofday(&expiration, 0);

  expiration.tv_sec += tmout / 1000;
  expiration.tv_usec += (tmout % 1000) * 1000;
  
  if (expiration.tv_usec >= 1000000)
  {
    expiration.tv_usec -= 1000000;
    expiration.tv_sec++;
  }
}

StandardDispatcher::~StandardDispatcher ()
{
  list<FileEvent *>::iterator i;
  for (i = fd_handlers_.begin(); i != fd_handlers_.end(); ++i)
  {
    (*i)->cb->callback(this, Remove);
    delete (*i);
  }

  list<TimerEvent *>::iterator j;
  for (j = tm_handlers_.begin(); j != tm_handlers_.end(); ++j)
  {
    (*j)->cb->callback(this, Remove);
    delete *j;
  }
}

void StandardDispatcher::add_input_handler(Handler *cb, int fd)
{
  FileEvent *ev = new FileEvent(this, cb, Read, fd);

  FD_SET(fd, &rd_fds_);
  
  fd_handlers_.push_back(ev);
}

void StandardDispatcher::add_output_handler(Handler *cb, int fd)
{
  FileEvent *ev = new FileEvent(this, cb, Write, fd);

  FD_SET(fd, &wr_fds_);
  
  fd_handlers_.push_back(ev);
}

void StandardDispatcher::add_exception_handler(Handler *cb, int fd)
{
  FileEvent *ev = new FileEvent(this, cb, Except, fd);

  FD_SET(fd, &ex_fds_);
  
  fd_handlers_.push_back(ev);
}

void StandardDispatcher::add_timeout_handler(Handler *cb,
                                             unsigned long tmout)
{
  TimerEvent *ev = new TimerEvent(this, cb, tmout);

  tm_handlers_.push_back(ev);
}

void StandardDispatcher::remove(Handler *cb, Event e)
{
  if (e == All || e == Timer)
  {
    list<TimerEvent *>::iterator i, next;
    for (i = tm_handlers_.begin(); i != tm_handlers_.end(); i = next)
    {
      next = i;
      ++next;
      if ((*i)->cb == cb)
      {
        delete *i;
        tm_handlers_.erase(i);
      }
    }
  }
  if (e == All || e == Read || e == Write || e == Except)
  {
    list<FileEvent *>::iterator i, next;
    for (i = fd_handlers_.begin(); i != fd_handlers_.end(); i = next)
    {
      next = i;
      ++next;
      if ((*i)->cb == cb && (e == All || e == (*i)->ev))
      {
        if ((e == All || e == Read) && FD_ISSET((*i)->fd, &rd_fds_))
          FD_CLR((*i)->fd, &rd_fds_);
        if ((e == All || e == Read) && FD_ISSET((*i)->fd, &wr_fds_))
          FD_CLR((*i)->fd, &wr_fds_);
        if ((e == All || e == Read) && FD_ISSET((*i)->fd, &ex_fds_))
          FD_CLR((*i)->fd, &ex_fds_);
        
        delete (*i);
        fd_handlers_.erase(i);
      }
    }
  }
}

static int timeval_subtract(struct timeval *result,
                            struct timeval *x,
                            struct timeval *y)
{
  /* Perform the carry for the later subtraction by updating Y. */
  if (x->tv_usec < y->tv_usec)
  {
    int nsec = (y->tv_usec - x->tv_usec) / 1000000 + 1;
    y->tv_usec -= 1000000 * nsec;
    y->tv_sec += nsec;
  }
  if (x->tv_usec - y->tv_usec > 1000000)
  {
    int nsec = (x->tv_usec - y->tv_usec) / 1000000;
    y->tv_usec += 1000000 * nsec;
    y->tv_sec -= nsec;
  }
     
  /* Compute the time remaining to wait.
     `tv_usec' is certainly positive. */
  result->tv_sec = x->tv_sec - y->tv_sec;
  result->tv_usec = x->tv_usec - y->tv_usec;
     
  /* Return 1 if result is negative. */
  return x->tv_sec < y->tv_sec;
}

void StandardDispatcher::run(bool infinite)
{
  struct timeval now, timeout, event_timeout;

  timeout.tv_sec = 0;
  timeout.tv_usec = 0;

  do
  {
    // We set it here, it may be unset in a handler by calling exit
    do_exit_ = false;

    fd_set rd = rd_fds_;
    fd_set wr = wr_fds_;
    fd_set ex = ex_fds_;
    
    gettimeofday(&now, 0);

    list<TimerEvent *>::iterator i;
    for (i = tm_handlers_.begin(); i != tm_handlers_.end(); ++i)
    {
      if (timeval_subtract(&event_timeout, &((*i)->expiration), &now) == 1)
      {
        (*i)->cb->callback(this, Timer);
        delete *i;
        tm_handlers_.erase(i);
      }
      
      // Set timeout to event_timeout, if it is smaller
      if ((timeout.tv_sec == 0 && timeout.tv_usec == 0) ||
          (event_timeout.tv_sec < timeout.tv_sec ||
           (event_timeout.tv_sec == timeout.tv_sec &&
            event_timeout.tv_sec < timeout.tv_sec)))
        timeout = event_timeout;
    }

    if (tm_handlers_.size() == 0)
      select(FD_SETSIZE, &rd, &wr, &ex, 0);
    else
      select(FD_SETSIZE, &rd, &wr, &ex, &timeout);
    
    list<FileEvent *>::iterator j;
#if 0
    for (int k = 0; k < FD_SETSIZE; k++)
      if (FD_ISSET(k, &rd))
        cout << "event on " << k << endl;
#endif
    for (j = fd_handlers_.begin(); j != fd_handlers_.end(); j++)
    {
#if 0 
      cout << "standard disp: select() done" << endl;
      cout << "  " << (*j) << ", ev = " << (*j)->ev << ", fd = "
           << (*j)->fd << endl;
#endif
      if ((*j)->ev == Read && FD_ISSET((*j)->fd, &rd))
        (*j)->cb->callback(this, Read);
      else if ((*j)->ev == Write && FD_ISSET((*j)->fd, &wr))
        (*j)->cb->callback(this, Write);
      else if ((*j)->ev == Except && FD_ISSET((*j)->fd, &ex))
        (*j)->cb->callback(this, Write);
    }
  } while (infinite && !do_exit_);
}

void StandardDispatcher::move(Dispatcher *)
{
  assert(0);
}

bool StandardDispatcher::idle () const
{
  return(fd_handlers_.size() + tm_handlers_.size() == 0);
}

void StandardDispatcher::exit()
{
  do_exit_ = true;
}

}

