/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *   Yabause - assem_arm.c                                                 *
 *   Copyright (C) 2009-2011 Ari64                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

extern void *dynarec_local;
extern u32 memory_map[1048576];
extern u32 mini_ht_master[32][2]  __attribute__((aligned(8)));
extern u32 mini_ht_slave[32][2]  __attribute__((aligned(8)));
extern u8 restore_candidate[512]  __attribute__((aligned(4)));

void FASTCALL WriteInvalidateLong(u32 addr, u32 val);
void FASTCALL WriteInvalidateWord(u32 addr, u32 val);
void FASTCALL WriteInvalidateByte(u32 addr, u32 val);
void FASTCALL WriteInvalidateByteSwapped(u32 addr, u32 val);

void jump_vaddr_r0_master();
void jump_vaddr_r1_master();
void jump_vaddr_r2_master();
void jump_vaddr_r3_master();
void jump_vaddr_r4_master();
void jump_vaddr_r5_master();
void jump_vaddr_r6_master();
void jump_vaddr_r7_master();
void jump_vaddr_r8_master();
void jump_vaddr_r9_master();
void jump_vaddr_r12_master();
void jump_vaddr_r0_slave();
void jump_vaddr_r1_slave();
void jump_vaddr_r2_slave();
void jump_vaddr_r3_slave();
void jump_vaddr_r4_slave();
void jump_vaddr_r5_slave();
void jump_vaddr_r6_slave();
void jump_vaddr_r7_slave();
void jump_vaddr_r8_slave();
void jump_vaddr_r9_slave();
void jump_vaddr_r12_slave();

const pointer jump_vaddr_reg[2][16] = {
  {
    (pointer)jump_vaddr_r0_master,
    (pointer)jump_vaddr_r1_master,
    (pointer)jump_vaddr_r2_master,
    (pointer)jump_vaddr_r3_master,
    (pointer)jump_vaddr_r4_master,
    (pointer)jump_vaddr_r5_master,
    (pointer)jump_vaddr_r6_master,
    (pointer)jump_vaddr_r7_master,
    (pointer)jump_vaddr_r8_master,
    (pointer)jump_vaddr_r9_master,
    0,
    0,
    (pointer)jump_vaddr_r12_master,
    0,
    0,
    0
  },{
    (pointer)jump_vaddr_r0_slave,
    (pointer)jump_vaddr_r1_slave,
    (pointer)jump_vaddr_r2_slave,
    (pointer)jump_vaddr_r3_slave,
    (pointer)jump_vaddr_r4_slave,
    (pointer)jump_vaddr_r5_slave,
    (pointer)jump_vaddr_r6_slave,
    (pointer)jump_vaddr_r7_slave,
    (pointer)jump_vaddr_r8_slave,
    (pointer)jump_vaddr_r9_slave,
    0,
    0,
    (pointer)jump_vaddr_r12_slave,
    0,
    0,
    0
  }
};

u32 needs_clear_cache[1<<(TARGET_SIZE_2-17)];

//#define JUMP_TABLE_SIZE (sizeof(jump_table_symbols)*2)
#define JUMP_TABLE_SIZE 0

/* Linker */

void set_jump_target(pointer addr,pointer target)
{
  u8 *ptr=(u8 *)addr;
  u32 *ptr2=(u32 *)ptr;
  if(ptr[3]==0xe2) {
    assert((target-(u32)ptr2-8)<1024);
    assert((addr&3)==0);
    assert((target&3)==0);
    *ptr2=(*ptr2&0xFFFFF000)|((target-(u32)ptr2-8)>>2)|0xF00;
    //printf("target=%x addr=%x insn=%x\n",target,addr,*ptr2);
  }
  else if(ptr[3]==0x72) {
    // generated by emit_jno_unlikely
    if((target-(u32)ptr2-8)<1024) {
      assert((addr&3)==0);
      assert((target&3)==0);
      *ptr2=(*ptr2&0xFFFFF000)|((target-(u32)ptr2-8)>>2)|0xF00;
    }
    else if((target-(u32)ptr2-8)<4096&&!((target-(u32)ptr2-8)&15)) {
      assert((addr&3)==0);
      assert((target&3)==0);
      *ptr2=(*ptr2&0xFFFFF000)|((target-(u32)ptr2-8)>>4)|0xE00;
    }
    else *ptr2=(0x7A000000)|(((target-(u32)ptr2-8)<<6)>>8);
  }
  else {
    assert((ptr[3]&0x0e)==0xa);
    *ptr2=(*ptr2&0xFF000000)|(((target-(u32)ptr2-8)<<6)>>8);
  }
}

// This optionally copies the instruction from the target of the branch into
// the space before the branch.  Works, but the difference in speed is
// usually insignificant.
void set_jump_target_fillslot(int addr,u32 target,int copy)
{
  u8 *ptr=(u8 *)addr;
  u32 *ptr2=(u32 *)ptr;
  assert(!copy||ptr2[-1]==0xe28dd000);
  if(ptr[3]==0xe2) {
    assert(!copy);
    assert((target-(u32)ptr2-8)<4096);
    *ptr2=(*ptr2&0xFFFFF000)|(target-(u32)ptr2-8);
  }
  else {
    assert((ptr[3]&0x0e)==0xa);
    u32 target_insn=*(u32 *)target;
    if((target_insn&0x0e100000)==0) { // ALU, no immediate, no flags
      copy=0;
    }
    if((target_insn&0x0c100000)==0x04100000) { // Load
      copy=0;
    }
    if(target_insn&0x08000000) {
      copy=0;
    }
    if(copy) {
      ptr2[-1]=target_insn;
      target+=4;
    }
    *ptr2=(*ptr2&0xFF000000)|(((target-(u32)ptr2-8)<<6)>>8);
  }
}

/* Literal pool */
add_literal(int addr,int val)
{
  literals[literalcount][0]=addr;
  literals[literalcount][1]=val;
  literalcount++; 
} 

void *kill_pointer(void *stub)
{
  int *ptr=(int *)(stub+4);
  assert((*ptr&0x0ff00000)==0x05900000);
  u32 offset=*ptr&0xfff;
  int **l_ptr=(void *)ptr+offset+8;
  int *i_ptr=*l_ptr;
  set_jump_target((int)i_ptr,(int)stub);
  return i_ptr;
}

pointer get_pointer(void *stub)
{
  //printf("get_pointer(%x)\n",(int)stub);
  int *ptr=(int *)(stub+4);
  assert((*ptr&0x0ff00000)==0x05900000);
  u32 offset=*ptr&0xfff;
  int **l_ptr=(void *)ptr+offset+8;
  int *i_ptr=*l_ptr;
  assert((*i_ptr&0x0f000000)==0x0a000000);
  return (pointer)i_ptr+((*i_ptr<<8)>>6)+8;
}

// Find the "clean" entry point from a "dirty" entry point
// by skipping past the call to verify_code
pointer get_clean_addr(pointer addr)
{
  int *ptr=(int *)addr;
  #ifdef ARMv5_ONLY
  ptr+=4;
  #else
  ptr+=6;
  #endif
  if((*ptr&0xFF000000)!=0xeb000000) ptr++;
  assert((*ptr&0xFF000000)==0xeb000000); // bl instruction
  ptr++;
  if((*ptr&0xFF000000)==0xea000000) {
    return (int)ptr+((*ptr<<8)>>6)+8; // follow jump
  }
  return (pointer)ptr;
}

int verify_dirty(pointer addr)
{
  u32 *ptr=(u32 *)addr;
  #ifdef ARMv5_ONLY
  // get from literal pool
  assert((*ptr&0xFFF00000)==0xe5900000);
  u32 offset=*ptr&0xfff;
  u32 *l_ptr=(void *)ptr+offset+8;
  u32 source=l_ptr[0];
  u32 copy=l_ptr[1];
  u32 len=l_ptr[2];
  ptr+=4;
  #else
  // ARMv7 movw/movt
  assert((*ptr&0xFFF00000)==0xe3000000);
  u32 source=(ptr[0]&0xFFF)+((ptr[0]>>4)&0xF000)+((ptr[2]<<16)&0xFFF0000)+((ptr[2]<<12)&0xF0000000);
  u32 copy=(ptr[1]&0xFFF)+((ptr[1]>>4)&0xF000)+((ptr[3]<<16)&0xFFF0000)+((ptr[3]<<12)&0xF0000000);
  u32 len=(ptr[4]&0xFFF)+((ptr[4]>>4)&0xF000);
  ptr+=6;
  #endif
  if((*ptr&0xFF000000)!=0xeb000000) ptr++;
  assert((*ptr&0xFF000000)==0xeb000000); // bl instruction
  //printf("verify_dirty: %x %x %x\n",source,copy,len);
  return !memcmp((void *)source,(void *)copy,len);
}

// This doesn't necessarily find all clean entry points, just
// guarantees that it's not dirty
int isclean(pointer addr)
{
  #ifdef ARMv5_ONLY
  int *ptr=((u32 *)addr)+4;
  #else
  int *ptr=((u32 *)addr)+6;
  #endif
  if((*ptr&0xFF000000)!=0xeb000000) ptr++;
  if((*ptr&0xFF000000)!=0xeb000000) return 1; // bl instruction
  if((int)ptr+((*ptr<<8)>>6)+8==(int)verify_code) return 0;
  return 1;
}

void get_bounds(pointer addr,u32 *start,u32 *end)
{
  u32 *ptr=(u32 *)addr;
  #ifdef ARMv5_ONLY
  // get from literal pool
  assert((*ptr&0xFFF00000)==0xe5900000);
  u32 offset=*ptr&0xfff;
  u32 *l_ptr=(void *)ptr+offset+8;
  u32 source=l_ptr[0];
  //u32 copy=l_ptr[1];
  u32 len=l_ptr[2];
  ptr+=4;
  #else
  // ARMv7 movw/movt
  assert((*ptr&0xFFF00000)==0xe3000000);
  u32 source=(ptr[0]&0xFFF)+((ptr[0]>>4)&0xF000)+((ptr[2]<<16)&0xFFF0000)+((ptr[2]<<12)&0xF0000000);
  //u32 copy=(ptr[1]&0xFFF)+((ptr[1]>>4)&0xF000)+((ptr[3]<<16)&0xFFF0000)+((ptr[3]<<12)&0xF0000000);
  u32 len=(ptr[4]&0xFFF)+((ptr[4]>>4)&0xF000);
  ptr+=6;
  #endif
  if((*ptr&0xFF000000)!=0xeb000000) ptr++;
  assert((*ptr&0xFF000000)==0xeb000000); // bl instruction
  *start=source;
  *end=source+len;
}

/* Register allocation */

// Note: registers are allocated clean (unmodified state)
// if you intend to modify the register, you must call dirty_reg().
void alloc_reg(struct regstat *cur,int i,signed char reg)
{
  int r,hr;
  int preferred_reg = (reg&7);
  if(reg==CCREG) preferred_reg=HOST_CCREG;
  if(reg==PTEMP) preferred_reg=12;
  
  // Don't allocate unused registers
  if((cur->u>>reg)&1) return;
  
  // see if it's already allocated
  for(hr=0;hr<HOST_REGS;hr++)
  {
    if(cur->regmap[hr]==reg) return;
  }
  
  // Keep the same mapping if the register was already allocated in a loop
  preferred_reg = loop_reg(i,reg,preferred_reg);
  
  // Try to allocate the preferred register
  if(cur->regmap[preferred_reg]==-1) {
    cur->regmap[preferred_reg]=reg;
    cur->dirty&=~(1<<preferred_reg);
    cur->isdoingcp&=~(1<<preferred_reg);
    return;
  }
  r=cur->regmap[preferred_reg];
  if(r<64&&((cur->u>>r)&1)) {
    cur->regmap[preferred_reg]=reg;
    cur->dirty&=~(1<<preferred_reg);
    cur->isdoingcp&=~(1<<preferred_reg);
    return;
  }
  
  // Clear any unneeded registers
  // We try to keep the mapping consistent, if possible, because it
  // makes branches easier (especially loops).  So we try to allocate
  // first (see above) before removing old mappings.  If this is not
  // possible then go ahead and clear out the registers that are no
  // longer needed.
  for(hr=0;hr<HOST_REGS;hr++)
  {
    r=cur->regmap[hr];
    if(r>=0) {
      if((cur->u>>r)&1)
        if(i==0||(unneeded_reg[i-1]>>r)&1) {cur->regmap[hr]=-1;break;}
    }
  }
  // Try to allocate any available register, but prefer
  // registers that have not been used recently.
  if(i>0) {
    for(hr=0;hr<HOST_REGS;hr++) {
      if(hr!=EXCLUDE_REG&&cur->regmap[hr]==-1) {
        if(regs[i-1].regmap[hr]!=rs1[i-1]&&regs[i-1].regmap[hr]!=rs2[i-1]&&regs[i-1].regmap[hr]!=rt1[i-1]&&regs[i-1].regmap[hr]!=rt2[i-1]) {
          cur->regmap[hr]=reg;
          cur->dirty&=~(1<<hr);
          cur->isdoingcp&=~(1<<hr);
          return;
        }
      }
    }
  }
  // Try to allocate any available register
  for(hr=0;hr<HOST_REGS;hr++) {
    if(hr!=EXCLUDE_REG&&cur->regmap[hr]==-1) {
      cur->regmap[hr]=reg;
      cur->dirty&=~(1<<hr);
      cur->isdoingcp&=~(1<<hr);
      return;
    }
  }
  
  // Ok, now we have to evict someone
  // Pick a register we hopefully won't need soon
  unsigned char hsn[MAXREG+1];
  memset(hsn,10,sizeof(hsn));
  int j;
  lsn(hsn,i,&preferred_reg);
  //printf("eax=%d ecx=%d edx=%d ebx=%d ebp=%d esi=%d edi=%d\n",cur->regmap[0],cur->regmap[1],cur->regmap[2],cur->regmap[3],cur->regmap[5],cur->regmap[6],cur->regmap[7]);
  //printf("hsn(%x): %d %d %d %d %d %d %d\n",start+i*4,hsn[cur->regmap[0]&63],hsn[cur->regmap[1]&63],hsn[cur->regmap[2]&63],hsn[cur->regmap[3]&63],hsn[cur->regmap[5]&63],hsn[cur->regmap[6]&63],hsn[cur->regmap[7]&63]);
  if(i>0) {
    // Don't evict the cycle count at entry points, otherwise the entry
    // stub will have to write it.
    if(bt[i]&&hsn[CCREG]>2) hsn[CCREG]=2;
    if(i>1&&hsn[CCREG]>2&&(itype[i-2]==RJUMP||itype[i-2]==UJUMP||itype[i-2]==CJUMP||itype[i-2]==SJUMP)) hsn[CCREG]=2;
    for(j=10;j>=3;j--)
    {
      // Alloc preferred register if available
      if(hsn[r=cur->regmap[preferred_reg]&63]==j) {
        for(hr=0;hr<HOST_REGS;hr++) {
          // Evict both parts of a 64-bit register
          if((cur->regmap[hr]&63)==r) {
            cur->regmap[hr]=-1;
            cur->dirty&=~(1<<hr);
            cur->isdoingcp&=~(1<<hr);
          }
        }
        cur->regmap[preferred_reg]=reg;
        return;
      }
      for(r=0;r<=MAXREG;r++)
      {
        if(hsn[r]==j&&r!=rs1[i-1]&&r!=rs2[i-1]&&r!=rt1[i-1]&&r!=rt2[i-1]) {
          for(hr=0;hr<HOST_REGS;hr++) {
            if(hr!=HOST_CCREG||j<hsn[CCREG]) {
              if(cur->regmap[hr]==r+64) {
                cur->regmap[hr]=reg;
                cur->dirty&=~(1<<hr);
                cur->isdoingcp&=~(1<<hr);
                return;
              }
            }
          }
          for(hr=0;hr<HOST_REGS;hr++) {
            if(hr!=HOST_CCREG||j<hsn[CCREG]) {
              if(cur->regmap[hr]==r) {
                cur->regmap[hr]=reg;
                cur->dirty&=~(1<<hr);
                cur->isdoingcp&=~(1<<hr);
                return;
              }
            }
          }
        }
      }
    }
  }
  for(j=10;j>=0;j--)
  {
    for(r=0;r<=MAXREG;r++)
    {
      if(hsn[r]==j) {
        for(hr=0;hr<HOST_REGS;hr++) {
          if(cur->regmap[hr]==r+64) {
            cur->regmap[hr]=reg;
            cur->dirty&=~(1<<hr);
            cur->isdoingcp&=~(1<<hr);
            return;
          }
        }
        for(hr=0;hr<HOST_REGS;hr++) {
          if(cur->regmap[hr]==r) {
            cur->regmap[hr]=reg;
            cur->dirty&=~(1<<hr);
            cur->isdoingcp&=~(1<<hr);
            return;
          }
        }
      }
    }
  }
  printf("This shouldn't happen (alloc_reg)");exit(1);
}

// Allocate a temporary register.  This is done without regard to
// dirty status or whether the register we request is on the unneeded list
// Note: This will only allocate one register, even if called multiple times
void alloc_reg_temp(struct regstat *cur,int i,signed char reg)
{
  int r,hr;
  int preferred_reg = -1;
  
  // see if it's already allocated
  for(hr=0;hr<HOST_REGS;hr++)
  {
    if(hr!=EXCLUDE_REG&&cur->regmap[hr]==reg) return;
  }
  
  // Try to allocate any available register
  for(hr=HOST_REGS-1;hr>=0;hr--) {
    if(hr!=EXCLUDE_REG&&cur->regmap[hr]==-1) {
      cur->regmap[hr]=reg;
      cur->dirty&=~(1<<hr);
      cur->isdoingcp&=~(1<<hr);
      return;
    }
  }
  
  // Find an unneeded register
  for(hr=HOST_REGS-1;hr>=0;hr--)
  {
    r=cur->regmap[hr];
    if(r>=0) {
      if((cur->u>>r)&1) {
        if(i==0||((unneeded_reg[i-1]>>r)&1)) {
          cur->regmap[hr]=reg;
          cur->dirty&=~(1<<hr);
          cur->isdoingcp&=~(1<<hr);
          return;
        }
      }
    }
  }
  
  // Ok, now we have to evict someone
  // Pick a register we hopefully won't need soon
  // TODO: we might want to follow unconditional jumps here
  // TODO: get rid of dupe code and make this into a function
  unsigned char hsn[MAXREG+1];
  memset(hsn,10,sizeof(hsn));
  int j;
  lsn(hsn,i,&preferred_reg);
  //printf("hsn: %d %d %d %d %d %d %d\n",hsn[cur->regmap[0]&63],hsn[cur->regmap[1]&63],hsn[cur->regmap[2]&63],hsn[cur->regmap[3]&63],hsn[cur->regmap[5]&63],hsn[cur->regmap[6]&63],hsn[cur->regmap[7]&63]);
  if(i>0) {
    // Don't evict the cycle count at entry points, otherwise the entry
    // stub will have to write it.
    if(bt[i]&&hsn[CCREG]>2) hsn[CCREG]=2;
    if(i>1&&hsn[CCREG]>2&&(itype[i-2]==RJUMP||itype[i-2]==UJUMP||itype[i-2]==CJUMP||itype[i-2]==SJUMP)) hsn[CCREG]=2;
    for(j=10;j>=3;j--)
    {
      for(r=0;r<=MAXREG;r++)
      {
        if(hsn[r]==j&&r!=rs1[i-1]&&r!=rs2[i-1]&&r!=rt1[i-1]&&r!=rt2[i-1]) {
          for(hr=0;hr<HOST_REGS;hr++) {
            if(hr!=HOST_CCREG||j<hsn[CCREG]) {
              if(cur->regmap[hr]==r+64) {
                cur->regmap[hr]=reg;
                cur->dirty&=~(1<<hr);
                cur->isdoingcp&=~(1<<hr);
                return;
              }
            }
          }
          for(hr=0;hr<HOST_REGS;hr++) {
            if(hr!=HOST_CCREG||j<hsn[CCREG]) {
              if(cur->regmap[hr]==r) {
                cur->regmap[hr]=reg;
                cur->dirty&=~(1<<hr);
                cur->isdoingcp&=~(1<<hr);
                return;
              }
            }
          }
        }
      }
    }
  }
  for(j=10;j>=0;j--)
  {
    for(r=0;r<=MAXREG;r++)
    {
      if(hsn[r]==j) {
        for(hr=0;hr<HOST_REGS;hr++) {
          if(cur->regmap[hr]==r+64) {
            cur->regmap[hr]=reg;
            cur->dirty&=~(1<<hr);
            cur->isdoingcp&=~(1<<hr);
            return;
          }
        }
        for(hr=0;hr<HOST_REGS;hr++) {
          if(cur->regmap[hr]==r) {
            cur->regmap[hr]=reg;
            cur->dirty&=~(1<<hr);
            cur->isdoingcp&=~(1<<hr);
            return;
          }
        }
      }
    }
  }
  printf("This shouldn't happen");exit(1);
}
// Allocate a specific ARM register.
void alloc_arm_reg(struct regstat *cur,int i,signed char reg,char hr)
{
  int n;
  u32 dirty=0;
  
  // see if it's already allocated (and dealloc it)
  for(n=0;n<HOST_REGS;n++)
  {
    if(n!=EXCLUDE_REG&&cur->regmap[n]==reg) {
      dirty=(cur->dirty>>n)&1;
      cur->regmap[n]=-1;
    }
  }
  
  cur->regmap[hr]=reg;
  cur->dirty&=~(1<<hr);
  cur->dirty|=dirty<<hr;
  cur->isdoingcp&=~(1<<hr);
}

// Alloc cycle count into dedicated register
alloc_cc(struct regstat *cur,int i)
{
  alloc_arm_reg(cur,i,CCREG,HOST_CCREG);
}

/* Special alloc */


/* Assembler */

char regname[16][4] = {
 "r0",
 "r1",
 "r2",
 "r3",
 "r4",
 "r5",
 "r6",
 "r7",
 "r8",
 "r9",
 "r10",
 "fp",
 "r12",
 "sp",
 "lr",
 "pc"};

void output_byte(u8 byte)
{
  *(out++)=byte;
}
void output_modrm(u8 mod,u8 rm,u8 ext)
{
  assert(mod<4);
  assert(rm<8);
  assert(ext<8);
  u8 byte=(mod<<6)|(ext<<3)|rm;
  *(out++)=byte;
}
void output_sib(u8 scale,u8 index,u8 base)
{
  assert(scale<4);
  assert(index<8);
  assert(base<8);
  u8 byte=(scale<<6)|(index<<3)|base;
  *(out++)=byte;
}
void output_w32(u32 word)
{
  *((u32 *)out)=word;
  out+=4;
}
u32 rd_rn_rm(unsigned int rd, unsigned int rn, unsigned int rm)
{
  assert(rd<16);
  assert(rn<16);
  assert(rm<16);
  return((rn<<16)|(rd<<12)|rm);
}
u32 rd_rn_imm_shift(unsigned int rd, unsigned int rn, u32 imm, unsigned int shift)
{
  assert(rd<16);
  assert(rn<16);
  assert(imm<256);
  assert((shift&1)==0);
  return((rn<<16)|(rd<<12)|(((32-shift)&30)<<7)|imm);
}
u32 genimm(u32 imm,u32 *encoded)
{
  if(imm==0) {*encoded=0;return 1;}
  int i=32;
  while(i>0)
  {
    if(imm<256) {
      *encoded=((i&30)<<7)|imm;
      return 1;
    }
    imm=(imm>>2)|(imm<<30);i-=2;
  }
  return 0;
}
u32 genjmp(u32 addr)
{
  if(addr<4) return 0;
  int offset=addr-(int)out-8;
  #if 0
  if(offset<-33554432||offset>=33554432) {
    int n;
    for (n=0;n<sizeof(jump_table_symbols)/4;n++)
    {
      if(addr==jump_table_symbols[n])
      {
        offset=BASE_ADDR+(1<<TARGET_SIZE_2)-JUMP_TABLE_SIZE+n*8-(int)out-8;
        break;
      }
    }
  }
  #endif
  assert(offset>=-33554432&&offset<33554432);
  return ((u32)offset>>2)&0xffffff;
}

void emit_mov(int rs,int rt)
{
  assem_debug("mov %s,%s\n",regname[rt],regname[rs]);
  output_w32(0xe1a00000|rd_rn_rm(rt,0,rs));
}

void emit_movs(int rs,int rt)
{
  assem_debug("movs %s,%s\n",regname[rt],regname[rs]);
  output_w32(0xe1b00000|rd_rn_rm(rt,0,rs));
}

void emit_add(int rs1,int rs2,int rt)
{
  assem_debug("add %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe0800000|rd_rn_rm(rt,rs1,rs2));
}

void emit_adds(int rs1,int rs2,int rt)
{
  assem_debug("adds %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe0900000|rd_rn_rm(rt,rs1,rs2));
}

void emit_adc(int rs1,int rs2,int rt)
{
  assem_debug("adc %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe0a00000|rd_rn_rm(rt,rs1,rs2));
}

void emit_adcs(int rs1,int rs2,int rt)
{
  assem_debug("adcs %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe0b00000|rd_rn_rm(rt,rs1,rs2));
}

void emit_sbc(int rs1,int rs2,int rt)
{
  assem_debug("sbc %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe0c00000|rd_rn_rm(rt,rs1,rs2));
}

void emit_sbcs(int rs1,int rs2,int rt)
{
  assem_debug("sbcs %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe0d00000|rd_rn_rm(rt,rs1,rs2));
}

void emit_neg(int rs, int rt)
{
  assem_debug("rsb %s,%s,#0\n",regname[rt],regname[rs]);
  output_w32(0xe2600000|rd_rn_rm(rt,rs,0));
}

void emit_negs(int rs, int rt)
{
  assem_debug("rsbs %s,%s,#0\n",regname[rt],regname[rs]);
  output_w32(0xe2700000|rd_rn_rm(rt,rs,0));
}

void emit_sub(int rs1,int rs2,int rt)
{
  assem_debug("sub %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe0400000|rd_rn_rm(rt,rs1,rs2));
}

void emit_subs(int rs1,int rs2,int rt)
{
  assem_debug("subs %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe0500000|rd_rn_rm(rt,rs1,rs2));
}

void emit_zeroreg(int rt)
{
  assem_debug("mov %s,#0\n",regname[rt]);
  output_w32(0xe3a00000|rd_rn_rm(rt,0,0));
}

void emit_loadlp(u32 imm,unsigned int rt)
{
  add_literal((int)out,imm);
  assem_debug("ldr %s,pc+? [=%x]\n",regname[rt],imm);
  output_w32(0xe5900000|rd_rn_rm(rt,15,0));
}
void emit_movw(u32 imm,unsigned int rt)
{
  assert(imm<65536);
  assem_debug("movw %s,#%d (0x%x)\n",regname[rt],imm,imm);
  output_w32(0xe3000000|rd_rn_rm(rt,0,0)|(imm&0xfff)|((imm<<4)&0xf0000));
}
void emit_movt(u32 imm,unsigned int rt)
{
  assem_debug("movt %s,#%d (0x%x)\n",regname[rt],imm&0xffff0000,imm&0xffff0000);
  output_w32(0xe3400000|rd_rn_rm(rt,0,0)|((imm>>16)&0xfff)|((imm>>12)&0xf0000));
}
void emit_movimm(u32 imm,unsigned int rt)
{
  u32 armval;
  if(genimm(imm,&armval)) {
    assem_debug("mov %s,#%d\n",regname[rt],imm);
    output_w32(0xe3a00000|rd_rn_rm(rt,0,0)|armval);
  }else if(genimm(~imm,&armval)) {
    assem_debug("mvn %s,#%d\n",regname[rt],imm);
    output_w32(0xe3e00000|rd_rn_rm(rt,0,0)|armval);
  }else if(imm<65536) {
    #ifdef ARMv5_ONLY
    assem_debug("mov %s,#%d\n",regname[rt],imm&0xFF00);
    output_w32(0xe3a00000|rd_rn_imm_shift(rt,0,imm>>8,8));
    assem_debug("add %s,%s,#%d\n",regname[rt],regname[rt],imm&0xFF);
    output_w32(0xe2800000|rd_rn_imm_shift(rt,rt,imm&0xff,0));
    #else
    emit_movw(imm,rt);
    #endif
  }else{
    #ifdef ARMv5_ONLY
    emit_loadlp(imm,rt);
    #else
    emit_movw(imm&0x0000FFFF,rt);
    emit_movt(imm&0xFFFF0000,rt);
    #endif
  }
}
void emit_pcreladdr(unsigned int rt)
{
  assem_debug("add %s,pc,#?\n",regname[rt]);
  output_w32(0xe2800000|rd_rn_rm(rt,15,0));
}

void emit_loadreg(int r, int hr)
{
  if(r==MMREG)
    emit_movimm(((int)memory_map-(int)&dynarec_local)>>2,hr);
  else {
    int addr=(slave?(int)slave_reg:(int)master_reg)+(r<<2);
    if(r==CCREG) addr=slave?(int)&slave_cc:(int)&master_cc;
    u32 offset = addr-(u32)&dynarec_local;
    assert(offset<4096);
    assem_debug("ldr %s,fp+%d\n",regname[hr],offset);
    output_w32(0xe5900000|rd_rn_rm(hr,FP,0)|offset);
  }
}
void emit_storereg(int r, int hr)
{
  int addr=(slave?(int)slave_reg:(int)master_reg)+(r<<2);
  if(r==CCREG) addr=slave?(int)&slave_cc:(int)&master_cc;
  u32 offset = addr-(u32)&dynarec_local;
  assert(offset<4096);
  assem_debug("str %s,fp+%d\n",regname[hr],offset);
  output_w32(0xe5800000|rd_rn_rm(hr,FP,0)|offset);
}

void emit_test(int rs, int rt)
{
  assem_debug("tst %s,%s\n",regname[rs],regname[rt]);
  output_w32(0xe1100000|rd_rn_rm(0,rs,rt));
}

void emit_testimm(int rs,int imm)
{
  u32 armval;
  assem_debug("tst %s,#%d\n",regname[rs],imm);
  assert(genimm(imm,&armval));
  output_w32(0xe3100000|rd_rn_rm(0,rs,0)|armval);
}

void emit_not(int rs,int rt)
{
  assem_debug("mvn %s,%s\n",regname[rt],regname[rs]);
  output_w32(0xe1e00000|rd_rn_rm(rt,0,rs));
}

void emit_and(unsigned int rs1,unsigned int rs2,unsigned int rt)
{
  assem_debug("and %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe0000000|rd_rn_rm(rt,rs1,rs2));
}

void emit_or(unsigned int rs1,unsigned int rs2,unsigned int rt)
{
  assem_debug("orr %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe1800000|rd_rn_rm(rt,rs1,rs2));
}
void emit_or_and_set_flags(int rs1,int rs2,int rt)
{
  assem_debug("orrs %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe1900000|rd_rn_rm(rt,rs1,rs2));
}

void emit_xor(unsigned int rs1,unsigned int rs2,unsigned int rt)
{
  assem_debug("eor %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe0200000|rd_rn_rm(rt,rs1,rs2));
}

void emit_addimm(unsigned int rs,int imm,unsigned int rt)
{
  assert(rs<16);
  assert(rt<16);
  if(imm!=0) {
    assert(imm>-65536&&imm<65536);
    u32 armval;
    if(genimm(imm,&armval)) {
      assem_debug("add %s,%s,#%d\n",regname[rt],regname[rs],imm);
      output_w32(0xe2800000|rd_rn_rm(rt,rs,0)|armval);
    }else if(genimm(-imm,&armval)) {
      assem_debug("sub %s,%s,#%d\n",regname[rt],regname[rs],imm);
      output_w32(0xe2400000|rd_rn_rm(rt,rs,0)|armval);
    }else if(imm<0) {
      assem_debug("sub %s,%s,#%d\n",regname[rt],regname[rs],(-imm)&0xFF00);
      assem_debug("sub %s,%s,#%d\n",regname[rt],regname[rt],(-imm)&0xFF);
      output_w32(0xe2400000|rd_rn_imm_shift(rt,rs,(-imm)>>8,8));
      output_w32(0xe2400000|rd_rn_imm_shift(rt,rt,(-imm)&0xff,0));
    }else{
      assem_debug("add %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF00);
      assem_debug("add %s,%s,#%d\n",regname[rt],regname[rt],imm&0xFF);
      output_w32(0xe2800000|rd_rn_imm_shift(rt,rs,imm>>8,8));
      output_w32(0xe2800000|rd_rn_imm_shift(rt,rt,imm&0xff,0));
    }
  }
  else if(rs!=rt) emit_mov(rs,rt);
}

void emit_addimm_and_set_flags(int imm,int rt)
{
  assert(imm>-65536&&imm<65536);
  u32 armval;
  if(genimm(imm,&armval)) {
    assem_debug("adds %s,%s,#%d\n",regname[rt],regname[rt],imm);
    output_w32(0xe2900000|rd_rn_rm(rt,rt,0)|armval);
  }else if(genimm(-imm,&armval)) {
    assem_debug("subs %s,%s,#%d\n",regname[rt],regname[rt],imm);
    output_w32(0xe2500000|rd_rn_rm(rt,rt,0)|armval);
  }else if(imm<0) {
    assem_debug("sub %s,%s,#%d\n",regname[rt],regname[rt],(-imm)&0xFF00);
    assem_debug("subs %s,%s,#%d\n",regname[rt],regname[rt],(-imm)&0xFF);
    output_w32(0xe2400000|rd_rn_imm_shift(rt,rt,(-imm)>>8,8));
    output_w32(0xe2500000|rd_rn_imm_shift(rt,rt,(-imm)&0xff,0));
  }else{
    assem_debug("add %s,%s,#%d\n",regname[rt],regname[rt],imm&0xFF00);
    assem_debug("adds %s,%s,#%d\n",regname[rt],regname[rt],imm&0xFF);
    output_w32(0xe2800000|rd_rn_imm_shift(rt,rt,imm>>8,8));
    output_w32(0xe2900000|rd_rn_imm_shift(rt,rt,imm&0xff,0));
  }
}
void emit_addimm_no_flags(u32 imm,unsigned int rt)
{
  emit_addimm(rt,imm,rt);
}

void emit_addnop(unsigned int r)
{
  assert(r<16);
  assem_debug("add %s,%s,#0 (nop)\n",regname[r],regname[r]);
  output_w32(0xe2800000|rd_rn_rm(r,r,0));
}

void emit_adcimm(unsigned int rs,int imm,unsigned int rt)
{
  u32 armval;
  assert(genimm(imm,&armval));
  assem_debug("adc %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0xe2a00000|rd_rn_rm(rt,rs,0)|armval);
}
void emit_sbcimm(unsigned int rs,int imm,unsigned int rt)
{
  u32 armval;
  assert(genimm(imm,&armval));
  assem_debug("sbc %s,%s,#%d\n",regname[rt],regname[rt],imm);
  output_w32(0xe2c00000|rd_rn_rm(rt,rt,0)|armval);
}
void emit_sbbimm(int imm,unsigned int rt)
{
  assem_debug("sbb $%d,%%%s\n",imm,regname[rt]);
  assert(rt<8);
  if(imm<128&&imm>=-128) {
    output_byte(0x83);
    output_modrm(3,rt,3);
    output_byte(imm);
  }
  else
  {
    output_byte(0x81);
    output_modrm(3,rt,3);
    output_w32(imm);
  }
}
void emit_rscimm(int rs,int imm,unsigned int rt)
{
  assert(0);
  u32 armval;
  assert(genimm(imm,&armval));
  assem_debug("rsc %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0xe2e00000|rd_rn_rm(rt,rs,0)|armval);
}

void emit_addimm64_32(int rsh,int rsl,int imm,int rth,int rtl)
{
  // TODO: if(genimm(imm,&armval)) ...
  // else
  emit_movimm(imm,HOST_TEMPREG);
  emit_adds(HOST_TEMPREG,rsl,rtl);
  emit_adcimm(rsh,0,rth);
}

void emit_sbb(int rs1,int rs2)
{
  assem_debug("sbb %%%s,%%%s\n",regname[rs2],regname[rs1]);
  output_byte(0x19);
  output_modrm(3,rs1,rs2);
}

void emit_andimm(int rs,int imm,int rt)
{
  u32 armval;
  if(imm==0) {
    emit_zeroreg(rt);
  }else if(genimm(imm,&armval)) {
    assem_debug("and %s,%s,#%d\n",regname[rt],regname[rs],imm);
    output_w32(0xe2000000|rd_rn_rm(rt,rs,0)|armval);
  }else if(genimm(~imm,&armval)) {
    assem_debug("bic %s,%s,#%d\n",regname[rt],regname[rs],imm);
    output_w32(0xe3c00000|rd_rn_rm(rt,rs,0)|armval);
  }else if(imm==65535) {
    #ifdef ARMv5_ONLY
    assem_debug("bic %s,%s,#FF000000\n",regname[rt],regname[rs]);
    output_w32(0xe3c00000|rd_rn_rm(rt,rs,0)|0x4FF);
    assem_debug("bic %s,%s,#00FF0000\n",regname[rt],regname[rt]);
    output_w32(0xe3c00000|rd_rn_rm(rt,rt,0)|0x8FF);
    #else
    assem_debug("uxth %s,%s\n",regname[rt],regname[rs]);
    output_w32(0xe6ff0070|rd_rn_rm(rt,0,rs));
    #endif
  }else{
    assert(imm>0&&imm<65535);
    #ifdef ARMv5_ONLY
    assem_debug("mov r14,#%d\n",imm&0xFF00);
    output_w32(0xe3a00000|rd_rn_imm_shift(HOST_TEMPREG,0,imm>>8,8));
    assem_debug("add r14,r14,#%d\n",imm&0xFF);
    output_w32(0xe2800000|rd_rn_imm_shift(HOST_TEMPREG,HOST_TEMPREG,imm&0xff,0));
    #else
    emit_movw(imm,HOST_TEMPREG);
    #endif
    assem_debug("and %s,%s,r14\n",regname[rt],regname[rs]);
    output_w32(0xe0000000|rd_rn_rm(rt,rs,HOST_TEMPREG));
  }
}

void emit_orimm(int rs,int imm,int rt)
{
  u32 armval;
  if(imm==0) {
    if(rs!=rt) emit_mov(rs,rt);
  }else if(genimm(imm,&armval)) {
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm);
    output_w32(0xe3800000|rd_rn_rm(rt,rs,0)|armval);
  }else{
    assert(imm>0&&imm<65536);
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF00);
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF);
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rs,imm>>8,8));
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rt,imm&0xff,0));
  }
}

void emit_xorimm(int rs,int imm,int rt)
{
  u32 armval;
  if(imm==0) {
    if(rs!=rt) emit_mov(rs,rt);
  }else if(genimm(imm,&armval)) {
    assem_debug("eor %s,%s,#%d\n",regname[rt],regname[rs],imm);
    output_w32(0xe2200000|rd_rn_rm(rt,rs,0)|armval);
  }else{
    assert(imm>0&&imm<65536);
    assem_debug("eor %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF00);
    assem_debug("eor %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF);
    output_w32(0xe2200000|rd_rn_imm_shift(rt,rs,imm>>8,8));
    output_w32(0xe2200000|rd_rn_imm_shift(rt,rt,imm&0xff,0));
  }
}

void emit_shlimm(int rs,unsigned int imm,int rt)
{
  assert(imm>0);
  assert(imm<32);
  if(imm==1)
    emit_add(rs,rs,rt);
  else {
    assem_debug("lsl %s,%s,#%d\n",regname[rt],regname[rs],imm);
    output_w32(0xe1a00000|rd_rn_rm(rt,0,rs)|(imm<<7));
  }
}
void emit_lsls_imm(int rs,unsigned int imm,int rt)
{
  assert(imm>0);
  assert(imm<32);
  if(imm==1)
    emit_adds(rs,rs,rt);
  else {
    assem_debug("lsls %s,%s,#%d\n",regname[rt],regname[rs],imm);
    output_w32(0xe1b00000|rd_rn_rm(rt,0,rs)|(imm<<7));
  }
}

void emit_shrimm(int rs,unsigned int imm,int rt)
{
  assert(imm>0);
  assert(imm<32);
  assem_debug("lsr %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0xe1a00000|rd_rn_rm(rt,0,rs)|0x20|(imm<<7));
}
void emit_lsrs_imm(int rs,unsigned int imm,int rt)
{
  assert(imm>0);
  assert(imm<32);
  assem_debug("lsrs %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0xe1b00000|rd_rn_rm(rt,0,rs)|0x20|(imm<<7));
}

void emit_sarimm(int rs,unsigned int imm,int rt)
{
  assert(imm>0);
  assert(imm<32);
  assem_debug("asr %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0xe1a00000|rd_rn_rm(rt,0,rs)|0x40|(imm<<7));
}
void emit_asrs_imm(int rs,unsigned int imm,int rt)
{
  assert(imm>0);
  assert(imm<32);
  assem_debug("asrs %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0xe1b00000|rd_rn_rm(rt,0,rs)|0x40|(imm<<7));
}

void emit_rorimm(int rs,unsigned int imm,int rt)
{
  assert(imm>0);
  assert(imm<32);
  assem_debug("ror %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0xe1a00000|rd_rn_rm(rt,0,rs)|0x60|(imm<<7));
}
void emit_rors_imm(int rs,unsigned int imm,int rt)
{
  assert(imm>0);
  assert(imm<32);
  assem_debug("rors %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0xe1b00000|rd_rn_rm(rt,0,rs)|0x60|(imm<<7));
}
void emit_rrxs(int rs, int rt)
{
  assem_debug("rrxs %s,%s\n",regname[rt],regname[rs]);
  output_w32(0xe1b00060|rd_rn_rm(rt,0,rs));
}

void emit_swapb(int rs,int rt)
{
  assem_debug("rev16 %s,%s\n",regname[14],regname[rs]);
  output_w32(0xe6bf0fb0|rd_rn_rm(14,0,rs));
  assem_debug("pkhbt %s,%s,%s\n",regname[rt],regname[14],regname[rs]);
  output_w32(0xe6800010|rd_rn_rm(rt,14,rs));
}

void emit_shldimm(int rs,int rs2,unsigned int imm,int rt)
{
  assem_debug("shld %%%s,%%%s,%d\n",regname[rt],regname[rs2],imm);
  assert(imm>0);
  assert(imm<32);
  //if(imm==1) ...
  assem_debug("lsl %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0xe1a00000|rd_rn_rm(rt,0,rs)|(imm<<7));
  assem_debug("orr %s,%s,%s,lsr #%d\n",regname[rt],regname[rt],regname[rs2],32-imm);
  output_w32(0xe1800020|rd_rn_rm(rt,rt,rs2)|((32-imm)<<7));
}

void emit_shrdimm(int rs,int rs2,unsigned int imm,int rt)
{
  assem_debug("shrd %%%s,%%%s,%d\n",regname[rt],regname[rs2],imm);
  assert(imm>0);
  assert(imm<32);
  //if(imm==1) ...
  assem_debug("lsr %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0xe1a00020|rd_rn_rm(rt,0,rs)|(imm<<7));
  assem_debug("orr %s,%s,%s,lsl #%d\n",regname[rt],regname[rt],regname[rs2],32-imm);
  output_w32(0xe1800000|rd_rn_rm(rt,rt,rs2)|((32-imm)<<7));
}

void emit_shl(unsigned int rs,unsigned int shift,unsigned int rt)
{
  assert(rs<16);
  assert(rt<16);
  assert(shift<16);
  //if(imm==1) ...
  assem_debug("lsl %s,%s,%s\n",regname[rt],regname[rs],regname[shift]);
  output_w32(0xe1a00000|rd_rn_rm(rt,0,rs)|0x10|(shift<<8));
}
void emit_shr(unsigned int rs,unsigned int shift,unsigned int rt)
{
  assert(rs<16);
  assert(rt<16);
  assert(shift<16);
  assem_debug("lsr %s,%s,%s\n",regname[rt],regname[rs],regname[shift]);
  output_w32(0xe1a00000|rd_rn_rm(rt,0,rs)|0x30|(shift<<8));
}
void emit_sar(unsigned int rs,unsigned int shift,unsigned int rt)
{
  assert(rs<16);
  assert(rt<16);
  assert(shift<16);
  assem_debug("asr %s,%s,%s\n",regname[rt],regname[rs],regname[shift]);
  output_w32(0xe1a00000|rd_rn_rm(rt,0,rs)|0x50|(shift<<8));
}
void emit_shlcl(int r)
{
  assem_debug("shl %%%s,%%cl\n",regname[r]);
  assert(0);
}
void emit_shrcl(int r)
{
  assem_debug("shr %%%s,%%cl\n",regname[r]);
  assert(0);
}
void emit_sarcl(int r)
{
  assem_debug("sar %%%s,%%cl\n",regname[r]);
  assert(0);
}

void emit_shldcl(int r1,int r2)
{
  assem_debug("shld %%%s,%%%s,%%cl\n",regname[r1],regname[r2]);
  assert(0);
}
void emit_shrdcl(int r1,int r2)
{
  assem_debug("shrd %%%s,%%%s,%%cl\n",regname[r1],regname[r2]);
  assert(0);
}
void emit_orrshl(unsigned int rs,unsigned int shift,unsigned int rt)
{
  assert(rs<16);
  assert(rt<16);
  assert(shift<16);
  assem_debug("orr %s,%s,%s,lsl %s\n",regname[rt],regname[rt],regname[rs],regname[shift]);
  output_w32(0xe1800000|rd_rn_rm(rt,rt,rs)|0x10|(shift<<8));
}
void emit_orrshr(unsigned int rs,unsigned int shift,unsigned int rt)
{
  assert(rs<16);
  assert(rt<16);
  assert(shift<16);
  assem_debug("orr %s,%s,%s,lsr %s\n",regname[rt],regname[rt],regname[rs],regname[shift]);
  output_w32(0xe1800000|rd_rn_rm(rt,rt,rs)|0x30|(shift<<8));
}

void emit_cmpimm(int rs,int imm)
{
  u32 armval;
  if(genimm(imm,&armval)) {
    assem_debug("cmp %s,#%d\n",regname[rs],imm);
    output_w32(0xe3500000|rd_rn_rm(0,rs,0)|armval);
  }else if(genimm(-imm,&armval)) {
    assem_debug("cmn %s,#%d\n",regname[rs],imm);
    output_w32(0xe3700000|rd_rn_rm(0,rs,0)|armval);
  }else if(imm>0) {
    assert(imm<65536);
    #ifdef ARMv5_ONLY
    emit_movimm(imm,HOST_TEMPREG);
    #else
    emit_movw(imm,HOST_TEMPREG);
    #endif
    assem_debug("cmp %s,r14\n",regname[rs]);
    output_w32(0xe1500000|rd_rn_rm(0,rs,HOST_TEMPREG));
  }else{
    assert(imm>-65536);
    #ifdef ARMv5_ONLY
    emit_movimm(-imm,HOST_TEMPREG);
    #else
    emit_movw(-imm,HOST_TEMPREG);
    #endif
    assem_debug("cmn %s,r14\n",regname[rs]);
    output_w32(0xe1700000|rd_rn_rm(0,rs,HOST_TEMPREG));
  }
}

void emit_cmovne(u32 *addr,int rt)
{
  assem_debug("cmovne %x,%%%s",(int)addr,regname[rt]);
  assert(0);
}
void emit_cmovl(u32 *addr,int rt)
{
  assem_debug("cmovl %x,%%%s",(int)addr,regname[rt]);
  assert(0);
}
void emit_cmovs(u32 *addr,int rt)
{
  assem_debug("cmovs %x,%%%s",(int)addr,regname[rt]);
  assert(0);
}
void emit_cmovne_imm(int imm,int rt)
{
  assem_debug("movne %s,#%d\n",regname[rt],imm);
  u32 armval;
  assert(genimm(imm,&armval));
  output_w32(0x13a00000|rd_rn_rm(rt,0,0)|armval);
}
void emit_cmovl_imm(int imm,int rt)
{
  assem_debug("movlt %s,#%d\n",regname[rt],imm);
  u32 armval;
  assert(genimm(imm,&armval));
  output_w32(0xb3a00000|rd_rn_rm(rt,0,0)|armval);
}
void emit_cmovb_imm(int imm,int rt)
{
  assem_debug("movcc %s,#%d\n",regname[rt],imm);
  u32 armval;
  assert(genimm(imm,&armval));
  output_w32(0x33a00000|rd_rn_rm(rt,0,0)|armval);
}
void emit_cmovs_imm(int imm,int rt)
{
  assem_debug("movmi %s,#%d\n",regname[rt],imm);
  u32 armval;
  assert(genimm(imm,&armval));
  output_w32(0x43a00000|rd_rn_rm(rt,0,0)|armval);
}
void emit_cmove_reg(int rs,int rt)
{
  assem_debug("moveq %s,%s\n",regname[rt],regname[rs]);
  output_w32(0x01a00000|rd_rn_rm(rt,0,rs));
}
void emit_cmovne_reg(int rs,int rt)
{
  assem_debug("movne %s,%s\n",regname[rt],regname[rs]);
  output_w32(0x11a00000|rd_rn_rm(rt,0,rs));
}
void emit_cmovl_reg(int rs,int rt)
{
  assem_debug("movlt %s,%s\n",regname[rt],regname[rs]);
  output_w32(0xb1a00000|rd_rn_rm(rt,0,rs));
}
void emit_cmovs_reg(int rs,int rt)
{
  assem_debug("movmi %s,%s\n",regname[rt],regname[rs]);
  output_w32(0x41a00000|rd_rn_rm(rt,0,rs));
}

void emit_movzbl_reg(int rs, int rt)
{
  emit_andimm(rs,0xff,rt);
}
void emit_movzwl_reg(int rs, int rt)
{
  emit_andimm(rs,0xffff,rt);
}
void emit_movsbl_reg(int rs, int rt)
{
  assem_debug("sxtb %s,%s\n",regname[rt],regname[rs]);
  output_w32(0xe6af0070|rd_rn_rm(rt,0,rs));
}
void emit_movswl_reg(int rs, int rt)
{
  assem_debug("sxth %s,%s\n",regname[rt],regname[rs]);
  output_w32(0xe6bf0070|rd_rn_rm(rt,0,rs));
}

void emit_slti32(int rs,int imm,int rt)
{
  if(rs!=rt) emit_zeroreg(rt);
  emit_cmpimm(rs,imm);
  if(rs==rt) emit_movimm(0,rt);
  emit_cmovl_imm(1,rt);
}
void emit_sltiu32(int rs,int imm,int rt)
{
  if(rs!=rt) emit_zeroreg(rt);
  emit_cmpimm(rs,imm);
  if(rs==rt) emit_movimm(0,rt);
  emit_cmovb_imm(1,rt);
}
void emit_slti64_32(int rsh,int rsl,int imm,int rt)
{
  assert(rsh!=rt);
  emit_slti32(rsl,imm,rt);
  if(imm>=0)
  {
    emit_test(rsh,rsh);
    emit_cmovne_imm(0,rt);
    emit_cmovs_imm(1,rt);
  }
  else
  {
    emit_cmpimm(rsh,-1);
    emit_cmovne_imm(0,rt);
    emit_cmovl_imm(1,rt);
  }
}
void emit_sltiu64_32(int rsh,int rsl,int imm,int rt)
{
  assert(rsh!=rt);
  emit_sltiu32(rsl,imm,rt);
  if(imm>=0)
  {
    emit_test(rsh,rsh);
    emit_cmovne_imm(0,rt);
  }
  else
  {
    emit_cmpimm(rsh,-1);
    emit_cmovne_imm(1,rt);
  }
}

void emit_cmp(int rs,int rt)
{
  assem_debug("cmp %s,%s\n",regname[rs],regname[rt]);
  output_w32(0xe1500000|rd_rn_rm(0,rs,rt));
}
void emit_set_gz32(int rs, int rt)
{
  //assem_debug("set_gz32\n");
  emit_cmpimm(rs,1);
  emit_movimm(1,rt);
  emit_cmovl_imm(0,rt);
}
void emit_set_nz32(int rs, int rt)
{
  //assem_debug("set_nz32\n");
  if(rs!=rt) emit_movs(rs,rt);
  else emit_test(rs,rs);
  emit_cmovne_imm(1,rt);
}
void emit_set_gz64_32(int rsh, int rsl, int rt)
{
  //assem_debug("set_gz64\n");
  emit_set_gz32(rsl,rt);
  emit_test(rsh,rsh);
  emit_cmovne_imm(1,rt);
  emit_cmovs_imm(0,rt);
}
void emit_set_nz64_32(int rsh, int rsl, int rt)
{
  //assem_debug("set_nz64\n");
  emit_or_and_set_flags(rsh,rsl,rt);
  emit_cmovne_imm(1,rt);
}
void emit_set_if_less32(int rs1, int rs2, int rt)
{
  //assem_debug("set if less (%%%s,%%%s),%%%s\n",regname[rs1],regname[rs2],regname[rt]);
  if(rs1!=rt&&rs2!=rt) emit_zeroreg(rt);
  emit_cmp(rs1,rs2);
  if(rs1==rt||rs2==rt) emit_movimm(0,rt);
  emit_cmovl_imm(1,rt);
}
void emit_set_if_carry32(int rs1, int rs2, int rt)
{
  //assem_debug("set if carry (%%%s,%%%s),%%%s\n",regname[rs1],regname[rs2],regname[rt]);
  if(rs1!=rt&&rs2!=rt) emit_zeroreg(rt);
  emit_cmp(rs1,rs2);
  if(rs1==rt||rs2==rt) emit_movimm(0,rt);
  emit_cmovb_imm(1,rt);
}
void emit_set_if_less64_32(int u1, int l1, int u2, int l2, int rt)
{
  //assem_debug("set if less64 (%%%s,%%%s,%%%s,%%%s),%%%s\n",regname[u1],regname[l1],regname[u2],regname[l2],regname[rt]);
  assert(u1!=rt);
  assert(u2!=rt);
  emit_cmp(l1,l2);
  emit_movimm(0,rt);
  emit_sbcs(u1,u2,HOST_TEMPREG);
  emit_cmovl_imm(1,rt);
}
void emit_set_if_carry64_32(int u1, int l1, int u2, int l2, int rt)
{
  //assem_debug("set if carry64 (%%%s,%%%s,%%%s,%%%s),%%%s\n",regname[u1],regname[l1],regname[u2],regname[l2],regname[rt]);
  assert(u1!=rt);
  assert(u2!=rt);
  emit_cmp(l1,l2);
  emit_movimm(0,rt);
  emit_sbcs(u1,u2,HOST_TEMPREG);
  emit_cmovb_imm(1,rt);
}

void emit_orreq_imm(int rs,int imm,int rt)
{
  u32 armval;
  if(genimm(imm,&armval)) {
    assem_debug("orreq %s,%s,#%d\n",regname[rt],regname[rs],imm);
    output_w32(0x03800000|rd_rn_rm(rt,rs,0)|armval);
  }else{
    assert(0); // FIXME
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF00);
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF);
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rs,imm>>8,8));
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rt,imm&0xff,0));
  }
}

void emit_orrne_imm(int rs,int imm,int rt)
{
  u32 armval;
  if(genimm(imm,&armval)) {
    assem_debug("orrne %s,%s,#%d\n",regname[rt],regname[rs],imm);
    output_w32(0x13800000|rd_rn_rm(rt,rs,0)|armval);
  }else{
    assert(0); // FIXME
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF00);
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF);
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rs,imm>>8,8));
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rt,imm&0xff,0));
  }
}

void emit_orrhi_imm(int rs,int imm,int rt)
{
  u32 armval;
  if(genimm(imm,&armval)) {
    assem_debug("orrhi %s,%s,#%d\n",regname[rt],regname[rs],imm);
    output_w32(0x83800000|rd_rn_rm(rt,rs,0)|armval);
  }else{
    assert(0); // FIXME
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF00);
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF);
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rs,imm>>8,8));
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rt,imm&0xff,0));
  }
}

void emit_orrge_imm(int rs,int imm,int rt)
{
  u32 armval;
  if(genimm(imm,&armval)) {
    assem_debug("orrge %s,%s,#%d\n",regname[rt],regname[rs],imm);
    output_w32(0xa3800000|rd_rn_rm(rt,rs,0)|armval);
  }else{
    assert(0); // FIXME
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF00);
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF);
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rs,imm>>8,8));
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rt,imm&0xff,0));
  }
}

void emit_orrgt_imm(int rs,int imm,int rt)
{
  u32 armval;
  if(genimm(imm,&armval)) {
    assem_debug("orrgt %s,%s,#%d\n",regname[rt],regname[rs],imm);
    output_w32(0xc3800000|rd_rn_rm(rt,rs,0)|armval);
  }else{
    assert(0); // FIXME
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF00);
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF);
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rs,imm>>8,8));
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rt,imm&0xff,0));
  }
}

void emit_orrmi_imm(int rs,int imm,int rt)
{
  u32 armval;
  if(genimm(imm,&armval)) {
    assem_debug("orrmi %s,%s,#%d\n",regname[rt],regname[rs],imm);
    output_w32(0x43800000|rd_rn_rm(rt,rs,0)|armval);
  }else{
    assert(0); // FIXME
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF00);
    assem_debug("orr %s,%s,#%d\n",regname[rt],regname[rs],imm&0xFF);
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rs,imm>>8,8));
    output_w32(0xe3800000|rd_rn_imm_shift(rt,rt,imm&0xff,0));
  }
}

void emit_sh2tst(int s1, int s2, int sr, int temp)
{
  emit_andimm(sr,~1,sr);
  emit_test(s1,s2);
  emit_orreq_imm(sr,1,sr);
}
void emit_sh2tstimm(int s, int imm, int sr, int temp)
{
  emit_andimm(sr,~1,sr);
  emit_testimm(s,imm);
  emit_orreq_imm(sr,1,sr);
}
void emit_cmpeq(int s1, int s2, int sr, int temp)
{
  emit_andimm(sr,~1,sr);
  emit_cmp(s1,s2);
  emit_orreq_imm(sr,1,sr);
}
void emit_cmpeqimm(int s, int imm, int sr, int temp)
{
  emit_andimm(sr,~1,sr);
  emit_cmpimm(s,imm);
  emit_orreq_imm(sr,1,sr);
}
void emit_cmpge(int s1, int s2, int sr, int temp)
{
  emit_andimm(sr,~1,sr);
  emit_cmp(s2,s1);
  emit_orrge_imm(sr,1,sr);
}
void emit_cmpgt(int s1, int s2, int sr, int temp)
{
  emit_andimm(sr,~1,sr);
  emit_cmp(s2,s1);
  emit_orrgt_imm(sr,1,sr);
}
void emit_cmphi(int s1, int s2, int sr, int temp)
{
  emit_andimm(sr,~1,sr);
  emit_cmp(s2,s1);
  emit_orrhi_imm(sr,1,sr);
}
void emit_cmphs(int s1, int s2, int sr, int temp)
{
  emit_andimm(sr,~1,sr);
  emit_cmp(s2,s1);
  emit_adcimm(sr,0,sr);
}
void emit_dt(int t, int sr)
{
  emit_andimm(sr,~1,sr);
  emit_addimm_and_set_flags(-1,t);
  emit_orreq_imm(sr,1,sr);
}
void emit_cmppz(int s, int sr)
{
  emit_andimm(sr,~1,sr);
  emit_testimm(s,0x80000000);
  emit_orreq_imm(sr,1,sr);
}
void emit_cmppl(int s, int sr, int temp)
{
  assert(temp>=0);
  emit_andimm(sr,~1,sr);
  emit_cmpimm(s,0);
  emit_orrgt_imm(sr,1,sr);
}
void emit_addc(int s, int t, int sr)
{
  emit_lsrs_imm(sr,1,sr);
  emit_adcs(s,t,t);
  emit_adc(sr,sr,sr);
}
void emit_subc(int s, int t, int sr)
{
  emit_xorimm(sr,1,sr);
  emit_lsrs_imm(sr,1,sr);
  emit_sbcs(t,s,t);
  emit_adc(sr,sr,sr);
  emit_xorimm(sr,1,sr);
}
void emit_shrsr(int t, int sr)
{
  emit_andimm(sr,~1,sr);
  emit_lsrs_imm(t,1,t);
  emit_adcimm(sr,0,sr);
}
void emit_sarsr(int t, int sr)
{
  emit_andimm(sr,~1,sr);
  emit_asrs_imm(t,1,t);
  emit_adcimm(sr,0,sr);
}
void emit_shlsr(int t, int sr)
{
  emit_andimm(sr,~1,sr);
  emit_lsls_imm(t,1,t);
  emit_adcimm(sr,0,sr);
}
void emit_rotl(int t)
{
  emit_rorimm(t,31,t);
}
void emit_rotlsr(int t, int sr)
{
  emit_andimm(sr,~1,sr);
  emit_test(t,t);
  emit_rotl(t);
  emit_orrmi_imm(sr,1,sr);
}
void emit_rotr(int t)
{
  emit_rorimm(t,1,t);
}
void emit_rotrsr(int t, int sr)
{
  emit_andimm(sr,~1,sr);
  emit_rors_imm(t,1,t);
  emit_adcimm(sr,0,sr);
}
void emit_rotclsr(int t, int sr)
{
  emit_lsrs_imm(sr,1,sr);
  emit_adcs(t,t,t);
  emit_adc(sr,sr,sr);
}
void emit_rotcrsr(int t, int sr)
{
  emit_lsrs_imm(sr,1,sr);
  emit_rrxs(t,t);
  emit_adc(sr,sr,sr);
}

void emit_call(int a)
{
  assem_debug("bl %x (%x+%x)\n",a,(int)out,a-(int)out-8);
  u32 offset=genjmp(a);
  output_w32(0xeb000000|offset);
}
void emit_jmp(int a)
{
  assem_debug("b %x (%x+%x)\n",a,(int)out,a-(int)out-8);
  u32 offset=genjmp(a);
  output_w32(0xea000000|offset);
}
void emit_jne(int a)
{
  assem_debug("bne %x\n",a);
  u32 offset=genjmp(a);
  output_w32(0x1a000000|offset);
}
void emit_jeq(int a)
{
  assem_debug("beq %x\n",a);
  u32 offset=genjmp(a);
  output_w32(0x0a000000|offset);
}
void emit_js(int a)
{
  assem_debug("bmi %x\n",a);
  u32 offset=genjmp(a);
  output_w32(0x4a000000|offset);
}
void emit_jns(int a)
{
  assem_debug("bpl %x\n",a);
  u32 offset=genjmp(a);
  output_w32(0x5a000000|offset);
}
void emit_jl(int a)
{
  assem_debug("blt %x\n",a);
  u32 offset=genjmp(a);
  output_w32(0xba000000|offset);
}
void emit_jge(int a)
{
  assem_debug("bge %x\n",a);
  u32 offset=genjmp(a);
  output_w32(0xaa000000|offset);
}
void emit_jno(int a)
{
  assem_debug("bvc %x\n",a);
  u32 offset=genjmp(a);
  output_w32(0x7a000000|offset);
}
void emit_jc(int a)
{
  assem_debug("bcs %x\n",a);
  u32 offset=genjmp(a);
  output_w32(0x2a000000|offset);
}
void emit_jcc(int a)
{
  assem_debug("bcc %x\n",a);
  u32 offset=genjmp(a);
  output_w32(0x3a000000|offset);
}

void emit_pushimm(int imm)
{
  assem_debug("push $%x\n",imm);
  assert(0);
}
void emit_pusha()
{
  assem_debug("pusha\n");
  assert(0);
}
void emit_popa()
{
  assem_debug("popa\n");
  assert(0);
}
void emit_pushreg(unsigned int r)
{
  assem_debug("push %%%s\n",regname[r]);
  assert(0);
}
void emit_popreg(unsigned int r)
{
  assem_debug("pop %%%s\n",regname[r]);
  assert(0);
}
void emit_callreg(unsigned int r)
{
  assem_debug("call *%%%s\n",regname[r]);
  assert(0);
}
void emit_jmpreg(unsigned int r)
{
  assem_debug("mov pc,%s\n",regname[r]);
  output_w32(0xe1a00000|rd_rn_rm(15,0,r));
}

void emit_cmpstr(int s1, int s2, int sr, int temp)
{
  // Compare s1 and s2.  If any byte is equal, set T.
  // Calculates the xor of the strings, then checks if any byte is zero
  emit_xor(s1,s2,14);
  emit_andimm(sr,~1,sr);
  emit_testimm(14,0xFF);
  emit_orreq_imm(sr,1,sr);
  emit_testimm(14,0xFF00);
  emit_orreq_imm(sr,1,sr);
  emit_testimm(14,0xFF0000);
  emit_orreq_imm(sr,1,sr);
  emit_testimm(14,0xFF000000);
  emit_orreq_imm(sr,1,sr);
  // Compare s1 and s2.  If any byte is equal, set T.
  // Calculates the xor of the strings, then checks if any byte is
  // zero by subtracting 1 from each byte.  If there is a carry/borrow
  // then a byte was zero.
  //assert(temp>=0);
  //emit_xor(s1,s2,s2);
  //emit_movimm(0x01010101,14);
  //emit_andimm(sr,~1,sr);
  //emit_subs(s2,14,temp);
  //emit_sbcimm(temp,0,temp);
  //emit_xor(temp,s2,temp);
  //emit_not(temp,temp);
  //emit_testimm(temp,14);
  //emit_xor(s1,s2,s2);
  //emit_orrne_imm(sr,1,sr);
}
void emit_negc(int rs, int rt, int sr)
{
assert(0);
/*
  assert(rs>=0&&rs<8);
  if(rt<0) {
    emit_shrimm(sr,1,sr); // Get C flag
    emit_jc((int)out+10); // 6
    emit_neg(rs,rs); // 2
    emit_neg(rs,rs); // 2
    emit_adc(sr,sr); // Save C flag
  }else{
    if(rs!=rt) emit_mov(rs,rt);
    emit_shrimm(sr,1,sr); // Get C flag
    emit_jc((int)out+9); // 6
    emit_addimm(rt,-1,rt); // 3
    emit_adc(sr,sr); // Save C flag
    emit_not(rt,rt);
  }
*/
}

void emit_readword_indexed(int offset, int rs, int rt)
{
  assert(offset>-4096&&offset<4096);
  assem_debug("ldr %s,%s+%d\n",regname[rt],regname[rs],offset);
  if(offset>=0) {
    output_w32(0xe5900000|rd_rn_rm(rt,rs,0)|offset);
  }else{
    output_w32(0xe5100000|rd_rn_rm(rt,rs,0)|(-offset));
  }
}
void emit_readword_dualindexedx4(int rs1, int rs2, int rt)
{
  assem_debug("ldr %s,%s,%s lsl #2\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe7900000|rd_rn_rm(rt,rs1,rs2)|0x100);
}
void emit_readword_indexed_map(int addr, int rs, int map, int rt)
{
  if(map<0) emit_readword_indexed(addr, rs, rt);
  else {
    assert(addr==0);
    emit_readword_dualindexedx4(rs, map, rt);
  }
}
void emit_movsbl_indexed(int offset, int rs, int rt)
{
  assert(offset>-256&&offset<256);
  assem_debug("ldrsb %s,%s+%d\n",regname[rt],regname[rs],offset);
  if(offset>=0) {
    output_w32(0xe1d000d0|rd_rn_rm(rt,rs,0)|((offset<<4)&0xf00)|(offset&0xf));
  }else{
    output_w32(0xe15000d0|rd_rn_rm(rt,rs,0)|(((-offset)<<4)&0xf00)|((-offset)&0xf));
  }
}
void emit_movsbl_indexed_map(int addr, int rs, int map, int rt)
{
  if(map<0) emit_movsbl_indexed(addr, rs, rt);
  else {
    if(addr==0) {
      emit_shlimm(map,2,HOST_TEMPREG);
      assem_debug("ldrsb %s,%s+%s\n",regname[rt],regname[rs],regname[HOST_TEMPREG]);
      output_w32(0xe19000d0|rd_rn_rm(rt,rs,HOST_TEMPREG));
    }else{
      assert(addr>-256&&addr<256);
      assem_debug("add %s,%s,%s,lsl #2\n",regname[rt],regname[rs],regname[map]);
      output_w32(0xe0800000|rd_rn_rm(rt,rs,map)|(2<<7));
      emit_movsbl_indexed(addr, rt, rt);
    }
  }
}
void emit_movswl_indexed(int offset, int rs, int rt)
{
  assert(offset>-256&&offset<256);
  assem_debug("ldrsh %s,%s+%d\n",regname[rt],regname[rs],offset);
  if(offset>=0) {
    output_w32(0xe1d000f0|rd_rn_rm(rt,rs,0)|((offset<<4)&0xf00)|(offset&0xf));
  }else{
    output_w32(0xe15000f0|rd_rn_rm(rt,rs,0)|(((-offset)<<4)&0xf00)|((-offset)&0xf));
  }
}
void emit_movswl_indexed_map(int addr, int rs, int map, int rt)
{
  if(map<0) emit_movswl_indexed(addr, rs, rt);
  else {
    if(addr==0) {
      emit_shlimm(map,2,HOST_TEMPREG);
      assem_debug("ldrsh %s,%s+%s\n",regname[rt],regname[rs],regname[HOST_TEMPREG]);
      output_w32(0xe19000f0|rd_rn_rm(rt,rs,HOST_TEMPREG));
    }else{
      assert(addr>-256&&addr<256);
      assem_debug("add %s,%s,%s,lsl #2\n",regname[rt],regname[rs],regname[map]);
      output_w32(0xe0800000|rd_rn_rm(rt,rs,map)|(2<<7));
      emit_movswl_indexed(addr, rt, rt);
    }
  }
}
void emit_movzbl_indexed(int offset, int rs, int rt)
{
  assert(offset>-4096&&offset<4096);
  assem_debug("ldrb %s,%s+%d\n",regname[rt],regname[rs],offset);
  if(offset>=0) {
    output_w32(0xe5d00000|rd_rn_rm(rt,rs,0)|offset);
  }else{
    output_w32(0xe5500000|rd_rn_rm(rt,rs,0)|(-offset));
  }
}
void emit_movzbl_dualindexedx4(int rs1, int rs2, int rt)
{
  assem_debug("ldrb %s,%s,%s lsl #2\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe7d00000|rd_rn_rm(rt,rs1,rs2)|0x100);
}
void emit_movzbl_indexed_map(int addr, int rs, int map, int rt)
{
  if(map<0) emit_movzbl_indexed(addr, rs, rt);
  else {
    if(addr==0) {
      emit_movzbl_dualindexedx4(rs, map, rt);
    }else{
      emit_addimm(rs,addr,rt);
      emit_movzbl_dualindexedx4(rt, map, rt);
    }
  }
}
void emit_movzwl_indexed(int offset, int rs, int rt)
{
  assert(offset>-256&&offset<256);
  assem_debug("ldrh %s,%s+%d\n",regname[rt],regname[rs],offset);
  if(offset>=0) {
    output_w32(0xe1d000b0|rd_rn_rm(rt,rs,0)|((offset<<4)&0xf00)|(offset&0xf));
  }else{
    output_w32(0xe15000b0|rd_rn_rm(rt,rs,0)|(((-offset)<<4)&0xf00)|((-offset)&0xf));
  }
}
void emit_readword(int addr, int rt)
{
  u32 offset = addr-(u32)&dynarec_local;
  assert(offset<4096);
  assem_debug("ldr %s,fp+%d\n",regname[rt],offset);
  output_w32(0xe5900000|rd_rn_rm(rt,FP,0)|offset);
}
void emit_movsbl(int addr, int rt)
{
  u32 offset = addr-(u32)&dynarec_local;
  assert(offset<256);
  assem_debug("ldrsb %s,fp+%d\n",regname[rt],offset);
  output_w32(0xe1d000d0|rd_rn_rm(rt,FP,0)|((offset<<4)&0xf00)|(offset&0xf));
}
void emit_movswl(int addr, int rt)
{
  u32 offset = addr-(u32)&dynarec_local;
  assert(offset<256);
  assem_debug("ldrsh %s,fp+%d\n",regname[rt],offset);
  output_w32(0xe1d000f0|rd_rn_rm(rt,FP,0)|((offset<<4)&0xf00)|(offset&0xf));
}
void emit_movzbl(int addr, int rt)
{
  u32 offset = addr-(u32)&dynarec_local;
  assert(offset<4096);
  assem_debug("ldrb %s,fp+%d\n",regname[rt],offset);
  output_w32(0xe5d00000|rd_rn_rm(rt,FP,0)|offset);
}
void emit_movzwl(int addr, int rt)
{
  u32 offset = addr-(u32)&dynarec_local;
  assert(offset<256);
  assem_debug("ldrh %s,fp+%d\n",regname[rt],offset);
  output_w32(0xe1d000b0|rd_rn_rm(rt,FP,0)|((offset<<4)&0xf00)|(offset&0xf));
}

void emit_xchg(int rs, int rt)
{
  assem_debug("xchg %%%s,%%%s\n",regname[rs],regname[rt]);
  assert(0);
}
void emit_writeword_indexed(int rt, int offset, int rs)
{
  assert(offset>-4096&&offset<4096);
  assem_debug("str %s,%s+%d\n",regname[rt],regname[rs],offset);
  if(offset>=0) {
    output_w32(0xe5800000|rd_rn_rm(rt,rs,0)|offset);
  }else{
    output_w32(0xe5000000|rd_rn_rm(rt,rs,0)|(-offset));
  }
}
void emit_writeword_dualindexedx4(int rt, int rs1, int rs2)
{
  assem_debug("str %s,%s,%s lsl #2\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe7800000|rd_rn_rm(rt,rs1,rs2)|0x100);
}
void emit_writeword_indexed_map(int rt, int addr, int rs, int map, int temp)
{
  if(map<0) emit_writeword_indexed(rt, addr, rs);
  else {
    assert(addr==0);
    emit_writeword_dualindexedx4(rt, rs, map);
  }
}
void emit_writehword_indexed(int rt, int offset, int rs)
{
  assert(offset>-256&&offset<256);
  assem_debug("strh %s,%s+%d\n",regname[rt],regname[rs],offset);
  if(offset>=0) {
    output_w32(0xe1c000b0|rd_rn_rm(rt,rs,0)|((offset<<4)&0xf00)|(offset&0xf));
  }else{
    output_w32(0xe14000b0|rd_rn_rm(rt,rs,0)|(((-offset)<<4)&0xf00)|((-offset)&0xf));
  }
}
void emit_writebyte_indexed(int rt, int offset, int rs)
{
  assert(offset>-4096&&offset<4096);
  assem_debug("strb %s,%s+%d\n",regname[rt],regname[rs],offset);
  if(offset>=0) {
    output_w32(0xe5c00000|rd_rn_rm(rt,rs,0)|offset);
  }else{
    output_w32(0xe5400000|rd_rn_rm(rt,rs,0)|(-offset));
  }
}
void emit_writebyte_dualindexedx4(int rt, int rs1, int rs2)
{
  assem_debug("strb %s,%s,%s lsl #2\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe7c00000|rd_rn_rm(rt,rs1,rs2)|0x100);
}
void emit_writebyte_indexed_map(int rt, int addr, int rs, int map, int temp)
{
  if(map<0) emit_writebyte_indexed(rt, addr, rs);
  else {
    if(addr==0) {
      emit_writebyte_dualindexedx4(rt, rs, map);
    }else{
      emit_addimm(rs,addr,temp);
      emit_writebyte_dualindexedx4(rt, temp, map);
    }
  }
}
void emit_writehword_indexed_map(int rt, int addr, int rs, int map, int temp)
{
  if(map<0) emit_writehword_indexed(rt, addr, rs);
  else {
    if(addr==0) {
      emit_shlimm(map,2,HOST_TEMPREG);
      assem_debug("strh %s,%s+%s\n",regname[rt],regname[rs],regname[HOST_TEMPREG]);
      output_w32(0xe18000b0|rd_rn_rm(rt,rs,HOST_TEMPREG));
    }else{
      assert(addr==0);
    }
  }
}
void emit_writeword(int rt, int addr)
{
  u32 offset = addr-(u32)&dynarec_local;
  assert(offset<4096);
  assem_debug("str %s,fp+%d\n",regname[rt],offset);
  output_w32(0xe5800000|rd_rn_rm(rt,FP,0)|offset);
}
void emit_writehword(int rt, int addr)
{
  u32 offset = addr-(u32)&dynarec_local;
  assert(offset<256);
  assem_debug("strh %s,fp+%d\n",regname[rt],offset);
  output_w32(0xe1c000b0|rd_rn_rm(rt,FP,0)|((offset<<4)&0xf00)|(offset&0xf));
}
void emit_writebyte(int rt, int addr)
{
  u32 offset = addr-(u32)&dynarec_local;
  assert(offset<4096);
  assem_debug("strb %s,fp+%d\n",regname[rt],offset);
  output_w32(0xe5c00000|rd_rn_rm(rt,FP,0)|offset);
}
void emit_writeword_imm(int imm, int addr)
{
  assem_debug("movl $%x,%x\n",imm,addr);
  assert(0);
}
void emit_writebyte_imm(int imm, int addr)
{
  assem_debug("movb $%x,%x\n",imm,addr);
  assert(0);
}

void emit_rmw_andimm(int addr, int map, int imm)
{
  if(map<0) {
    assert(map>=0);
  }
  else
  {
    emit_movzbl_dualindexedx4(addr, map, 14);
    emit_andimm(14,imm,14);
    emit_writebyte_dualindexedx4(14, addr, map);
  }
}
void emit_rmw_xorimm(int addr, int map, int imm)
{
  if(map<0) {
    assert(map>=0);
  }
  else
  {
    emit_movzbl_dualindexedx4(addr, map, 14);
    emit_xorimm(14,imm,14);
    emit_writebyte_dualindexedx4(14, addr, map);
  }
}
void emit_rmw_orimm(int addr, int map, int imm)
{
  if(map<0) {
    assert(map>=0);
  }
  else
  {
    emit_movzbl_dualindexedx4(addr, map, 14);
    emit_orimm(14,imm,14);
    emit_writebyte_dualindexedx4(14, addr, map);
  }
}
void emit_sh2tas(int addr, int map, int sr)
{
  if(map<0) {
    assert(map>=0);
  }
  else
  {
    emit_movzbl_dualindexedx4(addr, map, 14);
    emit_andimm(sr,~1,sr);
    emit_testimm(14,0xff);
    emit_orimm(14,0x80,14);
    emit_orreq_imm(sr,1,sr);
    emit_writebyte_dualindexedx4(14, addr, map);
  }
}

void emit_mul(int rs)
{
  assem_debug("mul %%%s\n",regname[rs]);
  assert(0);
}
void emit_imul(int rs)
{
  assem_debug("imul %%%s\n",regname[rs]);
  assert(0);
}
void emit_multiply(unsigned int rs1,unsigned int rs2,unsigned int rt)
{
  if(rs1==rt&&rs2==rt) {emit_mov(rs1,14);emit_multiply(14,14,rt);}
  else if(rs1==rt) {emit_multiply(rs2,rs1,rt);}
  else {
    assem_debug("mul %s, %s, %s\n",regname[rt],regname[rs1],regname[rs2]);
    assert(rs1<16);
    assert(rs2<16);
    assert(rt<16);
    output_w32(0xe0000090|(rt<<16)|(rs2<<8)|rs1);
  }
}
void emit_umull(unsigned int rs1, unsigned int rs2, unsigned int hi, unsigned int lo)
{
  assem_debug("umull %s, %s, %s, %s\n",regname[lo],regname[hi],regname[rs1],regname[rs2]);
  assert(rs1<16);
  assert(rs2<16);
  assert(hi<16);
  assert(lo<16);
  assert(hi!=rs1&&hi!=rs2);
  assert(lo!=rs1&&lo!=rs2);
  output_w32(0xe0800090|(hi<<16)|(lo<<12)|(rs2<<8)|rs1);
}
void emit_smull(unsigned int rs1, unsigned int rs2, unsigned int hi, unsigned int lo)
{
  assem_debug("smull %s, %s, %s, %s\n",regname[lo],regname[hi],regname[rs1],regname[rs2]);
  assert(rs1<16);
  assert(rs2<16);
  assert(hi<16);
  assert(lo<16);
  assert(hi!=rs1&&hi!=rs2);
  assert(lo!=rs1&&lo!=rs2);
  output_w32(0xe0c00090|(hi<<16)|(lo<<12)|(rs2<<8)|rs1);
}

void emit_div(int rs)
{
  assem_debug("div %%%s\n",regname[rs]);
  assert(0);
}
void emit_idiv(int rs)
{
  assem_debug("idiv %%%s\n",regname[rs]);
  assert(0);
}
void emit_cdq()
{
  assem_debug("cdq\n");
  assert(0);
}

void emit_teq(int rs, int rt)
{
  assem_debug("teq %s,%s\n",regname[rs],regname[rt]);
  output_w32(0xe1300000|rd_rn_rm(0,rs,rt));
}

void emit_div0s(int s1, int s2, int sr, int temp) {
  emit_andimm(sr,0xfe,sr);
  emit_testimm(s2,0x80000000);
  emit_orrne_imm(sr,0x100,sr);
  emit_testimm(s1,0x80000000);
  emit_orrne_imm(sr,0x200,sr);
  emit_teq(s1,s2);
  emit_orrmi_imm(sr,1,sr);
}

// Load return address
void emit_load_return_address(unsigned int rt)
{
  // (assumes this instruction will be followed by a branch instruction)
  emit_mov(15,rt);
  //assem_debug("add %s,pc,#0\n",regname[rt]);
  //output_w32(0xe2800000|rd_rn_rm(rt,15,0));
}

void emit_clz(int rs,int rt)
{
  assem_debug("clz %s,%s\n",regname[rt],regname[rs]);
  output_w32(0xe16f0f10|rd_rn_rm(rt,0,rs));
}

void emit_subcs(int rs1,int rs2,int rt)
{
  assem_debug("subcs %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0x20400000|rd_rn_rm(rt,rs1,rs2));
}

void emit_shrcc_imm(int rs,unsigned int imm,int rt)
{
  assert(imm>0);
  assert(imm<32);
  assem_debug("lsrcc %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0x31a00000|rd_rn_rm(rt,0,rs)|0x20|(imm<<7));
}

void emit_negmi(int rs, int rt)
{
  assem_debug("rsbmi %s,%s,#0\n",regname[rt],regname[rs]);
  output_w32(0x42600000|rd_rn_rm(rt,rs,0));
}

void emit_negsmi(int rs, int rt)
{
  assem_debug("rsbsmi %s,%s,#0\n",regname[rt],regname[rs]);
  output_w32(0x42700000|rd_rn_rm(rt,rs,0));
}

void emit_orreq(unsigned int rs1,unsigned int rs2,unsigned int rt)
{
  assem_debug("orreq %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0x01800000|rd_rn_rm(rt,rs1,rs2));
}

void emit_orrne(unsigned int rs1,unsigned int rs2,unsigned int rt)
{
  assem_debug("orrne %s,%s,%s\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0x11800000|rd_rn_rm(rt,rs1,rs2));
}

void emit_bic_lsl(unsigned int rs1,unsigned int rs2,unsigned int shift,unsigned int rt)
{
  assem_debug("bic %s,%s,%s lsl %s\n",regname[rt],regname[rs1],regname[rs2],regname[shift]);
  output_w32(0xe1C00000|rd_rn_rm(rt,rs1,rs2)|0x10|(shift<<8));
}

void emit_biceq_lsl(unsigned int rs1,unsigned int rs2,unsigned int shift,unsigned int rt)
{
  assem_debug("biceq %s,%s,%s lsl %s\n",regname[rt],regname[rs1],regname[rs2],regname[shift]);
  output_w32(0x01C00000|rd_rn_rm(rt,rs1,rs2)|0x10|(shift<<8));
}

void emit_bicne_lsl(unsigned int rs1,unsigned int rs2,unsigned int shift,unsigned int rt)
{
  assem_debug("bicne %s,%s,%s lsl %s\n",regname[rt],regname[rs1],regname[rs2],regname[shift]);
  output_w32(0x11C00000|rd_rn_rm(rt,rs1,rs2)|0x10|(shift<<8));
}

void emit_bic_lsr(unsigned int rs1,unsigned int rs2,unsigned int shift,unsigned int rt)
{
  assem_debug("bic %s,%s,%s lsr %s\n",regname[rt],regname[rs1],regname[rs2],regname[shift]);
  output_w32(0xe1C00000|rd_rn_rm(rt,rs1,rs2)|0x30|(shift<<8));
}

void emit_biceq_lsr(unsigned int rs1,unsigned int rs2,unsigned int shift,unsigned int rt)
{
  assem_debug("biceq %s,%s,%s lsr %s\n",regname[rt],regname[rs1],regname[rs2],regname[shift]);
  output_w32(0x01C00000|rd_rn_rm(rt,rs1,rs2)|0x30|(shift<<8));
}

void emit_bicne_lsr(unsigned int rs1,unsigned int rs2,unsigned int shift,unsigned int rt)
{
  assem_debug("bicne %s,%s,%s lsr %s\n",regname[rt],regname[rs1],regname[rs2],regname[shift]);
  output_w32(0x11C00000|rd_rn_rm(rt,rs1,rs2)|0x30|(shift<<8));
}

void emit_rsbimm(int rs, int imm, int rt)
{
  u32 armval;
  assert(genimm(imm,&armval));
  assem_debug("rsb %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0xe2600000|rd_rn_rm(rt,rs,0)|armval);
}

// Load 2 immediates optimizing for small code size
void emit_mov2imm_compact(int imm1,unsigned int rt1,int imm2,unsigned int rt2)
{
  emit_movimm(imm1,rt1);
  u32 armval;
  if(genimm(imm2-imm1,&armval)) {
    assem_debug("add %s,%s,#%d\n",regname[rt2],regname[rt1],imm2-imm1);
    output_w32(0xe2800000|rd_rn_rm(rt2,rt1,0)|armval);
  }else if(genimm(imm1-imm2,&armval)) {
    assem_debug("sub %s,%s,#%d\n",regname[rt2],regname[rt1],imm1-imm2);
    output_w32(0xe2400000|rd_rn_rm(rt2,rt1,0)|armval);
  }
  else emit_movimm(imm2,rt2);
}

// Conditionally select one of two immediates, optimizing for small code size
// This will only be called if HAVE_CMOV_IMM is defined
void emit_cmov2imm_e_ne_compact(int imm1,int imm2,unsigned int rt)
{
  u32 armval;
  if(genimm(imm2-imm1,&armval)) {
    emit_movimm(imm1,rt);
    assem_debug("addne %s,%s,#%d\n",regname[rt],regname[rt],imm2-imm1);
    output_w32(0x12800000|rd_rn_rm(rt,rt,0)|armval);
  }else if(genimm(imm1-imm2,&armval)) {
    emit_movimm(imm1,rt);
    assem_debug("subne %s,%s,#%d\n",regname[rt],regname[rt],imm1-imm2);
    output_w32(0x12400000|rd_rn_rm(rt,rt,0)|armval);
  }
  else {
    #ifdef ARMv5_ONLY
    emit_movimm(imm1,rt);
    add_literal((int)out,imm2);
    assem_debug("ldrne %s,pc+? [=%x]\n",regname[rt],imm2);
    output_w32(0x15900000|rd_rn_rm(rt,15,0));
    #else
    emit_movw(imm1&0x0000FFFF,rt);
    if((imm1&0xFFFF)!=(imm2&0xFFFF)) {
      assem_debug("movwne %s,#%d (0x%x)\n",regname[rt],imm2&0xFFFF,imm2&0xFFFF);
      output_w32(0x13000000|rd_rn_rm(rt,0,0)|(imm2&0xfff)|((imm2<<4)&0xf0000));
    }
    emit_movt(imm1&0xFFFF0000,rt);
    if((imm1&0xFFFF0000)!=(imm2&0xFFFF0000)) {
      assem_debug("movtne %s,#%d (0x%x)\n",regname[rt],imm2&0xffff0000,imm2&0xffff0000);
      output_w32(0x13400000|rd_rn_rm(rt,0,0)|((imm2>>16)&0xfff)|((imm2>>12)&0xf0000));
    }
    #endif
  }
}

// special case for checking invalid_code
void emit_cmpmem_indexedsr12_imm(int addr,int r,int imm)
{
  assert(0);
}

// special case for checking invalid_code
void emit_cmpmem_indexedsr12_reg(int base,int r,int imm)
{
  assert(imm<128&&imm>=0);
  assert(r>=0&&r<16);
  assem_debug("ldrb lr,%s,%s lsr #12\n",regname[base],regname[r]);
  output_w32(0xe7d00000|rd_rn_rm(HOST_TEMPREG,base,r)|0x620);
  emit_cmpimm(HOST_TEMPREG,imm);
}

// special case for memory map
void emit_addsr12(int rs1,int rs2,int rt)
{
  assem_debug("add %s,%s,%s lsr #12\n",regname[rt],regname[rs1],regname[rs2]);
  output_w32(0xe0800620|rd_rn_rm(rt,rs1,rs2));
}

void emit_callne(int a)
{
  assem_debug("blne %x\n",a);
  u32 offset=genjmp(a);
  output_w32(0x1b000000|offset);
}

// Used to preload hash table entries
void emit_prefetch(void *addr)
{
  assem_debug("prefetch %x\n",(int)addr);
  output_byte(0x0F);
  output_byte(0x18);
  output_modrm(0,5,1);
  output_w32((int)addr);
}
void emit_prefetchreg(int r)
{
  assem_debug("pld %s\n",regname[r]);
  output_w32(0xf5d0f000|rd_rn_rm(0,r,0));
}

// Special case for mini_ht
void emit_ldreq_indexed(int rs, u32 offset, int rt)
{
  assert(offset<4096);
  assem_debug("ldreq %s,[%s, #%d]\n",regname[rt],regname[rs],offset);
  output_w32(0x05900000|rd_rn_rm(rt,rs,0)|offset);
}

void emit_flds(int r,int sr)
{
  assem_debug("flds s%d,[%s]\n",sr,regname[r]);
  output_w32(0xed900a00|((sr&14)<<11)|((sr&1)<<22)|(r<<16));
} 

void emit_vldr(int r,int vr)
{
  assem_debug("vldr d%d,[%s]\n",vr,regname[r]);
  output_w32(0xed900b00|(vr<<12)|(r<<16));
} 

void emit_fsts(int sr,int r)
{
  assem_debug("fsts s%d,[%s]\n",sr,regname[r]);
  output_w32(0xed800a00|((sr&14)<<11)|((sr&1)<<22)|(r<<16));
} 

void emit_vstr(int vr,int r)
{
  assem_debug("vstr d%d,[%s]\n",vr,regname[r]);
  output_w32(0xed800b00|(vr<<12)|(r<<16));
} 

void emit_ftosizs(int s,int d)
{
  assem_debug("ftosizs s%d,s%d\n",d,s);
  output_w32(0xeebd0ac0|((d&14)<<11)|((d&1)<<22)|((s&14)>>1)|((s&1)<<5));
} 

void emit_ftosizd(int s,int d)
{
  assem_debug("ftosizd s%d,d%d\n",d,s);
  output_w32(0xeebd0bc0|((d&14)<<11)|((d&1)<<22)|(s&7));
} 

void emit_fsitos(int s,int d)
{
  assem_debug("fsitos s%d,s%d\n",d,s);
  output_w32(0xeeb80ac0|((d&14)<<11)|((d&1)<<22)|((s&14)>>1)|((s&1)<<5));
} 

void emit_fsitod(int s,int d)
{
  assem_debug("fsitod d%d,s%d\n",d,s);
  output_w32(0xeeb80bc0|((d&7)<<12)|((s&14)>>1)|((s&1)<<5));
} 

void emit_fcvtds(int s,int d)
{
  assem_debug("fcvtds d%d,s%d\n",d,s);
  output_w32(0xeeb70ac0|((d&7)<<12)|((s&14)>>1)|((s&1)<<5));
} 

void emit_fcvtsd(int s,int d)
{
  assem_debug("fcvtsd s%d,d%d\n",d,s);
  output_w32(0xeeb70bc0|((d&14)<<11)|((d&1)<<22)|(s&7));
} 

void emit_fsqrts(int s,int d)
{
  assem_debug("fsqrts d%d,s%d\n",d,s);
  output_w32(0xeeb10ac0|((d&14)<<11)|((d&1)<<22)|((s&14)>>1)|((s&1)<<5));
} 

void emit_fsqrtd(int s,int d)
{
  assem_debug("fsqrtd s%d,d%d\n",d,s);
  output_w32(0xeeb10bc0|((d&7)<<12)|(s&7));
} 

void emit_fabss(int s,int d)
{
  assem_debug("fabss d%d,s%d\n",d,s);
  output_w32(0xeeb00ac0|((d&14)<<11)|((d&1)<<22)|((s&14)>>1)|((s&1)<<5));
} 

void emit_fabsd(int s,int d)
{
  assem_debug("fabsd s%d,d%d\n",d,s);
  output_w32(0xeeb00bc0|((d&7)<<12)|(s&7));
} 

void emit_fnegs(int s,int d)
{
  assem_debug("fnegs d%d,s%d\n",d,s);
  output_w32(0xeeb10a40|((d&14)<<11)|((d&1)<<22)|((s&14)>>1)|((s&1)<<5));
} 

void emit_fnegd(int s,int d)
{
  assem_debug("fnegd s%d,d%d\n",d,s);
  output_w32(0xeeb10b40|((d&7)<<12)|(s&7));
} 

void emit_fadds(int s1,int s2,int d)
{
  assem_debug("fadds s%d,s%d,s%d\n",d,s1,s2);
  output_w32(0xee300a00|((d&14)<<11)|((d&1)<<22)|((s1&14)<<15)|((s1&1)<<7)|((s2&14)>>1)|((s2&1)<<5));
} 

void emit_faddd(int s1,int s2,int d)
{
  assem_debug("faddd d%d,d%d,d%d\n",d,s1,s2);
  output_w32(0xee300b00|((d&7)<<12)|((s1&7)<<16)|(s2&7));
} 

void emit_fsubs(int s1,int s2,int d)
{
  assem_debug("fsubs s%d,s%d,s%d\n",d,s1,s2);
  output_w32(0xee300a40|((d&14)<<11)|((d&1)<<22)|((s1&14)<<15)|((s1&1)<<7)|((s2&14)>>1)|((s2&1)<<5));
} 

void emit_fsubd(int s1,int s2,int d)
{
  assem_debug("fsubd d%d,d%d,d%d\n",d,s1,s2);
  output_w32(0xee300b40|((d&7)<<12)|((s1&7)<<16)|(s2&7));
} 

void emit_fmuls(int s1,int s2,int d)
{
  assem_debug("fmuls s%d,s%d,s%d\n",d,s1,s2);
  output_w32(0xee200a00|((d&14)<<11)|((d&1)<<22)|((s1&14)<<15)|((s1&1)<<7)|((s2&14)>>1)|((s2&1)<<5));
} 

void emit_fmuld(int s1,int s2,int d)
{
  assem_debug("fmuld d%d,d%d,d%d\n",d,s1,s2);
  output_w32(0xee200b00|((d&7)<<12)|((s1&7)<<16)|(s2&7));
} 

void emit_fdivs(int s1,int s2,int d)
{
  assem_debug("fdivs s%d,s%d,s%d\n",d,s1,s2);
  output_w32(0xee800a00|((d&14)<<11)|((d&1)<<22)|((s1&14)<<15)|((s1&1)<<7)|((s2&14)>>1)|((s2&1)<<5));
} 

void emit_fdivd(int s1,int s2,int d)
{
  assem_debug("fdivd d%d,d%d,d%d\n",d,s1,s2);
  output_w32(0xee800b00|((d&7)<<12)|((s1&7)<<16)|(s2&7));
} 

void emit_fcmps(int x,int y)
{
  assem_debug("fcmps s14, s15\n");
  output_w32(0xeeb47a67);
} 

void emit_fcmpd(int x,int y)
{
  assem_debug("fcmpd d6, d7\n");
  output_w32(0xeeb46b47);
} 

void emit_fmstat()
{
  assem_debug("fmstat\n");
  output_w32(0xeef1fa10);
} 

void emit_bicne_imm(int rs,int imm,int rt)
{
  u32 armval;
  assert(genimm(imm,&armval));
  assem_debug("bicne %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0x13c00000|rd_rn_rm(rt,rs,0)|armval);
}

void emit_biccs_imm(int rs,int imm,int rt)
{
  u32 armval;
  assert(genimm(imm,&armval));
  assem_debug("biccs %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0x23c00000|rd_rn_rm(rt,rs,0)|armval);
}

void emit_bicvc_imm(int rs,int imm,int rt)
{
  u32 armval;
  assert(genimm(imm,&armval));
  assem_debug("bicvc %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0x73c00000|rd_rn_rm(rt,rs,0)|armval);
}

void emit_bichi_imm(int rs,int imm,int rt)
{
  u32 armval;
  assert(genimm(imm,&armval));
  assem_debug("bichi %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0x83c00000|rd_rn_rm(rt,rs,0)|armval);
}

void emit_orrvs_imm(int rs,int imm,int rt)
{
  u32 armval;
  assert(genimm(imm,&armval));
  assem_debug("orrvs %s,%s,#%d\n",regname[rt],regname[rs],imm);
  output_w32(0x63800000|rd_rn_rm(rt,rs,0)|armval);
}

void emit_jno_unlikely(int a)
{
  //emit_jno(a);
  assem_debug("addvc pc,pc,#? (%x)\n",/*a-(int)out-8,*/a);
  output_w32(0x72800000|rd_rn_rm(15,15,0));
}

// Save registers before function call
void save_regs(u32 reglist)
{
  reglist&=0x100f; // only save the caller-save registers, r0-r3, r12
  if(!reglist) return;
  assem_debug("stmia fp,{");
  if(reglist&1) assem_debug("r0, ");
  if(reglist&2) assem_debug("r1, ");
  if(reglist&4) assem_debug("r2, ");
  if(reglist&8) assem_debug("r3, ");
  if(reglist&0x1000) assem_debug("r12");
  assem_debug("}\n");
  output_w32(0xe88b0000|reglist);
}
// Restore registers after function call
void restore_regs(u32 reglist)
{
  reglist&=0x100f; // only restore the caller-save registers, r0-r3, r12
  if(!reglist) return;
  assem_debug("ldmia fp,{");
  if(reglist&1) assem_debug("r0, ");
  if(reglist&2) assem_debug("r1, ");
  if(reglist&4) assem_debug("r2, ");
  if(reglist&8) assem_debug("r3, ");
  if(reglist&0x1000) assem_debug("r12");
  assem_debug("}\n");
  output_w32(0xe89b0000|reglist);
}

/* Write back consts using r14 so we don't disturb the other registers
void wb_consts(signed char i_regmap[],uint64_t i_is32,u_int i_dirty,int i)
{
  int hr;
  for(hr=0;hr<HOST_REGS;hr++) {
    if(hr!=EXCLUDE_REG&&i_regmap[hr]>=0&&((i_dirty>>hr)&1)) {
      if(((regs[i].isdoingcp>>hr)&1)&&i_regmap[hr]>0) {
        if(i_regmap[hr]<64 || !((i_is32>>(i_regmap[hr]&63))&1) ) {
          int value=cpmap[i][hr];
          if(value==0) {
            emit_zeroreg(HOST_TEMPREG);
          }
          else {
            emit_movimm(value,HOST_TEMPREG);
          }
          emit_storereg(i_regmap[hr],HOST_TEMPREG);
          if((i_is32>>i_regmap[hr])&1) {
            if(value!=-1&&value!=0) emit_sarimm(HOST_TEMPREG,31,HOST_TEMPREG);
            emit_storereg(i_regmap[hr]|64,HOST_TEMPREG);
          }
        }
      }
    }
  }
}*/

/* Stubs/epilogue */

void literal_pool(int n)
{
  if(!literalcount) return;
  if(n) {
    if((int)out-literals[0][0]<4096-n) return;
  }
  u32 *ptr;
  int i;
  for(i=0;i<literalcount;i++)
  {
    ptr=(u32 *)literals[i][0];
    u32 offset=(u32)out-(u32)ptr-8;
    assert(offset<4096);
    assert(!(offset&3));
    *ptr|=offset;
    output_w32(literals[i][1]);
  }
  literalcount=0;
}

void literal_pool_jumpover(int n)
{
  if(!literalcount) return;
  if(n) {
    if((int)out-literals[0][0]<4096-n) return;
  }
  int jaddr=(int)out;
  emit_jmp(0);
  literal_pool(0);
  set_jump_target(jaddr,(int)out);
}

emit_extjump(pointer addr, int target)
{
  u8 *ptr=(u8 *)addr;
  assert((ptr[3]&0x0e)==0xa);
  emit_loadlp(target,0);
  emit_loadlp(addr,1);
  //assert(addr>=0x7000000&&addr<0x7FFFFFF);
//DEBUG >
#ifdef DEBUG_CYCLE_COUNT
  emit_readword((int)&last_count,ECX);
  emit_add(HOST_CCREG,ECX,HOST_CCREG);
  emit_readword((int)&next_interupt,ECX);
  emit_writeword(HOST_CCREG,(int)&Count);
  emit_sub(HOST_CCREG,ECX,HOST_CCREG);
  emit_writeword(ECX,(int)&last_count);
#endif
//DEBUG <
  emit_jmp((pointer)dyna_linker);
}

do_readstub(int n)
{
  assem_debug("do_readstub %x\n",start+stubs[n][3]*2);
  literal_pool(256);
  set_jump_target(stubs[n][1],(int)out);
  int type=stubs[n][0];
  int i=stubs[n][3];
  int rs=stubs[n][4];
  struct regstat *i_regs=(struct regstat *)stubs[n][5];
  u32 reglist=stubs[n][7];
  signed char *i_regmap=i_regs->regmap;
  int addr=get_reg(i_regmap,AGEN1+(i&1));
  int rt;
  int ds;

  rt=get_reg(i_regmap,rt1[i]==TBIT?-1:rt1[i]);
  assert(rs>=0);
  if(addr<0) addr=rt;
  if(addr<0) addr=get_reg(i_regmap,-1);
  assert(addr>=0);
  save_regs(reglist);
  if(type==LOADB_STUB) emit_xorimm(rs,1,0);
  else {if(rs!=0) emit_mov(rs,0);}

  //ds=i_regs!=&regs[i];
  //int real_rs=(itype[i]==LOADLR)?-1:get_reg(i_regmap,rs1[i]);
  //u32 cmask=ds?-1:(0x100f|~i_regs->wasdoingcp);
  //if(!ds) load_all_consts(regs[i].regmap_entry,regs[i].was32,regs[i].wasdirty&~(1<<addr)&(real_rs<0?-1:~(1<<real_rs))&0x100f,i);
  //wb_dirtys(i_regs->regmap_entry,i_regs->was32,i_regs->wasdirty&cmask&~(1<<addr)&(real_rs<0?-1:~(1<<real_rs)));
  //if(!ds) wb_consts(regs[i].regmap_entry,regs[i].was32,regs[i].wasdirty&~(1<<addr)&(real_rs<0?-1:~(1<<real_rs))&~0x100f,i);
  int cc=get_reg(i_regmap,CCREG);
  if(cc<0) {
    emit_loadreg(CCREG,2);
  }
  if(type==LOADB_STUB)
    emit_call((int)MappedMemoryReadByte);
  if(type==LOADW_STUB)
    emit_call((int)MappedMemoryReadWord);
  if(type==LOADL_STUB)
    emit_call((int)MappedMemoryReadLong);
  if(type==LOADS_STUB)
  {
    // RTE instruction, pop PC and SR from stack
    int pc=get_reg(i_regmap,RTEMP);
    assert(pc>=0);
    if(rs<4||rs==12)
      emit_writeword(rs,(int)&dynarec_local+24);
    emit_call((int)MappedMemoryReadLong);
    if(rs==1||rs==2||rs==3||rs==12)
      emit_readword((int)&dynarec_local+24,rs);
    if(pc==0) {
      emit_writeword(0,(int)&dynarec_local+24);
    }
    else
    {
      if(pc==1||pc==2||pc==3||pc==12)
        emit_writeword(0,(int)&dynarec_local+24);
      else
        emit_mov(0,pc);
      if(rs==0) {
        emit_readword((int)&dynarec_local+24,rs);
        emit_addimm(rs,4,rs);
      }else
        emit_addimm(rs,4,0);
    }
    emit_call((int)MappedMemoryReadLong);
    assert(rt>=0);
    if(rt!=0) emit_mov(0,rt);
    if(pc<4||pc==12)
      emit_readword((int)&dynarec_local+24,pc);
  }
  else if(type==LOADB_STUB)
  {
    if(rt>=0) emit_movsbl_reg(0,rt);
  }
  else if(type==LOADW_STUB)
  {
    if(rt>=0) emit_movswl_reg(0,rt);
  }
  else
  {
    if(rt>0) emit_mov(0,rt);
  }
  restore_regs(reglist);
  if(type==LOADS_STUB) emit_addimm(rs,8,rs);
  emit_jmp(stubs[n][2]); // return address
}

inline_readstub(int type, int i, u32 addr, signed char regmap[], int target, int adj, u32 reglist)
{
  assem_debug("inline_readstub\n");
  //int rs=get_reg(regmap,target);
  int rt=get_reg(regmap,target);
  //if(rs<0) rs=get_reg(regmap,-1);
  if(rt<0) rt=get_reg(regmap,-1);
  //rt=get_reg(i_regmap,rt1[i]==TBIT?-1:rt1[i]);
  assert(rt>=0);
  //if(addr<0) addr=rt;
  //if(addr<0) addr=get_reg(i_regmap,-1);
  //assert(addr>=0);
  save_regs(reglist);
  emit_movimm(addr,0);
  if(type==LOADB_STUB)
    emit_call((int)MappedMemoryReadByte);
  if(type==LOADW_STUB)
    emit_call((int)MappedMemoryReadWord);
  if(type==LOADL_STUB)
    emit_call((int)MappedMemoryReadLong);
  assert(type!=LOADS_STUB);
  if(type==LOADB_STUB)
  {
    if(rt>=0) emit_movsbl_reg(0,rt);
  }
  else if(type==LOADW_STUB)
  {
    if(rt>=0) emit_movswl_reg(0,rt);
  }
  else
  {
    if(rt>0) emit_mov(0,rt);
  }

  restore_regs(reglist);
}

do_writestub(int n)
{
  assem_debug("do_writestub %x\n",start+stubs[n][3]*2);
  literal_pool(256);
  set_jump_target(stubs[n][1],(int)out);
  int type=stubs[n][0];
  int i=stubs[n][3];
  int rs=stubs[n][4];
  struct regstat *i_regs=(struct regstat *)stubs[n][5];
  u32 reglist=stubs[n][7];
  signed char *i_regmap=i_regs->regmap;
  int addr=get_reg(i_regmap,AGEN1+(i&1));
  int rt=get_reg(i_regmap,rs1[i]);
  assert(rs>=0);
  assert(rt>=0);
  if(addr<0) addr=get_reg(i_regmap,-1);
  assert(addr>=0);
  save_regs(reglist);
  // "FASTCALL" api: address in r0, data in r1
  if(rs!=0) {
    if(rt==0) {
      if(rs==1) {
        emit_mov(0,2);
        emit_mov(1,0);
        emit_mov(2,1);
      } else {
        emit_mov(rt,1);
        emit_mov(rs,0);
      }
    }
    else {
      emit_mov(rs,0);
      if(rt!=1) emit_mov(rt,1);
    }
  }
  else if(rt!=1) emit_mov(rt,1);
  //if(type==STOREB_STUB) emit_xorimm(EAX,1,EAX); // WriteInvalidateByteSwapped does this
  
  //if(i_regmap[HOST_CCREG]==CCREG) emit_storereg(CCREG,HOST_CCREG);//DEBUG
  //ds=i_regs!=&regs[i];
  //int real_rs=get_reg(i_regmap,rs2[i]);
  //if(!ds) load_all_consts(regs[i].regmap_entry,regs[i].was32,regs[i].wasdirty&~(1<<addr)&(real_rs<0?-1:~(1<<real_rs)),i);
  //wb_dirtys(i_regs->regmap_entry,i_regs->was32,i_regs->wasdirty&~(1<<addr)&(real_rs<0?-1:~(1<<real_rs)));
  
  if(type==STOREB_STUB)
    emit_call((int)WriteInvalidateByteSwapped);
  if(type==STOREW_STUB)
    emit_call((int)WriteInvalidateWord);
  if(type==STOREL_STUB)
    emit_call((int)WriteInvalidateLong);
  
  restore_regs(reglist);
  emit_jmp(stubs[n][2]); // return address
}

inline_writestub(int type, int i, u32 addr, signed char regmap[], int target, int adj, u32 reglist)
{
  assem_debug("inline_writestub\n");
  //int rs=get_reg(regmap,-1);
  int rt=get_reg(regmap,target);
  //assert(rs>=0);
  assert(rt>=0);
  save_regs(reglist);
  // "FASTCALL" api: address in r0, data in r1
  if(rt!=1) emit_mov(rt,1);
  emit_movimm(addr,0); // FIXME - should be able to move the existing value
  if(type==STOREB_STUB)
    emit_call((int)WriteInvalidateByte);
  if(type==STOREW_STUB)
    emit_call((int)WriteInvalidateWord);
  if(type==STOREL_STUB)
    emit_call((int)WriteInvalidateLong);
  restore_regs(reglist);
}

do_rmwstub(int n)
{
  assem_debug("do_rmwstub %x\n",start+stubs[n][3]*2);
  set_jump_target(stubs[n][1],(int)out);
  int type=stubs[n][0];
  int i=stubs[n][3];
  int rs=stubs[n][4];
  struct regstat *i_regs=(struct regstat *)stubs[n][5];
  u32 reglist=stubs[n][7];
  signed char *i_regmap=i_regs->regmap;
  int addr=get_reg(i_regmap,AGEN1+(i&1));
  //int rt=get_reg(i_regmap,rs1[i]);
  assert(rs>=0);
  //assert(rt>=0);
  if(addr<0) addr=get_reg(i_regmap,-1);
  assert(addr>=0);
  save_regs(reglist);
  // "FASTCALL" api: address in r0, data in r1 
  emit_xorimm(rs,1,rs);
  if(rs!=0) emit_mov(rs,0);
  if(rs<4||rs==12)
    emit_writeword(0,(int)&dynarec_local+24);
  
  //if(i_regmap[HOST_CCREG]==CCREG) emit_storereg(CCREG,HOST_CCREG);//DEBUG
  emit_call((int)MappedMemoryReadByte);
  //emit_mov(0,1);
  if(type==RMWA_STUB)
    emit_andimm(0,imm[i],1);
  if(type==RMWX_STUB)
    emit_xorimm(0,imm[i],1);
  if(type==RMWO_STUB)
    emit_orimm(0,imm[i],1);
  if(type==RMWT_STUB) { // TAS.B
    emit_writeword(0,(int)&dynarec_local+20);
    emit_orimm(0,0x80,1);
  }
  if(rs<4||rs==12)
    emit_readword((int)&dynarec_local+24,0);
  else
    emit_mov(rs,0);
  //emit_call((int)MappedMemoryWriteByte);
  emit_call((int)WriteInvalidateByte);
  
  restore_regs(reglist);

  if(opcode2[i]==11) { // TAS.B
    emit_readword((int)&dynarec_local+20,14);
    signed char sr;
    sr=get_reg(i_regs->regmap,SR);
    assert(sr>=0); // Liveness analysis?
    emit_andimm(sr,~1,sr);
    emit_testimm(14,0xff);
    emit_orreq_imm(sr,1,sr);
  }
  emit_jmp(stubs[n][2]); // return address
}

do_unalignedwritestub(int n)
{
  set_jump_target(stubs[n][1],(int)out);
  output_w32(0xef000000);
  emit_jmp(stubs[n][2]); // return address
}

void printregs(int edi,int esi,int ebp,int esp,int b,int d,int c,int a)
{
  printf("regs: %x %x %x %x %x %x %x (%x)\n",a,b,c,d,ebp,esi,edi,(&edi)[-1]);
}

int do_dirty_stub(int i)
{
  assem_debug("do_dirty_stub %x\n",start+i*2);
  u32 alignedlen=((((u32)source)+slen*2+2)&~2)-(u32)alignedsource;
  // Careful about the code output here, verify_dirty needs to parse it.
  #ifdef ARMv5_ONLY
  emit_loadlp(((int)source&~3),1);
  emit_loadlp((int)copy,2);
  emit_loadlp((((u32)source+slen*2+2)&~3)-((u32)source&~3),3);
  #else
  emit_movw(((u32)source&~3)&0x0000FFFF,1);
  emit_movw(((u32)copy)&0x0000FFFF,2);
  emit_movt(((u32)source&~3)&0xFFFF0000,1);
  emit_movt(((u32)copy)&0xFFFF0000,2);
  emit_movw((((u32)source+slen*2+2)&~3)-((u32)source&~3),3);
  #endif
  emit_movimm(start+i*2+slave,0);
  emit_call((int)&verify_code);
  int entry=(int)out;
  load_regs_entry(i);
  if(entry==(int)out) entry=instr_addr[i];
  emit_jmp(instr_addr[i]);
  return entry;
}

/* Memory Map */

int do_map_r(int s,int ar,int map,int cache,int x,int a,int shift,int c,u32 addr)
{
  if(c) {
    return -1; // No mapping
  }
  else {
    assert(s!=map);
    if(cache>=0) {
      // Use cached offset to memory map
      emit_addsr12(cache,s,map);
    }else{
      emit_movimm(((int)memory_map-(int)&dynarec_local)>>2,map);
      emit_addsr12(map,s,map);
    }
    // Schedule this while we wait on the load
    if(x) emit_xorimm(s,x,ar);
    //if(shift>=0) emit_shlimm(s,3,shift);
    //if(~a) emit_andimm(s,a,ar);
    emit_readword_dualindexedx4(FP,map,map);
  }
  return map;
}
int do_map_r_branch(int map, int c, u32 addr, int *jaddr)
{
  if(!c) {
    emit_test(map,map);
    *jaddr=(int)out;
    emit_js(0);
  }
  return map;
}

int gen_tlb_addr_r(int ar, int map) {
  if(map>=0) {
    assem_debug("add %s,%s,%s lsl #2\n",regname[ar],regname[ar],regname[map]);
    output_w32(0xe0800100|rd_rn_rm(ar,ar,map));
  }
}

int do_map_w(int s,int ar,int map,int cache,int x,int c,u32 addr)
{
  if(c) {
    if(can_direct_write(addr)) {
      // address_generation already loaded the const
      emit_readword_dualindexedx4(FP,map,map);
    }
    else
      return -1; // No mapping
  }
  else {
    assert(s!=map);
    if(cache>=0) {
      // Use cached offset to memory map
      emit_addsr12(cache,s,map);
    }else{
      emit_movimm(((int)memory_map-(int)&dynarec_local)>>2,map);
      emit_addsr12(map,s,map);
    }
    // Schedule this while we wait on the load
    if(x) emit_xorimm(s,x,ar);
    emit_readword_dualindexedx4(FP,map,map);
  }
  return map;
}
int do_map_w_branch(int map, int c, u32 addr, int *jaddr)
{
  if(!c||can_direct_write(addr)) {
    emit_testimm(map,0x40000000);
    *jaddr=(int)out;
    emit_jne(0);
  }
}

int gen_tlb_addr_w(int ar, int map) {
  if(map>=0) {
    assem_debug("add %s,%s,%s lsl #2\n",regname[ar],regname[ar],regname[map]);
    output_w32(0xe0800100|rd_rn_rm(ar,ar,map));
  }
}

// This reverses the above operation
int gen_orig_addr_w(int ar, int map) {
  if(map>=0) {
    assem_debug("sub %s,%s,%s lsl #2\n",regname[ar],regname[ar],regname[map]);
    output_w32(0xe0400100|rd_rn_rm(ar,ar,map));
  }
}

// Generate the address of the memory_map entry, relative to dynarec_local
generate_map_const(u32 addr,int reg) {
  //printf("generate_map_const(%x,%s)\n",addr,regname[reg]);
  emit_movimm((addr>>12)+(((u32)memory_map-(u32)&dynarec_local)>>2),reg);
}

/* Special assem */

void do_preload_rhash(int r) {
  // Don't need this for ARM.  On x86, this puts the value 0xf8 into the
  // register.  On ARM the hash can be done with a single instruction (below)
}

void do_preload_rhtbl(int ht) {
  if(slave) emit_addimm(FP,(int)&mini_ht_slave-(int)&dynarec_local,ht);
  else emit_addimm(FP,(int)&mini_ht_master-(int)&dynarec_local,ht);
}

void do_rhash(int rs,int rh) {
  emit_andimm(rs,0xf8,rh);
}

void do_miniht_load(int ht,int rh) {
  assem_debug("ldr %s,[%s,%s]!\n",regname[rh],regname[ht],regname[rh]);
  output_w32(0xe7b00000|rd_rn_rm(rh,ht,rh));
}

void do_miniht_jump(int rs,int rh,int ht) {
  emit_cmp(rh,rs);
  emit_ldreq_indexed(ht,4,15);
  #ifdef CORTEX_A8_BRANCH_PREDICTION_HACK
  emit_mov(rs,7);
  emit_jmp(jump_vaddr_reg[slave][7]);
  #else
  emit_jmp(jump_vaddr_reg[slave][rs]);
  #endif
}

void do_miniht_insert(u32 return_address,int rt,int temp) {
  #ifdef ARMv5_ONLY
  emit_movimm(return_address,rt); // PC into link register
  add_to_linker((int)out,return_address,1);
  emit_pcreladdr(temp);
  emit_writeword(rt,(int)&mini_ht[(return_address&0xFF)>>3][0]);
  emit_writeword(temp,(int)&mini_ht[(return_address&0xFF)>>3][1]);
  #else
  emit_movw(return_address&0x0000FFFF,rt);
  add_to_linker((int)out,return_address,1);
  emit_pcreladdr(temp);
  if(slave) emit_writeword(temp,(int)&mini_ht_slave[(return_address&0xFF)>>3][1]);
  else emit_writeword(temp,(int)&mini_ht_master[(return_address&0xFF)>>3][1]);
  emit_movt(return_address&0xFFFF0000,rt);
  if(slave) emit_writeword(rt,(int)&mini_ht_slave[(return_address&0xFF)>>3][0]);
  else emit_writeword(rt,(int)&mini_ht_master[(return_address&0xFF)>>3][0]);
  #endif
}

void wb_valid(signed char pre[],signed char entry[],u32 dirty_pre,u32 dirty,u64 u)
{
  //if(dirty_pre==dirty) return;
  int hr,reg,new_hr;
  for(hr=0;hr<HOST_REGS;hr++) {
    if(hr!=EXCLUDE_REG) {
      reg=pre[hr];
      if(((~u)>>(reg&63))&1) {
        if(reg>=0) {
          if(((dirty_pre&~dirty)>>hr)&1) {
            if(reg>=0&&reg<TBIT) {
              emit_storereg(reg,hr);
            }
          }
        }
      }
    }
  }
}


/* using strd could possibly help but you'd have to allocate registers in pairs
void wb_invalidate_arm(signed char pre[],signed char entry[],uint64_t dirty,uint64_t is32,uint64_t u,uint64_t uu)
{
  int hr;
  int wrote=-1;
  for(hr=HOST_REGS-1;hr>=0;hr--) {
    if(hr!=EXCLUDE_REG) {
      if(pre[hr]!=entry[hr]) {
        if(pre[hr]>=0) {
          if((dirty>>hr)&1) {
            if(get_reg(entry,pre[hr])<0) {
              if(pre[hr]<64) {
                if(!((u>>pre[hr])&1)) {
                  if(hr<10&&(~hr&1)&&(pre[hr+1]<0||wrote==hr+1)) {
                    if( ((is32>>pre[hr])&1) && !((uu>>pre[hr])&1) ) {
                      emit_sarimm(hr,31,hr+1);
                      emit_strdreg(pre[hr],hr);
                    }
                    else
                      emit_storereg(pre[hr],hr);
                  }else{
                    emit_storereg(pre[hr],hr);
                    if( ((is32>>pre[hr])&1) && !((uu>>pre[hr])&1) ) {
                      emit_sarimm(hr,31,hr);
                      emit_storereg(pre[hr]|64,hr);
                    }
                  }
                }
              }else{
                if(!((uu>>(pre[hr]&63))&1) && !((is32>>(pre[hr]&63))&1)) {
                  emit_storereg(pre[hr],hr);
                }
              }
              wrote=hr;
            }
          }
        }
      }
    }
  }
  for(hr=0;hr<HOST_REGS;hr++) {
    if(hr!=EXCLUDE_REG) {
      if(pre[hr]!=entry[hr]) {
        if(pre[hr]>=0) {
          int nr;
          if((nr=get_reg(entry,pre[hr]))>=0) {
            emit_mov(hr,nr);
          }
        }
      }
    }
  }
}
#define wb_invalidate wb_invalidate_arm
*/

// Clearing the cache is rather slow on ARM Linux, so mark the areas
// that need to be cleared, and then only clear these areas once.
void do_clear_cache()
{
  int i,j;
  for (i=0;i<(1<<(TARGET_SIZE_2-17));i++)
  {
    u32 bitmap=needs_clear_cache[i];
    if(bitmap) {
      u32 start,end;
      for(j=0;j<32;j++) 
      {
        if(bitmap&(1<<j)) {
          start=BASE_ADDR+i*131072+j*4096;
          end=start+4095;
          j++;
          while(j<32) {
            if(bitmap&(1<<j)) {
              end+=4096;
              j++;
            }else{
              __clear_cache((void *)start,(void *)end);
              break;
            }
          }
        }
      }
      needs_clear_cache[i]=0;
    }
  }
}

// CPU-architecture-specific initialization
void arch_init() {

  #if 0
  // Trampolines for jumps >32M
  int *ptr,*ptr2;
  ptr=(int *)jump_table_symbols;
  ptr2=(int *)((void *)BASE_ADDR+(1<<TARGET_SIZE_2)-JUMP_TABLE_SIZE);
  while((void *)ptr<(void *)jump_table_symbols+sizeof(jump_table_symbols))
  {
    int offset=*ptr-(int)ptr2-8;
    if(offset>=-33554432&&offset<33554432) {
      *ptr2=0xea000000|((offset>>2)&0xffffff); // direct branch
    }else{
      *ptr2=0xe51ff004; // ldr pc,[pc,#-4]
    }
    ptr2++;
    *ptr2=*ptr;
    ptr++;
    ptr2++;
  }

  // Jumping thru the trampolines created above slows things down by about 1%.
  // If part of the cache is beyond the 32M limit, avoid using this area
  // initially.  It will be used later if the cache gets full.
  if((u32)dyna_linker-33554432>(u32)BASE_ADDR) {
    if((u32)dyna_linker-33554432<(u32)BASE_ADDR+(1<<(TARGET_SIZE_2-1))) {
      out=(u8 *)(((u32)dyna_linker-33554432)&~4095);
      expirep=((((int)out-BASE_ADDR)>>(TARGET_SIZE_2-16))+16384)&65535;
    }
  }
  #endif
}
