/****************************************************************************/
/*                                                                          */
/*                              Device Mounter                              */
/*                                                                          */
/*                              (c) Thomas Uhl                              */
/*                                                                          */
/*                                                                          */
/*                             v1.0 - 7.2.1993                              */
/*                             v1.1 - 28.3.1993                             */
/*                             v2.0 - 17.4.1993                             */
/*                             v2.5 - 5.5.1993                              */
/*                             v3.0 - 22.5.1993                             */
/*                             v3.1 - 7.6.1993                              */
/*                             v3.5 - 14.8.1993                             */
/*                             v3.6 - 20.8.1994                             */
/*                                                                          */
/*                   v3.7 - 23.8.2003 -- Volker Ossenkopf                   */
/*                                                                          */
/****************************************************************************/


#include <xview/xview.h>
#include <xview/panel.h>

#include <string.h>
#include <fcntl.h>
#include <assert.h>
#include <errno.h>
#include <wait.h>

#include <mntent.h>

#include "iconimage.h"
#include "iconmask.h"

#define VERSION "3.7"
#define CONFIG_FILE "xvmounttab"
#define CONFIG_DIR "/etc"
#define XVMOUNTPATH "XVMOUNTPATH"
#define MOUNTCMD  "/bin/mount"
#define UMOUNTCMD "/bin/umount"

#define BOOL int
#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#define STRLEN 80
#define LINELEN 255
#define MAXITEMS 16

#define ITEM_KEY 1000
#define ITEM_NR_KEY 1001

#define streq(s, t)     (strcmp ((s), (t)) == 0)

/* types */
typedef struct {
  char name[STRLEN];
  char dir[STRLEN];
  char type[STRLEN];
  BOOL mounted;
  Panel_item Label, Button;
} MITEM;


/* function prototypes */
void all_mounted_fs ();
BOOL check_mounted_fs (MITEM*);
BOOL do_mount (char*);
BOOL do_umount (char*);
int execmount (char*, char*);
Icon create_icon (char*);
void create_buttons ();
void quit ();

/* callback prototypes */
void but_cb (Panel_item, Event*);

/* global variables */
Frame frame;
Panel panel;
Panel_item Floppy0Txt, Floppy1Txt;
Panel_item Floppy0But, Floppy1But;
Panel_item	quitbutton;

int devices;              /* number of devices */
MITEM mitems[MAXITEMS];   /* device list*/
int mounting;
BOOL setquitbutton = FALSE;
char *confdir = CONFIG_DIR;

main (int argc, char *argv[])
{
  Icon icon;
  int c;
  extern char *optarg;

  xv_init (XV_INIT_ARGC_PTR_ARGV, &argc, argv, NULL);

  while ((c = getopt(argc, argv, "qd:h?")) != EOF)
		switch (c) {
		case 'q':
			setquitbutton = TRUE;
			break;
		case 'd':
			confdir = optarg;
			break;
		case 'h':
		case '?':
		default:
			fprintf(stderr,
"usage: %s [-qh] \n\
  -q  put quit button in the window\n\
  -d <directory>   search for xvmounttab in <directory> instead of /etc\n\
  -h  and  -?  display this message\n",
  argv[0]);
			exit(1);
		}

  
  frame = (Frame) xv_create (XV_NULL, FRAME,
                                      FRAME_LABEL, "Tom's Device Mounter v" 
			                           VERSION,
                                      FRAME_SHOW_FOOTER, TRUE, 
                                      FRAME_SHOW_RESIZE_CORNER, FALSE,
                                      NULL);

  icon = create_icon (argv[0]);

  panel = (Panel) xv_create (frame, PANEL, 
                                    XV_HELP_DATA, "xvmount:panel",
                                    NULL);
                                     // XV_WIDTH, 300,

  create_buttons ();
  mounting=0;

  xv_set (frame, FRAME_ICON, icon, NULL);

  window_fit (panel);
  window_fit (frame);

  xv_main_loop (frame);
}

void but_cb (Panel_item item, Event *event)
{
  MITEM *mitem;
  char msg[STRLEN];

#ifdef DEBUG
  printf ("Selected Item-Nr: %d\n", xv_get (item, XV_KEY_DATA, ITEM_NR_KEY));
#endif

  /* Only one mount instance must be running at a time */
  if (mounting != 0) return;
  mounting = 1;

  mitem = (MITEM *) xv_get (item, XV_KEY_DATA, ITEM_KEY);

  if (!mitem->mounted)
  {
     mitem->mounted = do_mount (mitem->dir); 
     if (check_mounted_fs(mitem))
	   {
              xv_set (item, PANEL_LABEL_STRING, "umount", NULL);
              sprintf (msg, "%s %s mounted", mitem->dir, mitem->type);
              xv_set (frame, FRAME_LEFT_FOOTER, msg, NULL);
	   }
     else
	   {
              sprintf (msg, "mount %s failed", mitem->dir);
              xv_set (frame, FRAME_LEFT_FOOTER, msg, NULL);
	   }
  }
  else
  {
      mitem->mounted = !do_umount (mitem->dir);
      if (check_mounted_fs(mitem))
	   {
              sprintf (msg, "umount %s failed", mitem->dir);
              xv_set (frame, FRAME_LEFT_FOOTER, msg, NULL);
	   }
	   else
	   {
	       xv_set (item, PANEL_LABEL_STRING, "mount", NULL);
	       sprintf (msg, "%s unmounted", mitem->dir);
	       xv_set (frame, FRAME_LEFT_FOOTER, msg, NULL);
           }
  }
  mounting = 0;

}


BOOL do_mount (char* mdir)
{
  int  mret;
  char msg[STRLEN];

#ifdef DEBUG
  printf ("mount <%s>\n", mdir);
#endif

#ifndef DONT_MOUNT
  mret=execmount(MOUNTCMD,mdir);
  if (mret != 0)
    { 
      sprintf (msg, "Mount: %s", strerror(WEXITSTATUS(mret)));
      xv_set (frame, FRAME_LEFT_FOOTER, msg, NULL);
      return FALSE;
    }
#endif

  return TRUE;
}


BOOL do_umount (char *mdir)
{
  int  mret;
  char msg[80];

  sync ();

#ifdef DEBUG
  printf ("umount <%s>\n", mdir);
#endif

#ifndef DONT_MOUNT
  mret=execmount(UMOUNTCMD,mdir);
  if (mret != 0)
  {
    sprintf (msg, "Umount: %s", strerror(WEXITSTATUS(mret)));
    xv_set (frame, FRAME_LEFT_FOOTER, msg, NULL);
    return FALSE;
  }
#endif

  return TRUE;
}

Icon create_icon (char *label)
{
  Icon icon;
  Server_image icon_image, mask_image;

  if (strrchr (label, '/'))
  {
    label = strrchr (label, '/');
    label++;
  }

  icon_image = (Server_image) xv_create (XV_NULL, SERVER_IMAGE,
                                         XV_WIDTH, iconimage_width,
                                         XV_HEIGHT, iconimage_height,
                                         SERVER_IMAGE_X_BITS, iconimage_bits,
                                         NULL);

  mask_image = (Server_image) xv_create (XV_NULL, SERVER_IMAGE,
                                         XV_WIDTH, iconmask_width,
                                         XV_HEIGHT, iconmask_height,
                                         SERVER_IMAGE_X_BITS, iconmask_bits,
                                         NULL);

  icon = (Icon) xv_create (XV_NULL, ICON,
                                    ICON_IMAGE, icon_image,
                                    ICON_LABEL, label,
                                    ICON_MASK_IMAGE, mask_image,
                                    NULL);
  return icon;
}

void create_buttons ()
{
  int i, spac = 10, lin = 1, len = 0, nlen, ypos, res;
  FILE *fd;
  char *cp;
  char path[LINELEN];
  char line[LINELEN];

  /* Create file path for xvmounttab */
  if ((cp = getenv (XVMOUNTPATH)) != NULL)
  {
    res = strlen (cp) - 1;
    if (cp[res] == '/')
      cp[res] = '\0';
    sprintf (path, "%s/%s", cp, CONFIG_FILE); 
  } else { 
    sprintf (path, "%s/%s", confdir, CONFIG_FILE);
  } 

  devices = 0;

  /* Read devices in xvmounttab */
  if (fd = fopen (path, "r"))
  {
    while ((cp = fgets (line, LINELEN, fd)) && devices < MAXITEMS)
    {
      while (*cp)
      {
        if (*cp == '#' || *cp == '\n')
          *cp = '\0';
        cp++;
      }
      
      if (line[0])
      {
        res = sscanf (line, "%s%s", &mitems[devices].name, 
                                          &mitems[devices].dir);
        if (res > 0 && res < 2)
        {
          fprintf (stderr, "xvmount: error in line %d\n", lin);
          exit (1);
        }

#ifdef DEBUG
        printf ("%s - %s\n", mitems[devices].name, 
		                            mitems[devices].dir);
#endif

        devices++;
      }
      lin++;
    }
    fclose (fd);
  } else {
    fprintf (stderr, "xvmount: %s %s\n", path, strerror (errno));
    exit (1);
  }

  /* Create buttons in panel */
  len = spac;
  ypos = 0; 
  for (i = 0; i < devices; i++)
  {
    mitems[i].Label = xv_create (panel, PANEL_TEXT,
				 XV_X, spac,
				 XV_Y, ypos+spac,
                                 PANEL_LABEL_STRING, mitems[i].name,
                                 PANEL_VALUE_DISPLAY_WIDTH, spac,
                                 PANEL_VALUE, "",
                                 PANEL_VALUE_UNDERLINED, FALSE,
                                 PANEL_READ_ONLY, TRUE,
                                 XV_HELP_DATA, "xvmount:status",
                                 NULL);
    ypos=(int)(xv_get(mitems[i].Label,XV_Y)+xv_get(mitems[i].Label,XV_HEIGHT));
    nlen=(int)(xv_get(mitems[i].Label,XV_X)+xv_get(mitems[i].Label,XV_WIDTH));
    if ( nlen >len)
    {
      len=nlen;
    }
  }
  for (i = 0; i < devices; i++)
  {
     mitems[i].mounted = FALSE;
     strcpy(mitems[i].type,"");
     ypos = (int) xv_get(mitems[i].Label,XV_Y);
     mitems[i].Button = xv_create (panel, PANEL_BUTTON,
                                   XV_X, len,
				   XV_Y,ypos,
                                   PANEL_LABEL_STRING, "mount",
                                   PANEL_NOTIFY_PROC, but_cb, 
                                   XV_KEY_DATA, ITEM_KEY, &mitems[i],
                                   XV_KEY_DATA, ITEM_NR_KEY, i,
                                   XV_HELP_DATA, "xvmount:button",
                                   NULL);
  }

 /* Quit button if requested */
 if (setquitbutton)
 {
   i=devices-1;
   ypos=(int)(xv_get(mitems[i].Label,XV_Y)+xv_get(mitems[i].Label,XV_HEIGHT));
   quitbutton = xv_create(panel, PANEL_BUTTON,
			XV_X, len+spac,
			XV_Y, ypos+spac,
			PANEL_LABEL_STRING, "Quit", 
			PANEL_NOTIFY_PROC, quit, 
			NULL);
  }

  all_mounted_fs ();
}

void all_mounted_fs ()
{
  int i;
  FILE *F_mtab;
  struct mntent *mnt;
  char msg[80];

  if ((F_mtab = setmntent (MOUNTED, "r")) == NULL)
  {
    sprintf (msg, "Can't open %s: %s", MOUNTED, strerror (errno));
    xv_set (frame, FRAME_LEFT_FOOTER, msg, NULL);
    return;
  }

  for (i = 0; i < devices; i++) mitems[i].mounted = FALSE;

  while (mnt = getmntent (F_mtab))
  {
    for (i = 0; i < devices; i++)
    {
      if (!strcmp (mitems[i].dir, mnt->mnt_dir))
      {
        /* fs is already mounted */
        mitems[i].mounted = TRUE;
        strcpy (mitems[i].type,mnt->mnt_type);
      }
    }
  }
  endmntent (F_mtab);

  for (i = 0; i < devices; i++)
  {
    if (mitems[i].mounted)
        xv_set (mitems[i].Button, PANEL_LABEL_STRING, "umount", NULL);
    else
        xv_set (mitems[i].Button, PANEL_LABEL_STRING, "mount", NULL);
  }
}

BOOL check_mounted_fs (MITEM* mitem)
{
  BOOL mounted;
  FILE *F_mtab;
  struct mntent *mnt;
  char msg[80];

  if ((F_mtab = setmntent (MOUNTED, "r")) == NULL)
  {
    sprintf (msg, "Can't open %s: %s", MOUNTED, strerror (errno));
    xv_set (frame, FRAME_LEFT_FOOTER, msg, NULL);
    return FALSE;
  }

  mitem->mounted = FALSE;

  while (mnt = getmntent (F_mtab))
  {
      if (!strcmp (mitem->dir, mnt->mnt_dir))
      {
        mitem->mounted = TRUE;
        strcpy(mitem->type,mnt->mnt_type);
      }
  }
  endmntent (F_mtab);

  return mitem->mounted;
}

/*
 * This will cause the main loop to exit.
 */
void
quit( void )
{
	xv_destroy_safe(frame);
}

/*
 * Do actual call of external mount command
 */
int execmount (char *command, char *mdir) {
 int pid, status;
 char *argv[3];

 if (command == 0)
     return 1;
 pid = fork();
 if (pid == -1)
     return -1;
 if (pid == 0) {

#ifdef DEBUG
  printf("Command called: %s %s\n",command,mdir);
#endif

     argv[0] = command;
     argv[1] = mdir;
     argv[2] = 0;
     execv(argv[0], argv);
     exit(127);
 }
 do {
     if (waitpid(pid, &status, 0) == -1) {
         if (errno != EINTR)
             return -1;
     } else
         return status;
 } while(1);
}
