/*
 *	Xenophilia GTK+ Theme Engine
 * 
 *  xeno_style_draw.c:
 *		Drawing routines of XenoStyle.
 *
 *	Copyright  1999-2002 Johan Hanson <misagon@bahnhof.se>.
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include "xeno_rc_style.h"

#include "xeno_style.h"
#include "xeno_style_draw.h"
#include "xeno_style_fill.h"
#include "xeno_style_images.h"
#include "xeno_patches.h"

/* Macros */
#define SIGN(exp)	((exp)<0 ? -1 : 1)

#if !XENO_GTK2 && !defined(GTK_RANGE_GET_CLASS)
#define GTK_RANGE_GET_CLASS(_widget)	GTK_RANGE_CLASS(((GtkObject *)(_widget))->klass)
#endif

#define XENO_WINDOW_SIZE(_window, _width, _height) \
	if ((_width | _height) < 0) \
		gdk_window_get_size (_window, (_width < 0) ? &_width : NULL, (_height < 0) ? &_height : NULL)


/*
 *	Prototypes
 */

static void	xeno_style_draw_line		(GtkStyle		*style,
										 GdkWindow		*window,
										 GtkStateType	state_type,
										 GtkShadowType	shadow_type,
										 GdkRectangle	*area,
										 gint			x,
										 gint			y,
										 gint			length,
										 GtkOrientation	orientation);

/* Images */
static void xeno_style_draw_image		(XenoStyle		*style,
										 GdkWindow		*window,
										 GtkWidget		*widget,
										 GdkRectangle	*area,
										 XenoStyleImageType	image_type,
										 guint			variant_type,
										 gint			srcx,
										 gint			srcy,
										 gint			x,
										 gint			y,
										 gint			width,
										 gint			height);

/* Decoration */
static void xeno_style_draw_knob		(GtkStyle		*style,
										 GdkWindow		*window,
										 GtkStateType	state_type,
										 XenoKnobType	knob_type,
										 GdkRectangle	*area,
										 GtkWidget		*widget,
										 gboolean		fill,
										 GtkOrientation	orientation,
										 gboolean		center,
										 GtkOrientation	centering,
										 gint			x,
										 gint			y,
										 gint			width,
										 gint			height);

static void xeno_style_draw_lines	 	(GtkStyle		*style,
										 GdkWindow		*window,
										 GtkStateType	state_type,
										 gboolean		in,
										 GdkRectangle	*area,
										 GtkWidget		*widget,
										 gint			x,
										 gint			y,
										 gint			width,
										 gint			height,
										 GtkOrientation	orientation);
									 
static void xeno_style_draw_buds		(GtkStyle		*style,
										 GdkWindow		*window,
										 GtkStateType	state_type,
										 gboolean		in,
										 GdkRectangle	*area,
										 GtkWidget		*widget,
										 gint			x,
										 gint			y,
										 gint			width,
										 gint			height,
										 GtkOrientation	orientation,
										 gboolean		ns_style);

static void	xeno_style_draw_solid_arrow	(GtkStyle		*style,
										 GdkWindow		*window,
										 GtkStateType	state_type,
										 GdkRectangle	*area,
										 GtkWidget		*widget,
										 GtkArrowType	arrow_type,
										 gint			x, 
										 gint			y, 
										 gint			double_width,
										 gint			double_height);



/*** GtkStyleClass Methods ****************************************************/

/*	shadow_type:	IN:  as "thin in" rectangle drawn thin
					OUT: as separator between two "thin out" frames.
*/
static void
xeno_style_draw_line	(GtkStyle		*style,
						 GdkWindow		*window,
						 GtkStateType	state_type,
						 GtkShadowType	shadow_type,
						 GdkRectangle	*area,
						 gint			x,
						 gint			y,
						 gint			length,
						 GtkOrientation	orientation)
{
	GdkGC *light_gc, *dark_gc, *mid_gc;

	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	light_gc = style->light_gc[state_type];
	dark_gc  = style->dark_gc[state_type];
	mid_gc   = style->mid_gc[state_type];
	
	if (area) {
		gdk_gc_set_clip_rectangle (light_gc, area);
		gdk_gc_set_clip_rectangle (dark_gc, area);
		gdk_gc_set_clip_rectangle (mid_gc, area);
	}
	
	if (orientation == GTK_ORIENTATION_VERTICAL) {
		gint y2 = y + length - 1;
		
		gdk_draw_line (window, dark_gc,  x,   y, x,   y2);
		gdk_draw_line (window, light_gc, x+1, y, x+1, y2);
		if (shadow_type == GTK_SHADOW_OUT) {
			gdk_draw_point (window, mid_gc,  x,   y);
			gdk_draw_point (window, mid_gc,  x+1, y2);
		} else if (shadow_type == GTK_SHADOW_IN) {
			gdk_draw_point (window, mid_gc,  x+1, y);
			gdk_draw_point (window, mid_gc,  x,   y2);
		}
	} else {
		gint x2 = x + length - 1;
		
		gdk_draw_line (window, dark_gc,  x, y,   x2, y);
		gdk_draw_line (window, light_gc, x, y+1, x2, y+1);
		if (shadow_type == GTK_SHADOW_IN) {
			gdk_draw_point (window, mid_gc, x,  y);
			gdk_draw_point (window, mid_gc, x2, y+1);
		} else if (shadow_type == GTK_SHADOW_OUT) {
			gdk_draw_point (window, mid_gc, x,  y+1);
			gdk_draw_point (window, mid_gc, x2, y);
		}
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (mid_gc, NULL);
		gdk_gc_set_clip_rectangle (dark_gc, NULL);
		gdk_gc_set_clip_rectangle (light_gc, NULL);
	}
}


void
xeno_style_draw_hline	(GtkStyle     *style,
						 GdkWindow    *window,
						 GtkStateType  state_type,
						 GdkRectangle  *area,
						 GtkWidget     *widget,
						 xeno_char     *detail,
						 gint          x1,
						 gint          x2,
						 gint          y)
{
	GtkWidget		*cur;
	GtkShadowType	shadow_type;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if (detail) {
		if (!g_strcasecmp (detail, "label")) {
			if (area)
				gdk_gc_set_clip_rectangle (style->fg_gc[state_type], area);
			
			gdk_draw_line (window, style->fg_gc[state_type], x1, y, x2-1, y);
			
			if (area)
				gdk_gc_set_clip_rectangle (style->fg_gc[state_type], NULL);
			
			return;
		} else if (   !g_strcasecmp (detail, "vpaned")
				   && (xeno_patch_config & XENO_PATCH_PANED))
		{
			return;
		}
	}
	
	shadow_type = GTK_SHADOW_OUT;
	if (x1 == 0 && widget != NULL && (cur = widget->parent) != NULL) {
		shadow_type = GTK_SHADOW_NONE;
		while (cur != NULL && GTK_CONTAINER(cur)->border_width == 0) {
			if (GTK_IS_BOX(cur) || GTK_IS_TABLE(cur)) {
				cur = cur->parent;
				continue;
			} else if (GTK_IS_WINDOW(cur)) {
				if (XENO_STYLE_RC_DATA(style) && XENO_STYLE_RC_DATA(style)->flat_windows) {
					shadow_type = GTK_SHADOW_OUT;
				} else {
					shadow_type = GTK_SHADOW_IN;
				}
			}
			break;
		}
	}
	
	xeno_style_draw_line (style, window, state_type, shadow_type, area,
						  x1, y, x2-x1, GTK_ORIENTATION_HORIZONTAL);
}

void
xeno_style_draw_vline	(GtkStyle		*style,
						 GdkWindow		*window,
						 GtkStateType	state_type,
						 GdkRectangle	*area,
						 GtkWidget		*widget,
						 xeno_char		*detail,
						 gint			y1,
						 gint			y2,
						 gint			x)
{
	GtkWidget		*cur;
	GtkShadowType	shadow_type;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if (   (detail != NULL) && !g_strcasecmp (detail, "hpaned")
		&& (xeno_patch_config & XENO_PATCH_PANED))
	{
		return;
	}
	
	shadow_type = GTK_SHADOW_OUT;
	if (y1 == 0 && widget != NULL && (cur = widget->parent)) {
		while (cur != NULL && GTK_CONTAINER(cur)->border_width == 0) {
			if (GTK_IS_BOX(cur) || GTK_IS_TABLE(cur)) {
				cur = cur->parent;
				continue;
			} else if (GTK_IS_WINDOW(cur)) {
				if (XENO_STYLE_RC_DATA(style) && XENO_STYLE_RC_DATA(style)->flat_windows) {
					shadow_type = GTK_SHADOW_OUT;
				} else {
					shadow_type = GTK_SHADOW_IN;
				}
			} else if (GTK_IS_FRAME(cur) || GTK_IS_MENU(cur)) {
				shadow_type = GTK_SHADOW_NONE;
			}
			break;
		}
	}
	
	xeno_style_draw_line (style, window, state_type, shadow_type, area,
						  x, y1, y2-y1, GTK_ORIENTATION_VERTICAL);
}


void
xeno_style_draw_shadow	(GtkStyle      *style,
						 GdkWindow     *window,
						 GtkStateType   state_type,
						 GtkShadowType  shadow_type,
						 GdkRectangle  *area,
						 GtkWidget     *widget,
						 xeno_char     *detail,
						 gint           x,
						 gint           y,
						 gint           width,
						 gint           height)
{
	GdkPoint points[5];
	XenoRcData *rc_data;
	XenoShadowType xeno_shadow_type;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	XENO_WINDOW_SIZE(window, width, height);
	
	rc_data = XENO_STYLE_RC_DATA(style);
	xeno_shadow_type = (XenoShadowType)shadow_type;
	
	if (widget != NULL) {
		if (shadow_type == GTK_SHADOW_IN) {
		  #if !XENO_GTK2
			if (GTK_IS_RANGE(widget)) {
				if (   ((GtkRange *)widget)->in_child == GTK_RANGE_GET_CLASS(widget)->step_forw
					|| ((GtkRange *)widget)->in_child == GTK_RANGE_GET_CLASS(widget)->step_back)
				{
					xeno_shadow_type = XENO_SHADOW_BUTTON_IN;
					/* draw steppers as buttons */
				} else if (state_type != GTK_STATE_INSENSITIVE) {
					state_type = GTK_STATE_NORMAL;
					/* trough shadow should not be drawn in ACTIVE colors, just the trough background */
				}
			} else if (GTK_IS_TEXT(widget)) {
				if (GTK_WIDGET_HAS_FOCUS(widget)) {
					/* GtkText has extra thickness for the focus rect. Lets use it */
					x--;
					y--;
					width+=2;
					height+=2;
				}
			} else
		  #endif
			if (GTK_IS_ENTRY(widget) && GTK_WIDGET_STATE(widget) == GTK_STATE_INSENSITIVE) {
				/* use widget state for gods sake */
				state_type = GTK_STATE_INSENSITIVE;
			}
			
			/* user "in[<state_type>]" override the following */
			if (!rc_data || !(rc_data->config_shadows & (XENO_CONFIG_SHADOW_IN << state_type))) {
				if (GTK_IS_FRAME(widget) && !GTK_IS_PREVIEW(GTK_BIN(widget)->child)) {
					/* only clickable widgets should have fat frames */
					xeno_shadow_type = XENO_SHADOW_THIN_IN;
					x++;		y++;
					width-=2;	height-=2;
				} else if (GTK_IS_PROGRESS(widget)) {
					xeno_shadow_type = XENO_SHADOW_THIN_IN;
				} else if (   GTK_IS_BUTTON(widget)
						   && !GTK_IS_TOGGLE_BUTTON(widget)
						   && state_type == GTK_STATE_ACTIVE)
				{
					xeno_shadow_type = XENO_SHADOW_BUTTON_IN;
				}
			}
		} else if (shadow_type == GTK_SHADOW_OUT) {
			if (   GTK_IS_BUTTON(widget)
				&& ((GtkButton *)widget)->relief != GTK_RELIEF_NORMAL
				&& state_type == GTK_STATE_PRELIGHT)
			{
				/* "button_default" */
				xeno_shadow_type = (rc_data) ? rc_data->shadow_button_relief : XENO_SHADOW_BUTTON_RELIEF;
			} else if (GTK_IS_MENU_ITEM(widget)) {
				/* "menu_style" */
				xeno_shadow_type = (rc_data) ? rc_data->shadow_menu_item : XENO_SHADOW_IN;
			} else if (!rc_data || !(rc_data->config_shadows & (XENO_CONFIG_SHADOW_OUT << state_type))) {
				/* user "in[<state_type>]" override the following */
				if (   (GTK_IS_FRAME(widget) && !GTK_IS_EVENT_BOX(widget->parent))
					|| GTK_IS_RULER(widget)
					|| (detail && (   !g_strcasecmp(detail, "handlebox_bin")
								   || !g_strcasecmp(detail, "dockitem_bin")))
					)
				{
					/* only clickable widgets should have fat frames */
					xeno_shadow_type = XENO_SHADOW_THIN_OUT;
				} else if (GTK_IS_MENU(widget)) {
					if (   GTK_IS_MENU(widget)
					   && ((GtkMenu *)widget)->torn_off)
					{
						/* draw torn off menu as a GtkWindow */
						xeno_shadow_type = (rc_data && rc_data->flat_windows) ? XENO_SHADOW_NONE : XENO_SHADOW_THIN_OUT;
					}
				} else if (GTK_IS_MENU_BAR(widget)) {
					if (widget->parent && GTK_IS_HANDLE_BOX(widget->parent))
						return; /* do not draw nested frame in frame */
					
					if (rc_data && rc_data->flat_windows) {
						/* draw only a separating line a'la windoze */
						xeno_style_draw_hline (style, window, state_type, area, widget, detail,
											   x, x+width, y+height-2);
						return;
					} else if (XENO_STYLE_YTHICKNESS(style) > 1) {
						/* make it etched in case it has borders or isnt at the top */
						xeno_shadow_type = XENO_SHADOW_ETCHED_IN;
						x--;
						y--;
						height++;
						width+=2;
					}
				}
			}
		}
	}
	
	if (xeno_shadow_type == XENO_SHADOW_NONE)
		return;
	
	points[0].x = x;			points[0].y = y;
	points[1].x = x;			points[1].y = y+height-1;
	points[2].x = x+width-1;	points[2].y = y+height-1;
	points[3].x = x+width-1;	points[3].y = y;
	points[4].x = x;			points[4].y = y;
	
	xeno_style_draw_polygon (style, window, state_type, (GtkShadowType)xeno_shadow_type, area,
							 widget, NULL, points, 5, FALSE);
}

void
xeno_style_draw_polygon (GtkStyle      *style,
						 GdkWindow     *window,
						 GtkStateType   state_type,
						 GtkShadowType  shadow_type,
						 GdkRectangle  *area,
						 GtkWidget     *widget,
						 xeno_char     *detail,
						 GdkPoint      *points,
						 gint           npoints,
						 gboolean       fill)
{
	static const gdouble pi_over_4	 = M_PI_4;
	static const gdouble pi_3_over_4 = M_PI_4 * 3;
	
	XenoShadow		 ctx;
	XenoShadowGCRing *ring;
	GdkGC		*gc, *mid_gc;
	gdouble		angle;
	gint		i, j, x,y, x2,y2, xt, yt, mx=0, my, sign;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	g_return_if_fail (points != NULL);
	
	/* xeno_shadow_init() will fetch user settings for "in" and "out" */
	xeno_shadow_init (&ctx, style, window, state_type, (XenoShadowType)shadow_type, area, widget);
	
	if (fill) {
		GdkGC *bg = style->bg_gc[state_type];
		
		if (area) gdk_gc_set_clip_rectangle (bg, area);
		gdk_draw_polygon (window, bg, TRUE, points, npoints);
		
		/* don't trust gdk_draw_polygon(,,TRUE,,) to draw edge pixels correctly */
		if (ctx.thickness == 0) {
			for (i = 0; i < npoints - 1; ++i) {
				x  = points[i].x;   y  = points[i].y;
				x2 = points[i+1].x; y2 = points[i+1].y;
				
				if (y2 < y) {
					/* draw top-to-bottom always because bottom-to-top is NOT equivalent in X */
					xt = x; x = x2; x2 = xt;
					yt = y; y = y2; y2 = yt;
				}
				gdk_draw_line (window, bg, x, y, x2, y2);
			}
		}
		
		/* draw edge line if polygon is not closed */
		if (points[npoints-1].x != points[0].x || points[npoints-1].y != points[0].y) {
			gdk_draw_line (window, bg,
						   points[0].x, points[0].y,
						   points[npoints-1].x, points[npoints-1].y);
		}
		
		if (area) gdk_gc_set_clip_rectangle (bg, NULL);
	}
	
	/* draw internal rings first and go outwards,
	   in each ring, draw lines first and then points.
	   no need to optimize lines into loops or rectangles: Xlib will do that
	   "sign" is used to find transitions between light and dark
	   (mx,my) are used there, as they may be offset from (x,y)
	*/
	sign = 0;
	for (i = ctx.thickness - 1; i >= 0; --i) {
		ring = &ctx.ring[i];
		for (j = 0; j < npoints - 1; ++j) {
			x	= points[j].x;
			y	= points[j].y;
			x2	= points[j+1].x;
			y2	= points[j+1].y;
			
			if ((x == x2) && (y == y2)) {
				angle = 0;
			} else {
				angle = atan2 (y2 - y, x2 - x);
			}
			
			if ((angle > -pi_3_over_4 - 0.0625) && (angle < pi_over_4 - 0.0625)) {
				if (i > 0) {
					if (angle > -pi_over_4) {
						y  -= i;
						y2 -= i;
					} else {
						x  -= i;
						x2 -= i;
					}
				}
				mid_gc = gc = ring->gc[XENO_SHADOW_GC_BOTTOM];
				if (sign != 0) {
					mid_gc = ring->gc[XENO_SHADOW_GC_CORNER];
					mx = x + i;
					sign = 0;
				}
			} else {
				if (i > 0) {
					if ((angle < -pi_3_over_4) || (angle > pi_3_over_4)) {
						y  += i;
						y2 += i;
					} else {
						x  += i;
						x2 += i;
					}
				}
				mid_gc = gc = ring->gc[XENO_SHADOW_GC_TOP];
				if (sign != 1) {
					mid_gc = ring->gc[XENO_SHADOW_GC_CORNER];
					mx = x - i;
					sign = 1;
				}
			}
			my = y;
			
			if (gc) {
				if (y2 < y) {
					xt = x; x = x2; x2 = xt;
					yt = y; y = y2; y2 = yt;
				}
				gdk_draw_line (window, gc, x, y, x2, y2);
			}
			
			if (j > 0 && mid_gc != NULL && mid_gc != gc)
				gdk_draw_point (window, mid_gc, mx, my);
		}
	}
	xeno_shadow_done(&ctx, area);
}


static void
xeno_style_draw_solid_arrow	(GtkStyle		*style,
							 GdkWindow		*window,
							 GtkStateType	state_type,
							 GdkRectangle	*area,
							 GtkWidget		*widget,
							 GtkArrowType	arrow_type,
							 gint			x,
							 gint			y,
							 gint			width,
							 gint			height)
{
	XenoStyleImageType image_type;
	gint arrow_x, arrow_y, arrow_width, arrow_height, side;
	
	switch (state_type) {
	  case GTK_STATE_PRELIGHT:
		image_type = XENO_STYLE_IMAGE_TRIANGLE_PRELIGHT;
		break;
		
	  case GTK_STATE_INSENSITIVE:
		image_type = XENO_STYLE_IMAGE_TRIANGLE_INSENSITIVE;
		break;
		
	  case GTK_STATE_ACTIVE:
		image_type = XENO_STYLE_IMAGE_TRIANGLE_ACTIVE;
		break;
		
	  default:
		image_type = XENO_STYLE_IMAGE_TRIANGLE_NORMAL;
	}
	
	if (arrow_type == GTK_ARROW_UP || arrow_type == GTK_ARROW_DOWN) {
		width = (width - 1) | 1;
		side = MIN(width * XENO_STYLE_IMAGE_TRIANGLE_RATIO, height) - 2;
		arrow_width  = width;
		arrow_height = side;
		y += (height - side + 1)/2;
		if (arrow_type == GTK_ARROW_UP) {
			arrow_x = XENO_STYLE_IMAGE_TRIANGLE_UP_X - arrow_width/2;
			arrow_y = XENO_STYLE_IMAGE_TRIANGLE_UP_Y;
		} else {
			arrow_x = XENO_STYLE_IMAGE_TRIANGLE_DOWN_X - arrow_width/2;
			arrow_y = XENO_STYLE_IMAGE_TRIANGLE_DOWN_Y - arrow_height;
		}
	} else {
		height = (height - 1) | 1;
		side = MIN(width, height * XENO_STYLE_IMAGE_TRIANGLE_RATIO) - 2;
		arrow_width = side;
		arrow_height = height;
		x += (width - side + 1)/2;
		if (arrow_type == GTK_ARROW_LEFT) {
			arrow_x = XENO_STYLE_IMAGE_TRIANGLE_LEFT_X;
			arrow_y = XENO_STYLE_IMAGE_TRIANGLE_LEFT_Y - arrow_height/2;
		} else {
			arrow_x = XENO_STYLE_IMAGE_TRIANGLE_RIGHT_X - arrow_width;
			arrow_y = XENO_STYLE_IMAGE_TRIANGLE_RIGHT_Y - arrow_height/2;
		}
	}
	
	xeno_style_draw_image (style, window, widget, area, image_type, 0,
						   arrow_x, arrow_y, x, y, arrow_width, arrow_height);
}


void
xeno_style_draw_arrow	(GtkStyle      *style,
						 GdkWindow     *window,
						 GtkStateType   state_type,
						 GtkShadowType  shadow_type,
						 GdkRectangle  *area,
						 GtkWidget     *widget,
						 xeno_char     *detail,
						 GtkArrowType   arrow_type,
						 gboolean       fill,
						 gint           x,
						 gint           y,
						 gint           width,
						 gint           height)
{
	gint		t;
	gboolean	solid = FALSE;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	XENO_WINDOW_SIZE(window, width, height);
	
	if (detail) {
		if (detail[0] != '\0' && !g_strcasecmp(detail + 1, /* "v"|"h" */"scrollbar")) {
		  #if !XENO_GTK2
			if (widget && GTK_WIDGET_STATE(widget) == GTK_STATE_INSENSITIVE)
				state_type = GTK_STATE_INSENSITIVE;
		  #endif
			if (XENO_STYLE_RC_DATA(style)->stepper_box) {
				gint xt, yt;
				
				gtk_paint_box (style, window, state_type, shadow_type,
							   area, widget, detail, x, y, width, height);
				
				if (XENO_STYLE_RC_DATA(style)->stepper_arrows) {
					shadow_type = GTK_SHADOW_OUT;
					
					if (state_type == GTK_STATE_NORMAL || state_type == GTK_STATE_ACTIVE)
						state_type = GTK_STATE_SELECTED;
				}
				xt = XENO_STYLE_XTHICKNESS(style);
				yt = XENO_STYLE_YTHICKNESS(style);
				x += xt;
				y += yt;
				width -= xt + xt;
				height -= yt + yt;
				fill = FALSE;
			}
			if (!XENO_STYLE_RC_DATA(style)->stepper_arrows)
				solid = TRUE;
		} else if (   !g_strcasecmp(detail, "arrow")
				   && widget && widget->parent && GTK_IS_BUTTON(widget->parent))
		{
			GtkStyle *parent_style;
			
			GTK_WIDGET_STATE(widget->parent);
			parent_style = gtk_widget_get_style (widget->parent);
			if (XENO_STYLE_IS_XENO(parent_style))
				style = parent_style;
			
			if (widget->parent->parent && GTK_IS_COMBO(widget->parent->parent)) {
				t = height + (XENO_STYLE_YTHICKNESS(style) - XENO_STYLE_FONT_DESCENT(style) - 1) * 2;
			} else {
				t = XENO_STYLE_FONT_ASCENT(style);
			}
			if (t < height) {
				y += (height - t)/2;
				height = t;
			}
			fill = FALSE;
			solid = TRUE;
		} else if (!g_strcasecmp ("menuitem", detail)) {
			t = XENO_STYLE_FONT_ASCENT(style) + 1;
			y += (height - t)/2;
			height = t;
			fill = FALSE;
			solid = TRUE;
		}
	}
	
	if (solid) {
		if (fill)
			xeno_style_draw_flat_box (style, window, state_type, shadow_type, area, widget, detail,
									  x, y, width, height);
		
		xeno_style_draw_solid_arrow (style, window, state_type, area, widget, arrow_type,
									 x, y, width, height);
	} else {
		GdkPoint points[4];
		gint half_width, half_height, w;
		
		if (GTK_IS_RANGE(widget)) {
			x++;		y++;
			width-=2;	height-=2;
		} else if (   widget == NULL
				   && detail == NULL
				   && width == 10
				   && height == 10
				   && arrow_type == GTK_ARROW_LEFT)
		{
			/* possibly called from tearoff menu item!
			   if so then draw lines instead...
			*/
			GtkWidget	*window_widget;
			
			gdk_window_get_user_data (window, (gpointer)(&window_widget));
			if (GTK_IS_TEAROFF_MENU_ITEM(window_widget)) {
				y = GTK_CONTAINER(window_widget)->border_width;
				x = y + (GTK_MENU_ITEM(window_widget)->toggle_size > 10)
						? GTK_MENU_ITEM(window_widget)->toggle_size + 3
						: 20;
				y += ((window_widget->allocation.height - y * 2) - XENO_STYLE_YTHICKNESS(style)) / 2;
				
				while (x-5 > 0) {
					xeno_style_draw_hline (style, window, GTK_STATE_NORMAL, NULL, NULL, NULL,
										   x-10, x-5, y);
					x -= 10;
				}
				return;
			}
		}
		
		if (widget) {
			if (   GTK_IS_SPIN_BUTTON(widget)
				|| GTK_IS_NOTEBOOK(widget)
				|| GTK_IS_RANGE(widget))
			{
				if (shadow_type == GTK_SHADOW_IN) {
					shadow_type = (GtkShadowType)XENO_SHADOW_BUTTON_IN;
				} else if (shadow_type == GTK_SHADOW_ETCHED_IN) {
					state_type = GTK_STATE_INSENSITIVE;
					shadow_type = GTK_SHADOW_OUT;
				}
			}
		}
		
		/* finally ... draw arrow */
		width &= 0xfffe;
		height &= 0xfffe;

		w = MIN(width, height);
		x += (width-w)/2;
		y += (height-w)/2;
		width = height = w;
		
		half_width = width / 2;
		half_height = height / 2;

		switch (arrow_type) {
		  case GTK_ARROW_UP:
			points[0].x = x + half_width - 1;
			points[0].y = y;
			points[1].x = x;
			points[1].y = y + height - 1;
			points[2].x = x + width  - 1;
			points[2].y = y + height - 1;
			points[3].x = x + half_width;
			points[3].y = y;
			break;

		  case GTK_ARROW_DOWN:
			points[0].x = x + half_width ;
			points[0].y = y + height     - 1;
			points[1].x = x + width - 1;
			points[1].y = y;
			points[2].x = x;
			points[2].y = y;
			points[3].x = x + half_width - 1;
			points[3].y = y + height     - 1;
			break;

		  case GTK_ARROW_LEFT:
			points[0].x = x;
			points[0].y = y + half_height;
			points[1].x = x + width - 1;
			points[1].y = y + height - 1;
			points[2].x = x + width - 1;
			points[2].y = y;
			points[3].x = x;
			points[3].y = y + half_height - 1;
			break;

		  case GTK_ARROW_RIGHT:
			points[0].x = x + width - 1;
			points[0].y = y + half_height - 1;
			points[1].x = x;
			points[1].y = y;
			points[2].x = x;
			points[2].y = y + height - 1;
			points[3].x = x + width - 1;
			points[3].y = y + half_height;
			break;

		  default:
			return;
		}
		
		xeno_style_draw_polygon (style, window, state_type, XENO_SHADOW_USER(shadow_type), area, widget, detail,
								 points, 4, fill);
	}
}


void
xeno_style_draw_diamond (GtkStyle      *style,
						 GdkWindow     *window,
						 GtkStateType   state_type,
						 GtkShadowType  shadow_type,
						 GdkRectangle  *area,
						 GtkWidget     *widget,
						 xeno_char     *detail,
						 gint           x,
						 gint           y,
						 gint           width,
						 gint           height)
{
	GdkPoint	points[6];
	gint		half_width;
	gint		half_height;

	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	XENO_WINDOW_SIZE(window, width, height);
	
	half_width	= width / 2;
	half_height	= height / 2;

	width = half_width * 2;
	height = half_height * 2 - 1;

	points[0].x = x+half_width-1;	points[0].y = y;
	points[1].x = x;				points[1].y = y+half_height-1;
	points[2].x = x+half_width-1;	points[2].y = y+height-1;
	points[3].x = x+half_width;		points[3].y = y+height-1;
	points[4].x = x+width-1;		points[4].y = y+half_height-1;
	points[5].x = x+half_width;		points[5].y = y;

	xeno_style_draw_polygon (style, window, state_type, XENO_SHADOW_USER(shadow_type), area, widget, detail,
							 points, 3, FALSE);
	xeno_style_draw_polygon (style, window, state_type, XENO_SHADOW_USER(shadow_type), area, widget, detail,
							 points+3, 3, FALSE);
}

void
xeno_style_draw_oval	(GtkStyle      *style,
						 GdkWindow     *window,
						 GtkStateType   state_type,
						 GtkShadowType  shadow_type,
						 GdkRectangle  *area,
						 GtkWidget     *widget,
						 xeno_char     *detail,
						 gint           x,
						 gint           y,
						 gint           width,
						 gint           height)
{
	XenoShadow ctx;
	GdkGC *bg_gc;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	bg_gc = style->bg_gc[state_type];
	if (area)
		gdk_gc_set_clip_rectangle(bg_gc, area);
	
	shadow_type = XENO_SHADOW_USER(shadow_type);
	xeno_shadow_init (&ctx, style, window, state_type, (XenoShadowType)shadow_type, area, widget);
	
	if (ctx.thickness < 2) {
		gdk_draw_arc (window, bg_gc, TRUE, x, y, width, height, 0, 360*64);
		gdk_draw_arc (window, ctx.ring[0].gc[XENO_SHADOW_GC_CORNER], FALSE, x, y, width, height, 15*64, 60*64);
		gdk_draw_arc (window, ctx.ring[0].gc[XENO_SHADOW_GC_CORNER], FALSE, x, y, width, height, -105*64, -60*64);
	} else {
		gdk_draw_arc (window, ctx.ring[1].gc[XENO_SHADOW_GC_TOP], TRUE, x, y, width, height, 50*64, 150*64);
		gdk_draw_arc (window, ctx.ring[0].gc[XENO_SHADOW_GC_CORNER], TRUE, x, y, width, height, 20*64, 30*64);
		gdk_draw_arc (window, ctx.ring[1].gc[XENO_SHADOW_GC_BOTTOM], TRUE, x, y, width, height, 20*64, -150*64);
		gdk_draw_arc (window, ctx.ring[0].gc[XENO_SHADOW_GC_CORNER], TRUE, x, y, width, height, -130*64, -30*64);
		
		gdk_draw_arc (window, ctx.ring[1].gc[XENO_SHADOW_GC_TOP], FALSE, x, y, width, height, 55*64, 20*64);
		gdk_draw_arc (window, ctx.ring[0].gc[XENO_SHADOW_GC_CORNER], FALSE, x, y, width, height, 35*64, 20*64);
		gdk_draw_arc (window, ctx.ring[1].gc[XENO_SHADOW_GC_BOTTOM], FALSE, x, y, width, height, 15*64, 20*64);
		
		gdk_draw_arc (window, ctx.ring[1].gc[XENO_SHADOW_GC_BOTTOM], FALSE, x, y, width, height, -105*64, -20*64);
		gdk_draw_arc (window, ctx.ring[0].gc[XENO_SHADOW_GC_CORNER], FALSE, x, y, width, height, -125*64, -20*64);
		gdk_draw_arc (window, ctx.ring[1].gc[XENO_SHADOW_GC_TOP], FALSE, x, y, width, height, -145*64, -20*64);
		
		gdk_draw_arc (window, bg_gc, TRUE, x+2, y+2, width-4, height-4, 0, 360*64);
	}
	
	gdk_draw_arc (window, ctx.ring[0].gc[XENO_SHADOW_GC_TOP], FALSE, x, y, width, height, 75*64, 120*64);
	gdk_draw_arc (window, ctx.ring[0].gc[XENO_SHADOW_GC_BOTTOM], FALSE, x, y, width, height, 15*64, -120*64);
	
	xeno_shadow_done (&ctx, area);
	
	if (area)
		gdk_gc_set_clip_rectangle(bg_gc, NULL);
}

#if XENO_GTK2
static void
xeno_style_draw_text	(GtkStyle		*style,
						 GdkWindow		*window,
						 GtkStateType	state_type,
						 gboolean		use_text,
						 GdkRectangle	*area,
						 GtkWidget		*widget,
						 xeno_char		*detail,
						 gint			x,
						 gint			y,
						 const gchar	*string,
						 PangoLayout	*layout)
#else
void
xeno_style_draw_string (GtkStyle		*style,
						GdkWindow		*window,
						GtkStateType	state_type,
						GdkRectangle	*area,
						GtkWidget		*widget,
						xeno_char		*detail,
						gint			x,
						gint			y,
						const gchar		*string)
#endif
{
	GdkGC			*fg_gc, *bg_gc, *tmp_gc;
	XenoStyleData	*style_data;
	XenoRcData		*rc_data;
	XenoShadowType	shadow_type;
	
	style_data = XENO_STYLE_DATA(style);
	
  #if XENO_GTK2	
	if (use_text) {
		fg_gc = style->text_gc[state_type];
		goto draw_flat;
	}
	shadow_type = (widget && GTK_IS_FRAME(widget->parent) && widget == GTK_FRAME(widget->parent)->label_widget)
				? XENO_SHADOW_XENO_OUT : XENO_SHADOW_NONE;
  #else
	shadow_type = (detail && g_strcasecmp(detail, "frame") == 0) ? XENO_SHADOW_XENO_OUT : XENO_SHADOW_NONE;
  #endif
	if ((rc_data = XENO_STYLE_RC_DATA(style)) != NULL) {
		if ((rc_data->config_shadows & XENO_CONFIG_SHADOW_TEXT) != 0)
			shadow_type = rc_data->shadow_text[state_type];
	}
	
	switch (shadow_type) {
	  case XENO_SHADOW_XENO_IN:
	  case XENO_SHADOW_XENO_OUT:
		if ((style_data = XENO_STYLE_DATA(style)) != NULL) {
			if ((fg_gc = style_data->white_gc[state_type]) == NULL)
				fg_gc = style->white_gc;
			
			if ((bg_gc = style_data->black_gc[state_type]) == NULL)
				bg_gc = style->black_gc;
		} else {
			fg_gc = style->white_gc;
			bg_gc = style->black_gc;
		}
		break;
		
	  case XENO_SHADOW_THIN_IN:
	  case XENO_SHADOW_THIN_OUT:
	  case XENO_SHADOW_ETCHED_IN:
	  case XENO_SHADOW_ETCHED_OUT:
		fg_gc = style->light_gc[state_type];
		bg_gc = style->dark_gc[state_type];
		break;
		
	  default:
		fg_gc = style->fg_gc[state_type];
		goto draw_flat;
	}
	
	if (XENO_SHADOW_IS_IN(shadow_type)) {
		tmp_gc = fg_gc;
		fg_gc = bg_gc;
		bg_gc = tmp_gc;
	}
	
	if (area)
		gdk_gc_set_clip_rectangle (bg_gc, area);

  #if XENO_GTK2
	if (layout != NULL) {
		gdk_draw_layout (window, bg_gc, x + 1, y + 1, layout);
	} else
  #else
	{
		gdk_draw_string (window, style->font, bg_gc, x + 1, y + 1, string);
	}
  #endif
	if (area)
		gdk_gc_set_clip_rectangle (bg_gc, NULL);
	
  draw_flat:
	if (area)
		gdk_gc_set_clip_rectangle (fg_gc, area);

  #if XENO_GTK2	
	if (layout != NULL) {
		gdk_draw_layout (window, fg_gc, x, y, layout);
	} else
  #else
	{
		gdk_draw_string (window, style->font, fg_gc, x, y, string);
	}
  #endif
	if (area)
		gdk_gc_set_clip_rectangle (fg_gc, NULL);
}

#if XENO_GTK2
void
xeno_style_draw_string (GtkStyle      *style,
						GdkWindow     *window,
						GtkStateType   state_type,
						GdkRectangle  *area,
						GtkWidget     *widget,
						xeno_char     *detail,
						gint           x,
						gint           y,
						const gchar   *string)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (GTK_IS_STYLE(style));
	g_return_if_fail (window != NULL);
	g_return_if_fail (string != NULL);
	
	xeno_style_draw_text (style, window, state_type, FALSE, area, widget, detail,
						  x, y, string, NULL);
}

void
xeno_style_draw_layout	(GtkStyle		*style,
						 GdkWindow		*window,
						 GtkStateType	state_type,
						 gboolean		use_text,
						 GdkRectangle	*area,
						 GtkWidget		*widget,
						 const gchar	*detail,
						 gint			x,
						 gint			y,
						 PangoLayout	*layout)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (GTK_IS_STYLE(style));
	g_return_if_fail (window != NULL);
	g_return_if_fail (layout != NULL);
	
	xeno_style_draw_text (style, window, state_type, use_text, area, widget, detail,
						  x, y, NULL, layout);
}
#endif

void 
xeno_style_draw_box	   (GtkStyle      *style,
						GdkWindow     *window,
						GtkStateType   state_type,
						GtkShadowType  shadow_type,
						GdkRectangle  *area,
						GtkWidget     *widget,
						xeno_char     *detail,
						gint           x,
						gint           y,
						gint           width,
						gint           height)
{
	GtkStyle		*bg_style;
	XenoKnobType	knob_type;
	GtkOrientation	knob_orientation = GTK_ORIENTATION_HORIZONTAL;
	GtkOrientation	knob_centering = GTK_ORIENTATION_HORIZONTAL;
	gint			mouse_x, mouse_y;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	XENO_WINDOW_SIZE(window, width, height);
	
	bg_style = style;
	knob_type = XENO_KNOB_NONE;
	if (widget) {
		if (GTK_IS_BUTTON(widget)) {
			if ((detail) && !g_strcasecmp ("buttondefault", detail)) {
			  #if !XENO_GTK2
				x++; y++;
				width--; height--;
			  #endif
				if (shadow_type == GTK_SHADOW_IN) {
					if (XENO_STYLE_RC_DATA(style))
						shadow_type = XENO_STYLE_RC_DATA(style)->shadow_button_default;
				}
				
				if (state_type != GTK_STATE_INSENSITIVE)
					state_type = GTK_STATE_NORMAL;
				
				if (widget && widget->parent)
					bg_style = gtk_widget_get_style(widget->parent);
			} else if (   GTK_IS_TOGGLE_BUTTON(widget)
					   && ((GtkToggleButton *)widget)->active)
			{
				if (   ((GtkButton *)widget)->button_down
					&& ((GtkButton *)widget)->in_button)
				{
					state_type = GTK_STATE_ACTIVE;
				} else if (   XENO_STYLE_RC_DATA(style)
						   && !XENO_SHADOW_IS_IN(XENO_STYLE_RC_DATA(style)->shadow_in[GTK_STATE_NORMAL]))
				{
					state_type = GTK_STATE_SELECTED;
				} else {
					xeno_style_fill_base (style, window, state_type, area, widget,
											  x, y, width, height);
					goto draw_shadow;
				}
		  #if !XENO_GTK2
			} else if (GTK_IS_OPTION_MENU(widget) && detail && g_strcasecmp(detail, "optionmenu")==0) {
				y++;
				height -= 2;
		  #endif
			} else if (   ((GtkButton *)widget)->relief != GTK_RELIEF_NORMAL
					   && widget->parent)
			{
				bg_style = gtk_widget_get_style(widget->parent);
			}
		} else if (GTK_IS_PANED(widget)) {
			gdk_window_get_size (window, &width, &height);
			x = 0;
			y = 0;
			
			state_type = widget->state;
			gdk_window_get_pointer (window, &mouse_x, &mouse_y, NULL);
			if (mouse_x >= 0 && mouse_x < width && mouse_y >= 0 && mouse_y < height)
				state_type = GTK_STATE_PRELIGHT;
			
			if (XENO_STYLE_RC_DATA(style))
				knob_type = XENO_STYLE_RC_DATA(style)->paned_knobs[state_type];
			
			area = NULL;
			shadow_type = GTK_SHADOW_NONE;
			knob_orientation = GTK_IS_VPANED(widget) ? GTK_ORIENTATION_HORIZONTAL
													 : GTK_ORIENTATION_VERTICAL;
			knob_centering = knob_orientation;
		} else if (   GTK_IS_SCROLLBAR(widget)
				   && (detail && g_strcasecmp(detail, "slider")==0))
		{
			XenoRcData *rc_data;
			
			state_type = widget->state;
			rc_data = XENO_STYLE_RC_DATA(style);
			if (rc_data)
				knob_type = rc_data->scrollbar_knobs[state_type];
			
			knob_orientation = GTK_IS_VSCROLLBAR(widget) ? GTK_ORIENTATION_HORIZONTAL
														 : GTK_ORIENTATION_VERTICAL;
			knob_centering = knob_orientation ^ 1;
		} else if (GTK_IS_PROGRESS(widget) && shadow_type == GTK_SHADOW_OUT) {
			GtkProgressBarOrientation or = ((GtkProgressBar *)widget)->orientation;
			GtkProgressBarStyle		  st = ((GtkProgressBar *)widget)->bar_style;
			
			xeno_style_fill_background (style, window, GTK_STATE_SELECTED, area, widget,
				x, y,
				width  - (st==GTK_PROGRESS_DISCRETE && (or==GTK_PROGRESS_RIGHT_TO_LEFT || or==GTK_PROGRESS_LEFT_TO_RIGHT)),
				height - (st==GTK_PROGRESS_DISCRETE && (or==GTK_PROGRESS_BOTTOM_TO_TOP || or==GTK_PROGRESS_TOP_TO_BOTTOM))
				);
			return;
		}
	}
	
	/* draw already */
	xeno_style_fill_background (bg_style, window, state_type, area, widget, x, y, width, height);
	
  draw_shadow:
	if (shadow_type != XENO_SHADOW_NONE)
		xeno_style_draw_shadow (bg_style, window, state_type, shadow_type, area,
								widget, detail, x, y, width, height);
	
	if (knob_type != XENO_KNOB_NONE) {
		if (shadow_type != XENO_SHADOW_NONE) {
			gint xt, yt;
			xt = XENO_STYLE_XTHICKNESS(style) + 1;
			yt = XENO_STYLE_YTHICKNESS(style) + 1;
			
			x += xt;
			y += yt;
			width  -= xt*2;
			height -= yt*2;
		}
		xeno_style_draw_knob (style, window, state_type, knob_type, area, widget,
							  FALSE, knob_orientation, TRUE, knob_centering,
							  x, y, width, height);
	}
}


void 
xeno_style_draw_flat_box   (GtkStyle      *style,
							GdkWindow     *window,
							GtkStateType   state_type,
							GtkShadowType  shadow_type,
							GdkRectangle  *area,
							GtkWidget     *widget,
							xeno_char     *detail,
							gint           x,
							gint           y,
							gint           width,
							gint           height)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	XENO_WINDOW_SIZE(window, width, height);
	
	if (GTK_IS_RADIO_BUTTON(widget) || GTK_IS_CHECK_BUTTON(widget)) {
		return;
	  /*	
		if ((widget = GTK_BIN(widget)->child) == NULL)
			return;
		
		x = widget->allocation.x - 1;
		y = widget->allocation.y + (widget->allocation.height - widget->requisition.height) / 2;
		width  = widget->requisition.width+2;
		height = widget->requisition.height;
		if (GTK_IS_MISC(widget) && GTK_MISC(widget)->xalign > 0.0)
			width = widget->parent->allocation.width - x;
	  */
	}
	
	if (GTK_IS_EDITABLE(widget)) {
	  #if !XENO_GTK2
		if (!GTK_EDITABLE(widget)->editable && state_type != GTK_STATE_SELECTED)
			state_type = GTK_STATE_INSENSITIVE;
	  #endif
		
		xeno_style_fill_base (style, window, state_type, area, widget, x, y, width, height);
	} else {
		xeno_style_fill_background (style, window, state_type, area, widget, x, y, width, height);
		
		if (detail) {
			if (   GTK_IS_WINDOW(widget)
				&& !g_strcasecmp("base", detail)
				&& !(XENO_STYLE_RC_DATA(style) && XENO_STYLE_RC_DATA(style)->flat_windows))
			{
				xeno_style_draw_shadow (style, window, GTK_STATE_NORMAL,
										(GTK_WINDOW(widget)->type == GTK_WINDOW_POPUP)
										 ? XENO_SHADOW_OUT : XENO_SHADOW_THIN_OUT,
										area, widget, NULL, x, y, width, height);
			} else if (!g_strcasecmp("tooltip", detail)) {
				xeno_style_draw_shadow (style, window, GTK_STATE_NORMAL, XENO_SHADOW_NOTE, area, widget,
										NULL, x, y, width, height);
			}
		}
	}
}

void 
xeno_style_draw_check  (GtkStyle      *style,
						GdkWindow     *window,
						GtkStateType   state_type,
						GtkShadowType  shadow_type,
						GdkRectangle  *area,
						GtkWidget     *widget,
						xeno_char     *detail,
						gint           x,
						gint           y,
						gint           width,
						gint           height)
{
	GdkGC			*bg_gc;
	XenoStyleImageType	image_type;
	gint			xt, yt, size;
	guint			variant;
	gboolean		down=FALSE;
	gboolean		active=FALSE;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (XENO_IS_STYLE(style));
	g_return_if_fail (window != NULL);
	
	if (detail != NULL) {
		bg_gc = style->bg_gc[state_type];
		xt = XENO_STYLE_XTHICKNESS(style);
		yt = XENO_STYLE_YTHICKNESS(style);
		
		if (!g_strcasecmp ("checkbutton", detail)) {
			gint new_width, new_height;
			/*
			  GtkCheckButton allocates:
				x:  allocation.x + indicator sacing + border width + 1
				y:  allocation.y + (allocation.height - indicator size)/2 + 1
				width: indicator size
				height: indicator size
			  
			  Adjustment:
				y -= (new_size - height)/2;
				y += (-height/2) 
			*/
			
			variant = XENO_STYLE_DATA(style)->check_variant;
			size = variant * XENO_STYLE_IMAGE_CHECK_BUTTON_STEP + XENO_STYLE_IMAGE_CHECK_BUTTON_MIN;
			new_width  = size + xt*2 + 2;
			new_height = size + yt*2 + 2;
			
			y -= (new_height + 2 - height)/2;
			
			if (widget)
				state_type = GTK_WIDGET_STATE(widget);
			
			if (shadow_type == GTK_SHADOW_IN) {
				down = TRUE;
				if (state_type == GTK_STATE_ACTIVE)
					state_type = GTK_STATE_NORMAL;
			}
			
			if (   widget && GTK_IS_BUTTON(widget)
				&& ((GtkButton *)widget)->button_down
				&& ((GtkButton *)widget)->in_button)
			{
				active = TRUE;
				state_type = GTK_STATE_ACTIVE;
				shadow_type = GTK_SHADOW_IN;
			}
			
			xeno_style_draw_box (style, window, state_type, shadow_type,
								 area, widget, detail, x, y, new_width, new_height);
			
			if (down) {
				if (active) {
					image_type = XENO_STYLE_IMAGE_CHECK_BUTTON_ACTIVE;
				} else if (state_type == GTK_STATE_INSENSITIVE) {
					image_type = XENO_STYLE_IMAGE_CHECK_BUTTON_INSENSITIVE;
				} else if (state_type == GTK_STATE_PRELIGHT) {
					image_type = XENO_STYLE_IMAGE_CHECK_BUTTON_PRELIGHT;
				} else {
					image_type = XENO_STYLE_IMAGE_CHECK_BUTTON_NORMAL;
				}
				x += xt + 1;
				y += yt + 1;
			}
		} else if (!g_strcasecmp ("check", detail)) {
			variant = XENO_STYLE_DATA(style)->menu_variant;
			size = variant * XENO_STYLE_IMAGE_CHECK_MENU_STEP + XENO_STYLE_IMAGE_CHECK_MENU_MIN;
			y -= (size + 2 - height) / 2;
			
			if (widget && GTK_IS_CHECK_MENU_ITEM(widget)) {
				down = GTK_CHECK_MENU_ITEM(widget)->active;
			} else {
				if (   (shadow_type == GTK_SHADOW_IN  && state_type != GTK_STATE_PRELIGHT)
					|| (shadow_type == GTK_SHADOW_OUT && state_type == GTK_STATE_PRELIGHT))
					down = TRUE;
			}
			
			if (down) {
				if (state_type == GTK_STATE_INSENSITIVE) {
					image_type = XENO_STYLE_IMAGE_CHECK_MENU_INSENSITIVE;
				} else if (state_type == GTK_STATE_PRELIGHT) {
					image_type = XENO_STYLE_IMAGE_CHECK_MENU_PRELIGHT;
				} else {
					image_type = XENO_STYLE_IMAGE_CHECK_MENU_NORMAL;
				}
			}
		} else {
			goto fallback;
		}
		
		if (down) {
			xeno_style_draw_image (style, window, widget, area, image_type, variant,
								   0, 0, x, y, size, size);
		}
		return;
	}
	
  fallback:
	gtk_paint_box (style, window, state_type, shadow_type, area, widget, detail,
				   x, y, width, height);
}


void 
xeno_style_draw_option	(GtkStyle      *style,
						 GdkWindow     *window,
						 GtkStateType   state_type,
						 GtkShadowType  shadow_type,
						 GdkRectangle  *area,
						 GtkWidget     *widget,
						 xeno_char     *detail,
						 gint           x,
						 gint           y,
						 gint           width,
						 gint           height)
{
	XenoStyleImageType image_type;
	guint	variant;
	gint	size;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (XENO_IS_STYLE(style));
	g_return_if_fail (window != NULL);
	
	if (detail != NULL) {
		if (!g_strcasecmp ("radiobutton", detail)) {
			variant = XENO_STYLE_DATA(style)->radio_variant;
			size = variant * XENO_STYLE_IMAGE_RADIO_BUTTON_STEP + XENO_STYLE_IMAGE_RADIO_BUTTON_MIN;
			
			if (shadow_type == GTK_SHADOW_IN) {
				image_type = XENO_STYLE_IMAGE_RADIO_BUTTON_NORMAL_IN;
			} else {
				image_type = XENO_STYLE_IMAGE_RADIO_BUTTON_NORMAL_OUT;
			}
			
			if (state_type == GTK_STATE_INSENSITIVE) {
				image_type += XENO_STYLE_IMAGE_RADIO_BUTTON_INSENSITIVE_OUT - XENO_STYLE_IMAGE_RADIO_BUTTON_NORMAL_OUT;
			} else if (   widget
					   && GTK_IS_BUTTON(widget)
					   && GTK_BUTTON(widget)->button_down
					   && GTK_BUTTON(widget)->in_button)
			{
				image_type = XENO_STYLE_IMAGE_RADIO_BUTTON_ACTIVE_IN;
			} else if (state_type == GTK_STATE_PRELIGHT) {
				image_type += XENO_STYLE_IMAGE_RADIO_BUTTON_PRELIGHT_OUT - XENO_STYLE_IMAGE_RADIO_BUTTON_NORMAL_OUT;
			}
		} else if (!g_strcasecmp ("option", detail)) {
			variant = XENO_STYLE_DATA(style)->menu_variant;
			size = variant * XENO_STYLE_IMAGE_RADIO_MENU_STEP + XENO_STYLE_IMAGE_RADIO_MENU_MIN;
			
			if (shadow_type != GTK_SHADOW_IN)
				return;
			
			switch (state_type) {
			  case GTK_STATE_INSENSITIVE:	image_type = XENO_STYLE_IMAGE_RADIO_MENU_INSENSITIVE; break;
			  case GTK_STATE_PRELIGHT:		image_type = XENO_STYLE_IMAGE_RADIO_MENU_PRELIGHT; break;
			  default:						image_type = XENO_STYLE_IMAGE_RADIO_MENU_NORMAL;
			}
		} else {
			goto fallback;
		}
		
		y -= (size - height)/2;
		
		xeno_style_draw_image (style, window, widget, area, image_type, variant,
							   0, 0, x, y, size, size);
		return;
	}
	
  fallback:
	gtk_paint_diamond (style, window, state_type, shadow_type, area, widget, 
					   detail, x, y, width, height);
}


void 
xeno_style_draw_cross	(GtkStyle		*style,
						 GdkWindow		*window,
						 GtkStateType	 state_type,
						 GtkShadowType	 shadow_type,
						 GdkRectangle	*area,
						 GtkWidget		*widget,
						 xeno_char		*detail,
						 gint			 x,
						 gint			 y,
						 gint			 width,
						 gint			 height)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	g_warning ("xeno_style_draw_cross(): FIXME, this functionality is not implemented in GTK+.");
}

void 
xeno_style_draw_ramp	(GtkStyle		*style,
						 GdkWindow		*window,
						 GtkStateType	 state_type,
						 GtkShadowType	 shadow_type,
						 GdkRectangle	*area,
						 GtkWidget		*widget,
						 xeno_char		*detail,
						 GtkArrowType	 arrow_type,
						 gint			 x,
						 gint			 y,
						 gint			 width,
						 gint			 height)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	g_warning ("xeno_style_draw_ramp(): FIXME, this functionality is not implemented in GTK+.");
}

void
xeno_style_draw_tab	(GtkStyle      *style,
					 GdkWindow     *window,
					 GtkStateType   state_type,
					 GtkShadowType  shadow_type,
					 GdkRectangle  *area,
					 GtkWidget     *widget,
					 xeno_char     *detail,
					 gint           x,
					 gint           y,
					 gint           width,
					 gint           height)
{
	gint xt, yt, tab_width, tab_height, item_height;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if ((detail) && (!g_strcasecmp ("optionmenutab", detail))) {
		xt = XENO_STYLE_XTHICKNESS(style);
		yt = XENO_STYLE_YTHICKNESS(style);
		
		if (XENO_STYLE_RC_DATA(style) && XENO_STYLE_RC_DATA(style)->popup_arrows) {
			gint window_width, window_height;
			
			gdk_window_get_size (window, &window_width, &window_height);
			
			height = window_height - 2 - yt*2;
			width  = (height - 3) | 1;
			x = window_width - 1 - xt - width;
			y = 1 + yt;
			item_height = XENO_STYLE_FONT_HEIGHT(style) + yt*2 + 4;
			tab_height = (MIN(height, item_height) + 2)/3;
			tab_width = width;
			
			xeno_style_draw_line (style, window, state_type, GTK_SHADOW_OUT, area, 
								  x - 2, y, height, GTK_ORIENTATION_VERTICAL);
			
			xeno_style_draw_solid_arrow (style, window, state_type, area, widget, GTK_ARROW_UP,
										 x, y + height/2 - tab_height, tab_width, tab_height);
			
			xeno_style_draw_solid_arrow (style, window, state_type, area, widget, GTK_ARROW_DOWN,
										 x, y + height/2, tab_width, tab_height);
			return;
		}
		
		tab_height = yt*2 + 2 + (height & 1);
		tab_width = xt*2 + 8;
		y += (height - tab_height)/2;
		height = tab_height;
		x += (width - tab_width)/2;
		width = tab_width;
	}
	
	gtk_paint_shadow (style, window, state_type, XENO_SHADOW_USER(shadow_type),
					  area, widget, detail, x, y, width, height);
}

void 
xeno_style_draw_shadow_gap	(GtkStyle       *style,
							 GdkWindow      *window,
							 GtkStateType    state_type,
							 GtkShadowType   shadow_type,
							 GdkRectangle   *area,
							 GtkWidget      *widget,
							 xeno_char      *detail,
							 gint            x,
							 gint            y,
							 gint            width,
							 gint            height,
							 GtkPositionType gap_side,
							 gint            gap_x,
							 gint            gap_width)
{
	GdkPoint points[7];
	int i=0, n=4;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	XENO_WINDOW_SIZE(window, width, height);
	
	switch (gap_side) {
	  case GTK_POS_TOP:
		if (gap_x > 0) {
			n++;
			points[i].x   = x+gap_x;
			points[i++].y = y;
		}
		goto cont_top;
		
	  case GTK_POS_BOTTOM:
	    if (gap_x+gap_width < width) {
			n++;
			points[i].x   = x + gap_x + gap_width;
			points[i++].y = y + height-1;
		}
		goto cont_bottom;
	
	  case GTK_POS_LEFT:
		if (gap_x+gap_width < height) {
			n++;
			points[i].x   = x;
			points[i++].y = y + gap_x + gap_width;
		}
		goto cont_left;
	
	  case GTK_POS_RIGHT:
	    if (gap_x > 0) {
			n++;
			points[i].x   = x+width-1;
			points[i++].y = y+gap_x;
		}
		goto cont_right;
	}

	while(1) {
	  cont_top:
		points[i].x	  = x;
		points[i++].y = y;
		if (i==n) {
			if (gap_x > 0) {
				n++;
				points[i].x = x;
				points[i].y = y + gap_x;
			}
			goto done;
		}

	  cont_left:
		points[i].x	  = x;
		points[i++].y = y+height-1;
		if (i==n) {
			if (gap_x > 0) {
				n++;
				points[i].x = x + gap_x;
				points[i].y = y + height - 1;
			}
			goto done;
		}

	  cont_bottom:
		points[i].x	  = x+width-1;
		points[i++].y = y+height-1;
		if (i==n) {
			if (gap_x + gap_width < width) {
				n++;
				points[i].x = x + width - 1;
				points[i].y = y + gap_x + gap_width;
			}
			goto done;
		}

	  cont_right:
		points[i].x	  = x+width-1;
		points[i++].y = y;
		if (i==n) {
			if (gap_x + gap_width < width) {
				n++;
				points[i].x = x + gap_x + gap_width;;
				points[i].y = y;
			}
			goto done;
		}
	};
  done:
	xeno_style_draw_polygon (style, window, state_type, XENO_SHADOW_USER(shadow_type), area,
							 widget, detail, points, n, FALSE);
	
	return;
}


void 
xeno_style_draw_box_gap	(GtkStyle       *style,
						 GdkWindow      *window,
						 GtkStateType    state_type,
						 GtkShadowType   shadow_type,
						 GdkRectangle   *area,
						 GtkWidget      *widget,
						 xeno_char      *detail,
						 gint            x,
						 gint            y,
						 gint            width,
						 gint            height,
						 GtkPositionType gap_side,
						 gint            gap_x,
						 gint            gap_width)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	XENO_WINDOW_SIZE(window, width, height);
	
	xeno_style_fill_background (style, window, state_type, area, widget, x, y, width, height);
	
	xeno_style_draw_shadow_gap (style, window, state_type, shadow_type, area, widget, detail,
								x, y, width, height, gap_side, gap_x, gap_width);

	return;
}

#define TAB_LIFT	2
#define	TAB_OVERLAP	2

void 
xeno_style_draw_extension	(GtkStyle       *style,
							 GdkWindow      *window,
							 GtkStateType    state_type,
							 GtkShadowType   shadow_type,
							 GdkRectangle   *area,
							 GtkWidget      *widget,
							 xeno_char      *detail,
							 gint            x,
							 gint            y,
							 gint            width,
							 gint            height,
							 GtkPositionType gap_side)
{
	GtkStateType parent_state;
	GdkPoint	 points[8];
	GdkRectangle tab_area, draw_area;
	GtkStyle	 *parent_style;
	gint		 x2, y2;
	gint		 bx, by, bx2, by2, bh, bw;
  #if XENO_CONFIG_TRIANGULAR_TABS
	gboolean	 triangular;
  #endif
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	XENO_WINDOW_SIZE(window, width, height);
	
	tab_area.x = x;
	tab_area.y = y;
	tab_area.width = width;
	tab_area.height = height;
	
	if (area) {
		gdk_rectangle_intersect (&tab_area, area, &draw_area);
		area = &draw_area;
	} else {
		area = &tab_area;
	}
	
	x2 = x + width - 1;
	y2 = y + height - 1;
	
  #if XENO_CONFIG_TRIANGULAR_TABS
	triangular = FALSE;
	
	if (   detail 
		&& !g_strcasecmp(detail, "tab")
		&& (gap_side == GTK_POS_BOTTOM || gap_side == GTK_POS_TOP)
		&& widget
		&& GTK_IS_NOTEBOOK(widget))
	{
		GtkNotebook *notebook;
		gint i, h, t;
		
		t = height - 5 + 2;
		i = t / 3;
		if (!(i > 0))
			goto square;
		
		notebook = GTK_NOTEBOOK(widget);
		if (notebook->tab_hborder == 2) {
			gtk_notebook_set_tab_hborder (notebook, i + 2);
			goto square;
		}
		
		triangular = TRUE;
		h = i*3 + 5;
		bw = i + 5;
	  #if XENO_CONFIG_TRIANGULAR_TABS_OVERLAP
		if (h > height) {
			--x;
			++x2;
			--bw;
		}
	  #endif
	  #if XENO_CONFIG_TRIANGULAR_TABS_OVERLAP
		bh = h - 8;
	  #else
		bh = h - 5;
	  #endif
		if (gap_side == GTK_POS_BOTTOM) {
			y2 = y + h - 1;
			
			points[0].x = x2;		points[0].y = y2;
			points[1].x = x2-i;		points[1].y = y+4;
			points[2].x = x2-i-2;	points[2].y = y+2;
			points[3].x = x2-i-5;	points[3].y = y;
			points[4].x = x+i+5;	points[4].y = y;
			points[5].x = x+i+2;	points[5].y = y+2;
			points[6].x = x+i;		points[6].y = y+4;
			points[7].x = x;		points[7].y = y2;
			
			by = y;
		} else {
			y -= (h - height);
			
			points[0].x = x;		points[0].y = y;
			points[1].x = x+i;		points[1].y = y2-4;
			points[2].x = x+i+2;	points[2].y = y2-2;
			points[3].x = x+i+5;	points[3].y = y2;
			points[4].x = x2-i-5;	points[4].y = y2;
			points[5].x = x2-i-2;	points[5].y = y2-2;
			points[6].x = x2-i;		points[6].y = y2-4;
			points[7].x = x2;		points[7].y = y;
			
			by = y2 - bh + 1;
		}
		bx = x;
		bx2 = x2 - i - 4;
		by2 = by;
		x  += i;
		x2 -= i;
	} else
  #endif
	{
	  square:
		switch (gap_side) {
		  case GTK_POS_BOTTOM:
			points[0].x = x2;	points[0].y = y2;
			points[1].x = x2;	points[1].y = y+5;
			points[2].x = x2-2;	points[2].y = y+2;
			points[3].x = x2-5;	points[3].y = y;
			points[4].x = x+5;	points[4].y = y;
			points[5].x = x+2;	points[5].y = y+2;
			points[6].x = x;	points[6].y = y+5;
			points[7].x = x;	points[7].y = y2;
			bx	= x;	by	= y;
			bx2	= x2-4;	by2	= y;
			break;
	
		  case GTK_POS_TOP:
			points[0].x = x;	points[0].y = y;
			points[1].x = x;	points[1].y = y2-5;
			points[2].x = x+2;	points[2].y = y2-2;
			points[3].x = x+5;	points[3].y = y2;
			points[4].x = x2-5;	points[4].y = y2;
			points[5].x = x2-2;	points[5].y = y2-2;
			points[6].x = x2;	points[6].y = y2-5;
			points[7].x = x2;	points[7].y = y;
			bx	= x;	by	= y2-4;
			bx2 = x2-4;	by2	= y2-4;
			break;
			
		  case GTK_POS_RIGHT:
			points[0].y = y;	points[0].x = x2;
			points[1].y = y;	points[1].x = x+5;
			points[2].y = y+2;	points[2].x = x+2;
			points[3].y = y+5;	points[3].x = x;
			points[4].y = y2-5;	points[4].x = x;
			points[5].y = y2-2;	points[5].x = x+2;
			points[6].y = y2;	points[6].x = x+5;
			points[7].y = y2;	points[7].x = x2;
			bx	= x;	by	= y;
			bx2 = x;	by2	= y2-4;
			break;
			
		  case GTK_POS_LEFT:
			points[0].y = y2;	points[0].x = x;
			points[1].y = y2;	points[1].x = x2-5;
			points[2].y = y2-2;	points[2].x = x2-2;
			points[3].y = y2-5;	points[3].x = x2;
			points[4].y = y+5;	points[4].x = x2;
			points[5].y = y+2;	points[5].x = x2-2;
			points[6].y = y;	points[6].x = x2-5;
			points[7].y = y;	points[7].x = x;
			bx	= x2-4;	by	= y;
			bx2 = x2-4;	by2	= y2-4;
			break;
		
		  default:
			return;
		}
		bh = bw = 5;
	}
	
	parent_style = style;
	parent_state = GTK_STATE_NORMAL;
	if (widget) {
		if (widget->state == GTK_STATE_INSENSITIVE)
			state_type = GTK_STATE_INSENSITIVE;
		
		if (widget->parent) {
			parent_style = widget->parent->style;
			parent_state = widget->parent->state;
		}
	}
	
	xeno_style_fill_background (parent_style, window, parent_state, area, widget, bx, by, bw, bh);
	xeno_style_fill_background (parent_style, window, parent_state, area, widget, bx2, by2, bw, bh);
	
	xeno_style_draw_polygon (style, window, state_type, XENO_SHADOW_USER(shadow_type),
							 area, widget, detail, points, 8, TRUE);
	
	if (!xeno_theme_pseudocolor && XENO_STYLE_IS_XENO(style)) {
		XenoStyleImageType image_type;
		XenoPixmap		*pixmap;
		XenoPixmapMask	*mask;
		GdkGC			*bg_gc;
		
		switch (state_type) {
		  case GTK_STATE_ACTIVE:
			image_type = XENO_STYLE_IMAGE_NOTEBOOK_TAB_ACTIVE;
			break;
			
		  case GTK_STATE_INSENSITIVE:                              
			image_type = XENO_STYLE_IMAGE_NOTEBOOK_TAB_INSENSITIVE;
			break;
			
		  default:
			image_type = XENO_STYLE_IMAGE_NOTEBOOK_TAB_NORMAL;
		}
		
		pixmap = xeno_style_pixmap_get (style, parent_style, image_type, 0, &mask);
		if (pixmap) {
			bg_gc = style->bg_gc[state_type];
			
			if (gap_side == GTK_POS_BOTTOM || gap_side == GTK_POS_RIGHT)
				xeno_draw_pixmap (window, bg_gc, area, pixmap, mask, 0, 0, x, y, 7, 7);
			
			if (gap_side == GTK_POS_BOTTOM || gap_side == GTK_POS_LEFT)
				xeno_draw_pixmap (window, bg_gc, area, pixmap, mask, 6, 0, x2-6, y, 7, 7);
			
			if (gap_side == GTK_POS_TOP || gap_side == GTK_POS_RIGHT)
				xeno_draw_pixmap (window, bg_gc, area, pixmap, mask, 0, 6, x, y2-6, 7, 7);
			
			if (gap_side == GTK_POS_TOP || gap_side == GTK_POS_LEFT)
				xeno_draw_pixmap (window, bg_gc, area, pixmap, mask, 6, 6, x2-6, y2-6, 7, 7);
			
		  #if XENO_CONFIG_TRIANGULAR_TABS
			if (triangular) {
				if (gap_side == GTK_POS_BOTTOM) {
					y += 5;
					x2 -= 3;
					do {
						x  -= 2;
						x2 += 2;
						xeno_draw_pixmap (window, bg_gc, area, pixmap, mask, 13,0, x,y,  4,6);
						xeno_draw_pixmap (window, bg_gc, area, pixmap, mask, 21,0, x2,y, 4,6);
						y += 6;
					} while (y < y2);
				} else if (gap_side == GTK_POS_TOP) {
					y2 -= 10;
					x2 -= 3;
					do {
						x  -= 2;
						x2 += 2;
						xeno_draw_pixmap (window, bg_gc, area, pixmap, mask, 13,6, x,y2,  4,6);
						xeno_draw_pixmap (window, bg_gc, area, pixmap, mask, 21,6, x2,y2, 4,6);
						y2 -= 6;
					} while (y2 + 5 > y);
				}
			}
		  #endif
		} else {
			g_warning("no pixmap for notebook corner\n");
		}
	}
}


void 
xeno_style_draw_focus	(GtkStyle      *style,
						 GdkWindow     *window,
						 GdkRectangle  *area,
						 GtkWidget     *widget,
						 xeno_char     *detail,
						 gint           x,
						 gint           y,
						 gint           width,
						 gint           height)
{
	GdkGC *gc;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if (widget && GTK_WIDGET_STATE(widget) == GTK_STATE_INSENSITIVE)
		return;
	
	if ((width == -1) && (height == -1)) {
		gdk_window_get_size (window, &width, &height);
		width -= 1;
		height -= 1;
	} else if (width == -1) {
		gdk_window_get_size (window, &width, NULL);
		width -= 1;
	} else if (height == -1) {
		gdk_window_get_size (window, NULL, &height);
		height -= 1;
	}
	
	if (XENO_STYLE_HAS_DATA(style)) {
		gc = XENO_STYLE_DATA(style)->focus_gc;
	} else {
		gc = style->black_gc;
	}
	
	if (area)
		gdk_gc_set_clip_rectangle (gc, area);
	
	if ((detail) && !g_strcasecmp (detail, "add-mode")) {
		gdk_gc_set_line_attributes (gc, 1, GDK_LINE_ON_OFF_DASH, 0, 0);
		gdk_gc_set_dashes (gc, 0, "\4\4", 2);
		
		gdk_draw_rectangle (window, gc, FALSE, x, y, width, height);
		
		gdk_gc_set_line_attributes (gc, 1, GDK_LINE_SOLID, 0, 0);
	} else if ((detail) && !g_strcasecmp (detail, "trough")) {
		x += 1;
		y += 1;
		width -= 1;
		height -= 1;
	} else {
		if (widget) {
			if (   GTK_IS_CHECK_BUTTON(widget) || GTK_IS_RADIO_BUTTON(widget)
				|| (detail && !g_strcasecmp("tab", detail)))
			{
				/* draw dotted frame */
				static const gchar data[] = { 0x01, 0x02 };
				static GdkBitmap *bm = NULL;
				
				if (!bm)
					bm = gdk_bitmap_create_from_data (window, data, 2, 2);
				
				if (bm) {
					gdk_gc_set_stipple(gc, bm);
					gdk_gc_set_fill(gc, GDK_STIPPLED);
					gdk_gc_set_ts_origin(gc, x, y);
					
					gdk_draw_rectangle (window, gc, FALSE, x, y, width & ~1, height & ~1);
					
					gdk_gc_set_fill(gc, GDK_SOLID);
					goto done;
				}
			} else if (GTK_IS_OPTION_MENU(widget)) {
				y++;
				height-=2;
			} else if (GTK_IS_EDITABLE(widget)) {
				int t;
				xeno_style_draw_shadow (style, window, widget->state, XENO_SHADOW_IN, area, widget, detail,
										x,y, width+1,height+1);
			  #if XENO_GTK2
				t = 1;
			  #else
				t = (GTK_IS_TEXT(widget)) ? 2 : 1;
			  #endif
				x += t;
				y += t;
				t <<= 1;
				width -= t;
				height -= t;
			} else if (   GTK_IS_TOGGLE_BUTTON(widget)
					   && (   GTK_TOGGLE_BUTTON(widget)->active
						   || widget->state == GTK_STATE_ACTIVE))
			{
				gint xt, yt;
				
				xeno_style_draw_shadow (style, window, widget->state, GTK_SHADOW_IN, area, widget, detail,
										x,y, width+1,height+1);
				xt = XENO_STYLE_XTHICKNESS(style);
				yt = XENO_STYLE_YTHICKNESS(style);
				
				x += xt;
				y += yt;
				width -= xt * 2;
				height -= yt * 2;
			}
		}
		
		gdk_draw_rectangle (window, gc, FALSE, x, y, width, height);
	}
	
  done:
	if (area)
		gdk_gc_set_clip_rectangle (gc, NULL);
}


void 
xeno_style_draw_slider	(GtkStyle		*style,
						 GdkWindow		*window,
						 GtkStateType	 state_type,
						 GtkShadowType	 shadow_type,
						 GdkRectangle	*area,
						 GtkWidget		*widget,
						 xeno_char		*detail,
						 gint			 x,
						 gint			 y,
						 gint			 width,
						 gint			 height,
						 GtkOrientation	 orientation)
{
	GdkGC *lgc, *dgc, *mgc;
	gint i, w, t, xt, yt;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	XENO_WINDOW_SIZE(window, width, height);
	
	xt = MAX(0, XENO_STYLE_XTHICKNESS(style) - 1);
	yt = MAX(0, XENO_STYLE_YTHICKNESS(style) - 1);
	
	t = 0;
	if (widget) {
		if (GTK_WIDGET_HAS_FOCUS(widget)) {
			t = 1;
			xt++;
			yt++;
			xeno_style_draw_focus (style, window, area, widget, detail, x, y, width-1, height-1);
		}
		state_type = widget->state;
	}
	
	gtk_paint_box (style, window, state_type, shadow_type,
				   area, widget, detail, x+t, y+t, width-t-t, height-t-t);
	
	lgc = style->light_gc[state_type];
	dgc = style->dark_gc[state_type];
	mgc = style->mid_gc[state_type];
	
	if (area) {
		gdk_gc_set_clip_rectangle (lgc, area);
		gdk_gc_set_clip_rectangle (dgc, area);
		gdk_gc_set_clip_rectangle (mgc, area);
	}
	
	if (orientation == GTK_ORIENTATION_HORIZONTAL) {
		w = MIN(width, height + width/6);
		x = x + width/2 - w/2;
		y += yt;
		height -= yt + yt;
		for (i = x; i < x + w; i += 3) {
			gdk_draw_line (window, dgc,		i,  	y+1,	i,   	y+height-1);
			gdk_draw_line (window, lgc,		i+1,	y,		i+1,	y+height-2);
			gdk_draw_point (window, mgc,	i,		y);
			gdk_draw_point (window, mgc,	i+1,	y+height-1);
		}
	} else {
		w = MIN(height, width + height/6);
		y = y + height/2 - w/2;
		x += xt;
		width -= xt + xt;
		for (i = y; i < y + w; i += 3) {
			gdk_draw_line (window, dgc,		x+1,		i,		x+width-1,	i  );
			gdk_draw_line (window, lgc,		x,			i+1,	x+width-2,	i+1);
			gdk_draw_point (window, mgc,	x,			i);
			gdk_draw_point (window, mgc,	x+width-1,	i+1);
		}
	}
	if (area) {
		gdk_gc_set_clip_rectangle (lgc, NULL);
		gdk_gc_set_clip_rectangle (dgc, NULL);
		gdk_gc_set_clip_rectangle (mgc, NULL);
	}
}

void
xeno_style_draw_handle	(GtkStyle      *style,
						 GdkWindow     *window,
						 GtkStateType   state_type,
						 GtkShadowType  shadow_type,
						 GdkRectangle  *area,
						 GtkWidget     *widget,
						 xeno_char     *detail,
						 gint           x,
						 gint           y,
						 gint           width,
						 gint           height,
						 GtkOrientation orientation)
{
	XenoKnobType knob_type;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	XENO_WINDOW_SIZE(window, width, height);
	
	knob_type = XENO_KNOB_NONE;
	if (XENO_STYLE_RC_DATA(style))
		knob_type = XENO_STYLE_RC_DATA(style)->handle_knob;
	
	if (   knob_type >= XENO_KNOB_NS_BUDS_IN
		&& detail
		&& (   !g_strcasecmp(detail, "handlebox")
			|| !g_strcasecmp(detail, "dockitem")))
	{
		GdkGC *light_gc, *dark_gc, *mid_gc, *white_gc;
		
		light_gc	= style->light_gc[state_type];
		dark_gc		= style->dark_gc[state_type];
		mid_gc		= style->mid_gc[state_type];
		white_gc	= style->white_gc;
		
		gdk_gc_set_clip_rectangle (mid_gc, area);
		gdk_gc_set_clip_rectangle (light_gc, area);
		if (orientation == GTK_ORIENTATION_VERTICAL) {
			gdk_draw_line(window, light_gc, x+width-1, y, x+width-1, y+height-2);
			gdk_draw_point(window, mid_gc, x+width-1, y+height-1);
			width--;
		} else {
			gdk_draw_line(window, light_gc, x+1, y+height-1, x+width-2, y+height-1);
			gdk_draw_point(window, mid_gc, x+width-1, y+height-1);
			height--;
		}
		gdk_gc_set_clip_rectangle (light_gc, NULL);
		gdk_gc_set_clip_rectangle (mid_gc, NULL);
		
		shadow_type = XENO_SHADOW_THIN_OUT;
	} else if (knob_type == XENO_KNOB_SHADOW_OUT || knob_type == XENO_KNOB_SHADOW_THIN_OUT) {
		shadow_type = (GtkShadowType)knob_type;
		knob_type = XENO_KNOB_NONE;
		if (state_type != GTK_STATE_INSENSITIVE)
			state_type = GTK_STATE_PRELIGHT;
	} else {
		shadow_type = XENO_SHADOW_NONE;
	}
	
	if (shadow_type != XENO_SHADOW_NONE)
		xeno_style_draw_box (style, window, state_type, shadow_type, area, widget, NULL,
							 x, y, width, height);
	
	if (knob_type != XENO_KNOB_NONE) {
		gboolean fill = FALSE;
		
		if (knob_type <= XENO_KNOB_SHADOW_LAST) {
			switch (state_type) {
			  case GTK_STATE_NORMAL:
			  case GTK_STATE_ACTIVE:
				state_type = GTK_STATE_PRELIGHT;
		
				break;
			  default:
				;
			}
			fill = TRUE;
		}
		
		xeno_style_draw_knob (style, window, state_type, knob_type, area, widget,
							  fill, orientation, FALSE, orientation,
							  x+2, y+2, width-4, height-4);
	}
}


/*** Decoration ***************************************************************/

static void
xeno_style_draw_lines	(GtkStyle		*style,
						 GdkWindow		*window,
						 GtkStateType	state_type,
						 gboolean		in,
						 GdkRectangle	*area,
						 GtkWidget		*widget,
						 gint			x,
						 gint			y,
						 gint			width,
						 gint			height,
						 GtkOrientation	orientation)
{
	GdkGC	*light_gc, *dark_gc, *mid_gc;
	gint	i;

	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if (!in) {
		light_gc = style->light_gc[state_type];
		dark_gc  = style->dark_gc[state_type];
	} else {
		light_gc = style->dark_gc[state_type];
		dark_gc	 = style->light_gc[state_type];
	}
	mid_gc = style->mid_gc[state_type];
	
	if (area) {
		gdk_gc_set_clip_rectangle (light_gc, area);
		gdk_gc_set_clip_rectangle (dark_gc, area);
		gdk_gc_set_clip_rectangle (mid_gc, area);
	}
	
	if (orientation == GTK_ORIENTATION_HORIZONTAL) {
		for (i = y + ((height % 3) & 0x01); i < y + height; i += 3) {
			gdk_draw_line  (window, light_gc,	x,			i,		x+width-2,	i  );
			gdk_draw_line  (window, dark_gc,	x+1, 		i+1,	x+width-1,	i+1);
			gdk_draw_point (window, mid_gc,		x,			i+1);
			gdk_draw_point (window, mid_gc,		x+width-1,	i);
		}
	} else {
		for (i = x + ((width % 3) & 0x01); i < x + width; i += 3) {
			gdk_draw_line  (window, light_gc,	i,		y,   	i,   y+height-2);
			gdk_draw_line  (window, dark_gc,	i+1,	y+1, 	i+1, y+height-1);
			gdk_draw_point (window, mid_gc,		i+1,	y);
			gdk_draw_point (window, mid_gc,		i,		y+height-1);
		}
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (mid_gc, NULL);
		gdk_gc_set_clip_rectangle (dark_gc, NULL);
		gdk_gc_set_clip_rectangle (light_gc, NULL);
	}
}

static void
xeno_style_draw_buds   (GtkStyle		*style,
						GdkWindow		*window,
						GtkStateType	state_type,
						gboolean		in,
						GdkRectangle	*area,
						GtkWidget		*widget,
						gint			x,
						gint			y,
						gint			width,
						gint			height,
						GtkOrientation	orientation,
						gboolean		ns_style)
{
	GdkGC	*light_gc, *dark_gc, *mid_gc;
	gint	x2, y2;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if (!in) {
		light_gc = style->white_gc;
		mid_gc	 = style->mid_gc[state_type];
		dark_gc  = style->dark_gc[state_type];
	} else {
		light_gc = style->dark_gc[state_type];;
		mid_gc = style->mid_gc[state_type];
		dark_gc	 = style->light_gc[state_type];
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (dark_gc, area);
		gdk_gc_set_clip_rectangle (light_gc, area);
		gdk_gc_set_clip_rectangle (mid_gc, area);
	}
	
	if (orientation == GTK_ORIENTATION_VERTICAL && ns_style) {
		/* netscape - style */
		for (y2 = y+1; y2 < y+height-1; y2 += 3) {
			for (x2 = x; x2 < x+width-1; x2 += 6) {
				gdk_draw_point (window, light_gc,	x2,		y2);
				gdk_draw_point (window, dark_gc,	x2 + 1, y2 + 1);
				gdk_draw_point (window, mid_gc,		x2 + 1,	y2);
				gdk_draw_point (window, mid_gc,		x2,		y2 + 1);
			}
		}
		for (y2 = y; y2 < y+height-1; y2 += 3) {
			for (x2 = x+3; x2 < x+width-1; x2 += 6) {
				gdk_draw_point (window, light_gc,	x2,		y2    );
				gdk_draw_point (window, dark_gc,	x2 + 1, y2 + 1);
				gdk_draw_point (window, mid_gc,		x2 + 1,	y2    );
				gdk_draw_point (window, mid_gc,		x2,		y2 + 1);
			}
		}
	} else {
		/* mac - style */
		x += (width % 3) & 0x01;
		y += (height% 3) & 0x01;
		
		for (y2 = y; y2 < y+height-1; y2 += 3) {
			for (x2 = x; x2 < x+width-1; x2 += 3) {
				gdk_draw_point (window, light_gc,	x2,		y2	);
				gdk_draw_point (window, mid_gc,		x2+1,	y2	);
				gdk_draw_point (window, mid_gc,		x2,		y2+1);
				gdk_draw_point (window, dark_gc,	x2+1,	y2+1);
			}
		}
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (mid_gc, NULL);
		gdk_gc_set_clip_rectangle (light_gc, NULL);
		gdk_gc_set_clip_rectangle (dark_gc, NULL);
	}
}

static void
xeno_style_draw_knob	(GtkStyle		*style,
						 GdkWindow		*window,
						 GtkStateType	state_type,
						 XenoKnobType	knob_type,
						 GdkRectangle	*area,
						 GtkWidget		*widget,
						 gboolean		fill,
						 GtkOrientation	orientation,
						 gboolean		center,
						 GtkOrientation	centering,
						 gint			x,
						 gint			y,
						 gint			width,
						 gint			height)
{
	if (fill)
		xeno_style_fill_background (style, window, state_type, area, widget, x, y, width, height);
	
	if (knob_type <= XENO_KNOB_SHADOW_LAST) {
		xeno_style_draw_shadow (style, window, state_type, (XenoShadowType)knob_type, area, widget, NULL,
								x, y, width, height);
	} else if (knob_type == XENO_KNOB_DIMPLE_IN || knob_type == XENO_KNOB_DIMPLE_OUT) {
		XenoStyleImageType image_type;
		
		x += (width  - XENO_STYLE_IMAGE_DIMPLE_SIZE)/2;
		y += (height - XENO_STYLE_IMAGE_DIMPLE_SIZE)/2;
		
		switch (state_type) {
		  case GTK_STATE_PRELIGHT: image_type = XENO_STYLE_IMAGE_DIMPLE_IN_PRELIGHT; break;
		  case GTK_STATE_INSENSITIVE: image_type = XENO_STYLE_IMAGE_DIMPLE_IN_INSENSITIVE; break;
		  default: image_type = XENO_STYLE_IMAGE_DIMPLE_IN_NORMAL;
		}
		if (knob_type == XENO_KNOB_DIMPLE_OUT)
			image_type += 3;
		
		xeno_style_draw_image (style, window, widget, area, image_type, 0,
							   0, 0, x, y, XENO_STYLE_IMAGE_DIMPLE_SIZE, XENO_STYLE_IMAGE_DIMPLE_SIZE);
	} else {
		gint w, h;
		
		w = width;
		h = height;
		if (center) {
			if (centering == GTK_ORIENTATION_VERTICAL) {
				h = MIN(height, width + height/7);
				y = y + (height - h)/2;
			} else {
				w = MIN(width, width/7 + height);
				x = x + (width - w)/2;
			}
		}
		
		switch (knob_type) {
		  case XENO_KNOB_LINES_IN:
		  case XENO_KNOB_LINES_OUT:
			xeno_style_draw_lines (style, window, state_type, knob_type & 1, area, widget,
								   x, y, w, h, orientation);
			break;
			
		  case XENO_KNOB_BUDS_IN:
		  case XENO_KNOB_BUDS_OUT:
			xeno_style_draw_buds (style, window, state_type, knob_type & 1, area, widget,
								  x, y, w, h, orientation, FALSE);
			break;
		  
		  case XENO_KNOB_NS_BUDS_IN:
		  case XENO_KNOB_NS_BUDS_OUT:
			xeno_style_draw_buds (style, window, state_type, knob_type & 1, area, widget,
								  x, y, w, h, orientation, TRUE);
		  default:
			break;
		}
	}
}


/*
 *	Images
 */

static void
xeno_style_draw_image	(XenoStyle		*style,
						 GdkWindow		*window,
						 GtkWidget		*widget,
						 GdkRectangle	*area,
						 XenoStyleImageType	image_type,
						 guint			variant_type,
						 gint			src_x,
						 gint			src_y,
						 gint			x,
						 gint			y,
						 gint			width,
						 gint			height)
{
	XenoPixmap		*pixmap;
	XenoPixmapMask	*mask;
	GtkStyle		*parent_style;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (XENO_IS_STYLE(style));
	g_return_if_fail (window != NULL);
	
	parent_style = (GtkStyle *)style;
	if (widget != NULL) {
		if (widget->parent != NULL)
			parent_style = gtk_widget_get_style (widget->parent);
	}
	
	pixmap = xeno_style_pixmap_get (style, parent_style, image_type, variant_type, &mask);
	if (pixmap) {
		xeno_draw_pixmap (window, style->bg_gc[widget->state], area,
						  pixmap, mask, src_x, src_y, x, y, width, height);
	}
}


/* end */

