#!/bin/bash
#
# Create certificates suitable for testing of Webfsd.
#
# Copyright 2010: Mats Erik Andersson <debian@gisladisker.se>
# Licensed under GPL v2.

BITLEN=${BITS:-1024}
KEYFILE=${KEY:-key.pem}
HOST_NAME=${HOST:-$HOSTNAME}
CANAME=${CANAME:-webfs.local}

CAFILE=${CAFILE:-ca.pem}

[ $BITLEN -lt 384 ] && BITLEN=384

if [ "$1" = "-h" -o "$1" = "--help" -o "$1" = "help" ]; then
	cat <<-END_TEXT
		Full usage:   BITS=1024 KEY=key.pem HOST=\$HOSTNAME \\
		              CANAME=webfs.local CAFILE=ca.pem  $0  [names]

		Simple usage: $0  [names]

		The defaults used in the latter case are as indicated above.

		If non-empty, 'names' is a list of strings. For each of these
		a client certificate will be created, each signed with the same
		ca-certificate as is used for the server certificate. When
		empty, the string "localhost" is used.

		The resulting certificate 'server.pem' can directly be used
		with Webfs as in '-S -C ./server.pem'. Observer that this
		file contains certificate _and_ key, so it must be treated as
		confidential information.

		Warning: All these files are intended as test material, as they
		are all using one and the same cryptographic key.

		END_TEXT
	exit 0;
fi

if [ -z "$(which certtool)" ]; then
	echo -e "\nThe tool \"certtool\" is missing."
	echo -e "You need to install \"gnutls-cli\".\n"
	exit 1
fi

if [ -r "$KEYFILE" ]; then
	echo "The previous keyfile will be used."
else
	echo -e "Creating a crypto key.\n\t$KEYFILE"
	certtool --generate-privkey \
			 --bits $BITLEN \
			 --outfile $KEYFILE 2>/dev/null
	NYCKELBYGGE=ja
	chmod 600 $KEYFILE
fi

cat > ca.info <<-SLUT
	cn = $CANAME
	ca
	cert_signing_key
SLUT

cat > subca.info <<-SLUT
	cn = sub-$CANAME
	ca
	cert_signing_key
SLUT

cat > client-ca.info <<-SLUT
	cn = client-$CANAME
	ca
	cert_signing_key
SLUT

cat > server.info <<-SLUT
	organization = $CANAME
	cn = $HOST_NAME
	tls_www_server
	encryption_key
	signing_key
SLUT

for nn in "${@-localhost}"; do
	NN=$(echo -n $nn | tr ' ' '_')
	cat > $NN.info <<-SLUT
		organization = $CANAME
		cn = $nn
		tls_www_client
		encryption_key
		signing_key
	SLUT
	cat > pre-$NN.info <<-SLUT
		organization = $CANAME
		cn = $nn
		ca
		cert_signing_key
		tls_www_server
	SLUT
done

if [ -n "$NYCKELBYGGE" -o -n "$CA" -o ! -r "$CAFILE" ]; then
	echo -e "Creating certificate for authority.\n\t$CAFILE"
	certtool --generate-self-signed \
			--load-privkey $KEYFILE \
			--template ca.info \
			--outfile $CAFILE 2>/dev/null
else
	echo "Using old \"$CAFILE\"."
fi

# Create an intermediary ca-certificate.
echo -e "Creating certificate for sub-authority.\n\tsub-$CAFILE"
certtool --generate-certificate \
			--load-ca-certificate $CAFILE \
			--load-ca-privkey $KEYFILE \
			--load-privkey $KEYFILE \
			--template subca.info \
			--outfile sub-$CAFILE  2>/dev/null

echo -e "Creating chain for ca and sub-ca.\n\tca-chain.pem"
cat $CAFILE sub-$CAFILE > ca-chain.pem

echo "Creating certificate for Webfs."
echo -e "\tCN: $HOST_NAME"
echo -e "\twebfs.pem"
certtool --generate-certificate \
			--load-ca-certificate $CAFILE \
			--load-ca-privkey $KEYFILE \
			--load-privkey $KEYFILE \
			--template server.info \
			--outfile webfs.pem  2>/dev/null

# Create a second level certificate for the server.
echo "Creating secondary certificate for Webfs."
echo -e "\tCN: $HOST_NAME"
echo -e "\twebfs.pem"
certtool --generate-certificate \
			--load-ca-certificate sub-$CAFILE \
			--load-ca-privkey $KEYFILE \
			--load-privkey $KEYFILE \
			--template server.info \
			--outfile webfs2.pem  2>/dev/null

echo "Creating combined certificate with key:"
echo -e "\tserver.pem"
cat key.pem webfs.pem > server.pem
chmod 600 server.pem
echo -e "\tserver2.pem"
cat key.pem webfs2.pem > server2.pem
chmod 600 server2.pem

echo -e "Creating client side, secondary CA.\n\tclient-$CAFILE"
certtool --generate-certificate \
			--load-ca-certificate $CAFILE \
			--load-ca-privkey $KEYFILE \
			--load-privkey $KEYFILE \
			--template client-ca.info \
			--outfile client-$CAFILE  2>/dev/null

echo -e "Creating client side CA-chain.\n\tclient-chain.pem"
cat $CAFILE client-$CAFILE > client-chain.pem

echo "Creating a client certificate."
for nn in "${@-localhost}"; do
	NN=$(echo -n $nn | tr ' ' '_')
	echo -en "\t$NN.pem"
	certtool --generate-certificate \
			--load-ca-certificate $CAFILE \
			--load-ca-privkey $KEYFILE \
			--load-privkey $KEYFILE \
			--template $NN.info \
			--outfile $NN.pem  2>/dev/null
	echo -e "  ${NN}2.pem"
	certtool --generate-certificate \
			--load-ca-certificate client-$CAFILE \
			--load-ca-privkey $KEYFILE \
			--load-privkey $KEYFILE \
			--template $NN.info \
			--outfile ${NN}2.pem  2>/dev/null
done

rm *.info
