/*=========================================================================

  Program:   Visualization Toolkit
  Module:    $RCSfile: vtkQuadraturePointsGenerator.h,v $

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkQuadraturePointsGenerator
// .SECTION Description
//
// Create a vtkPolyData on its output containing the vertices 
// for the quadrature points for one of the vtkDataArrays present
// on its input vtkUnstructuredGrid. If the input data set has
// has FieldData generated by vtkQuadraturePointInterpolator then
// this will be set as point data. Note: Point sets are generated
// per field array. This is because each field array may contain
// its own dictionary.
// 
// .SECTION See also
// vtkQuadraturePointInterpolator, vtkQuadratureSchemeDefinition, vtkInformationQuadratureSchemeDefinitionVectorKey

#ifndef vtkQuadraturePointsGenerator_h
#define vtkQuadraturePointsGenerator_h

#include "vtkDataSetAlgorithm.h"

class vtkPolyData;
class vtkUnstructuredGrid;
class vtkInformation;
class vtkInformationVector;

class VTK_GRAPHICS_EXPORT vtkQuadraturePointsGenerator : public vtkDataSetAlgorithm
{
public:
  vtkTypeRevisionMacro(vtkQuadraturePointsGenerator,vtkDataSetAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent);
  static vtkQuadraturePointsGenerator *New();
  // Description:
  // Set/Get the source array. The source array provides
  // the dictionary from which the quadrature points
  // are generated.
  vtkGetStringMacro(SourceArrayName);
  void SetSourceArrayName(const char *name);

protected:
  int FillInputPortInformation(int port, vtkInformation *info);
  int FillOutputPortInformation(int port, vtkInformation *info);
  int RequestData(vtkInformation *req, vtkInformationVector **input, vtkInformationVector *output);
  vtkQuadraturePointsGenerator();
  virtual ~vtkQuadraturePointsGenerator();

private:
  vtkQuadraturePointsGenerator(const vtkQuadraturePointsGenerator &); // Not implemented
  void operator=(const vtkQuadraturePointsGenerator &); // Not implemented
  //
  void Clear();
  // Description:
  // Generate field data arrays that have all scalar/vector
  // fields interpolated to the quadrature points. The type
  // of quadrature used is found in the dictionary stored as
  // meta data in each data array.
  int Generate(vtkUnstructuredGrid *usgIn,
                char *sourceArrayName,
                vtkPolyData *pdOut);
  //
  int GetSourceArrayNameFromAlgorithm(vtkInformationVector **inputVector);
  // State:
  char *SourceArrayName;
  int HasSourceArrayName;
};

#endif
