
// vbvolregress.cpp
// carry out volume regression using regular GLM
// Copyright (c) 2007-2009 by The VoxBo Development Team

// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
//
// original version written by Dan Kimberg


// replace IND in the output file name if it's a perm
// if -o is omitted, generate statmap_contrastname.suffix, params_contrastname.suffix45, and resids_contrastname.suffix4d
// if -prm is there, generate params, if -res is there generate res
// if -prm=xxx is there, use xxx, same with res, but still replace IND if it's a perm


using namespace std;

#include <stdio.h>
#include <string.h>
#include <sstream>
#include "vbutil.h"
#include "vbio.h"
#include "glmutil.h"
#include "makestatcub.h"
#include "vbvolregress.hlp.h"

void vbvolregress_help();
void vbvolregress_version();

int
main(int argc,char *argv[])
{
  if (argc<2) {
    vbvolregress_help();
    exit(0);
  }

  int f_listvars=0;
  tokenlist args;
  string infile,outfile;
  string glmdir,dvname;
  string maskname;
  vector<string>ivnames;
  vector<string>contrastlist;
  args.Transfer(argc-1,argv+1);
  int part=1,nparts=1;
  vector<VBMatrix> ivmats;
  string perm_mat;
  int signperm_index=-1;
  int orderperm_index=-1;
  string stemname="noname";
  string suffix="cub";
  string prmsuffix="tes";
  string mapname,prmname,resname;
  int f_writeparams=0;
  int f_writeresids=0;
  float q=nan("nan");

  for (int i=0; i<args.size(); i++) {
    if (args[i]=="-v")
      vbvolregress_version();
    else if (args[i]=="-h")
      vbvolregress_help();
    else if (args[i]=="-s" && i<args.size()-1)
      suffix=args[++i];
    else if (args[i]=="-t" && i<args.size()-1)
      prmsuffix=args[++i];
    else if (args[i]=="-prm")
      f_writeparams=1;
    else if (args[i]=="-res")
      f_writeresids=1;
    else if ((args[i]=="-o" || args[i]=="-mapfile") && i<args.size()-1) {
      mapname=args[++i];
    }
    else if (args[i]=="-m" && i<args.size()-1) {
      maskname=args[++i];
    }
    else if (args[i]=="-prmfile" && i<args.size()-1) {
      prmname=args[++i];
      f_writeparams=1;
    }
    else if (args[i]=="-resfile" && i<args.size()-1) {
      resname=args[++i];
      f_writeresids=1;
    }
    else if (args[i]=="-sp" && i<args.size()-2) {
      perm_mat=args[++i];
      signperm_index=strtol(args[++i]);
    }
    else if (args[i]=="-p" && i<args.size()-2) {
      part=strtol(args[++i]);
      nparts=strtol(args[++i]);
    }
    else if (args[i]=="-q" && i<args.size()-1) {
      q=strtod(args[++i]);
    }
    else if (args[i]=="-op" && i<args.size()-2) {
      perm_mat=args[++i];
      orderperm_index=strtol(args[++i]);
    }
    else if (args[i]=="-c" && i<args.size()-1) {
      contrastlist.push_back(args[++i]);
    }
    else if (args[i]=="-dv"&&i<args.size()-1) {
      dvname=args[++i];
    }
    else if (args[i]=="-iv"&&i<args.size()-1) {
      ivnames.push_back((string)"V"+args[++i]);
    }
    else if (args[i]=="-gv"&&i<args.size()-1) {
      int gcnt=strtol(args[i+1]);
      if (gcnt<2 || gcnt>999) {
        printf("[E] vbvolregress: -gv arguments must have 2-999 groups\n");
        return 220;
      }
      int glen=0,tmp;
      vector<int> sizes;
      for (int g=i+2; g<i+2+gcnt; g++) {
        if (g>(int)args.size()-1) {
          printf("[E] vbvolregress: not enough arguments for your -gv group\n");
          exit(210);
        }
        tmp=strtol(args[g]);
        glen+=tmp;
        sizes.push_back(tmp);
      }
      VBMatrix mat(glen,sizes.size()-1);
      int gstart=sizes[0];
      for (int g=1; g<(int)sizes.size(); g++) {
        VB_Vector vv(glen);
        for (int j=gstart; j<gstart+sizes[g]; j++)
          vv[j]=1;
        gstart+=sizes[g];
        vv.meanCenter();
        mat.SetColumn(g-1,vv);
      }
      mat.filename="testmat";
      ivmats.push_back(mat);
      ivnames.push_back("G");
    }
    else if (args[i]=="-int") {
      ivnames.push_back("I");
    }
    else if (args[i]=="-x") {
      f_listvars=1;
    }
    else
      glmdir=args[i];
  }

  // just list the vars in order, don't do any actual processing
  if (f_listvars) {
    int matcnt=0;
    for (int i=0; i<(int)ivnames.size(); i++) {
      if (ivnames[i][0]=='I') {
        printf("[I] vbvolregress: IV: intercept\n");
      }
      else if (ivnames[i][0]=='G') {
        printf("[I] vbvolregress: IV: group %d (%d groups, %d variables)\n",
               matcnt,ivmats[matcnt].n+1,ivmats[matcnt].n);
        matcnt++;
      }
      else if (ivnames[i][0]=='V') {
        printf("[I] vbvolregress: IV: %s\n",ivnames[i].c_str()+1);
      }
    }
    printf("[I] vbvolregress: DV: %s\n",dvname.c_str());
    exit(0);
  }

  GLMInfo glmi;
  Cube tmask;
  Tes ts;
  // build our mask
  if (!(ts.ReadHeader(dvname))) {
    ts.ExtractMask(tmask);
  }
  for (int i=0; i<(int)ivnames.size(); i++) {
    if (ivnames[i].size()<2)
      continue;
    if (!(ts.ReadHeader(ivnames[i].substr(1)))) {
      if (tmask.dimx==0) {
        ts.ExtractMask(tmask);
      }
      else {
        Cube tmp;
        ts.ExtractMask(tmp);
        if (tmp.dimx!=tmask.dimx||tmp.dimy!=tmask.dimy||tmp.dimz!=tmask.dimz) {
          printf("[E] vbvolregress: IV file %s has inconsistent dimensions\n",ivnames[i].c_str());
          exit(201);
        }
        tmask.intersect(tmp);
      }
    }
  }
  if (maskname.size()) {
    Cube mask;
    if (mask.ReadFile(maskname)) {
      printf("[E] vbvolregress: error reading mask file %s\n",maskname.c_str());
      exit(200);
    }
    if (tmask.dimx && (mask.dimx!=tmask.dimx || mask.dimy!=tmask.dimy || mask.dimz!=tmask.dimz)) {
      printf("[E] vbvolregress: mask file %s has dimensions inconsistent with \n",maskname.c_str());
      exit(200);
    }
    if (tmask.data)
      tmask.intersect(mask);
    else
      tmask=mask;
  }

  if (signperm_index>-1) {
    VBMatrix vm(perm_mat,-1,-1,signperm_index,signperm_index);
    glmi.perm_signs=vm.GetColumn(0);
    if (glmi.perm_signs.size()==0) {
      printf("[E] vbvolregress: error getting permutation vector\n");
      exit(200);
    }
  }
  if (orderperm_index>-1) {
    VBMatrix vm(perm_mat,-1,-1,orderperm_index,orderperm_index);
    glmi.perm_order=vm.GetColumn(0);
    if (glmi.perm_order.size()==0) {
      printf("[E] vbvolregress: error getting permutation vector\n");
      exit(200);
    }
  }

  if (!(tmask.data)) {
    tmask.SetVolume(1,1,1,vb_short);
    tmask.SetValue(0,0,0,1);
  }

  string partstring;
  if (nparts>1)
    partstring="_part_"+strnum(part);
  
  // do the contrasts
  string w_mapname=mapname;
  string w_prmname=prmname;
  string w_resname=resname;
  for (int c=0; c<(int)contrastlist.size(); c++) {
    VBContrast cc;
    tokenlist cspec;
    cspec.ParseLine(contrastlist[c]);
    vector<int>ilist;
    // all vars of nominal interest
    for (int i=0; i<(int)ivnames.size(); i++)
      ilist.push_back(i);
    glmi.contrast.parsemacro(cspec,ilist.size(),ilist);

    // do the regression
    glmi.stemname=glmi.contrast.name;
    glmi.f_saveprm=0;
    glmi.rescount=0;
    int err=glmi.VolumeRegress(tmask,part,nparts,ivnames,dvname,ivmats);
    if (err) {
      printf("[E] vbvolregress: failed with %s (%d)!\n",glmi.stemname.c_str(),err);
      exit(err);
    }

    // print out FDR thresh if requested
    if (isfinite(q)) {
      double thresh=calc_fdr_thresh(glmi.rawcube,glmi.statcube,tmask,q);
      printf("[I] vbvolregress: FDR thresh is %g\n",thresh);
    }

    // set up the output filenames
    if (mapname=="")
      w_mapname="statmap_"+glmi.contrast.name+"."+suffix+partstring;
    if (prmname=="")
      w_prmname="params_"+glmi.contrast.name+"."+prmsuffix+partstring;
    if (resname=="")
      w_resname="resids_"+glmi.contrast.name+"."+prmsuffix+partstring;

    // write stat map
    if (glmi.statcube.WriteFile(w_mapname)) {
      printf("[E] vbvolregress: error writing file %s\n",w_mapname.c_str());
      exit(171);
    }
    else
      printf("[I] vbvolregress: wrote file %s\n",w_mapname.c_str());
    // write param file
    if (f_writeparams) {
      if (glmi.paramtes.WriteFile(w_prmname)) {
        printf("[E] vbvolregress: error writing file %s\n",w_prmname.c_str());
        exit(172);
      }
      else
        printf("[I] vbvolregress: wrote file %s\n",w_prmname.c_str());
    }
    // write resids
    if (f_writeresids) {
      if (glmi.residtes.WriteFile(w_resname)) {
        printf("[E] vbvolregress: error writing file %s\n",w_resname.c_str());
        exit(173);
      }
      else
        printf("[I] vbvolregress: wrote file %s\n",w_resname.c_str());
    }
    printf("[I] vbvolregress: done with %s\n",glmi.stemname.c_str());
  }
  
  exit(0);
}

void
vbvolregress_help()
{
  cout << boost::format(myhelp) % vbversion;
}

void
vbvolregress_version()
{
  printf("VoxBo vbvolregress (v%s)\n",vbversion.c_str());
}
