#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "../include/fio.h"
#include "../include/string.h"

#include "cdialog.h"

#include "vma.h"
#include "vmacfg.h"
#include "vmacfgfio.h"

#include "config.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


int VMACFGLoadFromFile(char *filename, vma_cfg_item_struct *parm_list);
int VMACFGSaveToFile(char *filename, vma_cfg_item_struct *parm_list);


/*
 *	Loads the preferences from file to the sequential parm
 *	list.
 *
 *	The given sequential parm list must have the last structure
 *	unit all 0's to indicate end of the list.
 */
int VMACFGLoadFromFile(char *filename, vma_cfg_item_struct *parm_list)
{
	int i, ignore_all_errors = 0;
	FILE *fp;
	struct stat stat_buf;
	char *parm = NULL;


	if((filename == NULL) || (parm_list == NULL))
	    return(-1);

	if(stat(filename, &stat_buf))
	{
/*
	    fprintf(
		stderr,
		"%s: No such file.\n",
		filename
	    );
 */
	    return(-1);
	}
	if(!S_ISREG(stat_buf.st_mode))
	{
            fprintf(
                stderr,
                "%s: Not a file.\n",
                filename
            );
            return(-1);
	}

	/* Open file. */
	fp = FOpen(filename, "rb");
	if(fp == NULL)
	{
/*
	    fprintf(
		stderr,
		"%s: Cannot open.\n",
		filename
	    );
 */
	    return(-1);
	}

	while(1)
	{
	    parm = FSeekNextParm(
		fp, parm,
		VMA_CFG_COMMENT_CHAR, VMA_CFG_DELIMINATOR_CHAR
	    );
	    if(parm == NULL)
		break;

	    i = VMACFGItemListMatchParameter(parm_list, parm);
	    if(i < 0)
	    {
		if(!ignore_all_errors)
		{
                    fprintf(stderr, 
                        "%s: Unsupported parameter `%s'.\n",
                        filename,
                        parm
                    );
		}

		/* No such parameter. */
		FSeekNextLine(fp);
	    }
	    else
	    {
		vma_cfg_item_struct *parm_item_ptr = &parm_list[i];
		int vi[10];
		double vd[10];

		int8_t		i8;
		u_int8_t	ui8;
		int16_t		i16;
		u_int16_t	ui16;
		int32_t		i32;
		u_int32_t	ui32;
		int64_t		i64;
		u_int64_t	ui64;
		float		f;
		double		d;
		vma_color_struct c;

		switch(parm_item_ptr->type)
		{
		  case VMA_CFG_ITEM_TYPE_INT_8:
		    FGetValuesI(fp, vi, 1);
		    i8 = (int8_t)vi[0];
		    VMACFGItemSetValue(parm_item_ptr, &i8);
		    break;
                  case VMA_CFG_ITEM_TYPE_U_INT_8:
                    FGetValuesI(fp, vi, 1);
                    ui8 = (u_int8_t)vi[0];
                    VMACFGItemSetValue(parm_item_ptr, &ui8);
                    break;

                  case VMA_CFG_ITEM_TYPE_INT_16:
                    FGetValuesI(fp, vi, 1);
                    i16 = (int16_t)vi[0];
                    VMACFGItemSetValue(parm_item_ptr, &i16);
                    break;
                  case VMA_CFG_ITEM_TYPE_U_INT_16:
                    FGetValuesI(fp, vi, 1);  
                    ui16 = (u_int16_t)vi[0];
                    VMACFGItemSetValue(parm_item_ptr, &ui16);
                    break;

                  case VMA_CFG_ITEM_TYPE_INT_32:
                    FGetValuesI(fp, vi, 1);
                    i32 = (int32_t)vi[0];
                    VMACFGItemSetValue(parm_item_ptr, &i32);
                    break;
                  case VMA_CFG_ITEM_TYPE_U_INT_32:
                    FGetValuesI(fp, vi, 1);
                    ui32 = (u_int32_t)vi[0];
                    VMACFGItemSetValue(parm_item_ptr, &ui32);
                    break;

                  case VMA_CFG_ITEM_TYPE_INT_64:
                    FGetValuesI(fp, vi, 1);
                    i64 = (int64_t)vi[0];
                    VMACFGItemSetValue(parm_item_ptr, &i64);
                    break;
                  case VMA_CFG_ITEM_TYPE_U_INT_64:
                    FGetValuesI(fp, vi, 1);
                    ui64 = (u_int64_t)vi[0];
                    VMACFGItemSetValue(parm_item_ptr, &ui64);
                    break;

                  case VMA_CFG_ITEM_TYPE_FLOAT:
                    FGetValuesF(fp, vd, 1);
                    f = (float)vd[0];
                    VMACFGItemSetValue(parm_item_ptr, &f);
                    break;
                  case VMA_CFG_ITEM_TYPE_DOUBLE:
                    FGetValuesF(fp, vd, 1);
                    d = (double)vd[0];
                    VMACFGItemSetValue(parm_item_ptr, &d);
                    break;

		  case VMA_CFG_ITEM_TYPE_STRING:
		    free(parm_item_ptr->value);
		    parm_item_ptr->value = FGetString(fp);
		    break;

                  case VMA_CFG_ITEM_TYPE_COLOR:
                    FGetValuesF(fp, vd, 4);
                    c.r = (double)vd[0];
                    c.g = (double)vd[1];
                    c.b = (double)vd[2];
                    c.a = (double)vd[3];
                    VMACFGItemSetValue(parm_item_ptr, &c);
                    break;

		   default:
		    FSeekNextLine(fp);
		    break;
		}
	    }
	}

	/* Close file. */
	FClose(fp);
	fp = NULL;

	return(0);
}


/*
 *	Saves configuration in the sequential parm list to the
 *	specified file.
 *
 *      The given sequential parm list must have the last structure
 *      unit all 0's to indicate end of the list.
 */
int VMACFGSaveToFile(char *filename, vma_cfg_item_struct *parm_list)
{
        int i;
        FILE *fp;
        struct stat stat_buf;
	vma_cfg_item_struct *parm_item_ptr;


        if((filename == NULL) ||
           (parm_list == NULL)
        )
            return(-1);

        if(!stat(filename, &stat_buf))
        {
            if(S_ISDIR(stat_buf.st_mode))
            {
                fprintf(
                    stderr,
                    "%s: Cannot save directory as a file.\n",
                    filename
                );
                return(-1);
	    }
        }

	fp = FOpen(filename, "wb");

	i = 0;
	do
	{
	    parm_item_ptr = &parm_list[i];

	    if(parm_item_ptr->parameter != NULL)
	    {
		char *parm = parm_item_ptr->parameter;
		void *value = parm_item_ptr->value;

		switch(parm_item_ptr->type)
		{
		  case VMA_CFG_ITEM_TYPE_INT_8:
		    fprintf(fp,
			"%s %c %i\n",
			parm, VMA_CFG_DELIMINATOR_CHAR,
			(int8_t)((value == NULL) ? 0 : *(int8_t *)value)
		    );
		    break;
                  case VMA_CFG_ITEM_TYPE_U_INT_8:
                    fprintf(fp,
                        "%s %c %i\n",
                        parm, VMA_CFG_DELIMINATOR_CHAR,
                        (u_int8_t)((value == NULL) ? 0 : *(u_int8_t *)value) 
                    ); 
                    break;

                  case VMA_CFG_ITEM_TYPE_INT_16:
                    fprintf(fp,
                        "%s %c %i\n",
                        parm, VMA_CFG_DELIMINATOR_CHAR,
                        (int16_t)((value == NULL) ? 0 : *(int16_t *)value)
                    );
                    break;
                  case VMA_CFG_ITEM_TYPE_U_INT_16:
                    fprintf(fp,
                        "%s %c %i\n",
                        parm, VMA_CFG_DELIMINATOR_CHAR,
                        (u_int16_t)((value == NULL) ? 0 : *(u_int16_t *)value)
                    );
                    break;

                  case VMA_CFG_ITEM_TYPE_INT_32:
                    fprintf(fp,
                        "%s %c %i\n",
                        parm, VMA_CFG_DELIMINATOR_CHAR,
                        (int32_t)((value == NULL) ? 0 : *(int32_t *)value)
                    );
                    break;
                  case VMA_CFG_ITEM_TYPE_U_INT_32:
                    fprintf(fp,
                        "%s %c %i\n",
                        parm, VMA_CFG_DELIMINATOR_CHAR,
                        (u_int32_t)((value == NULL) ? 0 : *(u_int32_t *)value)
                    );
                    break;

                  case VMA_CFG_ITEM_TYPE_INT_64:
                    fprintf(fp,
                        "%s %c %ld\n",
                        parm, VMA_CFG_DELIMINATOR_CHAR,
                        (long)((value == NULL) ? 0 : *(int64_t *)value)
                    );
                    break;
                  case VMA_CFG_ITEM_TYPE_U_INT_64:
                    fprintf(fp,
                        "%s %c %ld\n",
                        parm, VMA_CFG_DELIMINATOR_CHAR,
                        (unsigned long)((value == NULL) ? 0 : *(u_int64_t *)value)
                    );
                    break;

                  case VMA_CFG_ITEM_TYPE_FLOAT:
                    fprintf(fp,
                        "%s %c %f\n",
                        parm, VMA_CFG_DELIMINATOR_CHAR,
                        (float)((value == NULL) ? 0.0 : *(float *)value)
                    );
                    break;
                  case VMA_CFG_ITEM_TYPE_DOUBLE:
                    fprintf(fp,
                        "%s %c %f\n",
                        parm, VMA_CFG_DELIMINATOR_CHAR,
                        (double)((value == NULL) ? 0.0 : *(double *)value)
                    );
                    break;

                  case VMA_CFG_ITEM_TYPE_STRING:
                    fprintf(fp,
                        "%s %c %s\n",
                        parm, VMA_CFG_DELIMINATOR_CHAR,
                        (char *)((value == NULL) ? "" : (char *)value)
                    );
                    break;

                  case VMA_CFG_ITEM_TYPE_COLOR:
                    fprintf(fp,
                        "%s %c %f %f %f %f\n",
                        parm, VMA_CFG_DELIMINATOR_CHAR,
                        (double)((value == NULL) ? 0.0 :
			    ((vma_color_struct *)value)->r
			),
                        (double)((value == NULL) ? 0.0 :
                            ((vma_color_struct *)value)->g  
                        ),
                        (double)((value == NULL) ? 0.0 :
                            ((vma_color_struct *)value)->b  
                        ),
                        (double)((value == NULL) ? 0.0 :
                            ((vma_color_struct *)value)->a
                        )
                    );
                    break;


		}
	    }

	    i++;

	} while(!((parm_item_ptr->type == 0) &&
                  (parm_item_ptr->parameter == NULL)
                 )
	);

	FClose(fp);

	return(0);
}
