#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>

#include "fsd.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


/*
 *      Font selection dialog structure:
 */
typedef struct {

        gbool initialized;
        gbool map_state;

        GtkWidget *fsd;         /* Color selection dialog widget. */

} fsd_data_struct;


static void FSDOKCB(GtkWidget *widget, gpointer data);
static void FSDCancelCB(GtkWidget *widget, gpointer data);
static gint FSDCloseCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);

gint FSDInit(void);
gbool FSDIsQuery(void);
void FSDBreakQuery(void);
gbool FSDGetResponse(
        const gchar *title,
        const gchar *ok_label, const gchar *cancel_label,
        gchar *start_font_name,
        gchar **font_name_rtn
);
void FSDMap(void);
void FSDUnmap(void);
void FSDShutdown(void);


static gint block_loop_level;
static fsd_data_struct fsd_data;
static gbool fsd_got_user_response;
static gchar *response_font_name;


/*
 *	Ok button callback.
 */
static void FSDOKCB(GtkWidget *widget, gpointer data)
{
	fsd_data_struct *fsdd = (fsd_data_struct *)data;
        if(fsdd == NULL)
            return;

        if(!fsdd->initialized)
            return;

	/* Set user response code to TRUE, indicating OK. */
	fsd_got_user_response = TRUE;

	/* Get font name. */
	g_free(response_font_name);
	response_font_name = gtk_font_selection_dialog_get_font_name(
	    GTK_FONT_SELECTION_DIALOG(fsdd->fsd)
	);

	/* Unmap. */
	FSDUnmap();

	/* Break out of blocking loop. */
	gtk_main_quit();
	block_loop_level--;

        return;
}

/*
 *	Cancel callback.
 */
static void FSDCancelCB(GtkWidget *widget, gpointer data)
{
        fsd_data_struct *fsdd = (fsd_data_struct *)data;
        if(fsdd == NULL)
            return;

        if(!fsdd->initialized)
            return;

        /* Set user response code to FALSE, indicating Cancel. */
        fsd_got_user_response = FALSE;

	/* Unmap. */
        FSDUnmap();

        /* Break out of blocking loop. */
        gtk_main_quit();
	block_loop_level--;

        return;
}           

/*
 *      Close callback.
 */
static gint FSDCloseCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	FSDCancelCB(widget, data);
	return(TRUE);
}


/*
 *	Initializes font selection dialog..
 */
int FSDInit(void)
{
	GtkWidget *w;
	GtkFontSelectionDialog *fsd;
	fsd_data_struct *fsdd = &fsd_data;


	/* Reset local globals. */
        block_loop_level = 0;
        fsd_got_user_response = FALSE;

	g_free(response_font_name);
	response_font_name = NULL;

	memset(fsdd, 0x00, sizeof(fsd_data_struct));

	/* Reset values. */
        fsdd->initialized = TRUE;
        fsdd->map_state = FALSE;

	/* Create the font selection dialog. */
	w = gtk_font_selection_dialog_new("Select Font");
	fsdd->fsd = w;
	fsd = GTK_FONT_SELECTION_DIALOG(w);

	/* Connect "delete_event" on main fsd window. */
	gtk_signal_connect(
            GTK_OBJECT(w), "delete_event",
            GTK_SIGNAL_FUNC(FSDCloseCB), (gpointer)fsdd
        );

	/* Get GtkFontSelection widget. */
	w = fsd->fontsel;

/* Connect some signals to the GtkFontSelection widget? */

	/* Ok button. */
	gtk_signal_connect(
	    GTK_OBJECT(fsd->ok_button), "clicked",
            GTK_SIGNAL_FUNC(FSDOKCB), (gpointer)fsdd
        );
	gtk_widget_show(fsd->ok_button);

	/* Hide apply button. */
	if(fsd->apply_button != NULL)
            gtk_widget_hide(fsd->apply_button);

	/* Cancel button. */
        gtk_signal_connect(
            GTK_OBJECT(fsd->cancel_button), "clicked",
            GTK_SIGNAL_FUNC(FSDCancelCB), (gpointer)fsdd
        );
        gtk_widget_show(fsd->cancel_button);

	return(0);
}

/*
 *      Returns TRUE if currently blocking for query.
 */
gbool FSDIsQuery(void)
{
        if(block_loop_level > 0)
            return(TRUE);
        else
            return(FALSE);
}

/*
 *	Ends query if any and returns a not available response.
 */
void FSDBreakQuery(void)
{
	fsd_got_user_response = FALSE;

        /* Break out of an additional blocking loops. */
        while(block_loop_level > 0)
        {
            gtk_main_quit();
            block_loop_level--;
        }
        block_loop_level = 0;

	return;
}

/*
 *	Maps the font selection dialog and sets up the inital values.
 *
 *	Returns TRUE if a font was selected or FALSE if user canceled.
 *
 *	For most values that are set NULL, the value is left unchanged.
 *	All given values are coppied.
 *
 *	All returned pointer values should be considered statically
 *	allocated, do not deallocate them.
 */
gbool FSDGetResponse(
        const gchar *title,
        const gchar *ok_label, const gchar *cancel_label,
        gchar *start_font_name,
        gchar **font_name_rtn
)
{
	GtkWidget *w;
	fsd_data_struct *fsdd = &fsd_data;


        /* Do not handle response if already waiting for a response,
         * return with a not available response code.
         */
        if(block_loop_level > 0)
	{
            if(font_name_rtn != NULL)
                (*font_name_rtn) = NULL;

            return(FALSE);
	}

	/* Reset global responses values. */
	fsd_got_user_response = FALSE;

	g_free(response_font_name);
	response_font_name = NULL;

	/* Reset returns. */
	if(font_name_rtn != NULL)
	    (*font_name_rtn) = NULL;


	/* Font selection dialog must be initialized. */
	if(!fsdd->initialized)
	    return(fsd_got_user_response);

	/* Get pointer to font selection dialog widget. */
	w = fsdd->fsd;
	if(w == NULL)
            return(fsd_got_user_response);

	/* Update title if specified. */
	if(title != NULL)
	    gtk_window_set_title(GTK_WINDOW(w), title);

	/* Update initial start font name if specified. */
	if(start_font_name != NULL)
	    gtk_font_selection_dialog_set_font_name(
		GTK_FONT_SELECTION_DIALOG(w),
		start_font_name
	    );

        /* Map font selection dialog as needed. */
        FSDMap();

        /* Block GUI untill response. */
	block_loop_level++;
        gtk_main();

        /* Unmap font selection dialog just in case it was not unmapped
	 * from any of the callbacks.
	 */
        FSDUnmap();

        /* Break out of an additional blocking loops. */
        while(block_loop_level > 0)
        {
            gtk_main_quit();
            block_loop_level--;
        }
        block_loop_level = 0;


	/* Begin setting returns. */

	/* Font name return. */
	if(font_name_rtn != NULL)
	    (*font_name_rtn) = response_font_name;

	return(fsd_got_user_response);
}


/*
 *	Maps the font selection dialog as needed.
 */
void FSDMap(void)
{
        fsd_data_struct *fsdd = &fsd_data;

        if(!fsdd->initialized)
            return;

        if(!fsdd->map_state)
        {
            GtkWidget *w = fsdd->fsd;

            if(w != NULL)
                gtk_widget_show(w);

            fsdd->map_state = TRUE;
        }
}

/*
 *	Unmaps the font selection dialog as needed.
 */
void FSDUnmap(void)
{
        fsd_data_struct *fsdd = &fsd_data;

	if(!fsdd->initialized)
	    return;

	if(fsdd->map_state)
	{
	    GtkWidget *w = fsdd->fsd;

	    if(w != NULL)
		gtk_widget_hide(w);

	    fsdd->map_state = FALSE;
	}
}

/*
 *	Deallocates font selection dialog resources.
 */
void FSDShutdown(void)
{
	GtkWidget **w;
        fsd_data_struct *fsdd = &fsd_data;


	/* Reset local globals. */
        fsd_got_user_response = FALSE;

	g_free(response_font_name);
	response_font_name = NULL;

        /* Break out of an additional blocking loops. */
        while(block_loop_level > 0)
        {
            gtk_main_quit();
            block_loop_level--;
        }
        block_loop_level = 0;


	/* Is color selection dialog initialized? */
	if(fsdd->initialized)
	{
#define DO_DESTROY_WIDGET	\
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}
	    /* Begin destroying widgets. */

	    w = &fsdd->fsd;
	    DO_DESTROY_WIDGET

#undef DO_DESTROY_WIDGET
	}

	/* Clear font selection dialog structure. */
	memset(fsdd, 0x00, sizeof(fsd_data_struct));
}
