#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>

#include "v3dmodel.h"
#include "editor.h"
#include "editorp.h"
#include "editormodel.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


void EditorModelDelete(
	ma_editor_struct *editor, int model_num
);
void EditorModelDeleteAll(ma_editor_struct *editor);


/*
 *	Deletes the model from the given editor, thus unrealizing and
 *	deleting all its primitives and updating the models list on the
 *	editor (pointers on the editor's models list may be adjusted).
 */
void EditorModelDelete(
        ma_editor_struct *editor, int model_num
)
{
	gint i;
	v3d_model_struct *model_ptr;
	GtkCList *clist;

	if(editor == NULL)
	    return;

	if(!editor->initialized)
	    return;

	/* Unselect model as needed. */
	if(model_num == EditorSelectedModelIndex(editor))
	    editor->selected_model_item = -1;

	/* Get pointer to model. */
	model_ptr = V3DModelListGetPtr(
	    editor->model, editor->total_models, model_num
	);
	if(model_ptr == NULL)
	    return;

	/* Unrealize all primitives on the model. */
	EditorPrimitiveUnrealizeAll(editor, model_num);

	/* Destroy the model, thus deallocating all primitives and
	 * resources on it.
	 */
	V3DModelDestroy(model_ptr);

	/* Mark model as NULL on the editor, to indicate it has been
	 * destroyed.
	 */
	editor->model[model_num] = NULL;

	/* Shift pointers. */
	editor->total_models--;
	for(i = model_num; i < editor->total_models; i++)
	    editor->model[i] = editor->model[i + 1];

	if(editor->total_models < 0)
	    editor->total_models = 0;

	/* Delete model from editor's gui models list. */
	clist = (GtkCList *)editor->models_list;
	if(clist != NULL)
	{
	    /* Remove the row on the clist that reffers to this model,
	     * the row number matches the model_num value.
	     */
	    if((model_num >= 0) && (model_num < clist->rows))
		gtk_clist_remove(clist, model_num);
	}

	return;
}

/*
 *	Deletes all models on the editor, calling EditorModelDelete() to
 *	delete each model.
 */
void EditorModelDeleteAll(ma_editor_struct *editor)
{
	int i;
	GtkCList *clist;


        if(editor == NULL)
            return;

        if(!editor->initialized)
            return;

	/* We need to freeze the models clist while we do this. */
        clist = (GtkCList *)editor->models_list;
        if(clist != NULL)
	    gtk_clist_freeze(clist);

	/* Go through all models on the editor. */
	for(i = 0; i < editor->total_models; i++)
	    EditorModelDelete(editor, i);

	free(editor->model);
	editor->model = NULL;
	editor->total_models = 0;

        /* Thaw models clist. */
        clist = (GtkCList *)editor->models_list;
        if(clist != NULL)
            gtk_clist_thaw(clist);

	return;
}
