/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_commandLine.h"

#include "visu_tools.h"
#include "visu_basic.h"

#include <stdlib.h>
#include <getopt.h>
#include <stdio.h>
#include <locale.h>
#include <string.h>

/**
 * SECTION:visu_commandLine
 * @short_description: All methods needed to parse options from the
 * command line.
 *
 * <para>V_Sim parses the command line at startup and store data in
 * private variables. All this values can be retrieve later by the
 * program through calls to commandLineGet_* methods.</para>
 */

static char *argFileName;

/* Forced resources file. */
static gchar *argResources;

/* Spin arguments. */
static char *argSpinFileName;
static int spinHidingMode;
static gboolean spinAndAtomic;

/* Miscelaneous arguments. */
static gchar *argExportFileName;
static gchar *argWindowMode;
static unsigned int argISet;
static gchar *argValueFile;

/* Colourisation tool. */
static gchar *argColorizeFileName;
static int argColorizeColUsed[3];
static int argColorizePresetColor;
static gboolean argColorizeColUsed_isPresent;

/* Iso-surfaces tool. */
static gchar *argScalarFieldFileName;
static gchar *argIsoSurfacesFileName;
static float *argIsoValues;
static gchar **argIsoNames;
static float argNbIsoValues;
static gboolean argFitToBox;

/* Translations stuffs. */
static gboolean argTranslationsIsSet;
static float argTranslations[3];
/* Extension stuffs. */
static gboolean argExpandIsSet;
static float argExtension[3];

/* Coloured map. */
static int *argMapPlaneId;
static matrix_scaleflag argLogScale;
static int argNIsoLines;
static float *argIsoLinesColor;
static guint argMapPrecision;

/* Background image. */
static gchar *argBgImageFile;

/* Extended options. */
static OptionTable *argOptionTable;

static gchar* argPlanesFileName;

static int withGtk;

static int xWindowWidth, xWindowHeight;

struct option_desc
{
  struct option *opt;
  const gchar *desc;
  const gchar *arg;
  const gchar *def;
  float version;
};

/* Remaining letters...
   j, k, l, q, y, z */
#define N_OPTIONS 28
static GString *short_options;
static struct option *long_options;
static struct option_desc *ext_options;

void alignPrint(GString *str, const gchar* value, int ln, gchar *pad)
{
  gchar *sp;
  gchar *tmp;
  
  if (g_utf8_strlen(value, -1) <= ln)
    {
      g_string_append_printf(str, "%s%s", pad, value);
      return;
    }

  sp = g_utf8_strrchr(value, ln, ' ');
  g_return_if_fail(sp);

  tmp = g_strndup(value, (gsize)(sp - value));
  g_string_append_printf(str, "%s%s\n", pad, tmp);
  g_free(tmp);
  alignPrint(str, sp + 1, ln, pad);
}

#define OUT stdout
#define P   fprintf
#define offset 25
void printInfoMessage()
{
  int i;
  GString *desc, *expl;
  gchar format[128], pad[128], arg[128];

  P(OUT, _("V_Sim is a software to visualize atomic structures with"
		    " OpenGl rendering.\n\n"));
  desc = g_string_new(_("usage:"));
  g_string_append_printf(desc, " %s [", PACKAGE_TARNAME);
  for (i = 0; ext_options[i].opt->name; i++)
    if (ext_options[i].opt->val)
      {
	if (i > 0)
	  g_string_append(desc, " | ");
	g_string_append_printf(desc, "-%c", ext_options[i].opt->val);
	if (ext_options[i].arg)
	  g_string_append_printf(desc, " %s", ext_options[i].arg);
      }
  g_string_append_printf(desc, "] [fileToRender]\n\n");
  P(OUT, desc->str);

  expl = g_string_new("");
  sprintf(format, "%%%ds", offset);
  sprintf(pad, format, " ");
  for (i = 0; ext_options[i].opt->name; i++)
    {
      if (ext_options[i].opt->val)
	g_string_printf(desc, "  -%c,", ext_options[i].opt->val);
      else
	g_string_assign(desc, "     ");
      if (ext_options[i].arg)
	sprintf(arg, "%s %s", ext_options[i].opt->name, ext_options[i].arg);
      else
	sprintf(arg, "%s", ext_options[i].opt->name);
      g_string_erase(expl, 0, -1);
      alignPrint(expl, ext_options[i].desc, 80 - offset, pad);
      g_string_append_printf(desc, " --%s (from v%3.1f.0)\n%s\n", arg, ext_options[i].version, expl->str);
      sprintf(format, "%%%ds", offset);
      g_string_append_printf(desc, format, " ");
      if (ext_options[i].def)
	g_string_append_printf(desc, _("(Default value: %s)\n\n"), ext_options[i].def);
      else
	g_string_append(desc, _("(Default value: unset)\n\n"));
      P(OUT, desc->str);
    }
  g_string_free(expl, TRUE);
  g_string_free(desc, TRUE);
}

void optionSet(int i, const gchar *lg, gchar sh, const gchar *desc,
	       const gchar *arg, const gchar *def, float version)
{
  long_options[i].name = lg;
  long_options[i].has_arg = (arg)?required_argument:no_argument;
  long_options[i].flag = (int*)0;
  long_options[i].val = (int)sh;
  ext_options[i].opt = long_options + i;
  ext_options[i].desc = desc;
  ext_options[i].arg = arg;
  ext_options[i].def = def;
  ext_options[i].version = version;
  if (arg && sh)
    g_string_append_printf(short_options, "%c:", sh);
  else if (sh)
    g_string_append_printf(short_options, "%c", sh);
}

void optionsInit()
{
  int i;

  short_options = g_string_new("");
  ext_options = g_malloc(sizeof(struct option_desc) * N_OPTIONS);
  long_options = g_malloc(sizeof(struct option) * N_OPTIONS);
  i = 0;
  optionSet(i++, "export", 'e',
	    _("make an image from the fileToRender argument. The format is"
	      " specified through the extension of the argument."),
	    _("file"), (gchar*)0, 3.0f);
  optionSet(i++, "resources", 'r',
	    _("load the given resources file on startup instead of looking"
	      " for a valid resources file in the standard locations."),
	    _("file"), (gchar*)0, 3.4f);
  optionSet(i++, "help", 'h',
	    _("show this little help."),
	    (gchar*)0, (gchar*)0, 3.0f);
  optionSet(i++, "geometry", 'g',
	    _("specify the size of the rendering window, the size argument must"
	      " have the following format: <width>x<height> with positive non null"
	      " values."),
	    _("<w>x<h>"), "600x600", 3.0f);
  optionSet(i++, "spin-file", 's',
	    _("use the given argument as a spin indicator. If this option is used,"
	      " V_Sim switches automatically to spin rendering whatever method"
	      " is specified in the parameter file."),
	    _("file"), (gchar*)0, 3.1f);
  optionSet(i++, "hiding-mode", 'm',
	    _("policy used to show or not null modulus spins possible values are"
	      " positives."),
	    _("id"), "0", 3.2f);
  optionSet(i++, "spin-and-atomic", 'a',
	    _("always draws atomic rendering on node position in addition to spin"
	      " rendering."),
	    (gchar*)0, (gchar*)0, 3.3f);
  optionSet(i++, "colorize", 'c',
	    _("the argument fileToRender must be called, then the given file of"
	      " the option is used to colorize the elements."),
	    _("file"), (gchar*)0, 3.1f);
  optionSet(i++, "use-column", 'u',
	    _("must be used with the '--colorize' option, it specifies the"
	      " columns to use from the data file for the three colour"
	      " channels [l;m;n]. Columns are counted from 1. Use -3, -2, -1 and 0"
	      " to use the special values, constant 1, coord. x, coord. y,"
	      " and coord. z, respectively."),
	    "l:m:n", (gchar*)0, 3.1f);
  optionSet(i++, "color-preset", 'd',
	    _("this option can be used with the '--colorize' one or the"
	      " '--build-map' one. It chooses a preset color scheme. The id"
	      " argument is an integer that corresponds to a defined color"
	      " shade (ranging from 0)."),
	    _("id"), (gchar*)0, 3.1f);
  optionSet(i++, "translate", 't',
	    _("a file must be loaded. It applies the given translations to the"
	      " loaded file. The units are those of the file. This is available"
	      " for periodic file formats only."),
	    "x:y:z", (gchar*)0, 3.3f);
  optionSet(i++, "expand", 'x',
	    _("a file must be loaded. It applies the given expansion to the loaded"
	      " file. The values are given in box coordinates. This is available"
	      " for periodic file formats only."),
	    "x:y:z", (gchar*)0, 3.4f);
  optionSet(i++, "planes", 'p',
	    _("the argument fileToRender must be called, then the given file of"
	      " the option is parsed as a list of planes and they are rendered."),
	    _("file"), (gchar*)0, 3.2f);
  optionSet(i++, "scalar-field", 'f',
	    _("the argument fileToRender must be called, then the given file of"
	      " the option is parsed as a scalar field and loaded."),
	    _("file"), (gchar*)0, 3.3f);
  optionSet(i++, "iso-values", 'v',
	    _("must be used with the '--scalar-field' option, then the given"
	      " surfaces are built and rendered. If a name is appended to a"
	      " value using '#' as a separator, this name is used as the name"
	      " for the iso-surface (i.e. 0.25#Blue)."),
	    "v[:v]", (gchar*)0, 3.3f);
  optionSet(i++, "iso-surfaces", 'i',
	    _("the argument fileToRender must be given, then the given file of"
	      " the option is parsed and surfaces are rendered."),
	    _("file"), (gchar*)0, 3.2f);
  optionSet(i++, "build-map", 'b',
	    _("the argument fileToRender must be given, as the '--planes',"
	      " '--color-preset' and '--scalar-field' options used, then the"
	      " given plane 'id' is replaced by a coloured map using given"
	      " scalar field and shade. 'id' ranges from 0. If several ids"
	      " are given, several maps are built."),
	    _("id[:id]"), (gchar*)0, 3.4f);
  optionSet(i++, "log-scale", 0,
	    _("if given, plots are done with the logarithm scale method that"
	      " is given (range from 0)."),
	    _("id"), (gchar*)0, 3.4f);
  optionSet(i++, "n-iso-lines", 'n',
	    _("when positive, val isolines are plotted on the coloured map."),
	    _("val"), (gchar*)0, 3.4f);
  optionSet(i++, "color-iso-lines", 0,
	    _("when given, generated iso-lines are colourised [R:G:B] or auto"
	      " with the values. The specific value 'auto' will produced"
	      " iso-lines in inversed colours."),
	    _("[R:G:B] or auto"), "[0:0:0]", 3.5f);
  optionSet(i++, "fit-to-box", 0,
	    _("if val is not TRUE, the surfaces use their own bounding box."),
	    _("val"), "TRUE", 3.3f);
  optionSet(i++, "bg-image", 0,
	    _("draw the given image on the background."),
	    _("file"), (gchar*)0, 3.4f);
  optionSet(i++, "option", 'o',
	    _("this is a generic way to give extended option. to V_Sim. As much"
	      " as -o can be used. Each one store a key and its value (boolean,"
	      " integer or float)."),
	    _("id=value"), (gchar*)0, 3.3f);
  optionSet(i++, "window-mode", 'w',
	    _("used to choose the windowing mode. By default the command panel "
	      "and the rendering window are separated. In the 'oneWindow' mode "
	      "they are joined. In the 'renderOnly' mode, the command panel is "
	      "not used."),
	    _("mode"), "classic", 3.5f);
  optionSet(i++, "i-set", 0,
	    _("this flag is used to choose the id of the loaded file if the"
	      " format has support for multiple ids in one file (see XYZ"
	      " format or -posi.d3 ones)."),
	    _("i"), "0", 3.5f);
  optionSet(i++, "value-file", 0,
	    _("specify an XML file with some value information for V_Sim,"
	      " like a list of planes, highlighted nodes... It replaces"
	      " and extend the previous --planes option."),
	    _("file"), (gchar*)0, 3.5f);
  optionSet(i++, "map-precision", 0,
	    _("Give the precision in percent to render the coloured map."),
	    _("prec"), "100", 3.5f);
  optionSet(i++, NULL, 0, NULL, NULL, NULL, 0.f);
  g_return_if_fail(i == N_OPTIONS);
}

gboolean commandLineExport(const gchar *filename, GError **error)
{
  GString *xml;
  int i;
  gboolean status;
  gchar *desc, *def, *arg;

  xml = g_string_new("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n");
  xml = g_string_append(xml, "<commandLine>\n");
  for (i = 0; ext_options[i].opt->name; i++)
    {
      if (ext_options[i].opt->val)
	g_string_append_printf
	  (xml, "  <option name=\"%s\" short=\"%c\" version=\"%3.1f\">\n",
	   ext_options[i].opt->name, ext_options[i].opt->val, ext_options[i].version);
      else
	g_string_append_printf
	  (xml, "  <option name=\"%s\" version=\"%3.1f\">\n",
	   ext_options[i].opt->name, ext_options[i].version);

      g_string_append(xml, "    <description");
      if (ext_options[i].arg)
	{
	  arg = g_markup_escape_text(ext_options[i].arg, -1);
	  g_string_append_printf(xml, " arg=\"%s\"", arg);
	  g_free(arg);
	}
      if (ext_options[i].def)
	{
	  def = g_markup_escape_text(ext_options[i].def, -1);
	  g_string_append_printf(xml, " default=\"%s\"", def);
	  g_free(def);
	}
      desc = g_markup_escape_text(ext_options[i].desc, -1);
      g_string_append_printf(xml, ">%s</description>\n", desc);
      g_free(desc);
      g_string_append(xml, "  </option>\n");
    }
  g_string_append(xml, "</commandLine>\n");

  status = g_file_set_contents(filename, xml->str, -1, error);
  g_string_free(xml, TRUE);
  return status;
}

int parseCommandLine(int argc, char **argv)
{
  int res, i, nb, valueInt;
  int option_index;
  gchar **tokens, **tokens2;
  Option *option;
  float valueFloat;

  /* We want to read . as floating point separator : in french it is , */
  setlocale(LC_NUMERIC, "C");

  argFileName                  = (char*)0;
  argSpinFileName              = (char *)0;
  spinHidingMode               = 1;
  spinAndAtomic                = FALSE;
  argExportFileName            = (char*)0;
  argColorizeFileName          = (gchar*)0;
  for (i = 0; i < 2; i++)
    argColorizeColUsed[i]      = 0;
  argColorizeColUsed_isPresent = FALSE;
  argColorizePresetColor       = -1;
  argTranslationsIsSet         = FALSE;
  argExpandIsSet               = FALSE;
  xWindowWidth                 = 600;
  xWindowHeight                = 600;
  argPlanesFileName            = (gchar*)0;
  argScalarFieldFileName       = (gchar*)0;
  argIsoSurfacesFileName       = (gchar*)0;
  argIsoValues                 = (float*)0;
  argFitToBox                  = TRUE;
  argOptionTable               = (OptionTable*)0;
  argMapPlaneId                = (int*)0;
  argLogScale                  = FALSE;
  argNIsoLines                 = 0;
  argBgImageFile               = (gchar*)0;
  argIsoLinesColor             = g_malloc0(sizeof(float) * 3);
  argWindowMode                = g_strdup("classic");
  argISet                      = 0;
  argValueFile                 = (gchar*)0;
  argMapPrecision              = 100;

  withGtk                      = 1;

  optionsInit();

  option = (Option*)0;
  option_index = 0;
  opterr = 0;
  DBG_fprintf(stderr, "Visu CommandLine: short options '%s'.\n", short_options->str);
  while (1)
    {
      res = getopt_long (argc, argv, short_options->str,
			 long_options, &option_index);

      /* test if there's no more option available. */
      if (res == -1)
	break;

      switch (res)
	{
	case 'e':
	  DBG_fprintf(stderr, "Visu Command: option '%s' found with arg '%s'.\n",
		      long_options[option_index].name, optarg);
	  if (!optarg)
	    g_error("The option 'export' needs a parameter.\n");
	  else
	    argExportFileName = g_strdup(optarg);
	  /* This option does not need gtk. */
	  withGtk = 0;
	  break;
	case 'r':
	  DBG_fprintf(stderr, "Visu Command: option '%s' found with arg '%s'.\n",
		      long_options[option_index].name, optarg);
	  if (!optarg)
	    g_error("The option 'resources' needs a parameter.\n");
	  else
	    argResources = g_strdup(optarg);
	  break;
	case 'h':
	  DBG_fprintf(stderr, "Visu Command: option '%s' found.\n",
		      long_options[option_index].name);
	  printInfoMessage();
	  exit(0);
	  break;
	case 'g':
	  DBG_fprintf(stderr, "Visu Command: set the geometry of the X window (%s).\n", optarg);
	  res = sscanf(optarg, "%dx%d", &xWindowWidth, &xWindowHeight);
	  if (res != 2 || xWindowWidth <= 0 || xWindowHeight <=0)
	    {
	      g_warning("Wrong format for geometry"
			" option (<width>x<height> awaited).\n");
	      xWindowWidth = 600;
	      xWindowHeight = 600;
	      break;
	    }
	  break;
	case 's':
	  DBG_fprintf(stderr, "Visu Command: set the filenane for"
		      " spin rendering to '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'spin-file' needs a parameter.\n");
	    }
	  argSpinFileName = g_strdup(optarg);
	  break;
	case 'm':
	  DBG_fprintf(stderr, "Visu Command: set hiding-mode to '%s' in spin rendering.\n", optarg);
	  res = sscanf(optarg, "%d", &spinHidingMode);
	  if (res != 1 || spinHidingMode < 0)
	    {
	      g_warning("Wrong format for hiding-mode"
			" option (integer awaited).\n");
	      spinHidingMode = 0;
	    }
	  break;
	case 'a':
	  DBG_fprintf(stderr, "Visu Command: set spin-and-atomic to TRUE in spin rendering.\n");
	  spinAndAtomic = TRUE;
	  break;
	case 'b':
	  DBG_fprintf(stderr, "Visu Command: get the values for"
		      " maps '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'build-map' needs a parameter.\n");
	    }
	  tokens = g_strsplit(optarg, ":", -1);
	  /* Count number of tokens. */
	  for (nb = 0; tokens[nb]; nb++);
	  argMapPlaneId = g_malloc(sizeof(int) * (nb + 1));
	  nb = 1;
	  for (i = 0; tokens[i]; i++)
	    {
	      res = sscanf(tokens[i], "%d", argMapPlaneId + nb);
	      if (res != 1 || argMapPlaneId[nb] < 0)
		g_warning("Parse error reading values from option "
			  "'build-map', unknown plane id '%s'.\n", tokens[i]);
	      else
		nb += 1;
	    }
	  argMapPlaneId[0] = nb - 1;
	  g_strfreev(tokens);
	  break;
	case 'n':
	  res = sscanf(optarg, "%d", &argNIsoLines);
	  if (res != 1 || argNIsoLines < 0)
	    {
	      g_warning("Wrong format for n-iso-lines option (id >= 0 awaited).\n");
	      argNIsoLines = 0;
	      break;
	    }
	  DBG_fprintf(stderr, "Visu Command: set the number of isolines"
		      " for coloured map '%d'.\n", argNIsoLines);
	  break;
	case 'c':
	  DBG_fprintf(stderr, "Visu Command: set the filenane for colorization to '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'colorize' needs a parameter.\n");
	    }
	  argColorizeFileName = g_strdup(optarg);
	  break;
	case 'u':
	  DBG_fprintf(stderr, "Visu Command: set the used columns for"
		      " colorize data (%s).\n", optarg);
	  res = sscanf(optarg, "%d:%d:%d", argColorizeColUsed,
		       argColorizeColUsed + 1, argColorizeColUsed + 2);
	  if (res != 3 || argColorizeColUsed[0] < -3 ||
	      argColorizeColUsed[1] < -3 || argColorizeColUsed[2] < -3)
	    {
	      g_warning("Wrong format for use-column"
			" option (<channel-1>:<channel-2>:<channel-3> awaited).\n");
	      for (i = 0; i < 2; i++)
		argColorizeColUsed[i] = 0;
	      break;
	    }
	  argColorizeColUsed_isPresent = TRUE;
	  break;
	case 'd':
	  DBG_fprintf(stderr, "Visu Command: set a previously defined color scheme (%s).\n", optarg);
	  res = sscanf(optarg, "%d", &argColorizePresetColor);
	  if (res != 1 || argColorizePresetColor < 0)
	    {
	      g_warning("Wrong format for color-preset"
			" option (positive integer awaited).\n");
	      argColorizePresetColor = -1;
	      break;
	    }
	  break;
	case 't':
	  DBG_fprintf(stderr, "Visu Command: set the translations (%s).\n", optarg);
	  res = sscanf(optarg, "%f:%f:%f", argTranslations,
		       argTranslations + 1, argTranslations + 2);
	  if (res != 3)
	    {
	      g_warning("Wrong format for translation"
			" option (<x>:<y>:<z> awaited).\n");
	      break;
	    }
	  else
	    argTranslationsIsSet = TRUE;
	  break;
	case 'x':
	  DBG_fprintf(stderr, "Visu Command: set the extension (%s).\n", optarg);
	  res = sscanf(optarg, "%f:%f:%f", argExtension,
		       argExtension + 1, argExtension + 2);
	  if (res != 3 || argExtension[0] < 0. ||
	      argExtension[1] < 0. || argExtension[2] < 0.)
	    {
	      g_warning("Wrong format for expand"
			" option (<x>:<y>:<z> awaited >= 0.).\n");
	      break;
	    }
	  else
	    argExpandIsSet = TRUE;
	  break;
	case 'p':
	  DBG_fprintf(stderr, "Visu Command: set the filenane for planes to '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'planes' needs a parameter.\n");
	    }
	  argPlanesFileName = g_strdup(optarg);
	  break;
	case 'f':
	  DBG_fprintf(stderr, "Visu Command: set the filenane for"
		      " a scalar field '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'scalar-field' needs a parameter.\n");
	    }
	  argScalarFieldFileName = g_strdup(optarg);
	  break;
	case 'v':
	  DBG_fprintf(stderr, "Visu Command: get the values for"
		      " a scalar field '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'iso-values' needs a parameter.\n");
	    }
	  tokens = g_strsplit(optarg, ":", -1);
	  /* Count number of tokens. */
	  for (nb = 0; tokens[nb]; nb++);
	  argIsoValues = g_malloc(sizeof(float) * (nb + 1));
	  argIsoNames = g_malloc(sizeof(gchar*) * (nb + 1));
	  nb = 0;
	  for (i = 0; tokens[i]; i++)
	    {
	      tokens2 = g_strsplit(tokens[i], "#", 2);
	      res = sscanf(tokens2[0], "%f", argIsoValues + nb);
	      if (res != 1)
		g_warning("Parse error reading values from option "
			  "'iso-values', unknown number '%s'.\n", tokens[i]);
	      else
		{
		  if (tokens2[1])
		    /* A name for the value is given. */
		    argIsoNames[nb] = g_strdup(tokens2[1]);
		  else
		    argIsoNames[nb] = (gchar*)0;
		  nb += 1;
		}
	      g_strfreev(tokens2);
	    }
	  argNbIsoValues = nb;
	  g_strfreev(tokens);
	  break;
	case 'i':
	  DBG_fprintf(stderr, "Visu Command: set the filenane for"
		      " an isosurfaces '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'iso-surfaces' needs a parameter.\n");
	    }
	  argIsoSurfacesFileName = g_strdup(optarg);
	  break;
	case 'o':
	  DBG_fprintf(stderr, "Visu Command: read an extended option.\n");
	  if (!optarg)
	    {
	      g_error("The option 'option' needs a parameter.\n");
	    }
	  tokens = g_strsplit(optarg, "=", 2);
	  if (!argOptionTable)
	    argOptionTable = toolOptionsNew_table();
	  if (tokens[1] && tokens[1][0] == 'T')
	    {
	      option = toolOptionsNew_optionBoolean(tokens[0], tokens[0]);
	      toolOptionsSet_valueBoolean(option, TRUE);
	    }
	  else if (tokens[1] && tokens[1][0] == 'F')
	    {
	      option = toolOptionsNew_optionBoolean(tokens[0], tokens[0]);
	      toolOptionsSet_valueBoolean(option, FALSE);
	    }
	  else if (tokens[1] && sscanf(tokens[1], "%f", &valueFloat) == 1)
	    {
	      option = toolOptionsNew_optionFloat(tokens[0], tokens[0]);
	      toolOptionsSet_valueFloat(option, valueFloat);
	    }
	  else if (tokens[1] && sscanf(tokens[1], "%d", &valueInt) == 1)
	    {
	      option = toolOptionsNew_optionInteger(tokens[0], tokens[0]);
	      toolOptionsSet_valueInteger(option, valueInt);
	    }
	  else if (!tokens[1])
	    {
	      option = toolOptionsNew_optionBoolean(tokens[0], tokens[0]);
	      toolOptionsSet_valueBoolean(option, TRUE);
	    }
	  else
	    {
	      g_error("Unparsable value for option 'option'.\n");
	    }
	  toolOptionsAdd_optionToTable(argOptionTable, option);
	  g_strfreev(tokens);
	  break;
	case 'w':
	  DBG_fprintf(stderr, "Visu Command: set the window mode to"
		      " '%s'.\n", optarg);
	  if (!optarg)
	    {
	      g_error("The option 'window-mode' needs a parameter.\n");
	    }
	  argWindowMode = g_strdup(optarg);
	  if (strcmp(argWindowMode, "classic") &&
	      strcmp(argWindowMode, "renderOnly") &&
	      strcmp(argWindowMode, "oneWindow"))
	    {
	      g_error("The option 'window-mode' accepts only 'calssic'"
		      ", 'oneWindow' and 'renderOnly' as a parameter.\n");
	    }
	  break;
	case 0:
	  /* Long option only. */
	  if (!strcmp(long_options[option_index].name, "fit-to-box"))
	    argFitToBox = (!strcmp(optarg, "TRUE"));
	  else if (!strcmp(long_options[option_index].name, "log-scale"))
	    {
	      if (sscanf(optarg, "%ud", &argLogScale) != 1)
		argLogScale = linear;
	      if (argLogScale >= n_scaleflag)
		argLogScale = linear;
	    }
	  else if (!strcmp(long_options[option_index].name, "bg-image"))
	    {
	      if (strcmp(optarg, "V_Sim"))
		argBgImageFile = g_strdup(optarg);
	      else
		argBgImageFile = g_build_filename(V_SIM_PIXMAPS_DIR,
						  "logo_grey.png", NULL);
	    }
	  else if (!strcmp(long_options[option_index].name, "color-iso-lines"))
	    {
	      if (sscanf(optarg, "[%f:%f:%f]", argIsoLinesColor,
			 argIsoLinesColor + 1, argIsoLinesColor + 2) != 3)
		{
		  if (!strcmp(optarg, "auto") || !strcmp(optarg, "Auto"))
		    {
		      g_free(argIsoLinesColor);
		      argIsoLinesColor = (float*)0;
		    }
		  else
		    {
		      g_error("Can't read any color from '%s' following"
			      " the [R:G:B] scheme or the 'auto' keyword.\n", optarg);
		    }
		}
	      if (argIsoLinesColor)
		{
		  argIsoLinesColor[0] = CLAMP(argIsoLinesColor[0], 0.f, 1.f);
		  argIsoLinesColor[1] = CLAMP(argIsoLinesColor[1], 0.f, 1.f);
		  argIsoLinesColor[2] = CLAMP(argIsoLinesColor[2], 0.f, 1.f);
		}
	    }
	  else if (!strcmp(long_options[option_index].name, "i-set"))
	    {
	      if (sscanf(optarg, "%u", &argISet))
		{
		  g_warning("Wrong format for i-set"
			    " option (integer awaited >= 0).\n");
		  break;
		}
	      else
		argISet = 0;
	    }
	  else if (!strcmp(long_options[option_index].name, "value-file"))
	    argValueFile = g_strdup(optarg);
	  else if (!strcmp(long_options[option_index].name, "map-precision"))
	    {
	      if (sscanf(optarg, "%u", &argMapPrecision) != 1 || argMapPrecision == 0)
		{
		  g_warning("Wrong value for map-precision"
			    " option (integer awaited > 0).\n");
		  argMapPrecision = 100;
		  break;
		}
	      else
		argMapPrecision = 100;
	    }
	  else
	    {
	      g_error("Unknown long option '--%s'.\n",
		      long_options[option_index].name);
	    }
	  break;
	default:
	  g_warning("Unknown option '%s'.", argv[optind - 1]);
	  printInfoMessage();
	  exit(0);
	}
    }

  if (argc - optind == 1)
    {
      DBG_fprintf(stderr, "Visu Command: there is one argument '%s'.\n", argv[optind]);
      argFileName = normalize_path(argv[optind]);
    }
  else if (argc - optind == 0)
    return 0;
  else
    g_warning("This program allows only up to one argument.\n");

  /* Consistency check between options. */
  if (argScalarFieldFileName && argIsoSurfacesFileName)
    {
      g_error("The options --iso-surfaces and --scalar-field are exclusive.");
    }
  if (argValueFile && argPlanesFileName)
    {
      g_error("The options --planes and --value-file are exclusive.");
    }

  return 0;
}

char* commandLineGet_ArgFilename()
{
  return argFileName;
}

char* commandLineGet_ArgSpinFileName()
{
  return argSpinFileName;
}

int commandLineGet_WithGtk()
{
  return withGtk;
}

char* commandLineGet_ExportFileName()
{
  return argExportFileName;
}

void commandLineGet_XWindowGeometry(int *width, int *height)
{
  *width = xWindowWidth;
  *height = xWindowHeight;
}
gchar* commandLineGet_colorizeFileName()
{
  return argColorizeFileName;
}
int* commandLineGet_colorizeColUsed()
{
  if (argColorizeColUsed_isPresent)
    return argColorizeColUsed;
  else
    return (int*)0;
}
float* commandLineGet_translation()
{
  if (argTranslationsIsSet)
    return argTranslations;
  else
    return (float*)0;
}
float* commandLineGet_extension()
{
  if (argExpandIsSet)
    return argExtension;
  else
    return (float*)0;
}
int commandLineGet_presetColor()
{
  return argColorizePresetColor;
}
gchar* commandLineGet_planesFileName()
{
  return argPlanesFileName;
}
int commandLineGet_spinHidingMode()
{
  return spinHidingMode;
}
gboolean commandLineGet_spinAndAtomic()
{
  return spinAndAtomic;
}
float* commandLineGet_isoValues(int *nb)
{
  g_return_val_if_fail(nb, (float*)0);

  *nb = argNbIsoValues;
  return argIsoValues;
}
gchar** commandLineGet_isoNames(int *nb)
{
  g_return_val_if_fail(nb, (gchar**)0);

  *nb = argNbIsoValues;
  return argIsoNames;
}
gchar* commandLineGet_scalarFieldFileName()
{
  return argScalarFieldFileName;
}
gchar* commandLineGet_isoSurfacesFileName()
{
  return argIsoSurfacesFileName;
}
gboolean commandLineGet_fitToBox()
{
  return argFitToBox;
}
OptionTable* commandLineGet_options()
{
  return argOptionTable;
}
gchar* commandLineGet_resourcesFile()
{
  return argResources;
}
int* commandLineGet_coloredMap()
{
  if (argMapPlaneId && argMapPlaneId[0])
    return argMapPlaneId;
  else
    return (int*)0;
}
matrix_scaleflag commandLineGet_logScale()
{
  return argLogScale;
}
guint commandLineGet_nIsoLines()
{
  return (guint)argNIsoLines;
}
gchar* commandLineGet_bgImage()
{
  return argBgImageFile;
}
float* commandLineGet_isoLinesColor()
{
  return argIsoLinesColor;
}
gchar* commandLineGet_windowMode()
{
  return argWindowMode;
}
guint commandLineGet_iSet()
{
  return (guint)argISet;
}
gchar* commandLineGet_valueFile()
{
  return argValueFile;
}
guint commandLineGet_mapPrecision()
{
  return argMapPrecision;
}

void commandLineFree_all()
{
  DBG_fprintf(stderr, "Command Line: free all.\n - the arguments\n");

  if (argFileName)
    g_free(argFileName);

  if (argResources)
    g_free(argResources);

  if (argSpinFileName)
    g_free(argSpinFileName);

  if (argExportFileName)
    g_free(argExportFileName);
  if (argWindowMode)
    g_free(argWindowMode);
  if (argValueFile)
    g_free(argValueFile);

  if (argColorizeFileName)
    g_free(argColorizeFileName);

  if (argScalarFieldFileName)
    g_free(argScalarFieldFileName);
  if (argIsoSurfacesFileName)
    g_free(argIsoSurfacesFileName);
  if (argIsoValues)
    g_free(argIsoValues);
  if (argIsoNames)
    g_strfreev(argIsoNames);

  if (argMapPlaneId)
    g_free(argMapPlaneId);
  if (argIsoLinesColor)
    g_free(argIsoLinesColor);

  if (argBgImageFile)
    g_free(argBgImageFile);

  if (argOptionTable)
    g_free(argOptionTable);

  if (argPlanesFileName)
    g_free(argPlanesFileName);

  DBG_fprintf(stderr, " - the descriptions\n");
  g_string_free(short_options, TRUE);
  g_free(ext_options);
  g_free(long_options);
}
