<?php
// $Horde: turba/lib/Driver.php,v 1.27.2.4 2004/10/10 04:12:01 chuck Exp $

require_once TURBA_BASE . '/lib/Turba.php';

/**
 * The Turba_Driver:: class provides a common abstracted interface to the
 * various directory search drivers.  It includes functions for searching,
 * adding, removing, and modifying directory entries.
 *
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @author  Jon Parise <jon@csh.rit.edu>
 * @version $Revision: 1.27.2.4 $
 * @since   Turba 0.0.1
 * @package turba
 */
class Turba_Driver {

    /** String holding the name of the current driver. */
    var $type = '';

    /** Hash holding the driver's additional parameters. */
    var $params = array();

    /**
     * Attempts to return a concrete Turba_Driver instance based on $driver.
     *
     * @param $driver   The type of Turba_Driver subclass to return.  The
     *                  code is dynamically included.
     * @param $params   Hash containing additional paramters to be passed to
     *                  the subclass's constructor.
     *
     * @return          The newly created concrete Turba_Driver instance, or
     *                  false on an error.
     */
    function &factory($driver, $params = array())
    {
        $driver = strtolower(basename($driver));
        include_once dirname(__FILE__) . '/Driver/' . $driver . '.php';
        $class = 'Turba_Driver_' . $driver;
        if (class_exists($class)) {
            return new $class($params);
        } else {
            Horde::fatal(new PEAR_Error(sprintf(_("Unable to load the definition of %s."), $class)), __FILE__, __LINE__);
        }
    }

    /**
     * Attempts to return a reference to a concrete Turba_Driver instance
     * based on $driver. It will only create a new instance if no
     * Turba_Driver instance with the same parameters currently exists.
     *
     * This method must be invoked as: $var = &Turba_Driver::singleton()
     *
     * @param $driver   The type of concrete Turba_Driver subclass to return.
     *                  This is based on the storage driver ($driver). The
     *                  code is dynamically included.
     * @param $params   A hash containing additional parameters for the
     *                  subclass.
     *
     * @return          The concrete Turba_Driver reference, or false on an
     *                  error.
     */
    function &singleton($driver, $params)
    {
        static $instances;

        if (!isset($instances)) $instances = array();

        $signature = md5(strtolower($driver) . '][' . @implode('][', $params));
        if (!isset($instances[$signature])) {
            $instances[$signature] = &Turba_Driver::factory($driver, $params);
        }

        return $instances[$signature];
    }

    /**
     * Returns the name of the current driver type.
     *
     * @return          String containing the name of the current driver type.
     */
    function getType()
    {
        return $this->type;
    }

    /**
     * Returns the current driver's additional parameters.
     *
     * @return          Hash containing the driver's additional parameters.
     */
    function getParams()
    {
        return $this->params;
    }


    /**
     ** Abstract stubs to be implemented by subclasses.
     **/

    /**
     * Searches the backend with the given criteria and returns a
     * filtered list of results. If no criteria are specified, all
     * records are returned.
     *
     * @param $criteria      Array containing the search criteria.
     * @param $fields        List of fields to return.
     * @param $strict_fields (optional) Array of fields which must be matched
     *                       exactly.
     * @param const $match   Do an 'and' or an 'or' search (defaults to TURBA_SEARCH_AND).
     *
     * @return               Hash containing the search results.
     */
    function search($criteria, $fields, $strict_fields = array(), $match = TURBA_SEARCH_AND)
    {
        // abstract
    }

    /**
     * Read the given data from the backend and returns
     * the result's fields.
     *
     * @param $criteria      Search criteria.
     * @param $id      	     Data identifier.
     * @param $fields        List of fields to return.
     *
     * @return               Hash containing the search results.
     */
    function read($criteria, $id, $fields)
    {
        // abstract
    }

    /**
     * Adds the specified object to the backend.
     */
    function addObject($attributes)
    {
        // abstract
    }

    /**
     * Deletes the specified object from the backend.
     */
    function removeObject($object_key, $object_id)
    {
        // abstract
    }

    /**
     * Saves the specified object to the backend.
     */
    function setObject($attributes)
    {
        // abstract
    }

}
