/*

Copyright (C) 2001  Gopal Narayanan <gopal@astro.umass.edu>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

*/

/* 
 * microstrip.c - Puts up window for microstrip and 
 * performs the associated calculations
 */


#include <stdio.h>
#include <gtk/gtk.h>
#include <string.h>
#include <math.h>

#include "microstrip.h"

char *drawing_filename;
char *image_dir;


/*
 * ereff - returns effective dielectric constant
 */

gfloat 
ereff (microstrip *ms)
{
  gfloat er, h, w, ht, t;
  gfloat ereff, w_h, ht_h;
  gfloat q, qc, qinf, qT;
  gfloat awh, b, j;
  gfloat fwh,c;

  er = ms->er;
  h = ms->h;
  w = ms->w;
  t = ms->t;
  ht = ms->ht;
  ht_h = ht/h;
  w_h = w/h;

  /* the following q formulae are from Brian Wadell's book
   * and seem to give too low ereff's
   * so go back to using Gupta, Garg and Bahl calculations 
   */
  
  /* if top height is not a large number use calculation for enclosure
   * er_eff = (er+1)/2 + (q*(er-1)/2),
   * where q = qc * (qinf - qT)
   * where qc = correction for effect of shiel
   * qinf = filling factor with no cover and zero-thickness strip
   * qT = correction for finite thickness of strip
   */

  if ((w/h) <= 1.0) {
    fwh = (pow((1. + (12.*h/w)),-0.5)) +
	   (0.04 * pow((1.  - (w/h)),2.0));
  } else {
    fwh = pow((1. + (12.*h/w)),-0.5);
  }

  c = ((er - 1.)/4.6) * ((t/h)/sqrt(w_h));

  if (ht_h <= 1000.0) {

/*      awh = 1.0 + ((1.0/49.0)* log(((pow(w_h,2.0)*(pow(w_h,2.0) + (pow((1.0/52.),2.0))))/ */
/*  				  (pow(w_h,4.0) + 0.432)) +  */
/*  				 ((1.0/18.7) * log(1.0 + pow((w_h/18.1),3.0))))); */
/*      b = -0.564 * pow(((er-0.9)/(er+3.0)),0.053); */
/*      j = awh * b; */
    
/*      qinf = pow((1.0 + (10.0/w_h)), j); */

/*      qT = (2.0 * t * log(2.))/(M_PI * h * sqrt(w_h)); */

/*      qc = tanh(1.043 + (0.121 * ht_h) - (1.164/ht_h)); */
    
/*      q = qc * (qinf - qT); */
/*      ereff = ((er + 1)/2.0) + (q * (er - 1.)/2.); */
/*      printf ("ereff = %g\n",ereff); */


    ereff = ((er + 1.)/2.) + (((er - 1.0)/2.) *
			      fwh * 
			      tanh(0.18 + (0.235*ht_h) - (0.415/pow(ht_h,2.))));
    ereff = ereff - c;
    return ereff;
  } else {
    ereff = ((er + 1.)/2.) + (((er - 1.0)/2.) * fwh) - c;
    return ereff;
  }
}

/*
 * mur_eff - returns effective magnetic permeability
 */

gfloat 
mur_eff (microstrip *ms)
{
  gfloat mur, h, w;
  gfloat mureff;
  
  mur = ms->mur;
  h = ms->h;
  w = ms->w;
  
  mureff = (2.0 * mur) / ((1.0 + mur) + ((1.0 - mur) *
					 pow((1.0 + (10.0 * h/w)),-0.5)));    
  return mureff;
}


/*
 * w_eff - returns effective width after correcting for 
 * thickness of trace
 */

gfloat 
w_eff (microstrip *ms)
{
  gfloat deltaw, deltaw_p, weff;
  gfloat er, w, h, t, w_h;
  
  er = ms->er;
  w = ms->w;
  h = ms->h;
  t = ms->t;
  w_h = w/h;

  /* stay with Gupta, Garg & Bahl for thickness based change of width */
  if (t > 0.0) {
/*      deltaw = (t/M_PI) * log((4.0 * M_E) / */
/*  			    sqrt(pow((t/h),2.0) +  */
/*  				 pow(((1./M_PI)/((w/t) + 1.1)),2.0))); */
/*      deltaw_p = deltaw * ((1.0 + (1.0/er))/2.0); */
/*      weff = w + deltaw_p; */
    if (w_h <= (1/(2.*M_PI))) {
      weff = w + ((1.25/M_PI) * t * (1 + log(4. * M_PI * w/t)));
    } else {
      weff = w + ((1.25/M_PI) * t * (1 + log(2 * h/t)));
    }
    return weff;
  } else {
    return w;
  }
}

/* 
 * er_eff_freq - returns modified er_eff as a function 
 * of frequency
 */

gfloat 
er_eff_freq (microstrip *ms)
{
  gfloat P1, P2, P3, P4, Pf;
  gfloat er_eff_fr;
  gfloat er, er_eff, w, h, f;

  er = ms->er;
  w = ms->w;
  h = ms->h;
  f = ms->f;
  er_eff = ms->er_eff;

  f = f * conv_freq[FREQ_HZ][FREQ_GHZ];
  h = h * conv_length[LENGTH_M][LENGTH_CM];
  w = w * conv_length[LENGTH_M][LENGTH_CM];

  P1 = 0.27488 + (0.6315 + ((0.525/(pow((1. + (0.157 * f * h)), 20.0))) * (w/h)))
    -(0.065683 * exp(-0.87513 * w/h));
  
  P2 = 0.33622 * (1.0 - exp(-0.03442 * er));

  P3 = 0.0363 * (exp(-4.6 * w/h)) * (1.0 - exp(-pow((f*h/3.87),4.97)));

  P4 = 1.0 + 2.751 * (1.0 - exp(-(pow((er/15.916),8.0))));

  Pf = P1 * P2 * pow(((0.1844 + (P3 * P4)) * (10.0 * f * h)),1.5763);

  er_eff_fr = er - ((er - er_eff)/(1.0 + Pf));
  
  return er_eff_fr;
}

/*
 * alpha_c - returns conductor losses in microstrip
 */

gfloat 
alpha_c (microstrip *ms)
{
  gfloat mur, h, w, Z0, t, sigma, f;
  gfloat weff;
  gfloat w_h, weff_h;
  gfloat Rs;
  gfloat dw_dt;
  gfloat ac;
  gfloat a, b;
  gfloat ereff;

  w = ms->w;
  weff = ms->w_eff;
  mur = ms->mur;
  h = ms->h;
  t = ms->t;  
  f = ms->f;  
  sigma = ms->sigma;
  Z0 = ms->Z0;
  w_h = w/h;
  weff_h = weff/h;
  ereff = ms->er_eff;

  Rs = sqrt((M_PI * f * mur* MU0)/sigma);
  
  /* Following from Wadell - for consistency use Gupta, Garg & Bahl */
  /*
  if ((w/h) <= (1./(2.*M_PI))) {
    dw_dt = (1.0/M_PI) * log((4.0 * M_PI * w)/t);
  } else {
    dw_dt = (1.0/M_PI) * log((2.0 * h)/t);
  }

  if ((w/h) <= 1) {
    ac = ((10.0 * Rs)/(M_PI * log(10.))) * ((((8. * h/w) - (w/(4. * h))) *
					     (1.0 + (h/w) + ((h/w)*dw_dt))) /
					    (h * Z0 * exp(Z0/60.)));
  } else {
    ac = ((Z0 * Rs)/(720.0 * pow(M_PI,2.) * h * log(10.0))) * 
      (1.0 + (0.44 * pow(h,2.0)/pow(w,2.0)) + ((6.0*pow(h,2.0)/pow(w,2.0)) * 
					       pow((1.0 - (h/w)), 5.0))) *
      (1.0 + (w/h) + dw_dt);
  }
  */
  
  if (w_h <= (1./(2.*M_PI))) {
    b = 2. * M_PI * w;
  } else {
    b = h;
  }


  if (t > 0.0)  {
    a = 1 + ((1./weff_h) * (1. + ((1.25/M_PI) * log(2. * b/t))));
  } else {
    return -1;
  }
  if (w_h <= 1) {
    ac = 1.38 * a * (Rs / (h * Z0)) * (32. - pow(weff_h,2.0)/
				       (32. + pow(weff_h,2.0)));
  } else {
    ac = 6.1e-5 * a * ((Rs*Z0*ereff)/h) *
      (weff_h + ((0.667 * weff_h)/(weff_h + 1.444)));
  }
  return ac;
}

/*
 * alpha_c_roughness - returns conductor losses in microstrip
 * due to conductor roughness
 */

gfloat 
alpha_c_roughness (microstrip *ms)
{
  gfloat mur, sigma, f, rough;
  gfloat ac, delta;
  gfloat ac_rough;

  mur = ms->mur;
  f = ms->f;  
  sigma = ms->sigma;
  ac = ms->atten_cond;
  rough = ms->rough;

  
  delta = 1.0 / sqrt(M_PI * f * mur * MU0 * sigma);
  ac_rough = ac * (1.0 + ((2.0/M_PI)* atan(1.40 * pow((rough/delta), 2.0))));

}


/*
 * alpha_dielectric - returns dielectric losses in microstrip
 */

gfloat 
alpha_dielectric (microstrip *ms)
{
  gfloat er_eff, er, tand, mur, f;
  gfloat q, lambda_g, ad;

  er_eff  = ms->er_eff;
  er = ms->er;
  tand = ms->tand;
  mur = ms->mur;
  f = ms->f;  

  /* from Wadell */
  /*
  q = ((er_eff - 1.0)/(er_eff - (er_eff/er)));
  lambda_g = C/(f * sqrt(er*mur));

  ad = ((20.0 * M_PI)/log(10.)) * ((q * tand)/lambda_g);
  */

  /* from Gupta, Garg & Bahl */
  ad = 27.3 * (er/(er - 1.)) * ((er_eff - 1.0)/sqrt(er_eff)) * tand * (C/f);
  return ad;
}

/*
 * impedance of covered microstrip line
 * Ref: Pg. 120-122 Wadell
 */

gfloat 
char_impedance_ht (microstrip *ms)
{
  gfloat h, weff, w, er_eff, mur_eff;
  gfloat imp, w_h, weff_h, ht, ht_h, fwh, P, Q;

  h = ms->h;
  w = ms->w;
  weff = ms->w_eff;
  ht = ms->ht;
  er_eff = ms->er_eff;
  mur_eff = ms->mur_eff;
  w_h = w/h;
  weff_h = weff/h;
  ht_h = ht/h;

  /* imp for er=1, ht = infinity*/
  /* Following from Wadell - gives too low answers something not right*/

  /*
  fwh = 6.0 + ((2.0 * M_PI) - 6.0) * exp(-pow((30.666/w_h),0.7528));
  imp = ((120. * M_PI)/(2.0*M_PI)) * log((fwh/w_h) + sqrt(1.0 + 
							  pow((2.0/w_h),2.0)));
  P = 270.0 * (1.0 - tanh(1.192 + (0.706*sqrt(1. - (1/ht_h))) - 
			  (1.389/(1.0 + ht_h))));
  Q = 1.0109 - atanh(((0.012*w_h) + (0.177*pow(w_h,2.0)) - (0.027*pow(w_h,3.0)))/
		     (pow((1.0+ht_h),2.0)));
  imp = (imp - (P*Q)) * sqrt(mur_eff/er_eff);
  */


  /* The following is from Gupta, Garg and Bahl - use this instead */


  P = 270.0 * (1.0 - tanh(0.28 + (1.2 * sqrt(ht_h))));
  Q = 1 - tanh(1.0 + ((0.48 * sqrt(w_h - 1.0))/pow((1+ht_h),2.0)));
  if (w_h <= 1.){
    imp = (((120.0 * M_PI)/(2 * M_PI * sqrt(er_eff))) * 
	   log((8./weff_h) + (0.25 * weff_h))) - (P/sqrt(er_eff));
  } else {
    imp = ((120. * M_PI)/sqrt(er_eff))*1./(weff_h+1.393+(0.667*log(weff_h+1.444)));
    imp = imp - (P*Q/sqrt(er_eff));
  }


  return imp;
}


/*
 * Z0
 */

gfloat 
char_impedance (microstrip *ms)
{
  gfloat h, weff, w, er_eff, mur_eff;
  gfloat imp, weff_h, w_h, ht;

  h = ms->h;
  weff = ms->w_eff;
  w = ms->w;
  ht = ms->ht;
  er_eff = ms->er_eff;
  mur_eff = ms->mur_eff;
  w_h = w/h;
  weff_h = weff/h;

  /*
   * The following is from Brian Wadell's book: Transmission Line Design Handbook
   * and seems to give slightly higher impedances than other formulae 
  imp = (((120.0 * M_PI) / (2. * sqrt(2.) * M_PI * sqrt(er_eff + 1.0))) * 
	 log((1.0 + (((4. * h)/w) * ((((14.0 + (8.0/er_eff))/11.0) * (4.*h/w)) +
				     sqrt((pow(((14.0 + (8.0/er_eff))/11.0),2.0) * 
					   pow((4.0 * h /w), 2.0) + 
					   (((1.0 + (1.0/er_eff))/2.) * pow(M_PI,2.)))))))));
  */

  /* account for cover height - I use an arbitrary ht/h determination to 
   * decide if I should use the more accurate formula
   */
  
  if ((ht/h) <= 1000.0) {
    imp = char_impedance_ht (ms);
    return imp;
  } else {

    /* if mur > 1.0, then Z0 = Z0(er=1) * sqrt(mur_eff/er_eff) ref: Wadell*/
    if (mur_eff != 1.0) er_eff = 1.0;
    
    /* The following is from
     *  ref. Microstrip Lines & Slotlines
     *  by Gupta, Garg and Bahl, Pub: Artech (1979); pp 89-93
     */
    if (w_h <= 1.){ 
      imp = ((120.0 * M_PI)/(2 * M_PI * sqrt(er_eff))) * log((8./weff_h) + (0.25 * weff_h));
    } else {
      imp = ((120. * M_PI)/sqrt(er_eff))*1./(weff_h+1.393+(0.667*log(weff_h+1.444)));
    }

    if (mur_eff != 1.0) imp = imp * sqrt(mur_eff/ms->er_eff);
    return imp;
  }
}


/*
 * synth_width - calculate width given Z0 and e_r
 */

gfloat 
synth_width (microstrip *ms)
{
  gfloat Z0, e_r, a, b, h;
  gfloat w_h, w;

  e_r = ms->er;
  Z0 = ms->Z0;
  h = ms->h;


  a = ((Z0/60.) * sqrt((e_r+1)/2.)) + ((e_r-1)/(e_r+1) * (0.23 +
							 (0.11/e_r)));
  b = 60. * pow(M_PI,2)/(Z0 * sqrt(e_r));

  if (a > 1.52){
    w_h = 8 * exp(a) / (exp(2. * a) - 2);
  } else {
    w_h = (2./M_PI) * (b - 1. -log((2 * b) - 1.) + ((e_r - 1)/(2 * e_r))*
                   (log(b - 1.) + 0.39 - 0.61/e_r));
  }
  
  if (h > 0.0) {
    w = w_h * h;
    return w;
  } else {
    exit(-1);
  }
}


/*
 * skin_depth - calculate skin depth
 */

gfloat 
skin_depth (microstrip *ms)
{
  gfloat mur, sigma, f;
  gfloat depth;

  mur = ms->mur;
  f = ms->f;  
  sigma = ms->sigma;

  depth = 1.0/(sqrt(M_PI * f * mur* MU0 * sigma));
  return depth;
}
  
/*
 * dispersion - calculate frequency dependencies of Z0 & er_eff 
 */
void 
dispersion (microstrip *ms)
{
  gfloat h, ereff, er, Z0;
  gfloat w, f;
  gfloat fereff, fZ0, fp, g;
  gfloat Z0T, we;


  h = ms->h;
  ereff = ms->er_eff;
  er = ms->er;
  Z0 = ms->Z0;
  w = ms->w;
  f = ms->f;

  fp = Z0 / (2.0 * MU0 * h);
  /*  g = (pow(M_PI,2.0) * (er - 1.0)/(12.0 * ereff)) * 
      sqrt((2.0 * M_PI * Z0)/(120.* M_PI));*/
  g = sqrt((Z0 - 5.)/60.0) + (0.004 * Z0);

  fereff = er - ((er - ereff)/(1.0 + (g * pow((f/fp),2.0))));


  /*Z0T = 2 * char imp of stripline of width w and height 2*h 
   * Reference for stripline impedance  Pozar p.179
   */

  if ((w/(2*h)) < 0.35) {
    we = w - pow((0.35 - (w/(2.*h))), 2.0);
  } else {
    we = w;
  }

  Z0T = (2. * 30.0 * M_PI * 2. * h)/(sqrt(er) * (we + (0.441 * 2. * h)));

  fZ0 = Z0T - ((Z0T - Z0)/(1. + (g * pow((f/fp),2.0))));

  ms->er_eff = fereff;
  ms->Z0 = fZ0;
}

/*
 * simple error message
 */
void 
error_mes (gchar *text)
{
  perror(text);
  exit(-1);
}


/*
 * get_microstrip_sub
 * get and assign microstrip substrate parameters
 * into microstrip structure
 */

void 
get_microstrip_sub (trans_win *mwin,
			 microstrip *ms)
{
  short curr_unit;


  if (sscanf (gtk_entry_get_text(GTK_ENTRY (mwin->subparam_text[0])), 
	      "%g", &ms->er) != 1) error_mes("Error: ms->er");

  if (sscanf (gtk_entry_get_text(GTK_ENTRY (mwin->subparam_text[1])), 
	      "%g", &ms->mur) != 1) error_mes("Error: ms->mur");

  if (sscanf (gtk_entry_get_text(GTK_ENTRY (mwin->subparam_text[2])), 
	      "%g", &ms->h) != 1) error_mes("Error: ms->h");
  curr_unit = getunit (gtk_entry_get_text (GTK_ENTRY 
					   (GTK_COMBO(mwin->subparam_combo[2])->entry)));
  ms->h = ms->h * conv_length[curr_unit][LENGTH_M];

  if (sscanf (gtk_entry_get_text(GTK_ENTRY (mwin->subparam_text[3])), 
	      "%g", &ms->ht) != 1) error_mes("Error: ms->ht");
  curr_unit = getunit (gtk_entry_get_text (GTK_ENTRY 
					   (GTK_COMBO(mwin->subparam_combo[3])->entry)));
  ms->ht = ms->ht * conv_length[curr_unit][LENGTH_M];

  if (sscanf (gtk_entry_get_text(GTK_ENTRY (mwin->subparam_text[4])), 
	      "%g", &ms->t) != 1) error_mes("Error: ms->t");
  curr_unit = getunit (gtk_entry_get_text (GTK_ENTRY 
					   (GTK_COMBO(mwin->subparam_combo[4])->entry)));
  ms->t = ms->t * conv_length[curr_unit][LENGTH_M];

  if (sscanf (gtk_entry_get_text(GTK_ENTRY (mwin->subparam_text[5])), 
	      "%g", &ms->sigma) != 1) error_mes("Error: ms->sigma");
  if (sscanf (gtk_entry_get_text(GTK_ENTRY (mwin->subparam_text[6])), 
	      "%g", &ms->tand) != 1) error_mes("Error: ms->tand");
  if (sscanf (gtk_entry_get_text(GTK_ENTRY (mwin->subparam_text[7])), 
	      "%g", &ms->rough) != 1) error_mes("Error: ms->rough");
  curr_unit = getunit (gtk_entry_get_text (GTK_ENTRY 
					   (GTK_COMBO(mwin->subparam_combo[7])->entry)));
  ms->rough = ms->rough * conv_length[curr_unit][LENGTH_M];
}

/*
 * get_microstrip_comp
 * get and assign microstrip component parameters
 * into microstrip structure
 */

void 
get_microstrip_comp (trans_win *mwin,
			  microstrip *ms)
{
  short curr_unit;

  if (sscanf (gtk_entry_get_text(GTK_ENTRY (mwin->component_param_text[0])), 
	      "%g", &ms->f) != 1) error_mes("Error: ms->f");
  curr_unit = getunit (gtk_entry_get_text (GTK_ENTRY 
					   (GTK_COMBO(mwin->component_param_combo[0])->entry)));
  ms->f = ms->f * conv_freq[curr_unit][FREQ_HZ];
}

/*
 * get_microstrip_elec
 * get and assign microstrip electrical parameters
 * into microstrip structure
 */

void 
get_microstrip_elec (trans_win *mwin,
			  microstrip *ms)
{
  short curr_unit;

  if (sscanf (gtk_entry_get_text(GTK_ENTRY (mwin->electrical_param_text[0])), 
	      "%g", &ms->Z0) != 1) error_mes("Error: ms->Z0");
  curr_unit = getunit (gtk_entry_get_text (GTK_ENTRY 
					   (GTK_COMBO(mwin->electrical_param_combo[0])->entry)));
  ms->Z0 = ms->Z0 * conv_res[curr_unit][RES_OHM];

  if (sscanf (gtk_entry_get_text(GTK_ENTRY (mwin->electrical_param_text[1])), 
	      "%g", &ms->ang_l) != 1) error_mes("Error: ms->ang_l");
  curr_unit = getunit (gtk_entry_get_text (GTK_ENTRY 
					   (GTK_COMBO(mwin->electrical_param_combo[1])->entry)));
  ms->ang_l = ms->ang_l * conv_ang[curr_unit][ANG_RAD];
}


/*
 * get_microstrip_phys
 * get and assign microstrip physical parameters
 * into microstrip structure
 */

void 
get_microstrip_phys (trans_win *mwin,
			  microstrip *ms)
{
  short curr_unit;

  if (sscanf (gtk_entry_get_text(GTK_ENTRY (mwin->physical_param_text[0])), 
	      "%g", &ms->w) != 1) error_mes("Error: ms->w");
  curr_unit = getunit (gtk_entry_get_text (GTK_ENTRY 
					   (GTK_COMBO(mwin->physical_param_combo[0])->entry)));
  ms->w = ms->w * conv_length[curr_unit][LENGTH_M];

  if (sscanf (gtk_entry_get_text(GTK_ENTRY (mwin->physical_param_text[1])), 
	      "%g", &ms->l) != 1) error_mes("Error: ms->l");
  curr_unit = getunit (gtk_entry_get_text (GTK_ENTRY 
					   (GTK_COMBO(mwin->physical_param_combo[1])->entry)));
  ms->l = ms->l * conv_length[curr_unit][LENGTH_M];

}


/*
 * analysis function
 */

void 
analyze_microstrip (GtkWidget *widget,
			 trans_win *mwin)
{
  /*  trans_win *mwin = (trans_win *) data;*/
  gchar *text, *results;
  microstrip *ms;
  short required_unit;
  gfloat lambda_g;


  /*allocate memory for text */
  if ((text = (char *) malloc(10*sizeof(char)))  == NULL){
    perror("text error: malloc");
    exit -1;
  }


  
  /*allocate memory for pointer mw */
  if ((ms = g_malloc(sizeof *ms)) != NULL)
    {

      /* Get and assign substrate parameters */
      get_microstrip_sub(mwin, ms);

      /* Get and assign component parameters */
      get_microstrip_comp(mwin, ms);
      
      /* Get and assign physical parameters */
      get_microstrip_phys(mwin, ms);

      
      /* correct width for thickness of trace */
      ms->w_eff = w_eff (ms);

      
      /*Get effective dielectric constant */
      ms->er_eff = ereff(ms);

      /*ms->er_eff = er_eff_freq(ms);*/

      /* Get effective magnetic permeability */
      ms->mur_eff = mur_eff(ms);
      
      /* calculate Z0 */
      ms->Z0 = char_impedance (ms);
      
      /* calculate freq dependence of Z0 & er */
      dispersion (ms);


      required_unit = getunit (gtk_entry_get_text 
			       (GTK_ENTRY (GTK_COMBO(mwin->electrical_param_combo[0])->entry)));
      sprintf(text,"%g", (float) (ms->Z0*conv_res[RES_OHM][required_unit]));
      gtk_entry_set_text (GTK_ENTRY (mwin->electrical_param_text[0]), text);
      
      
      /* calculate electrical angle */
      lambda_g = C/(ms->f * sqrt(ms->mur_eff * ms->er_eff));
      ms->ang_l = (2.0 * M_PI * ms->l)/lambda_g;    /* in radians */


      required_unit = getunit (gtk_entry_get_text 
			       (GTK_ENTRY (GTK_COMBO(mwin->electrical_param_combo[1])->entry)));
      sprintf(text,"%g", (float) (ms->ang_l * conv_ang[ANG_RAD][required_unit]));
      gtk_entry_set_text (GTK_ENTRY (mwin->electrical_param_text[1]), text);  
      
      /* calculate conductor losses */
      ms->atten_cond = (alpha_c (ms) + alpha_c_roughness (ms)) * ms->l;

      
      /*calculate dielectric losses */
      ms->atten_dielectric = alpha_dielectric (ms) * ms->l;

      
      /* calculate skin depth */
      ms->skindepth = skin_depth (ms);

      
      free(text);
      if ((text = (char *) malloc(500*sizeof(char)))  == NULL){
	perror("results text error: malloc");
	exit -1;
      }


      if ((results = (char *) malloc(500*sizeof(char)))  == NULL){
	perror("results text error: malloc");
	exit -1;
      }


      required_unit = getunit(gtk_entry_get_text (GTK_ENTRY 
						  (GTK_COMBO 
						   (mwin->physical_param_combo[1])->entry)));
      sprintf(results, "Effective dielectic constant, er_eff = %.4g\n", ms->er_eff);
      sprintf(text,"Conductor Losses = %.4g dB\n", 
	      ms->atten_cond);
      strcat(results,text);
      sprintf(text, "Dielectric Losses = %.4g dB\n", 
	      ms->atten_dielectric);
      
      strcat(results,text);

      required_unit = getunit(gtk_entry_get_text (GTK_ENTRY 
						  (GTK_COMBO(mwin->subparam_combo[4])->entry)));
      sprintf(text, "Skin Depth = %.4g %s\n", 
	      (ms->skindepth * conv_length[LENGTH_M][required_unit]), 
	      length_unit_text (getunit (gtk_entry_get_text 
					 (GTK_ENTRY 
					  (GTK_COMBO(mwin->subparam_combo[4])->entry)))));
	strcat(results,text);
	gtk_label_set (GTK_LABEL (mwin->results_text), results);

	free(results);
	free(text);

	if (statusexists){
	  if (statusint != CONSISTENT) {
	    gtk_label_set_text (GTK_LABEL (mwin->status), "Values are consistent");
	  } 
	}
	statusint = CONSISTENT;
    } else {
      perror("malloc ms");
      exit -1;
    }

}


/*
 * synthesis function
 */

void 
synthesize_microstrip (GtkWidget *widget,
			 trans_win *mwin)
{
  gchar *text, *results;
  microstrip *ms;
  short required_unit;
  gfloat lambda_g;


  /*allocate memory for text */
  if ((text = (char *) malloc(10*sizeof(char)))  == NULL){
    perror("text error: malloc");
    exit -1;
  }


  
  /*allocate memory for pointer mw */
  if ((ms = g_malloc(sizeof *ms)) != NULL)
    {

      /* Get and assign substrate parameters */
      get_microstrip_sub(mwin, ms);

      /* Get and assign component parameters */
      get_microstrip_comp(mwin, ms);
      
      /* Get and assign electrical parameters */
      get_microstrip_elec(mwin, ms);

      /* calculate width */
      ms->w = synth_width (ms);
      required_unit = getunit (gtk_entry_get_text 
			       (GTK_ENTRY (GTK_COMBO(mwin->physical_param_combo[0])->entry)));
      sprintf(text,"%g", (float) (ms->w * conv_length[LENGTH_M][required_unit]));
      gtk_entry_set_text (GTK_ENTRY (mwin->physical_param_text[0]), text); 
      

      /* Get effective dielectric constant */
      ms->er_eff = ereff(ms);
      ms->er_eff = er_eff_freq(ms);
      
      /* Get effective magnetic permeability */
      ms->mur_eff = mur_eff(ms);

      /* calculate physical length */

      lambda_g = C/(ms->f * sqrt(ms->mur_eff * ms->er_eff));
      ms->l = (ms->ang_l * lambda_g)/(2.0 * M_PI);    /* in m */

      required_unit = getunit (gtk_entry_get_text 
			       (GTK_ENTRY (GTK_COMBO(mwin->physical_param_combo[1])->entry)));
      sprintf(text,"%g", (float) (ms->l * conv_length[LENGTH_M][required_unit]));
      gtk_entry_set_text (GTK_ENTRY (mwin->physical_param_text[1]), text);  

      free(text);
      if ((text = (char *) malloc(500*sizeof(char)))  == NULL){
	perror("results text error: malloc");
	exit -1;
      }

      if ((results = (char *) malloc(500*sizeof(char)))  == NULL){
	perror("results text error: malloc");
	exit -1;
      }

      sprintf(results, "Effective dielectic constant, er_eff = %.4g\n", ms->er_eff);
      /*      strcat (results,"\nAnalyze and tweak values \nto get more accurate results\n");*/
      
      gtk_label_set (GTK_LABEL (mwin->results_text), results);
      
      free(results);
      free(text);
      if (statusexists){
	if (statusint != CONSISTENT) {
	  gtk_label_set_text (GTK_LABEL (mwin->status), "Tweak values and analyze to get more accurate synthesis");
	} 
      }
      statusint = CONSISTENT;
    } else {
      perror("malloc ms");
      exit -1;
    }

}

/*
 * the window aspect 
 */

/*void microstrip_win (GtkWidget *parent)*/
void 
microstrip_win (GtkWidget *parent)
/*microstrip_win (trans_gui *tg)*/
{
  short row, col;



  /* if there is a window that already exists kill it first */
  if (main_body_window != NULL)   {
    gtk_widget_destroy(main_body_window);
    twin = g_malloc(sizeof(*twin));
  }

  
  setup_transgui(MICROSTRIP, parent, twin);
  
  for (row = 0; row<=1; row++){
    gtk_widget_set_sensitive (GTK_WIDGET (twin->physical_param_fix[row]), FALSE);
  }

  gtk_signal_connect (GTK_OBJECT (twin->Analbutton), "clicked",
		      GTK_SIGNAL_FUNC (analyze_microstrip), twin); 

  gtk_signal_connect (GTK_OBJECT (twin->Synbutton), "clicked",
		      GTK_SIGNAL_FUNC (synthesize_microstrip), twin); 
  /*		      GTK_SIGNAL_FUNC (synthesize_microstrip), twin); */
  if (statusint == INCONSISTENT)  {
    analyze_microstrip(parent, (trans_win *) twin);
  }
  
  for (row = 0; row<=1; row++){
  gtk_signal_connect (GTK_OBJECT (twin->electrical_param_text[row]), "changed",
		      GTK_SIGNAL_FUNC (setstatus), twin);
  }


  /*  return twin;*/
  /*  free(twin);*/
}


