/*
 *  Copyright (C) 2000 by Marco G"otze.
 *
 *  This code is part of the ThoughtTracker source package, which is
 *  distributed under the terms of the GNU GPL2.
 */

#ifndef MENUBAR_H_INCLUDED
#define MENUBAR_H_INCLUDED

#include <cstring>
#include <vector>

#include <gtk--/editable.h>
#include <gtk--/menubar.h>
#include <gtk--/menuitem.h>

#include "thoughttracker.h"
#include "database.h"

class TTApplication;

class TTMenuBar : public Gtk::MenuBar {
  public:
    TTMenuBar(TTApplication *application);
    virtual ~TTMenuBar() {};

    /* menu item functionalities */
    virtual void menu_options();
    virtual void menu_quit(); 
    virtual void menu_copy();
    virtual void menu_paste();
    virtual void menu_add_bookmark();
    virtual void menu_del_bookmark(bool complain = false);
    virtual void menu_about();

    /* updates the state of some menu items (disabled/enabled) */
    virtual void update();

    /* enables or disabled clipboard functions; the type int return value
       and the parameter <dummy> are provided for convenience since this
       function is meant to be connected to focus change signals */
    virtual int set_cb_widget(GdkEventFocus *dummy, Gtk::Editable *widget);

    /* updates record #<id>'s bookmark both internally and in the menu;
       calling this function is also the way to go after an entry has been
       deleted; if <id> is < 0, ALL bookmarks will be updated */
    virtual void update_bookmark(dbid_t id);

    /* clears the list/menu of bookmarks */
    virtual void clear_bookmarks();

    /* returns a string representing the current list of bookmarks, intended
       for use in a config file */
    virtual string get_bm_string();

    /* sets the bookmark list (and builds the menu) by parsing a string as
       returned by get_bm_string(); returns the number of bookmarks deduced
       from the string (if lower than expected, there either were parse
       errors, or some records didn't exist any more) */
    virtual int parse_bm_string(const string&);

  protected:
    friend TTEntryWidget;

    typedef struct {
      dbid_t id;    // record id
      string text;  // summary = bookmark menu item text
    } ttbm_t;

    TTApplication *app;
    struct {  // relevant widgets
      Gtk::MenuItem *im, *ex;                // File/{Import,Export}
      Gtk::MenuItem *copy, *paste, *revert;  // Edit/{Copy,Paste;Revert}
      Gtk::Menu *bm;                         // Bookmarks
      Gtk::MenuItem *add, *del;              // Bookmarks/{Add,Del}
    } w;
    struct {  // clipboard data
      Gtk::Editable *widget;
      string buffer;
    } cb;
    vector<ttbm_t> bm;  // bookmarks (record IDs)

    /* opens a bookmarked entry in the entry widget */
    virtual void bm_recall(dbid_t);

    /* clears the bookmark menu (but not the variable bm) */
    virtual void bm_clear_menu();

    /* builds the bookmark menu (from the variable bm); clears it beforehand,
       sorts bm first if <sort_first> is true */
    virtual void bm_build_menu(bool sort_first = true);

    /* adds record #<id> to the (internal) list of bookmarks; returns true
       if the bookmarks was successfully added, false otherwise; does neither
       check whether #<id> is already bookmarked, nor sorts bm after the
       addition */
    virtual bool bm_add(dbid_t id);

    /* removes record #<id> from the (internal) list of bookmarks; returns
       true if the bookmark existed, false otherwise */
    virtual bool bm_del(dbid_t id);

    /* compare function for sorting ttbm_t records */
    static int bm_cmp(const ttbm_t &a, const ttbm_t &b) {
#ifdef HAVE_STRCOLL
      return strcoll(a.text.c_str(), b.text.c_str()) < 0;
#else
      return strcmp(a.text.c_str(), b.text.c_str()) < 0;
#endif
    }
};

#endif  /* MENUBAR_H_INCLUDED */

