
/******************************************************************************
* MODULE     : widget.gen.h
* DESCRIPTION: Definition of abstract widgets
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include <event.gen.h>

#module widget
#import event
#import array (string)

/******************************************************************************
* The abstract widget class
******************************************************************************/

class widget_rep;

struct widget {
#import abstract_null (widget, widget_rep)
  inline widget operator [] (int i);
         widget operator [] (string s);
  inline operator tree ();
  inline bool operator == (widget w);
  inline bool operator != (widget w);
};

#import array (widget)

struct widget_rep: abstract_struct {
  display       dis;              // underlying display
  window        win;              // underlying window
  SI            ox, oy;           // origin of widget in window
  SI            w, h;             // width and height of widget
  gravity       grav;             // position of the origin in the widget
  array<widget> a;                // children of widget
  array<string> name;             // names for the children

  widget_rep (display dis, array<widget> a, array<string> name, gravity grav);
  virtual ~widget_rep ();

  virtual operator tree () = 0;
  virtual bool handle (event ev) = 0;

  SI       x1 (); SI y1 (); // lower left window coordinates of widget
  SI       x2 (); SI y2 (); // upper right window coordinates of widget
  bool     attached ();
  volatile void fatal_error (string message, string in="", string fname="");

  friend   class widget;
};

#import code_abstract_null (widget, widget_rep)
inline widget widget::operator [] (int i) { return rep->a[i]; }
inline widget::operator tree () { return (tree) (*rep); }
inline bool widget::operator == (widget w) { return rep == w.rep; }
inline bool widget::operator != (widget w) { return rep != w.rep; }

ostream& operator << (ostream& out, widget w);
widget operator << (widget w, event ev);

/******************************************************************************
* Exported special widgets
******************************************************************************/

widget horizontal_list (display dis, array<widget> a);
widget horizontal_list (display dis, array<widget> a, array<string> name);
widget vertical_list (display dis, array<widget> a);
widget vertical_list (display dis, array<widget> a, array<string> name);
widget vertical_menu (display dis, array<widget> a);
widget tile (display dis, array<widget> a, int cols);
widget tile (display dis, array<widget> a, int cols, array<string> name);
widget horizontal_array (display dis, array<widget> a, int stretch_me= -1);
widget horizontal_array (display dis, array<widget> a, array<string> name,
			 int stretch_me= -1);
widget glue_widget (display dis, bool hx=TRUE, bool vx=TRUE, SI w=0, SI h=0);
widget separator_widget (display dis, SI pre=0, SI post=0, bool vert=FALSE);
widget text_widget (display dis, string s, bool transp= FALSE,
		    string lan="", bool tt= FALSE);
widget xpm_widget (display dis, string file_name, bool transp= TRUE);
widget command_button (widget w, command cmd, bool button_flag= FALSE);
widget command_button (widget lw, widget rw, command cmd);
widget pulldown_button (widget w, widget m);
widget pullright_button (widget w, widget m);
widget popup_widget (widget w, gravity quit=center);
widget canvas_widget (widget w, gravity grav=north_west);
widget input_text_widget (display dis, command call_back);
widget resize_widget (widget w, SI w, SI h, bool xr= FALSE, bool yr= FALSE);
widget file_chooser_widget (display dis, command cmd,
			    string type="TeXmacs", string magn="");
widget balloon_widget (widget w, widget help);
widget wait_widget (display dis, SI w, SI h, string message);

#endmodule // widget
