
/******************************************************************************
* MODULE     : composite_widget.gen.cc
* DESCRIPTION: composite list widgets
*              If there is space left in a list widget,
*              then all items are stretched proportionally
*              according to their sizes.
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#module code_list_widget

/******************************************************************************
* Horizontal lists
******************************************************************************/

class horizontal_list_rep: public composite_widget_rep {
public:
  horizontal_list_rep (display dis, array<widget> a);
  horizontal_list_rep (display dis, array<widget> a, array<string> name);
  operator tree ();

  void handle_get_size (get_size_event ev);
  void handle_position (position_event ev);
  void handle_find_child (find_child_event ev);
};

horizontal_list_rep::horizontal_list_rep (display dis, array<widget> a):
  composite_widget_rep (dis, a) {}

horizontal_list_rep::horizontal_list_rep (display dis, array<widget> a,
  array<string> n):
  composite_widget_rep (dis, a, n) {}

horizontal_list_rep::operator tree () {
  int i;
  tree t (TUPLE, N(a)+1);
  t[0]= "horizontal list";
  for (i=0; i<N(a); i++) t[i+1]= (tree) a[i];
  return t;
}

void
horizontal_list_rep::handle_get_size (get_size_event ev) {
  SI& w (ev->w);
  SI& h (ev->h);

  if (ev->mode==0) {
    SI ww= w, hh= h;
    this << get_size (ww, hh, 1);
    w= min (w, ww);
    h= min (h, hh);
    ww= w; hh= h;
    this << get_size (ww, hh, -1);
    w= max (w, ww);
    h= hh;
  }
  else {
    int i, ww=0, hh=0;
    for (i=0; i<N(a); i++) {
      int www= w/N(a), hhh= h;
      a[i] << get_size (www, hhh, ev->mode);
      ww= ww+ www;
      hh= max (hh, hhh);
    }
    w= ww; h= hh;
  }
}

void
horizontal_list_rep::handle_position (position_event ev) {
  (void) ev;
  if (N(a)==0) return;

  SI min_w=w, min_h= h;
  this << get_size (min_w, min_h, -1);
  SI max_w=w, max_h= h;
  this << get_size (max_w, max_h,  1);
  double stretch;
  if ((max_w==min_w) || (w<min_w)) stretch= 0.0;
  else if (w>max_w) stretch=1.0;
  else stretch= ((double) (w-min_w))/((double) (max_w-min_w));

  int i;
  SI  cur_w=0;
  for (i=0; i<N(a); i++) {
    SI the_w, the_h= h;
    if (i<N(a)-1) {
      min_w= w/N(a), min_h= h;
      a[i] << get_size (min_w, min_h, -1);
      max_w= w/N(a), max_h= h;
      a[i] << get_size (max_w, max_h,  1);
      the_w= (SI) (min_w+ stretch* (max_w- min_w));
    }
    else the_w= w- cur_w;
    abs_round (the_w);
    a[i] << emit_position (cur_w, 0, the_w, the_h);
    cur_w+=the_w;
  }
}

void
horizontal_list_rep::handle_find_child (find_child_event ev) {
  int& i (ev->which);
  for (i=0; i<N(a); i++)
    if ((ev->x >= a[i]->x1()-ox) && (ev->x < a[i]->x2()-ox)) return;
  i= -1;
}

/******************************************************************************
* Vertical lists
******************************************************************************/

class vertical_list_rep: public composite_widget_rep {
  bool menu_flag;
public:
  vertical_list_rep (display dis, array<widget> a, bool mf= FALSE);
  vertical_list_rep (display dis, array<widget> a, array<string> name);
  operator tree ();

  void handle_get_size (get_size_event ev);
  void handle_position (position_event ev);
  void handle_find_child (find_child_event ev);
};

vertical_list_rep::vertical_list_rep (display dis, array<widget> a, bool mf):
  composite_widget_rep (dis, a), menu_flag (mf) {}

vertical_list_rep::vertical_list_rep (display dis, array<widget> a,
  array<string> name):
    composite_widget_rep (dis, a, name), menu_flag (FALSE) {}

vertical_list_rep::operator tree () {
  int i;
  tree t (TUPLE, N(a)+1);
  t[0]= "vertical list";
  for (i=0; i<N(a); i++) t[i+1]= (tree) a[i];
  return t;
}

void
vertical_list_rep::handle_get_size (get_size_event ev) {
  SI& w (ev->w);
  SI& h (ev->h);

  if (menu_flag) {
    int i;
    SI m=0, test;
    for (i=0; i<N(a); i++) {
      a[i] << get_coord1 ("extra width", test);
      m= max (m, test);
    }
    for (i=0; i<N(a); i++)
      a[i] << set_coord1 ("extra width", m);
  }

  if (ev->mode==0) {
    SI ww= w, hh= h;
    this << get_size (ww, hh, 1);
    w= min (w, ww);
    h= min (h, hh);
    ww= w; hh= h;
    this << get_size (ww, hh, -1);
    w= ww;
    h= max (h, hh);
  }
  else {
    int i, ww=0, hh=0;
    for (i=0; i<N(a); i++) {
      int www= w, hhh= h/N(a);
      a[i] << get_size (www, hhh, ev->mode);
      ww= max (ww, www);
      hh= hh+ hhh;
    }
    w= ww; h= hh;
  }
}

void
vertical_list_rep::handle_position (position_event ev) {
  (void) ev;
  if (N(a)==0) return;

  SI min_w=w, min_h= h;
  this << get_size (min_w, min_h, -1);
  SI max_w=w, max_h= h;
  this << get_size (max_w, max_h,  1);
  double stretch;
  if ((max_h==min_h) || (h<min_h)) stretch= 0.0;
  else if (h>max_h) stretch=1.0;
  else stretch= ((double) (h-min_h))/((double) (max_h-min_h));

  int i;
  SI  cur_h=0;
  for (i=0; i<N(a); i++) {
    SI the_w= w, the_h;
    if (i<N(a)-1) {
      min_w= w, min_h= h/N(a);
      a[i] << get_size (min_w, min_h, -1);
      max_w= w, max_h= h/N(a);
      a[i] << get_size (max_w, max_h,  1);
      the_h= (SI) (min_h+ stretch* (max_h- min_h));
    }
    else the_h= h+ cur_h;
    abs_round (the_h);
    a[i] << emit_position (0, cur_h, the_w, the_h);
    cur_h-=the_h;
  }
}

void
vertical_list_rep::handle_find_child (find_child_event ev) {
  int& i (ev->which);
  for (i=0; i<N(a); i++)
    if ((ev->y >= a[i]->y1()-oy) && (ev->y < a[i]->y2()-oy)) return;
  i= -1;
}

/******************************************************************************
* Interface
******************************************************************************/

widget
horizontal_list (display dis, array<widget> a) {
  return new horizontal_list_rep (dis, a);
}

widget
horizontal_list (display dis, array<widget> a, array<string> name) {
  return new horizontal_list_rep (dis, a, name);
}

widget
vertical_list (display dis, array<widget> a) {
  return new vertical_list_rep (dis, a);
}

widget
vertical_list (display dis, array<widget> a, array<string> name) {
  return new vertical_list_rep (dis, a, name);
}

widget
vertical_menu (display dis, array<widget> a) {
  return new vertical_list_rep (dis, a, TRUE);
}

#endmodule // code_list_widget
