/*
	SRG - Squid Report Generator
	Copyright 2005 University of Waikato

	This file is part of SRG.

	SRG is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	SRG is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with SRG; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

*/
#include "srg.h"
#include "prototypes.h"

/* Check the environment for outputting reports */
void check_environment(void)
{
	
	/* Ensure Base Directory Exists */
	if (mkdir(srg.outputDir, 0755) == -1) {
		if (errno != EEXIST) {
			fprintf(stderr, "%s: Error creating base output "
					"directory: %s\n", progname, srg.outputDir);
			exit(1);
		}
	}

    /* Check for resources directory */
    if (access(srg.resourceDir, R_OK)!=0) {
        fprintf(stderr, "%s: Unable to access resource directory (%s)!\n",
                progname, srg.resourceDir);
        exit(1);
    }

	/* Check for stylesheet file */
	if (!file_present(SRG_CSS_FILE, "SRG Default Stylesheet - ")) {
		init_file(SRG_CSS_FILE);
	}
	
	/* Check for javascript file */
	if (srg.usejs && !file_present(SRG_JS_FILE, "SRG Default Javascript - ")) {
		init_file(SRG_JS_FILE);
	}
	
	/* Check for PHP header / footer if required */
	if (srg.outputMode == OUTPUT_PHP) {
		if (!file_present(SRG_HEADER_FILE, "SRG Default PHP Header - ")) {
			init_file(SRG_HEADER_FILE);
		}
		if (!file_present(SRG_FOOTER_FILE, "SRG Default PHP Footer - ")) {
			init_file(SRG_FOOTER_FILE);
		}
	}

	/* Setup Paths */
	asprintf(&srg.phpheader, "%s/%s", srg.outputDir, srg.phpheader);
	asprintf(&srg.phpfooter, "%s/%s", srg.outputDir, srg.phpfooter);
	asprintf(&srg.cssfile, "%s/%s", srg.outputURL, srg.cssfile);
	asprintf(&srg.jsfile, "%s/%s", srg.outputURL, srg.jsfile);
	
}

/* Copy a file from the resource directory to the base directory */
void init_file(const char *filename)
{

	char sfilename[1024] = {'\0'};
	char dfilename[1024] = {'\0'};
	FILE *source = NULL;
	FILE *dest = NULL;
	
	/* Create filenames */
	snprintf(&sfilename[0], 1023, "%s/%s", srg.resourceDir, filename);
	snprintf(&dfilename[0], 1023, "%s/%s", srg.outputDir, filename);
	
	/* Open files */
	source = fopen(sfilename, "r");
	if (!source) {
		fprintf(stderr, "ERROR: Could not open file: %s\n", sfilename);
		exit(1);
	}
	dest = fopen(dfilename, "w");
	if (!dest) {
		fprintf(stderr, "ERROR: Could not open file for writing: %s\n",
				dfilename);
		exit(1);
	}
	
	/* Copy data */
	while (!feof(source)) {
		char *buf[4096] = {'\0'};
		int c = fread((void *)&buf[0], 1, 4096, source);
		fwrite((void *)&buf[0], 1, c, dest);
	}

	/* Close files */
	fclose(source);
	fclose(dest);
	
}

/* Write the HTML header */
void html_header(FILE *outfile, const char *rootpath) {

	/* rootpath is used when the user is generating a stock standard
	 * report with no outputURL set. Each report sends through a string
	 * like ../../ that provides a path from the directory that this
	 * file will reside in to the top level output directory. This string
	 * is only used if srg.outputURL is not set
	 */
	char *base="";

	if (strlen(srg.outputURL)==0 && strlen(rootpath)>0) {
		base = strdup(rootpath);
	}

	/* Header & Title */
	if (srg.outputMode == OUTPUT_PHP) {
		/* Include the specified PHP header */
		fprintf(outfile, "<?php\ninclude_once('%s');\n?>\n", 
				srg.phpheader);
		if (srg.authenticate) {
			fprintf(outfile, "<?php if (can_view(\"-1\")) { "
					"?>\n\t");
		}
	} else {
		/* Output a static HTML Header */
		fprintf(outfile, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML "
				"4.01 Transitional//EN\" \"http://www.w3."
				"org/TR/html4/loose.dtd\">\n");
		fprintf(outfile, "<html>\n<head>\n<title>%s</title>\n", 
				srg.title);
		fprintf(outfile, "<meta name=\"generator\" content=\""
				"SRG %s (%s)\">\n", version, HOME_URL); 
		fprintf(outfile, "<meta name=\"robots\" content=\"noindex,"
				"nofollow\">\n");
		fprintf(outfile, "<link rel=\"stylesheet\" href=\"%s%s\" type=\""
				"text/css\">\n", base, srg.cssfile);
        if (srg.usejs == 1) {
            fprintf(outfile, "<script language=\"javascript\" src=\"%s%s\" "
                    "type=\"text/javascript\"></script></head>\n", 
                    base, srg.jsfile);
            fprintf(outfile, "<body onload=\"setupSort();\">\n");
        } else {
            fprintf(outfile, "</head>\n\n<body>\n");
        }
		/* Output some basic layout and headers */
		fprintf(outfile, "<div align=\"center\"><h1>%s</h1></div>\n\n", 
                srg.title);
	}

	if (strlen(base)>0)
		free(base);

}

/* Write the HTML footer */
void html_footer(FILE *outfile, const char *rootpath) {

	/* rootpath is used when the user is generating a stock standard
	 * report with no outputURL set. Each report sends through a string
	 * like ../../ that provides a path from the directory that this
	 * file will reside in to the top level output directory. This string
	 * is only used if srg.outputURL is not set
	 */
	char *base="";

	if (strlen(srg.outputURL)==0 && strlen(rootpath)>0)
		base = strdup(rootpath);

	fprintf(outfile, "<div align=\"center\" class=\"srgfooter\"><a href=\""
			"%s\">Generated by SRG %s</a></div>\n\n", HOME_URL, version);

	if (srg.outputMode == OUTPUT_PHP) {
		/* Include the specified PHP footer */
		fprintf(outfile, "<?php\ninclude_once('%s');\n\n",
				srg.phpfooter);
	} else {
		fprintf(outfile, "\n</body>\n");
		if (srg.usejs) {
			fprintf(outfile, "<script src=\"%s%s\" type=\""
					"text/javascript\"></script>\n", base, srg.jsfile);
		}                
		fprintf(outfile, "</html>\n");
	}

	if (strlen(base)>0)
		free(base);

}
