/* specter_PWSNIFF.c
 *
 * specter logging interpreter for POP3 / FTP like plaintext passwords.
 *
 * (C) 2000-2003 by Harald Welte <laforge@gnumonks.org>
 *
 *  11 Jun 2004, Michal Kwiatkowski <ruby@joker.linuxstuff.pl>:
 *      fixed for specter.
 *
 *  14 Jun 2004, Michal Kwiatkowski <ruby@joker.linuxstuff.pl>:
 *      no malloc() during packet receive - only static buffers
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/ip.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include "chtons.h"
#include <specter/specter.h>


#define PWSNIFF_RETS_NUM 2
static specter_iret_t pwsniff_rets[PWSNIFF_RETS_NUM] = {
	{ .type = SPECTER_IRET_STRING, .name = "pwsniff.user" },
	{ .type = SPECTER_IRET_STRING, .name = "pwsniff.pass" },
};


#define PORT_POP3	110
#define PORT_FTP	21

static u_int16_t pwsniff_ports[] = {
	__constant_htons(PORT_POP3),
	__constant_htons(PORT_FTP),
	/* feel free to include any other ports here, provided that their
	 * user/password syntax is the same */
};

#define PWSNIFF_MAX_PORTS 2


#define PWSNIFF_MAX_USERNAME 256 /* more than enough */
#define PWSNIFF_MAX_PASSWORD 256
static char pwsniff_username[PWSNIFF_MAX_USERNAME];
static char pwsniff_password[PWSNIFF_MAX_PASSWORD];


static char *_get_next_blank(char* begp, char *endp)
{
	char *ptr;

	for (ptr = begp; ptr < endp; ptr++) {
		if (*ptr == ' ' || *ptr == '\n' || *ptr == '\r') {
			return ptr-1;	
		}
	}
	return NULL;
}

static int _interp_pwsniff(ulog_packet_msg_t *pkt)
{
	struct iphdr *iph = (struct iphdr *) pkt->payload;
	void *protoh = (u_int32_t *)iph + iph->ihl;
	struct tcphdr *tcph = protoh;
	u_int32_t tcplen = ntohs(iph->tot_len) - iph->ihl * 4;
	unsigned char  *ptr, *begp, *pw_begp, *endp, *pw_endp;
	int len, pw_len, i, cont = 0;
	specter_iret_t *ret = pwsniff_rets;

	len = pw_len = 0;
	begp = pw_begp = NULL;

	if (iph->protocol != IPPROTO_TCP) {
		return 0;
	}

	for (i = 0; i < PWSNIFF_MAX_PORTS; i++)
	{
		if (tcph->dest == pwsniff_ports[i]) {
			cont = 1; 
			break;
		}
	}
	if (!cont) {
		return 0;
	}

	for (ptr = (unsigned char *) tcph + sizeof(struct tcphdr); 
			ptr < (unsigned char *) tcph + tcplen; ptr++)
	{
		if (!strncasecmp(ptr, "USER ", 5)) {
			begp = ptr+5;
			endp = _get_next_blank(begp, (char *)tcph + tcplen);
			if (endp)
				len = endp - begp + 1;
		}
		if (!strncasecmp(ptr, "PASS ", 5)) {
			pw_begp = ptr+5;
			pw_endp = _get_next_blank(pw_begp, 
					(char *)tcph + tcplen);
			if (pw_endp)
				pw_len = pw_endp - pw_begp + 1;
		}
	}

	if (len) {
		if (len > PWSNIFF_MAX_USERNAME) {
			specter_log(SPECTER_DEBUG, "Username longer than %i.\n",
					PWSNIFF_MAX_USERNAME);
			return 0;
		}
		ret[0].value.ptr = pwsniff_username;
		strncpy(ret[0].value.ptr, begp, len);
		*((char *)ret[0].value.ptr + len) = '\0';
		ret[0].flags |= SPECTER_RETF_VALID;
	}
	if (pw_len) {
		if (len > PWSNIFF_MAX_PASSWORD) {
			specter_log(SPECTER_DEBUG, "Password longer than %i.\n",
					PWSNIFF_MAX_USERNAME);
			return 0;
		}
		ret[1].value.ptr = pwsniff_password;
		strncpy(ret[1].value.ptr, pw_begp, pw_len);
		*((char *)ret[1].value.ptr + pw_len) = '\0';
		ret[1].flags |= SPECTER_RETF_VALID;

	}

	return 0;
}

static specter_input_t pwsniff_ip = { 
	.name = "pwsniff",
	.input = &_interp_pwsniff
};

void _init(void)
{
	if (register_input(&pwsniff_ip, pwsniff_rets, PWSNIFF_RETS_NUM, 0) == -1) {
		specter_log(SPECTER_FATAL, "Couldn't register.\n");
		exit(EXIT_FAILURE);
	}
}

