<?php
/**
 * Forwards_Driver_forwards:: implements the Forwards_Driver API for ftp
 * driven dot-forward compliant mail servers.
 *
 * $Horde: forwards/lib/Driver/forwards.php,v 1.36.2.3 2007/01/02 13:54:06 jan Exp $
 *
 * Copyright 2001-2007 Eric Rostetter <eric.rostetter@physics.utexas.edu>
 *
 * See the enclosed file LICENSE for license information (BSDL). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Eric Rostetter <eric.rostetter@physics.utexas.edu>
 * @since   Forwards 2.1
 * @package Forwards
 */
class Forwards_Driver_forwards extends Forwards_Driver {

    /**
     * The FTP stream we open via the VFS class.
     *
     * @var VFS_ftp
     */
    var $_vfs;

    /**
     * Checks if the realm has a specific configuration.
     *
     * If not, try to fall back on the default configuration.  If
     * still not a valid configuration then exit with an error.
     *
     * @param string $realm  The realm of the user, or "default" if none.
     *                       Note: passed by reference so we can change
     *                       its value!
     *
     * @return boolean  True on success, false otherwise.
     */
    function checkConfiguration(&$realm)
    {
        // If no realm passed in, or no host config for the realm
        // passed in, then we fall back to the default realm.
        if (empty($realm) || empty($this->_params[$realm]['host'])) {
            $realm = 'default';
        }

        // If still no host/port, then we have a misconfigured module.
        if (empty($this->_params[$realm]['host']) ||
            empty($this->_params[$realm]['port'])) {
            $this->_error = _("The module is not properly configured!");
            return false;
        }

        return true;
    }

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $user        The username of the user.
     * @param string $realm       The realm of the user.
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     *
     * @return boolean  True on success, false otherwise.
     */
    function enableForwarding($user, $realm = 'default', $password, $target,
                              $keeplocal = false)
    {
        // Make sure the configuration file is correct
        if (!$this->checkConfiguration($realm)) {
            return false;
        }

        // Build the FTP array to pass to VFS.
        $_args = array('hostspec' => $this->_params[$realm]['host'],
                       'port' => $this->_params[$realm]['port'],
                       'pasv' => $this->_params[$realm]['pasv'],
                       'username' => $user,
                       'password' => $password);

        // Create the VFS ftp driver.
        require_once 'VFS.php';
        $_vfs = &VFS::singleton('ftp', $_args);
        if (is_a($_vfs, 'PEAR_Error')) {
            $this->_error = $_vfs->getMessage();
            return false;
        }

        // Try to login with the username/password, no realm
        // This isn't really needed, but allows for a better error message
        $status = $_vfs->checkCredentials();
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            $this->_error .= '  ' .  _("Check your username and password");
            return false;
        }

        // Create the forwarding information
        $address = $this->_make_email_address($target, $realm);
        if ($address === false) {
            return false;
        }

        // Append the user if they want to save a copy to themselves...
        if ($keeplocal == "on") {
            $address = $address . ", \\$user";
        }

        // Set the forward
        $status = $_vfs->writeData('', '.forward', $address);
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            return false;
        }

        // Try to change the permissions, but ignore any errors
        $_vfs->changePermissions('', '.forward', '0600');

        return true;
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True on success, false otherwise.
     */
    function disableForwarding($user, $realm = 'default', $password)
    {
        if (!$this->checkConfiguration($realm)) {
            return false;
        }

        $_args = array('hostspec' => $this->_params[$realm]['host'],
                       'port' => $this->_params[$realm]['port'],
                       'pasv' => $this->_params[$realm]['pasv'],
                       'username' => $user,
                       'password' => $password);

        require_once 'VFS.php';
        $_vfs = &VFS::singleton('ftp', $_args);

        // Try to login with the username/password, no realm.
        $status = $_vfs->checkCredentials();
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            $this->_error .= '  ' .  _("Check your username and password");
            return false;
        }

        $status = $_vfs->deleteFile('', '.forward');
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            $this->_error .= '  ' .  _("Maybe you didn't have a forward enabled?");
            return false;
        }

        return true;
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return mixed  The current forwarding mail address, or false.
     */
    function currentTarget($user, $realm = 'default', $password)
    {
        // Make sure the configuration file is correct.
        if (!$this->checkConfiguration($realm)) {
            return false;
        }

        // Build the FTP array to pass to VFS.
        $_args = array('hostspec' => $this->_params[$realm]['host'],
                       'port' => $this->_params[$realm]['port'],
                       'pasv' => $this->_params[$realm]['pasv'],
                       'username' => $user,
                       'password' => $password);

        // Create the VFS ftp driver.
        require_once 'VFS.php';
        $_vfs = &VFS::singleton('ftp', $_args);

        // Try to login with the username/password, no realm.
        $status = $_vfs->checkCredentials();
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            $this->_error .= '  ' .  _("Check your username and password");
            return false;
        }

        $status = $_vfs->read('', '.forward');
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            return false;
        }

        return preg_replace('/ *(^|, *)\\\\' . $user . '( *, *|$)/', "", $status);
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @TODO FIXME: This function is implemented poorly, and should be
     * rewritten.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return mixed  'Y' if forwarding is enabled, or false otherwise.
     */
    function isEnabledForwarding($user, $realm, $password)
    {
        $yn = $this->currentTarget($user, $realm, $password);
        if ($yn) {
            return 'Y';
        } else {
            return false;
        }
    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($user, $realm, $password)
    {
        // Make sure the configuration file is correct.
        if (!$this->checkConfiguration($realm)) {
            return false;
        }

        // Build the FTP array to pass to VFS.
        $_args = array('hostspec' => $this->_params[$realm]['host'],
                       'port' => $this->_params[$realm]['port'],
                       'pasv' => $this->_params[$realm]['pasv'],
                       'username' => $user,
                       'password' => $password);

        // Create the VFS ftp driver.
        require_once 'VFS.php';
        $_vfs = &VFS::singleton('ftp', $_args);

        // Try to login with the username/password, no realm.
        $status = $_vfs->checkCredentials();
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            $this->_error .= '  ' .  _("Check your username and password");
            return false;
        }

        $status = $_vfs->read('', '.forward');
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            return false;
        }

        return preg_match('/(^|, *)\\\\' . $user . '( *,|$)/', $status);
    }

}
