/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#ifdef ENABLE_OGG

#include "../../player.h"
#include "../../other.h"
#include "../../util.h"

audio_data_vorbis_encode *
vorbis_config (long rate, int channels, int mode, int quality, int bitrate,
	       int lowpass, char *name, audio * ao, int (*output) (audio *,
								   void *,
								   size_t))
{
  audio_data_vorbis_encode *vorbis;
  ogg_packet header;
  ogg_packet comment;
  ogg_packet code;
  ogg_page page;

  if (!output)
    fatal (_("Internal error."));

  if (!
      (vorbis =
       (audio_data_vorbis_encode *)
       malloc (sizeof (audio_data_vorbis_encode))))
    fatal (_("Error: memory."));

  vorbis_info_init (&vorbis->vi);

  if (quality)
    {
      if (vorbis_encode_setup_vbr
	  (&vorbis->vi, channels, rate, (float) quality * 0.1))
	{
	  msg_error (_("Vorbis quality init error."));
	  free (vorbis);
	  return NULL;
	}

    }
  else if (bitrate)
    {
      if (vorbis_encode_setup_managed
	  (&vorbis->vi, channels, rate, -1, bitrate * 1000, -1))
	{
	  msg_error (_("Vorbis bitrate init error."));
	  free (vorbis);
	  return NULL;
	}
    }

  if (lowpass)
    vorbis_encode_ctl (&vorbis->vi, OV_ECTL_LOWPASS_SET, &lowpass);

  if (vorbis_encode_setup_init (&vorbis->vi))
    {
      msg_error (_("Vorbis analysis init error"));
      free (vorbis);
      return NULL;
    }

  if (vorbis_analysis_init (&vorbis->dsp, &vorbis->vi))
    {
      msg_error (_("Vorbis analysis init error"));
      free (vorbis);
      return NULL;
    }

  if (vorbis_block_init (&vorbis->dsp, &vorbis->block))
    {
      msg_error (_("Vorbis block init error"));
      free (vorbis);
      return NULL;
    }

  if (ogg_stream_init (&vorbis->ogg, 0))
    {
      msg_error (_("Ogg stream init error"));
      free (vorbis);
      return NULL;
    }

  vorbis_comment_init (&vorbis->comment);

  if (name)
    vorbis_comment_add_tag (&vorbis->comment, "title", name);

  vorbis_comment_add_tag (&vorbis->comment, "encoder", NAME);


  if (vorbis_analysis_headerout
      (&vorbis->dsp, &vorbis->comment, &header, &comment, &code))
    {
      msg_error (_("Vorbis header init error."));
      free (vorbis);
      return NULL;
    }

  ogg_stream_packetin (&vorbis->ogg, &header);
  ogg_stream_packetin (&vorbis->ogg, &comment);
  ogg_stream_packetin (&vorbis->ogg, &code);

  while (ogg_stream_flush (&vorbis->ogg, &page))
    {
      output (ao, page.header, page.header_len);
      output (ao, page.body, page.body_len);
    }

  vorbis_comment_clear (&vorbis->comment);

  return vorbis;
}

int
vorbis_write (audio_data_vorbis_encode * data, int channels, int bitrate,
	      void *buffer, size_t length, audio * ao, int (*output) (audio *,
								      void *,
								      size_t))
{
  unsigned int sample = (bitrate / 8) * channels;
  unsigned char *b = (unsigned char *) buffer;
  unsigned int processed = length - (length % sample);

  unsigned int n_samples = processed / sample;
  float **buf;
  unsigned int total_samples = n_samples * channels;
  short int short_buffer[total_samples];

  conv (bitrate, b, processed, short_buffer);

  if (!data)
    fatal (_("Internal error."));

  if (!output)
    fatal (_("Internal error."));

  buf = vorbis_analysis_buffer (&data->dsp, n_samples);
  conv_2 (short_buffer, total_samples, buf, channels);
  vorbis_analysis_wrote (&data->dsp, n_samples);

  while (vorbis_analysis_blockout (&data->dsp, &data->block) == 1)
    {
      ogg_packet packet;
      ogg_page page;

      vorbis_analysis (&data->block, NULL);
      vorbis_bitrate_addblock (&data->block);

      while (vorbis_bitrate_flushpacket (&data->dsp, &packet))
	{

	  ogg_stream_packetin (&data->ogg, &packet);

	  while (ogg_stream_pageout (&data->ogg, &page))
	    {
	      output (ao, page.header, page.header_len);
	      output (ao, page.body, page.body_len);
	    }
	}
    }

  return length;
}

void
vorbis_quit (audio_data_vorbis_encode * data)
{
  if (!data)
    fatal (_("Internal error."));

  ogg_stream_clear (&data->ogg);
  vorbis_block_clear (&data->block);
  vorbis_dsp_clear (&data->dsp);
  vorbis_info_clear (&data->vi);
}

#endif
