// SPDX-License-Identifier: BSD-3-Clause
//
// Copyright 2014 Raritan Inc. All rights reserved.

package com.raritan.json_rpc.examples;

import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import com.raritan.idl.AsyncRpcResponse;
import com.raritan.idl.TypeInfo;
import com.raritan.idl.pdumodel.Inlet;
import com.raritan.idl.pdumodel.Inlet_2_0_0;
import com.raritan.idl.pdumodel.Outlet;
import com.raritan.idl.pdumodel.Outlet_2_0_0;
import com.raritan.idl.pdumodel.Pdu;
import com.raritan.idl.pdumodel.Pdu_2_0_0;
import com.raritan.idl.pdumodel.Pdu_3_0_0;
import com.raritan.idl.pdumodel.Pdu_4_0_0;
import com.raritan.json_rpc.Agent;
import com.raritan.json_rpc.ObjectProxy;
import com.raritan.json_rpc.pdumodel.Inlet_Proxy;
import com.raritan.json_rpc.pdumodel.Inlet_2_0_0_Proxy;
import com.raritan.json_rpc.pdumodel.Outlet_Proxy;
import com.raritan.json_rpc.pdumodel.Outlet_2_0_0_Proxy;
import com.raritan.json_rpc.pdumodel.Pdu_2_0_0_Proxy;
import com.raritan.json_rpc.pdumodel.Pdu_3_0_0_Proxy;
import com.raritan.json_rpc.pdumodel.Pdu_4_0_0_Proxy;
import com.raritan.json_rpc.pdumodel.Pdu_Proxy;
import com.raritan.util.AsyncDoneListener;
import com.raritan.util.AsyncFailureListener;
import com.raritan.util.AsyncMultiRequest;
import com.raritan.util.AsyncRequest;
import com.raritan.util.AsyncSuccessListener;

public class DumpPduAsync {

    private static class InletCache {
	private final ObjectProxy proxy;
	private String label;
	private String plug;
	private int minVoltage;
	private int maxVoltage;
	private int currentRating;
	private String name;

	public InletCache(ObjectProxy proxy) {
	    this.proxy = proxy;
	}

	private AsyncRequest updateV1(Inlet inlet) {
	    AsyncMultiRequest getDataTask = new AsyncMultiRequest("get_inlet_data");

	    getDataTask.add(inlet.getMetaData(new AsyncRpcResponse<Inlet.GetMetaDataResult>() {
		@Override
		public void onSuccess(Inlet.GetMetaDataResult result) {
		    label = result._ret_.label;
		    plug = result._ret_.plugType;
		    minVoltage = result._ret_.rating.minVoltage;
		    maxVoltage = result._ret_.rating.maxVoltage;
		    currentRating = result._ret_.rating.current;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(inlet.getSettings(new AsyncRpcResponse<Inlet.GetSettingsResult>() {
		@Override
		public void onSuccess(Inlet.GetSettingsResult result) {
		    name = result._ret_.name;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    return getDataTask.started();
	}

	private AsyncRequest updateV2(Inlet_2_0_0 inlet) {
	    AsyncMultiRequest getDataTask = new AsyncMultiRequest("get_inlet_data");

	    getDataTask.add(inlet.getMetaData(new AsyncRpcResponse<Inlet_2_0_0.GetMetaDataResult>() {
		@Override
		public void onSuccess(Inlet_2_0_0.GetMetaDataResult result) {
		    label = result._ret_.label;
		    plug = result._ret_.plugType;
		    minVoltage = result._ret_.rating.minVoltage;
		    maxVoltage = result._ret_.rating.maxVoltage;
		    currentRating = result._ret_.rating.current;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(inlet.getSettings(new AsyncRpcResponse<Inlet_2_0_0.GetSettingsResult>() {
		@Override
		public void onSuccess(Inlet_2_0_0.GetSettingsResult result) {
		    name = result._ret_.name;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    return getDataTask.started();
	}
	public AsyncRequest update() {
	    TypeInfo type = proxy.getStaticTypeInfo();
	    if (type.isCallCompatible(Inlet.typeInfo)) {
		return updateV1(Inlet_Proxy.staticCast(proxy));
	    } else if (type.isCallCompatible(Inlet_2_0_0.typeInfo)) {
		return updateV2(Inlet_2_0_0_Proxy.staticCast(proxy));
	    } else {
		System.out.println("ERROR: Unsupported Inlet interface version: " + type.toString());
		return new AsyncRequest("update_inlet").failed(null);
	    }
	}

	public void dump() {
	    System.out.printf("Inlet %s (%s)%n", label, name);
	    System.out.printf("  Type:   %s%n", plug);
	    System.out.printf("  Rating: %d-%d V, %d A%n", minVoltage, maxVoltage, currentRating);
	    System.out.println();
	}
    };

    private static class OutletCache {
	private final ObjectProxy proxy;
	private String label;
	private String receptacle;
	private int currentRating;
	private String name;

	public OutletCache(ObjectProxy proxy) {
	    this.proxy = proxy;
	}

	private AsyncRequest updateV1(Outlet outlet) {
	    AsyncMultiRequest getDataTask = new AsyncMultiRequest("get_outlet_data");

	    getDataTask.add(outlet.getMetaData(new AsyncRpcResponse<Outlet.GetMetaDataResult>() {
		@Override
		public void onSuccess(Outlet.GetMetaDataResult result) {
		    label = result._ret_.label;
		    receptacle = result._ret_.receptacleType;
		    currentRating = result._ret_.rating.current;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(outlet.getSettings(new AsyncRpcResponse<Outlet.GetSettingsResult>() {
		@Override
		public void onSuccess(Outlet.GetSettingsResult result) {
		    name = result._ret_.name;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    return getDataTask.started();
	}

	private AsyncRequest updateV2(Outlet_2_0_0 outlet) {
	    AsyncMultiRequest getDataTask = new AsyncMultiRequest("get_outlet_data");

	    getDataTask.add(outlet.getMetaData(new AsyncRpcResponse<Outlet_2_0_0.GetMetaDataResult>() {
		@Override
		public void onSuccess(Outlet_2_0_0.GetMetaDataResult result) {
		    label = result._ret_.label;
		    receptacle = result._ret_.receptacleType;
		    currentRating = result._ret_.rating.current;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(outlet.getSettings(new AsyncRpcResponse<Outlet_2_0_0.GetSettingsResult>() {
		@Override
		public void onSuccess(Outlet_2_0_0.GetSettingsResult result) {
		    name = result._ret_.name;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    return getDataTask.started();
	}

	public AsyncRequest update() {
	    TypeInfo type = proxy.getStaticTypeInfo();
	    if (type.isCallCompatible(Outlet.typeInfo)) {
		return updateV1(Outlet_Proxy.staticCast(proxy));
	    } else if (type.isCallCompatible(Outlet_2_0_0.typeInfo)) {
		return updateV2(Outlet_2_0_0_Proxy.staticCast(proxy));
	    } else {
		System.out.println("ERROR: Unsupported Outlet interface version: " + type.toString());
		return new AsyncRequest("update_outlet").failed(null);
	    }
	}

	public void dump() {
	    System.out.printf("Outlet %s (%s)%n", label, name);
	    System.out.printf("  Type:   %s%n", receptacle);
	    System.out.printf("  Rating: %d A%n", currentRating);
	    System.out.println();
	}
    };

    private static class PduCache {
	private final ObjectProxy proxy;
	private String model;
	private String serialNumber;
	private String name;
	private List<InletCache> inlets = new ArrayList<InletCache>();
	private List<OutletCache> outlets = new ArrayList<OutletCache>();

	public PduCache(ObjectProxy proxy) {
	    this.proxy = proxy;
	}

	private AsyncRequest updateV1(Pdu pdu) {
	    AsyncMultiRequest getDataTask = new AsyncMultiRequest("get_pdu_v1_data");

	    getDataTask.add(pdu.getMetaData(new AsyncRpcResponse<Pdu.GetMetaDataResult>() {
		@Override
		public void onSuccess(Pdu.GetMetaDataResult result) {
		    model = result._ret_.nameplate.model;
		    serialNumber = result._ret_.nameplate.serialNumber;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(pdu.getSettings(new AsyncRpcResponse<Pdu.GetSettingsResult>() {
		@Override
		public void onSuccess(Pdu.GetSettingsResult result) {
		    name = result._ret_.name;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(pdu.getInlets(new AsyncRpcResponse<Pdu.GetInletsResult>() {
		@Override
		public void onSuccess(Pdu.GetInletsResult result) {
		    for (Object o : result._ret_) {
			inlets.add(new InletCache((ObjectProxy)o));
		    }
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(pdu.getOutlets(new AsyncRpcResponse<Pdu.GetOutletsResult>() {
		@Override
		public void onSuccess(Pdu.GetOutletsResult result) {
		    for (Object o : result._ret_) {
			outlets.add(new OutletCache((ObjectProxy)o));
		    }
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    return getDataTask.started();
	}

	private AsyncRequest updateV2(Pdu_2_0_0 pdu) {
	    AsyncMultiRequest getDataTask = new AsyncMultiRequest("get_pdu_v2_data");

	    getDataTask.add(pdu.getMetaData(new AsyncRpcResponse<Pdu_2_0_0.GetMetaDataResult>() {
		@Override
		public void onSuccess(Pdu_2_0_0.GetMetaDataResult result) {
		    model = result._ret_.nameplate.model;
		    serialNumber = result._ret_.nameplate.serialNumber;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(pdu.getSettings(new AsyncRpcResponse<Pdu_2_0_0.GetSettingsResult>() {
		@Override
		public void onSuccess(Pdu_2_0_0.GetSettingsResult result) {
		    name = result._ret_.name;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(pdu.getInlets(new AsyncRpcResponse<Pdu_2_0_0.GetInletsResult>() {
		@Override
		public void onSuccess(Pdu_2_0_0.GetInletsResult result) {
		    for (Object o : result._ret_) {
			inlets.add(new InletCache((ObjectProxy)o));
		    }
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(pdu.getOutlets(new AsyncRpcResponse<Pdu_2_0_0.GetOutletsResult>() {
		@Override
		public void onSuccess(Pdu_2_0_0.GetOutletsResult result) {
		    for (Object o : result._ret_) {
			outlets.add(new OutletCache((ObjectProxy)o));
		    }
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    return getDataTask.started();
	}

	private AsyncRequest updateV3(Pdu_3_0_0 pdu) {
	    AsyncMultiRequest getDataTask = new AsyncMultiRequest("get_pdu_v3_data");

	    getDataTask.add(pdu.getMetaData(new AsyncRpcResponse<Pdu_3_0_0.GetMetaDataResult>() {
		@Override
		public void onSuccess(Pdu_3_0_0.GetMetaDataResult result) {
		    model = result._ret_.nameplate.model;
		    serialNumber = result._ret_.nameplate.serialNumber;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(pdu.getSettings(new AsyncRpcResponse<Pdu_3_0_0.GetSettingsResult>() {
		@Override
		public void onSuccess(Pdu_3_0_0.GetSettingsResult result) {
		    name = result._ret_.name;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(pdu.getInlets(new AsyncRpcResponse<Pdu_3_0_0.GetInletsResult>() {
		@Override
		public void onSuccess(Pdu_3_0_0.GetInletsResult result) {
		    for (Object o : result._ret_) {
			inlets.add(new InletCache((ObjectProxy)o));
		    }
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(pdu.getOutlets(new AsyncRpcResponse<Pdu_3_0_0.GetOutletsResult>() {
		@Override
		public void onSuccess(Pdu_3_0_0.GetOutletsResult result) {
		    for (Object o : result._ret_) {
			outlets.add(new OutletCache((ObjectProxy)o));
		    }
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    return getDataTask.started();
	}

	private AsyncRequest updateV4(Pdu_4_0_0 pdu) {
	    AsyncMultiRequest getDataTask = new AsyncMultiRequest("get_pdu_v4_data");

	    getDataTask.add(pdu.getMetaData(new AsyncRpcResponse<Pdu_4_0_0.GetMetaDataResult>() {
		@Override
		public void onSuccess(Pdu_4_0_0.GetMetaDataResult result) {
		    model = result._ret_.nameplate.model;
		    serialNumber = result._ret_.nameplate.serialNumber;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(pdu.getSettings(new AsyncRpcResponse<Pdu_4_0_0.GetSettingsResult>() {
		@Override
		public void onSuccess(Pdu_4_0_0.GetSettingsResult result) {
		    name = result._ret_.name;
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(pdu.getInlets(new AsyncRpcResponse<Pdu_4_0_0.GetInletsResult>() {
		@Override
		public void onSuccess(Pdu_4_0_0.GetInletsResult result) {
		    for (Object o : result._ret_) {
			inlets.add(new InletCache((ObjectProxy)o));
		    }
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    getDataTask.add(pdu.getOutlets(new AsyncRpcResponse<Pdu_4_0_0.GetOutletsResult>() {
		@Override
		public void onSuccess(Pdu_4_0_0.GetOutletsResult result) {
		    for (Object o : result._ret_) {
			outlets.add(new OutletCache((ObjectProxy)o));
		    }
		}

		@Override
		public void onFailure(Exception e) {}
	    }));

	    return getDataTask.started();
	}

	private AsyncRequest updateChildren() {
	    AsyncMultiRequest updateChildrenTask = new AsyncMultiRequest("update_pdu_children");

	    for (InletCache inlet : inlets) {
		updateChildrenTask.add(inlet.update());
	    }

	    for (OutletCache outlet : outlets) {
		updateChildrenTask.add(outlet.update());
	    }

	    return updateChildrenTask.started();
	}

	public AsyncRequest update() {
	    final AsyncRequest updateFullTask = new AsyncRequest("update_pdu_full");

	    AsyncRequest getPduDataTask = null;

	    // First step: query Pdu properties (metadata, settings, outlets and inlets);
	    // this must be coded separately for each supported major interface version.
	    TypeInfo type = proxy.getStaticTypeInfo();
	    if (type.isCallCompatible(Pdu.typeInfo)) {
		getPduDataTask = updateV1(Pdu_Proxy.staticCast(proxy));
	    } else if (type.isCallCompatible(Pdu_2_0_0.typeInfo)) {
		getPduDataTask = updateV2(Pdu_2_0_0_Proxy.staticCast(proxy));
	    } else if (type.isCallCompatible(Pdu_3_0_0.typeInfo)) {
		getPduDataTask = updateV3(Pdu_3_0_0_Proxy.staticCast(proxy));
	    } else if (type.isCallCompatible(Pdu_4_0_0.typeInfo)) {
		getPduDataTask = updateV4(Pdu_4_0_0_Proxy.staticCast(proxy));
	    } else {
		System.out.println("ERROR: Unsupported PDU interface version: " + type.toString());
		return updateFullTask.failed(null);
	    }

	    getPduDataTask.addSuccessListener(new AsyncSuccessListener() {
		@Override
		public void onSuccess(Object data) {
		    // Second step: ask the inlet and outlet cache objects to update themselves;
		    // those objects are responsible for dealing with different interface versions.
		    AsyncRequest updateChildrenTask = updateChildren();

		    updateChildrenTask.addSuccessListener(new AsyncSuccessListener() {
			@Override
			public void onSuccess(Object data) {
			    // all is well; all data fields and child objects are now up-to-date
			    updateFullTask.succeeded(null);
			}
		    });

		    updateChildrenTask.addFailureListener(new AsyncFailureListener() {
			@Override
			public void onFailure(Exception e) {
			    System.out.println("Failed to update Pdu children");
			    updateFullTask.failed(e);
			}
		    });
		}
	    });

	    getPduDataTask.addFailureListener(new AsyncFailureListener() {
		@Override
		public void onFailure(Exception e) {
		    System.out.println("Failed to update Pdu members");
		    updateFullTask.failed(e);
		}
	    });

	    return updateFullTask.started();
	}

	public void dump() {
	    System.out.println("Model:            " + model);
	    System.out.println("Serial number:    " + serialNumber);
	    System.out.println("Device name:      " + name);
	    System.out.println();

	    for (InletCache inlet : inlets) {
		inlet.dump();
	    }

	    for (OutletCache outlet : outlets) {
		outlet.dump();
	    }
	}
    }

    public static void main(String[] args) {
	String host;
	if (args.length >= 1) {
	    host = args[0];
	} else {
	    System.out.println("Please enter PDU URL (default: https://10.0.42.2):");
	    Scanner input = new Scanner(System.in);
	    host = input.nextLine();
	    input.close();
	}
	if (host.isEmpty()) {
	    host = "https://10.0.42.2";
	}
	if (!host.contains("://")) {
	    host = "https://" + host;
	}
	System.out.printf("Connecting to %s ...%n", host);

	// Create an agent for performing JSON-RPC objects over HTTP(S)
	Agent agent = new Agent(host);
	agent.setUsername("admin");
	agent.setPassword("raritan");

	ObjectProxy pduProxy = null;
	try {
	    // Create a proxy for the root PDU object, synchronously query its type information
	    System.out.println("Querying Pdu interface version ...");
	    pduProxy = new ObjectProxy(agent, "/model/pdu/0");
	    TypeInfo type = pduProxy.getTypeInfo();
	    System.out.println("Pdu interface type: " + type.toString());
	} catch (Exception e) {
	    e.printStackTrace();
	    System.exit(1);
	}

	// Create a PDU cache object and request an asynchronous update
	final PduCache pdu = new PduCache(pduProxy);
	AsyncRequest task = pdu.update();

	// When the PDU cache has been successfully updated, dump the PDU
	task.addSuccessListener(new AsyncSuccessListener() {
	    @Override
	    public void onSuccess(Object data) {
		pdu.dump();
	    }
	});

	// In case of a update failure, show an error message
	task.addFailureListener(new AsyncFailureListener() {
	    @Override
	    public void onFailure(Exception e) {
		System.out.println("Error: PDU cache update failed!");
		e.printStackTrace();
	    }
	});

	final Lock lock = new ReentrantLock();
	final Condition cond = lock.newCondition();

	// In either case, wake up the main thread
	task.addDoneListener(new AsyncDoneListener() {
	    @Override
	    public void onDone() {
		lock.lock();
		cond.signal();
		lock.unlock();
	    }
	});

	// wait for all background tasks to finish
	while (!task.isDone()) {
	    lock.lock();
	    try {
		cond.await();
	    } catch (Exception e) {
		e.printStackTrace();
		System.exit(1);
	    }
	    lock.unlock();
	}
	agent.cleanup();
    }

}
