
#include <applet-widget.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <esd.h>
#include <sys/ioctl.h>   /* For the mixer */
#include <sys/soundcard.h>
#include "quickrecord_applet.h"

/* Directly set by record_set_options: */
static Properties *record_props=NULL;

/* Calculated from options: */
static gint buffer_size;        /* Size of one buffer block */
static gint buffer_amount;
static gint buffer_total_size;
static gint buffer_1sec;   /* rate*16bits*stereo */

/* Buffer static data: */
gboolean buffering=FALSE;    /* Current state */
static GList *buffers=NULL;
static gpointer *one_buffer=NULL;
static gint full_buffers=0;

/* Recording data: */
gboolean recording=FALSE;
FILE *rec_fp;
static gint recbytes=0;

static int esd_handle=-1;
static gint idle_handle=0;




/* Internal functions: */
static void record_buffers_allocate(void);
static void record_buffers_free(void);
static gint record_idle_function(gpointer data);

static gboolean record_file_open(gchar *file);
static void record_file_write(gpointer data,gint len);
static void record_file_close(void);

static gboolean record_actual_start(void);
static void record_actual_read(gpointer buf,gint amount);
static void record_actual_stop(void);



/*
 * Start prerecording. Returns TRUE on success.
 */
gboolean record_start_prerecord(void) {
	if (buffering)
		return TRUE;
	record_buffers_allocate();
	full_buffers=0;
	buffering=TRUE;
	if (!recording) {
		if (!record_actual_start()) {
			buffering=recording=FALSE;
			record_buffers_free();
			return FALSE;
		}
	}
	return TRUE;
}

/*
 * Stop buffering.
 */
void record_stop_prerecord(void) {
	if (!buffering)
		return;
	record_buffers_free();
	full_buffers=0;
	buffering=FALSE;
	if (!recording)
		record_actual_stop();
	return;
}

/*
 * Start recording. Returns FALSE on error.
 */
gboolean record_start_record(gchar *file) {
	if (recording)
		return TRUE;
	if (record_file_open(file)==FALSE)
		return FALSE;
	if (buffering) {
		GList *list;

		for (list=g_list_nth(g_list_first(buffers),full_buffers-1);
		     list; list=g_list_previous(list)) {
			record_file_write(list->data,buffer_size);
		}
	}
	recording=TRUE;
	if (!buffering) {
		if (!record_actual_start()) {
			record_file_close();
			recording=buffering=FALSE;
			return FALSE;
		}
	}

	return TRUE;
}

/*
 * Stop recording. Continue buffering if doing so.
 */
void record_stop_record(void) {
	if (!recording)
		return;
	record_file_close();
	recording=FALSE;
	if (!buffering)
		record_actual_stop();
	return;
}



/*
 * Set the options given.
 */
void record_set_options(Properties *props) {
	gint i;

	if (record_props)
		g_free(record_props);
	record_props=g_memdup(props,sizeof(Properties));

	/* Before calculating new buffer sizes, free the old ones. */
	if (buffering)
		record_buffers_free();

	/* Calculate buffer size, amount etc. */
	/* One buffer is 0.1 seconds, rounded downwards to the nearest
	 * power of two. The rounding doesn't affect much common
	 * bitrates (11025*n bytes/sec). */

	buffer_1sec=record_props->rate;
	if (record_props->bits==16)
		buffer_1sec*=2;
	if (record_props->stereo)
		buffer_1sec*=2;
	buffer_size=buffer_1sec/10;

	for (i=1; (1<<i)<buffer_size; i++)
		;
	buffer_size=1<<(i-1);
	buffer_amount=(buffer_1sec*record_props->prerec_amount)/buffer_size;
	buffer_total_size=buffer_size*buffer_amount;

	full_buffers=0;
	if (buffering)
		record_buffers_allocate();  /* Freed previously */

	return;
}





/************ Internal functions ************/

/*
 * Allocate all the buffers. All memory is set to 0.
 * one_buffer is freed, if allocated.
 */
static void record_buffers_allocate(void) {
	gpointer data;
	gint i;

	if (one_buffer) {
		g_free(one_buffer);
		one_buffer=NULL;
	}
	
	for (i=0; i<buffer_amount; i++) {
		data=g_malloc0(buffer_size);
		buffers=g_list_append(buffers,data);
	}
	return;
}

/*
 * Frees all the buffers in the buffer list. Also frees one_buffer, if
 * allocated.
 */
static void record_buffers_free(void) {
	GList *next;

	for (next=buffers; next; next=g_list_next(next)) {
		if (next->data)
			g_free(next->data);
	}
	g_list_free(buffers);
	buffers=NULL;
	full_buffers=0;   /* Can't set it too often... */
	if (one_buffer) {
		g_free(one_buffer);
		one_buffer=NULL;
	}
	return;
}


/*
 * The idle function. Read data from audio device, stuff it into the
 * buffers and optionally write it to the file.
 * Returns TRUE so that function will be kept on calling.
 */
static gint record_idle_function(gpointer data) {
	gpointer ptr;


	if (buffering) {
		/* Buffering: take the last block from buffers and move
		 * it to the beginning of the list. */
		ptr=g_list_last(buffers)->data;
		g_assert(ptr!=NULL);
		buffers=g_list_remove(buffers,ptr);
		buffers=g_list_prepend(buffers,ptr);
		if (full_buffers<buffer_amount)
			full_buffers++;
	} else {
		if (one_buffer==NULL)
			one_buffer=g_malloc0(buffer_size);
		ptr=one_buffer;
	}

	/* Record it... */
	record_actual_read(ptr,buffer_size);
	if (recording) {
		record_file_write(ptr,buffer_size);
	}
	return TRUE;   /* Continue calling... */
}


/*********************************************************
 * Functions related to file handling.
 *
 * Previously I used the libaudiofile functions, but had to change to
 * custom-made .wav-handling functions, because libaudiofile is
 * SLOW AS HELL! Saving a 500kB prerecording buffer took OVER 1 SECOND
 * to write!!! YEEAARGH....
 *
 *********************************************************/

/*
 * Open file as a WAV file for writing. Returns TRUE on success.
 */
static gboolean record_file_open(gchar *file) {
	gint n;
	unsigned char wave_header[]={'R','I','F','F',00,00,00,00, /* 0,sign */
				     'W','A','V','E', /*  8,sign */
				     'f','m','t',' ', /* 12,chunkID */
				     16,0,0,0,        /* 16,chunkSize */
				     1,0,             /* 20,wFormatTag */
				     0,0,             /* 22,wChannels */
				     0,0,0,0,         /* 24,dwSamplesPerSec */
				     0,0,0,0,         /* 28,dwAvgBytesPerSec */
				     0,0,             /* 32,wBlockAlign */
				     0,0,             /* 34,wBitsPerSample */
				     'd','a','t','a', /* 36,chunkID */
				     0,0,0,0          /* 40,chunkSize */
	};
#define WAVHDR_CHANNELS 22
#define WAVHDR_SMP_PER_SEC 24
#define WAVHDR_BYTES_PER_SEC 28
#define WAVHDR_BLOCKALIGN 32
#define WAVHDR_BITS_PER_SMP 34
#define WAVHDR_DATASIZE 40
#define WAVHDR_LENGTH 44

	rec_fp=fopen(file,"wb");
	if (rec_fp==NULL)
		return FALSE;
	/* Endian-independent construction of header. */
	if (record_props->stereo)
		wave_header[WAVHDR_CHANNELS]=2;
	else
		wave_header[WAVHDR_CHANNELS]=1;
	wave_header[WAVHDR_SMP_PER_SEC]=record_props->rate&0xFF;
	wave_header[WAVHDR_SMP_PER_SEC+1]=(record_props->rate>>8)&0xFF;
	n=record_props->rate;
	if (record_props->stereo)
		n*=2;
	if (record_props->bits==16)
		n*=2;
	wave_header[WAVHDR_BYTES_PER_SEC]=n&0xFF;
	wave_header[WAVHDR_BYTES_PER_SEC+1]=(n>>8)&0xFF;
	wave_header[WAVHDR_BYTES_PER_SEC+2]=(n>>16)&0xFF;
	n=1;
	if (record_props->stereo)
		n*=2;
	if (record_props->bits==16)
		n*=2;
	wave_header[WAVHDR_BLOCKALIGN]=n;
	wave_header[WAVHDR_BITS_PER_SMP]=record_props->bits;

	if (fwrite(wave_header,1,WAVHDR_LENGTH,rec_fp)<WAVHDR_LENGTH) {
		fclose(rec_fp);
		return FALSE;
	}

	return TRUE;
}

/*
 * Write data to the WAV file.
 */
static void record_file_write(gpointer data,gint len) {
	fwrite(data,1,len,rec_fp);
	recbytes+=len;
	return;
}

/*
 * Close the file, fix the header, etc.
 * TODO: No error handling...
 */
static void record_file_close(void) {
	unsigned char buf[4];

	fseek(rec_fp,WAVHDR_DATASIZE,SEEK_SET);
	buf[0]=recbytes&0xFF;
	buf[1]=(recbytes>>8)&0xFF;
	buf[2]=(recbytes>>16)&0xFF;
	buf[3]=(recbytes>>24)&0xFF;
	fwrite(buf,1,4,rec_fp);
	fclose(rec_fp);

	rec_fp=NULL;
	recbytes=0;
	return;
}






#if 0     /***** The libaudiofile foolishness... *****/

#include <audiofile.h>
static AFfilehandle recfile=AF_NULL_FILEHANDLE;

/*
 * Open file as a WAV file for writing. Returns TRUE on success.
 */
static gboolean record_file_open(gchar *file) {
	AFfilesetup setup;

	/* Yeech, that audiofile function naming scheme is horrible... */
	setup=afNewFileSetup();
	afInitFileFormat(setup,AF_FILE_WAVE);
	afInitRate(setup,AF_DEFAULT_TRACK,record_props->rate);
	afInitChannels(setup,AF_DEFAULT_TRACK,record_props->stereo?2:1);
	afInitSampleFormat(setup,AF_DEFAULT_TRACK,AF_SAMPFMT_TWOSCOMP,
			   record_props->bits);

	recfile=afOpenFile(file,"w",setup);
	afFreeFileSetup(setup);
	if (recfile==AF_NULL_FILEHANDLE) {
		error_open();
		return FALSE;
	}
	/* TODO: set endianess? */
	recsamples=0;
	recfilename=file;
	return TRUE;
}

/*
 * Write data to the WAV file.
 */
static void record_file_write(gpointer data,gint len) {
	if (record_props->bits==16)
		len/=2;
	if (record_props->stereo)
		len/=2;
	afWriteFrames(recfile,AF_DEFAULT_TRACK,data,len);
	recsamples+=len;
	return;
}

/*
 * Close the file, fix the header, etc.
 */
static void record_file_close(void) {
	afCloseFile(recfile);
	recfile=AF_NULL_FILEHANDLE;
	recsamples=0;
	return;
}

#endif    /***** libaudiofile foolishness ends here *****/





/*******************************************************
 * Routines that actually handle the audio stuff...
 * Currently we use ESD with the fallback functions.
 * Mixer is set directly by /dev/mixer, if possible.
 *******************************************************/


/*
 * Starts the actual recording. Opens audio device, sets options and
 * starts the idle function (the function itself stalls).
 * Also sets the mixer settings in /dev/mixer (TODO: no error checks).
 */
static gboolean record_actual_start(void) {
	esd_format_t frmt;
	int fd,i;

/*
	g_print("Starting the recording...\n");
	g_print("   %dHz %d-bit %s, sources:%s%s%s\n",
		record_props->rate,record_props->bits,
		record_props->stereo?"stereo":"mono",
		(record_props->source&REC_LINE)?" LINE":"",
		(record_props->source&REC_MIC)?" MIC":"",
		(record_props->source&REC_CD)?" CD":"");
		*/

	/* Try to set mixer settings */
	if (record_props->source) {
		fd=open("/dev/mixer",O_RDONLY);
		if (fd>=0) {
			ioctl(fd,SOUND_MIXER_READ_RECSRC,&i);
			if (record_props->source&REC_LINE)
				i|=SOUND_MASK_LINE;
			if (record_props->source&REC_MIC)
				i|=SOUND_MASK_MIC;
			if (record_props->source&REC_CD)
				i|=SOUND_MASK_CD;
			ioctl(fd,SOUND_MIXER_WRITE_RECSRC,&i);
			close(fd);
		}
	}

	frmt=0;
	if (record_props->bits==16)
		frmt|=ESD_BITS16;
	else
		frmt|=ESD_BITS8;
	if (record_props->stereo)
		frmt|=ESD_STEREO;
	else
		frmt|=ESD_MONO;
	frmt|=ESD_STREAM;
	frmt|=ESD_RECORD;
	esd_handle=esd_record_stream_fallback(frmt,record_props->rate,
					      NULL,"quickrecord");
	if (esd_handle<0) {
		error_sound();
		return FALSE;
	}
	
	/* TODO: BUG IN GNOME: If anything over G_PRIORITY_LOW is used,
	 * the menu options stop working! */
	idle_handle=gtk_idle_add_priority(G_PRIORITY_LOW,record_idle_function,
					  NULL);
	return TRUE;
}

/*
 * Reads amount bytes to buf.
 * TODO: no error checks
 */
static void record_actual_read(gpointer buf,gint amount) {
	int i=0;

	while (i<amount) {
		i+=read(esd_handle,buf+i,amount-i);
		usleep(1);
	}
	return;
}



/*
 * Remove the idle handler and close the audio device.
 */
static void record_actual_stop(void) {

	esd_close(esd_handle);
	esd_handle=-1;

	gtk_idle_remove(idle_handle);
	idle_handle=0;
	return;
}

