#	Abstract DataBase
#
# ADB.pm
#
# Copyright 1999 -- 2001, onShore Development Inc. <URL:http://www.onshore-devel.com/>
#
#
# This program is free software under the terms of the GNU General Public
# License (GPL). A copy of the GPL, "COPYING", should have been made
# available with this software.  If not, a copy may be obtained at 
# http://www.fsf.org/copyleft/gpl.html
#
# 
#	Provides abstracted interface to SQL DBs to make life a little
#	easier.
#
#	Usage:
#	$database = new ADB(hostname:port, options);
#	$result = $database->query("some sql statement");
#	$value = $result->get_value(fielname, row);
#	etc....
#
#	Read thru for docs on individual functions


use strict;
my $PGRES_EMPTY_QUERY    = 0 ;
my $PGRES_COMMAND_OK     = 1 ;
my $PGRES_TUPLES_OK      = 2 ;
my $PGRES_COPY_OUT       = 3 ;
my $PGRES_COPY_IN        = 4 ;
my $PGRES_BAD_RESPONSE   = 5 ;
my $PGRES_NONFATAL_ERROR = 6 ;
my $PGRES_FATAL_ERROR    = 7 ;

#########################
# ADBQuery class
#########################
package ADBResult;
use Pg;
#########
# Create new ADBResult object;
# $newresult = new ADBQuery(DatabaseSpecificResult);
#########
sub new {
    my $class = shift;
    my $self = {};
    return 0 if ( $_[0] == 0 );
    $self->{result} = $_[0];
    $self->{error} = "";
    $self->{is_ok} = 1;
    $self->{numrows} = PQntuples($self->{result});
    $self->{numfields} = PQnfields($self->{result});
    bless $self, $class;
    return $self;
}

#########
# Free memory used by internal data of ADBResult
# $ADBResult->Free;
#########
sub free {
    my $self = shift;
    PQclear($self->{result});
    return 0;
}

#########
# Get a Row of results from ADBResult
# %array = $ADBResult->GetRow(rownumber);
#########
sub get_row {
    my $self = shift;
    my $row = $_[0];
    my $i = 0;
    my %vals = ();
    return 0 if ( $row >= $self->{'numrows'} );
    while ( $i < $self->{numfields} ) {
	$vals{PQfname($self->{result}, $i)} = PQgetvalue($self->{result}, $row, $i);
	$i++;
    }
    return %vals;
}


sub get_field_type {
    my $self = shift;
    my $fname = $_[0];
    my $fnum;
    my $type;
    $fnum = PQfnumber($self->{result}, $fname);
    $type = PQftype($self->{result}, $fnum);
    return 0 if ( ! $type );
    return $type;
}

##########
# Get a value from a row by its fieldname
# $scalar = $ADBResult->GetValue(rownumber, fieldname)
##########
sub get_value {
    my $self = shift;
    my $row = $_[0];
    my $fname = $_[1];
    my %vals;
    return 0 if ($row >= $self->{'numrows'});
    %vals = $self->get_row($row);
    return $vals{$fname} if $vals{$fname};
    return 0;
}

##########
# Get the number of rows in a return
# $scalar = $ADBResult->GetNumRows
##########
sub get_num_rows {
    my $self = shift;
    return $self->{numrows};
}

##########
# Get the number of fields in a return
# $scalar = $ADbResult = $ADBResult->GetNumFields
##########
sub get_num_fields {
    my $self = shift;
    return $self->{numfields};
}



###############################################
# ADB Superclass, or rather connection class
# We make a new one of these everytime we wanna connect to a database
###############################################
package ADB;
use Pg;

#############
# $ADB = new ADB(hostname:port, dbname, options)
#############
sub new {
    my @junk;
    my ($host, $port) = ("localhost", "5432");
    my $class = shift;
    my $self = {};
    my $address = $_[0];
    @junk = split(/:/, $address);
    $self->{host} = $junk[0] if ( $#junk >= 0 );
    $self->{port} = $junk[1] if ( $#junk >= 1 );
    $self->{dbname} = $_[1];
    $self->{options} = "";
    $self->{tty} = "";
    $self->{err} = "";
    $self->{status} = 1;
    if ( $_[2] eq "sql_escape_data" ) { 
	$self->{"sql_escape_data"} = 1; 
    }
    $self->{dbconn} = PQsetdb ($self->{host},
			       $self->{port}, 
			       $self->{options},
			       $self->{tty},
			       $self->{dbname});
    if ( PQstatus($self->{dbconn}) != PGRES_CONNECTION_OK ) {
	$self->{'err'} = PQerrorMessage($self->{dbconn});
	$self->{'status'} = 0;
    }
    bless $self, $class;
    $self->query("set datestyle='US';");
    return $self;		
}


############
# Return the error string for the ADB object
############
sub error {
    my $self = shift;
    return $self->{'err'} if $self->{'err'};
    return 0;
}

sub errorstring {
    my $self = shift;
    my $err = $self->error;

    return "$self->{'host'}:$self->{'port'}:$self->{'dbname'}: $err\n";
}
	
sub is_ok {
    my $self = shift;
    return $self->{'status'};
}

sub status {
    my $self = shift;
    return $self->{'status'};
}

############
# Execute an SQL query
# $ADBResult = $ADB->Query("sql satment")
############
sub query {
    my $self = shift;
    my $sql = $_[0];
    my $query;
    my $rc;
    my $result;
    my $status;
	
    $result = PQexec($self->{dbconn}, $sql);
    $status = PQresultStatus($result);	
    unless( $status == PGRES_COMMAND_OK ||
	    $status == PGRES_TUPLES_OK ) {
	$self->{'err'} = PQerrorMessage($self->{dbconn});
	return 0;
    }
    $query = new ADBResult($result);
    return $query if ( $query );
    $self->{'err'} = "ADBResult creation failed";
    return 0;
}	


sub get_record {
    my $self = shift;
    my $key = $_[0];
    my $value = $_[1];
    my $table = $_[2];
    my $result;
    my $trash;
    my %hash; 
    my $type = 0;

    $result = $self->query("select $key from $table where 1 = 2;");
    return 0 unless ( $result );
    $type = $result->get_field_type($key);
    if ( $type == 23 ) {
	$result = $self->query("select * from $table where $key = $value;");
    } else {
	$result = $self->query("select * from $table where $key = '$value';");
    }
    $trash = $result->get_num_rows;
    if ( $trash ) {
	%hash = $result->get_row(0);
    }
    return \%hash if ( %hash );
    return 0;
}

sub get_sql_type {
    my $self = shift;
    my $field = $_[0];
    my $table = $_[1];
    my $result;
    my $type;
    $result = $self->query("select $field from $table where 1 = 2");
    return 0 unless ( $result );
    $type = $result->get_field_type($field);
    return $type;
}

sub insert_record {
    my $self = shift;
    my $key = $_[0];
    my $table = $_[1];
    my $h = $_[2];
    my %hash = %$h;
    my $sql;
    my $i;
    my $j;
    my $val;
    my @keys;
    my $result;
    
    if ( $self->is_unique($key, $hash{$key}, $table) ) {
	$j = 0;
	$sql = "INSERT into $table (";
	@keys = keys %hash;
	foreach $i (@keys) {
	    if ( $j == 0 ) {
		$sql .= "$i";
		$j = 1;
	    } else {
		$sql .= ", $i";
	    }
	}
	$sql .= ") VALUES (";
	$j = 0;
	foreach $i (@keys) {
	    if ( $self->{sql_escape_data} ) { 
		$hash{$i} =~ s/'/\\'/g; 
		$hash{$i} =~ s/\\\\'/\\\\\\'/g; 
	    }
	    $val = "'$hash{$i}'";
	    if ( $j == 0 ) {
		$sql .= "$val";
		$j = 1;
	    } else {
		$sql .= ", $val";
	    }
	}
	$sql .= ");";
	$result = $self->query($sql);
	return 0 unless ( $result );
	return 1;
    } else {
	$self->{'err'} = "insert_record: key value was not unique";
	return 0;
    }
}	

sub insert_record_seq {
    my $self = shift;
    my $key = $_[0];
    my $table = $_[1];
    my $h = $_[2];
    my %hash = %$h;
    my $sql;
    my $result;
    my $oldmax;
    my $type;

    $type = $self->get_sql_type($key, $table);
    if ( $type != 23 ) {
	$self->{'err'} = "insert_record_seq: key was not of type int";
	return 0;
    }
    $result = $self->query("select max($key) from $table;");
    return 0 unless ( $result );
    $oldmax = $result->get_value(0, "max");
    $oldmax = 0 unless ( $oldmax );
    $oldmax++;
    $hash{$key} = $oldmax;
    $result = $self->insert_record($key, $table, \%hash);
    return 0 unless ( $result );
    return $oldmax;
}


#################
# Tells you if value is unique for field in table
# returns '-1' if error
# $bool = $ADB->is_unique(table, fieldname, value)
#################
sub is_unique {
    my $self = shift;
    my ($field, $value, $table) = @_;
    my $result;
    my $status;
    my $type;

    $result = $self->query("select $field from $table where 1 = 2;");
    $type = $result->get_field_type($field);
    if ( $type == 23 ) {
	$result = $self->query("select $field from $table where $field = $value");
    } else {
	$result = $self->query("select $field from $table where $field = '$value'");
    }
    $status = PQresultStatus($result->{result});  
    if ( $status == PGRES_BAD_RESPONSE ||
	 $status == PGRES_NONFATAL_ERROR ||
	 $status == PGRES_FATAL_ERROR ) {
        $self->{'err'} = PQerrorMessage($self->{dbconn});
        return -1;
    }
    return 1 if ( $result->{numrows} == 0 );
    return 0;
}

sub list {
    my $self = shift;
    my $field = $_[0];
    my $table = $_[1];
    my $criteria = $_[2];
    my $result;
    my @vals;
    my $i;
    my $j;
    my $numrows;

    if ( $criteria ) {
	$criteria = "Where " . $criteria;
    } else {
	$criteria = '';
    }

    $result = $self->query("select $field from $table $criteria order by $field;");

    unless ( $result ) {
	return 0;
    }
    $numrows = $result->get_num_rows;
    $i = 0;
    while ( $i < $numrows ) {
	$j = $result->get_value($i, $field);
	push @vals, $j;
	$i++;
    }
    return @vals;
}

sub delete_record
{
    my $self = shift;
    my $key = $_[0];
    my $value = $_[1];
    my $table = $_[2];
    my $result;
    my $type;

    $result = $self->query("select $key from $table where 1 = 2;");
    $type = $result->get_field_type($key);
    if ( $type == 23 ) {
	$result = $self->query("delete from $table where $key = $value;");
    } else {
	$result = $self->query("delete from $table where $key = '$value';");
    }
    return 0 unless ( $result );
    return 1;
}

################
# Properly close a ADB connection
# $ADB->Close;
################
sub close {
    my $self = shift;
    PQfinish ($self->{dbconn});
    $self->{dbconn} = '';
}

1;	
