## Copyright (C) 1994-2012 John W. Eaton
##
## This file is part of Octave.
##
## Octave is free software; you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 3 of the License, or (at
## your option) any later version.
##
## Octave is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with Octave; see the file COPYING.  If not, see
## <http://www.gnu.org/licenses/>.

## -*- texinfo -*-
## @deftypefn  {Function File} {} polyderiv (@var{p})
## @deftypefnx {Function File} {[@var{k}] =} polyderiv (@var{a}, @var{b})
## @deftypefnx {Function File} {[@var{q}, @var{d}] =} polyderiv (@var{b}, @var{a})
## Return the coefficients of the derivative of the polynomial whose
## coefficients are given by the vector @var{p}.  If a pair of polynomials
## is given, return the derivative of the product @math{@var{a}*@var{b}}.
## If two inputs and two outputs are given, return the derivative of the
## polynomial quotient @math{@var{b}/@var{a}}.  The quotient numerator is
## in @var{q} and the denominator in @var{d}.
## @seealso{poly, polyint, polyreduce, roots, conv, deconv, residue,
## filter, polygcd, polyval, polyvalm}
## @end deftypefn

## Author: Tony Richardson <arichard@stark.cc.oh.us>
## Created: June 1994
## Adapted-By: jwe

function [q, d] = polyderiv (p, a)

  persistent warned = false;
  if (! warned)
    warned = true;
    warning ("Octave:deprecated-function",
             "polyderiv is obsolete and will be removed from a future version of Octave; please use polyder instead");
  endif

  if (nargin == 1 || nargin == 2)
    if (! isvector (p))
      error ("polyderiv: argument must be a vector");
    endif
    if (nargin == 2)
      if (! isvector (a))
        error ("polyderiv: argument must be a vector");
      endif
      if (nargout == 1)
        ## derivative of p*a returns a single polynomial
        q = polyderiv (conv (p, a));
      else
        ## derivative of p/a returns numerator and denominator
        d = conv (a, a);
        if (numel (p) == 1)
          q = -p * polyderiv (a);
        elseif (numel (a) == 1)
          q = a * polyderiv (p);
        else
          q = conv (polyderiv (p), a) - conv (p, polyderiv (a));
          q = polyreduce (q);
        endif

        ## remove common factors from numerator and denominator
        x = polygcd (q, d);
        if (length(x) != 1)
          q = deconv (q, x);
          d = deconv (d, x);
        endif

        ## move all the gain into the numerator
        q = q/d(1);
        d = d/d(1);
      endif
    else
      lp = numel (p);
      if (lp == 1)
        q = 0;
        return;
      elseif (lp == 0)
        q = [];
        return;
      endif

      ## Force P to be a row vector.
      p = p(:).';

      q = p(1:(lp-1)) .* [(lp-1):-1:1];
    endif
  else
    print_usage ();
  endif

endfunction

%!assert(all (all (polyderiv ([1, 2, 3]) == [2, 2])));

%!assert(polyderiv (13) == 0);

%!error polyderiv ([]);

%!error polyderiv ([1, 2; 3, 4]);

