/*
    nwnet.c - NWDS Calls implementation
    Copyright (C) 1999, 2000  Petr Vandrovec

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Revision history:

	0.00  1999, June 1		Petr Vandrovec <vandrove@vc.cvut.cz>
		Initial release

	0.05  1999, September 5		Philip R. Wilson <prw@home.com>
		Added DCK_CONFIDENCE flag.

	1.00  1999, November 20		Petr Vandrovec <vandrove@vc.cvut.cz>
		Added copyright, couple of important functions and so on...

	1.01  1999, November 20 	Petr Vandrovec <vandrove@vc.cvut.cz>
		Moved wide character functions to wcs.c.
		Moved DN handling functions to rdn.c.

	1.02  1999, December 14		Petr Vandrovec <vandrove@vc.cvut.cz>
		Fixed __NWDSFinishLoginV2, it can return NWE_PASSWORD_EXPIRED on success.
						
	1.03  2000, January 26		Petr Vandrovec <vandrove@vc.cvut.cz>
		Added NWDSGetObjectHostServerAddress and NWDSDuplicateContextInt calls.
		Fixed NWDSDuplicateContext call to copy DCK_NAME_CONTEXT.
		
	1.04  2000, January 29		Petr Vandrovec <vandrove@vc.cvut.cz>
		Fixed NWDXFindConnection to actually work (we always opened
			new connection).

	1.05  2000, April 26		Petr Vandrovec <vandrove@vc.cvut.cz>
		Added NWDSGetCountByClassAndName.

	1.06  2000, April 27		Petr Vandrovec <vandrove@vc.cvut.cz>
		Added NWDSGetServerName.

	1.07  2000, May 6		Petr Vandrovec <vandrove@vc.cvut.cz>
		Modified NWDSGetAttrVal and NWDSComputeAttrValSize to
		support DSV_READ_CLASS_DEF buffers.
		
	1.08  2000, May 21		Petr Vandrovec <vandrove@vc.cvut.cz>
		Fixed NWDSModifyObject iteration handle.

 */

#include "config.h"

#include <stddef.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include <sys/mman.h>

#include "nwnet_i.h"
#include "ncplib_i.h"
#include "ncpcode.h"

static const char* wchar_encoding = NULL;
static const char* default_encoding = NULL;
static const wchar_t* default_tree = NULL;
static const wchar_t* default_context = NULL;

static NWDSCCODE NWDSXlateFromCtx(NWDSContextHandle ctx, wchar_t* dst,
		size_t maxlen, const void* src);
static NWDSCCODE NWDSXlateToCtx(NWDSContextHandle ctx, void* data, 
		size_t maxlen, const wchar_t* src, size_t* ln);
static NWDSCCODE __NWCCGetServerAddressPtr(NWCONN_HANDLE conn, 
		NWObjectCount* count, nuint8** data);

/* debug fprintf */
#if 1
#define dfprintf(X...)
#else
#define dfprintf(X...) fprintf(X)
#endif

size_t unilen(const unicode* x) {
	const unicode* start = x;
	if (x)
		while (*x) x++;
	return x - start;
}

unicode* unirev(unicode* str) {
	unicode* end = str + unilen(str) - 1;
	unicode* oldstr = str;
	while (end > str) {
		unicode tmp;
		
		tmp = *str;
		*str++ = *end;
		*end-- = tmp;
	}
	return oldstr;
}
/* Charset conversion support for libc5 and libc6.0 */
/* libc6.1 contains iconv interface itself (but buggy and non-working) */
static int iconv_88591_to_wchar_t(const char** inp, size_t* inl,
		char** outp, size_t* outl) {
	const char* i;
	size_t il;
	wchar_t *o;
	size_t ol;
	int ret;
	
	ret = 0;
	ol = *outl;
	o = (wchar_t*)*outp;
	il = *inl;
	i = *inp;
	while (il) {
		if (ol < sizeof(*o)) {
			errno = E2BIG;
			ret = -1;
			goto end;
		}
		*o++ = (*i++) & 0xFF;
		il--;
		ol-=sizeof(*o);
		ret++;
	}
end:;
	*inp = i;
	*inl = il;
	*outp = (char*)o;
	*outl = ol;
	return ret;
}

static int iconv_wchar_t_to_88591(const char** inp, size_t* inl,
		char** outp, size_t* outl) {
	const wchar_t* i;
	size_t il;
	char *o;
	size_t ol;
	int ret;
	
	ret = 0;
	ol = *outl;
	o = *outp;
	il = *inl;
	i = (const wchar_t*)*inp;
	while (il >= sizeof(*i)) {
		if (ol < sizeof(*o)) {
			errno = E2BIG;
			ret = -1;
			goto end;
		}
		*o++ = (*i++) & 0xFF;
		il-=sizeof(*i);
		ol-=sizeof(*o);
		ret++;
	}
end:;
	*inp = (const char*)i;
	*inl = il;
	*outp = o;
	*outl = ol;
	return ret;
}

static int iconv_wchar_t_to_wchar_t(const char** inp, size_t* inl,
		char** outp, size_t* outl) {
	const wchar_t* i;
	size_t il;
	wchar_t *o;
	size_t ol;
	int ret;
	
	ret = 0;
	ol = *outl;
	o = (wchar_t*)*outp;
	il = *inl;
	i = (const wchar_t*)*inp;
	while (il >= sizeof(*i)) {
		if (ol < sizeof(*o)) {
			errno = E2BIG;
			ret = -1;
			goto end;
		}
		*o++ = *i++;
		il-=sizeof(*i);
		ol-=sizeof(*o);
		ret++;
	}
end:;
	*inp = (const char*)i;
	*inl = il;
	*outp = (char*)o;
	*outl = ol;
	return ret;
}

my_iconv_t my_iconv_open(const char* to, const char* from) {
	int (*p)(const char** inp, size_t* inl,
		char** outp, size_t* outl) = NULL;
	my_iconv_t ret;
		
	if (!strcmp(from, "WCHAR_T//")) {
		if (!strcmp(to, "ISO_8859-1//"))
			p=iconv_wchar_t_to_88591;
		else if (!strcmp(to, "WCHAR_T//"))
			p=iconv_wchar_t_to_wchar_t;
/*
		if (!strcmp(to, "UTF-8//"))
			return iconv_internal_to_utf8;
*/
	} else if (!strcmp(to, "WCHAR_T//")) {
		if (!strcmp(from, "ISO_8859-1//"))
			p=iconv_88591_to_wchar_t;
/* 
		if (!strcmp(from, "UTF-8//"))
			return iconv_utf8_to_internal;
*/
	}
	/* this conversion is not supported */
	if (!p) {
#ifdef HAVE_ICONV_H
		iconv_t h = iconv_open(to, from);
		if (h == (iconv_t)-1)
			return (my_iconv_t)-1;
		ret = (my_iconv_t)malloc(sizeof(*ret));
		if (!ret) {
			iconv_close(h);
			errno = ENOMEM;
			return (my_iconv_t)-1;
		}
		ret->lowlevel.h = h;
		ret->type = MY_ICONV_LIBC;
		return ret;
#else
		errno = EINVAL;
		return (my_iconv_t)-1;
#endif
	}
	ret = (my_iconv_t)malloc(sizeof(*ret));
	if (!ret) {
		errno = ENOMEM;
		return (my_iconv_t)-1;
	}
	ret->lowlevel.proc = p;
	ret->type = MY_ICONV_INTERNAL;
	return ret;
}

static int my_iconv_is_wchar(my_iconv_t filter) {
	return (filter->type == MY_ICONV_INTERNAL) && (filter->lowlevel.proc == iconv_wchar_t_to_wchar_t);
}

int my_iconv_close(my_iconv_t filter) {
	if (filter->type == MY_ICONV_INTERNAL) {
		;
#ifdef HAVE_ICONV_H		
	} else if (filter->type == MY_ICONV_LIBC) {
		iconv_close(filter->lowlevel.h);
#endif
	}
	free(filter);
	return 0;
}

int my_iconv(my_iconv_t filter, const char** inbuf, size_t* inbytesleft,
		char** outbuf, size_t* outbytesleft) {
	if (filter->type == MY_ICONV_INTERNAL) {
		if (inbuf && outbuf)
			return (filter->lowlevel.proc)(inbuf, inbytesleft, outbuf, outbytesleft);
		else
			return 0;
#ifdef HAVE_ICONV_H			
	} else if (filter->type == MY_ICONV_LIBC) {
		return iconv(filter->lowlevel.h, inbuf, inbytesleft, 
			outbuf, outbytesleft);
#endif			
	} else {
		errno = EBADF;
		return -1;
	}
}

static int iconv_is_wchar_encoding(const char* to, const char* from) {
	static const wchar_t expected[] = L"Test";
	my_iconv_t h;
	size_t il;
	size_t ol;
	const char* i;
	char* o;
	char obuf[40];
	int q;
	
	h = my_iconv_open(to, from);
	if (h == (my_iconv_t)-1)
		return -1;
	dfprintf(stderr, "open ok\n");
	i = "Test";
	il = 4;
	o = obuf;
	ol = sizeof(obuf);
	q = my_iconv(h, &i, &il, &o, &ol);
	my_iconv_close(h);
	if (q == -1)
		return -1;
	dfprintf(stderr, "conv ok, q=%d, il=%d, ol=%d\n", q, il, ol);
	if (sizeof(obuf)-ol != 4 * sizeof(wchar_t))
		return -1;
	dfprintf(stderr, "len ok\n");
	if (memcmp(obuf, expected, 4 * sizeof(wchar_t)))
		return -1;
	dfprintf(stderr, "ok ok\n");
	return 0;
}

static const char* iconv_search_wchar_name(const char* from) {
	static const char *(names[]) = {
		"WCHAR_T//",
		"WCHAR//",
		"UCS4//",
		"UCS4LE//",
		"UCS4LITTLE//",
		"UCS4BE//",
		"UCS4BIG//",
		"ISO-10646//",
		"ISO-10646-LE//",
		"ISO-10646-LITTLE//",
		"ISO-10646-BE//",
		"ISO-10646-BIG//",
		"INTERNAL//",
		NULL };
	const char **x;
	
	/* Yes, it is unbelievable...
	   with glibc up to glibc-2.1.1 (2.1.1 is latest at the time of writting)
	   iconv_open("ASD", "FGH");
	   iconv_open("ASD", "FGH");
	   coredumps in second iconv_open... So it is completely UNUSABLE */
	for (x = names; *x; x++) {
		dfprintf(stderr, "Testing: %s\n", *x);
		if (!iconv_is_wchar_encoding(*x, from))
			break;
	}
	return *x;
}
			
int __NWUUnicodeToInternal(wchar_t* dest, wchar_t* destEnd,
		const unicode* src, const unicode* srcEnd, wchar_t* noMap /*ignored*/,
		wchar_t** destPtr, const unicode** srcPtr) {
	if (!srcEnd) {
		srcEnd = src;
		while (*srcEnd++);
	}
	while (src < srcEnd) {
		if (dest < destEnd) {
			wint_t chr = WVAL_LH(src++, 0);
			*dest++ =  chr;
			continue;
		}
		if (srcPtr)
			*srcPtr = src;
		if (destPtr)
			*destPtr = dest;
		return E2BIG;
	}
	if (srcPtr)
		*srcPtr = src;
	if (destPtr)
		*destPtr = dest;
	return 0;
}

int __NWUInternalToUnicode(unicode* dest, unicode* destEnd,
		const wchar_t* src, const wchar_t* srcEnd, unicode* noMap, 
		unicode** destPtr, const wchar_t** srcPtr) {
	if (!srcEnd) {
		srcEnd = src + wcslen(src) + 1;
	}
	while (src < srcEnd) {
		int err;
		wint_t chr = *src;
		if (chr < 0x10000) {
			if (dest < destEnd) {
				src++;
				WSET_LH(dest++, 0, chr);
				continue;
			}
			err = E2BIG;
		} else {
			/* TODO: utf-16 add here... */
			if (noMap) {
				unicode* p = noMap;
				unicode* tp;
				
				tp = dest;
				while (*p && (dest < destEnd)) {
					*dest++ = *p++;
				}
				if (!*p) {
					src++;
					continue;
				}
				dest = tp;
				err = E2BIG;
			} else {
				err = EILSEQ;
			}
		}
		if (srcPtr)
			*srcPtr = src;
		if (destPtr)
			*destPtr = dest;
		return err;
	}
	if (srcPtr)
		*srcPtr = src;
	if (destPtr)
		*destPtr = dest;
	return 0;
}

void __NWULocalInit(my_iconv_t h) {
	my_iconv(h, NULL, NULL, NULL, NULL);
}

int __NWULocalToInternal(my_iconv_t h, wchar_t* dest, wchar_t* destEnd,
		const char* src, const char* srcEnd, wchar_t* noMap, 
		wchar_t** destPtr, const char** srcPtr) {
	int err = 0;
	size_t destLen = (destEnd - dest) * sizeof(*dest);
	size_t srcLen;
	
	if (!srcEnd) {
		if (my_iconv_is_wchar(h))
			srcEnd = src + (wcslen((const wchar_t*)src) + 1) * sizeof(wchar_t);
		else
			srcEnd = src + strlen(src) + 1;
	}
	srcLen = (srcEnd - src) * sizeof(*src);
	
	while (srcLen > 0) {
		size_t n = my_iconv(h, &src, &srcLen, (char**)&dest, &destLen);
		if (n != (size_t)-1)
			break;
		err = errno;
		if ((err == EILSEQ) && noMap) {
			wchar_t* p = noMap;
			wchar_t* tp;
			size_t tdl;
			
			tp = dest;
			tdl = destLen;
			while (*p && (destLen >= sizeof(*dest))) {
				*dest++ = *p++;
				destLen -= sizeof(*dest);
			}
			if (!*p) {
				src++;
				srcLen -= sizeof(*src);
				continue;
			}
			dest = tp;
			destLen = tdl;
			err = E2BIG;
		}
		break;
	}
	if (srcPtr)
		*srcPtr = src;
	if (destPtr)
		*destPtr = dest;
	return err;
}

int __NWUInternalToLocal(my_iconv_t h, char* dest, char* destEnd,
		const wchar_t* src, const wchar_t* srcEnd, char* noMap, 
		char** destPtr, const wchar_t** srcPtr) {
	int err = 0;
	size_t destLen = (destEnd - dest) * sizeof(*dest);
	size_t srcLen;
	
	if (!srcEnd) {
		srcEnd = src + wcslen(src) + 1;
	}
	srcLen = (srcEnd - src) * sizeof(*src);
	
	/* GRRRR: why is not INTERNAL available for iconv?! */
	while (srcLen > 0) {
		size_t n = my_iconv(h, (const char**)&src, &srcLen, &dest, &destLen);
		if (n != (size_t)-1)
			break;
		err = errno;
		if ((err == EILSEQ) && noMap) {
			char *p = noMap;
			char* tp;
			size_t tdl;
			
			tp = dest;
			tdl = destLen;
			while (*p && (destLen >= sizeof(*dest))) {
				*dest++ = *p++;
				destLen -= sizeof(*dest);
			}
			if (!*p) {
				src++;
				srcLen -= sizeof(*src);
				continue;
			}
			dest = tp;
			destLen = tdl;
			err = E2BIG;
		}
		break;
	}
	if (srcPtr)
		*srcPtr = src;
	if (destLen)
		*destPtr = dest;
	return err;
}

NWDSCCODE NWDSInitRequester(void) {
	static int dsinit = 0;

	if (dsinit)
		return 0;
	if (!default_encoding)
		default_encoding = strdup("ISO_8859-1//");
	if (!wchar_encoding) {
		wchar_encoding = iconv_search_wchar_name(default_encoding);
		if (!wchar_encoding) {
			wchar_encoding = iconv_search_wchar_name("US-ASCII//");
			/* return error... */
		}
	}
	dfprintf(stderr, "iconv: %s\n", wchar_encoding);
	dsinit = 1;
	return 0;
}

static NWDSCCODE NWDXGetConnection(NWDS_HANDLE dsh, NWCONN_HANDLE* result) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
	struct list_head* ptr;
	
	err = NWDXIsValid(dsh);
	if (err)
		return err;
	ptr = dsh->conns.next;
	if (list_empty(ptr))
		return ERR_NO_CONNECTION;
	conn = list_entry(ptr, struct ncp_conn, nds_ring);
	/* FIXME: mark connection as 'authentication disabled' ? */
	ncp_conn_use(conn);
	*result = conn;
	return 0;
}

static NWDSCCODE NWDSSetLastConnection(NWDSContextHandle ctx, NWCONN_HANDLE conn) {
	NWCONN_HANDLE connold;
	
	connold = ctx->dck.last_connection.conn;
	if (conn != connold) {
		if (ctx->dck.flags & DCV_DISALLOW_REFERRALS)
			return ERR_NO_REFERRALS;	/* FIXME! Compare with NWClient! */
		if (conn) {
			ncp_conn_store(conn);
			ctx->dck.last_connection.conn = conn;
			ctx->dck.last_connection.state = conn->state;
		} else {
			ctx->dck.last_connection.conn = NULL;
		}
		if (connold)
			ncp_conn_release(connold);
	}
	return 0;
}

NWDSCCODE __NWDSGetConnection(NWDSContextHandle ctx, NWCONN_HANDLE* result) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	conn = ctx->dck.last_connection.conn;
	if (conn) {
		if (conn->state == ctx->dck.last_connection.state) {
			ncp_conn_use(conn);
			/* FIXME: authentication disabled */
			*result = conn;
			return 0;
		}
		ncp_conn_release(conn);
		ctx->dck.last_connection.conn = NULL;
	}
	err = NWDXGetConnection(ctx->ds_connection, &conn);
	if (err)
		return err;
	err = NWDSSetLastConnection(ctx, conn);
	if (err) {
		NWCCCloseConn(conn);
		return err;
	}
	*result = conn;
	return 0;
}

static inline NWDSCCODE NWDSConnectionFinished(NWDSContextHandle ctx, NWCONN_HANDLE conn) {
	NWCCCloseConn(conn);
	return 0;
}

/* FIXME: Internal only! */
static NWDSCCODE NWDXAddConnection(NWDS_HANDLE dsh, NWCONN_HANDLE conn) {
	list_del(&conn->nds_ring);
	conn->state++;
	list_add(&conn->nds_ring, &dsh->conns);
	return 0;
}

NWDSCCODE NWDSAddConnection(NWDSContextHandle ctx, NWCONN_HANDLE conn) {
	return NWDXAddConnection(ctx->ds_connection, conn);
}

static void NWDXAddContext(NWDS_HANDLE dsh, NWDSContextHandle ctx) {
	if (ctx->ds_connection)
		list_del(&ctx->context_ring);
	ctx->ds_connection = dsh;
	list_add(&ctx->context_ring, &dsh->contexts);
}

static NWDSCCODE __NWDSCreateDSConnection(NWDS_HANDLE *dsh) {
	NWDS_HANDLE tmp;
	
	tmp = (NWDS_HANDLE)malloc(sizeof(*tmp));
	if (!tmp) return ERR_NOT_ENOUGH_MEMORY;
	memset(tmp, 0, sizeof(*tmp));
	tmp->dck.tree_name = default_tree ? wcsdup(default_tree) : NULL;
	INIT_LIST_HEAD(&tmp->contexts);
	INIT_LIST_HEAD(&tmp->conns);
	*dsh = tmp;
	return 0;
}
	
static NWDSCCODE __NWDSReleaseDSConnection(NWDS_HANDLE dsh) {
	/* TODO: mutex */
	/* FIXME: ?? conns ?? */
	if (list_empty(&dsh->contexts) && list_empty(&dsh->conns)) {
		if (dsh->dck.tree_name)
			free(dsh->dck.tree_name);
		if (dsh->authinfo) {
			size_t tlen = dsh->authinfo->header.total;
			memset(dsh->authinfo, 0, tlen);
			munlock(dsh->authinfo, tlen);
			free(dsh->authinfo);
		}
		free(dsh);
	}
	return 0;
}

/* NWDSSetContext supports only 2 transports */
NWDSCCODE NWDSSetTransport(NWDSContextHandle ctx, size_t len, const NET_ADDRESS_TYPE* transports) {
	NWDSCCODE err;
	nuint32* ptr;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	if (len > 20) /* some reasonable limit */
		return NWE_PARAM_INVALID;
	if (len) {
		size_t cnt;
		nuint32* ptr2;
		
		ptr2 = ptr = (nuint32*)malloc(len * sizeof(nuint32));
		if (!ptr)
			return ERR_NOT_ENOUGH_MEMORY;
		for (cnt = len; cnt; cnt--)
			DSET_LH(ptr2++, 0, *transports++);
	} else {
		ptr = NULL;
	}
	if (ctx->dck.transport_types)
		free(ctx->dck.transport_types);
	ctx->dck.transport_types = ptr;
	ctx->dck.transports = len;
	return 0;
}

NWDSCCODE NWDSCreateContextHandle(NWDSContextHandle *ctx) {
	NWDSContextHandle tmp;
	NWDSCCODE err;
	NWDS_HANDLE dsh;

	/* everyone must call NWDSCreateContextHandle before another
	   reasonable NWDS function, so do initialization here... */
	wcscasecmp(L"[Root]", L"[Root]2");
	NWDSInitRequester();
	wcscasecmp(L"[Root]", L"[Root]2");
	err = __NWDSCreateDSConnection(&dsh);
	if (err)
		return err;	
	tmp = (NWDSContextHandle)malloc(sizeof(*tmp));
	if (!tmp) {
		__NWDSReleaseDSConnection(dsh);
		return ERR_NOT_ENOUGH_MEMORY;
	}
	memset(tmp, 0, sizeof(*tmp));
	INIT_LIST_HEAD(&tmp->context_ring);
	tmp->dck.flags = DCV_DEREF_ALIASES | DCV_XLATE_STRINGS | DCV_CANONICALIZE_NAMES;
	tmp->dck.name_form = 0;
	tmp->dck.last_connection.conn = NULL;
	tmp->dck.local_charset = NULL;
	tmp->dck.confidence = DCV_LOW_CONF;
	tmp->dck.dsi_flags = DSI_ENTRY_FLAGS | DSI_OUTPUT_FIELDS | 
		DSI_SUBORDINATE_COUNT | DSI_MODIFICATION_TIME | 
		DSI_BASE_CLASS | DSI_ENTRY_RDN | DSI_ENTRY_DN;
	tmp->xlate.from = (my_iconv_t)-1;
	tmp->xlate.to = (my_iconv_t)-1;
	ncpt_mutex_init(&tmp->xlate.fromlock);
	ncpt_mutex_init(&tmp->xlate.tolock);
	NWDXAddContext(dsh, tmp);
	wcscasecmp(L"[Root]", L"[Root]2");
	err = NWDSSetContext(tmp, DCK_LOCAL_CHARSET, default_encoding);
	if (err) {
		NWDSFreeContext(tmp);
		return err;
	}
	/* set to something reasonable */
	wcscasecmp(L"[Root]", L"[Root]2");
	err = NWDSSetContext(tmp, DCK_NAME_CONTEXT, "[Root]");
	if (err) {
		NWDSFreeContext(tmp);
		return err;
	}
	{
		static const nuint32 t[] = {
#ifdef NCP_IPX_SUPPORT
			NT_IPX, 
#endif
#ifdef NCP_IN_SUPPORT
			NT_UDP,
#endif
		};

		err = NWDSSetTransport(tmp, sizeof(t)/sizeof(t[0]), t);
	}
	if (err) {
		NWDSFreeContext(tmp);
		return err;
	}
	*ctx = tmp;
	return 0;
}

static NWDSCCODE NWDSDuplicateContextHandleInt(NWDSContextHandle srcctx,
		NWDSContextHandle *ctx) {
	NWDSContextHandle tmp;
	NWDSCCODE err;
	
	if (!srcctx)
		return ERR_NULL_POINTER;
	tmp = (NWDSContextHandle)malloc(sizeof(*tmp));
	if (!tmp)
		return ERR_NOT_ENOUGH_MEMORY;

	memset(tmp, 0, sizeof(*tmp));
	INIT_LIST_HEAD(&tmp->context_ring);
	/* return typed absolute names */
	tmp->dck.flags = srcctx->dck.flags & ~(DCV_CANONICALIZE_NAMES | DCV_TYPELESS_NAMES);
	tmp->dck.name_form = srcctx->dck.name_form;
	tmp->dck.last_connection.conn = srcctx->dck.last_connection.conn;
	tmp->dck.local_charset = NULL;
	tmp->dck.confidence = srcctx->dck.confidence;
	tmp->dck.dsi_flags = srcctx->dck.dsi_flags;
	tmp->xlate.from = (my_iconv_t)-1;
	tmp->xlate.to = (my_iconv_t)-1;
	ncpt_mutex_init(&tmp->xlate.fromlock);
	ncpt_mutex_init(&tmp->xlate.tolock);
	NWDXAddContext(srcctx->ds_connection, tmp);
	/* select wchar_t encoding for NWDS* functions */
	err = NWDSSetContext(tmp, DCK_LOCAL_CHARSET, NULL);
	if (err) {
		NWDSFreeContext(tmp);
		return err;
	}
	err = NWDSSetContext(tmp, DCK_NAME_CONTEXT, srcctx->dck.namectx);
	if (err) {
		NWDSFreeContext(tmp);
		return err;
	}
	/* FIXME! Maybe other fields... */
	{
		size_t tlen = srcctx->dck.transports * sizeof(nuint32);
		void* tt = malloc(tlen);
		if (tt) {
			tmp->dck.transport_types = tt;
			tmp->dck.transports = srcctx->dck.transports;
			memcpy(tt, srcctx->dck.transport_types, tlen);
		} else
			err = ERR_NOT_ENOUGH_MEMORY;
	}
	if (err) {
		NWDSFreeContext(tmp);
		return err;
	}
	*ctx = tmp;
	return 0;
}

NWDSCCODE NWDSDuplicateContextHandle(NWDSContextHandle srcctx,
		NWDSContextHandle *ctx) {
	NWDSContextHandle tmp;
	NWDSCCODE err;

	err = NWDSDuplicateContextHandleInt(srcctx, &tmp);	
	if (err)
		return err;

	err = NWDSSetContext(tmp, DCK_LOCAL_CHARSET, srcctx->dck.local_charset);
	if (err) {
		NWDSFreeContext(tmp);
		return err;
	}
	/* DCK_LOCAL_CHARSET modifies dck.flags */
	tmp->dck.flags = srcctx->dck.flags;
	*ctx = tmp;
	return 0;
}

/* must be able to free partially allocated context handle */
NWDSCCODE NWDSFreeContext(NWDSContextHandle ctx) {
	if (ctx->ds_connection) {
		list_del(&ctx->context_ring);
		__NWDSReleaseDSConnection(ctx->ds_connection);
	}
	if (ctx->xlate.from != (my_iconv_t)-1)
		my_iconv_close(ctx->xlate.from);
	if (ctx->xlate.to != (my_iconv_t)-1)
		my_iconv_close(ctx->xlate.to);
	ncpt_mutex_destroy(&ctx->xlate.fromlock);
	ncpt_mutex_destroy(&ctx->xlate.tolock);
	if (ctx->dck.local_charset)
		free(ctx->dck.local_charset);
	if (ctx->dck.transport_types)
		free(ctx->dck.transport_types);
	__NWDSDestroyRDN(&ctx->dck.rdn);
	if (ctx->dck.namectx)
		free(ctx->dck.namectx);
	free(ctx);
	return 0;
}

NWDSContextHandle NWDSCreateContext(void) {
	NWDSContextHandle ctx;
	NWDSCCODE err;
	
	err = NWDSCreateContextHandle(&ctx);
	if (err)
		return (NWDSContextHandle)ERR_CONTEXT_CREATION;
	return ctx;
}

NWDSCCODE NWDSSetContext(NWDSContextHandle ctx, int key, const void* ptr) {
	NWDSCCODE err;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	switch (key) {
		case DCK_FLAGS:
			{
				ctx->dck.flags = *(const nuint32*)ptr;
			}
			return 0;
		case DCK_CONFIDENCE:
			{
				/* confidence can be any value... */
				ctx->dck.confidence = *(const nuint32*)ptr;
			}
			return 0;
		case DCK_NAME_CONTEXT:
			{
				wchar_t *namectx;
				struct RDNInfo ctxRDN;
				
				namectx = (wchar_t*)malloc(sizeof(wchar_t) * 1024);
				if (!namectx)
					return ERR_NOT_ENOUGH_MEMORY;
				err = NWDSXlateFromCtx(ctx, namectx, sizeof(wchar_t)*1024, ptr);
				if (err)
					return err;
				if (!wcscasecmp(namectx, L"[Root]")) {
					ctxRDN.end = NULL;
					ctxRDN.depth = 0;
				} else {
					err = __NWDSCreateRDN(&ctxRDN, namectx, NULL);
					if (err) {
						free(namectx);
						return err;
					}
				}
				__NWDSDestroyRDN(&ctx->dck.rdn);
				memcpy(&ctx->dck.rdn, &ctxRDN, sizeof(ctxRDN));
				free(ctx->dck.namectx);
				ctx->dck.namectx = namectx;
			}
			return 0;
		case DCK_DSI_FLAGS:
			{
				ctx->dck.dsi_flags = *(const nuint32*)ptr | DSI_OUTPUT_FIELDS;
			}
			return 0;
		case DCK_NAME_FORM:
			{
				nuint32 v = *(const nuint32*)ptr;
				nuint32 x;
				
				if (v == DCV_NF_PARTIAL_DOT)
					x = 0;
				else if (v == DCV_NF_FULL_DOT)
					x = 4;
				else if (v == DCV_NF_SLASH)
					x = 2;
				else
					x = 0;
	/* 0: CN=TEST.OU=VC.O=CVUT.C=CZ (NF_PARTIAL_DOT)
	 * 1: TEST.VC.CVUT.CZ
	 * 2: \T=CVUT\C=CZ\O=CVUT\OU=VC\CN=TEST (NF_SLASH)
	 * 3: \CVUT\CZ\CVUT\VC\TEST
	 * 4: CN=TEST.OU=VC.O=CVUT.C=CZ.T=CVUT. (NF_FULL_DOT)
	 * 5: TEST.VC.CVUT.CZ.CVUT.
	 * 6 & 7 == 4 & 5
	 * 8..9:  Structured (1), unsupported (?)
	 * 10..15: Mirror of 8..9
	 * 16..17: Structured (2), unsupported (?)
	 * 18..31: Mirror of 16..17
	 * other:  Only low 5 bits are significant on NW5 (DS 7.28)
	 * Structured (1) and (2) are not supported ATM TODO */
				ctx->dck.name_form = x;
			}
			return 0;
		case DCK_LOCAL_CHARSET:
			{
				my_iconv_t f;
				my_iconv_t t;
				const char* name = ptr;
				
				if (!name)
					name = wchar_encoding;

				if (ctx->dck.local_charset && !strcmp(name, ctx->dck.local_charset))
					return 0;
				f = my_iconv_open(wchar_encoding, name);
				if (f == (my_iconv_t)-1)
					return ERR_UNICODE_FILE_NOT_FOUND;
				t = my_iconv_open(name, wchar_encoding);
				if (t == (my_iconv_t)-1) {
					my_iconv_close(f);
					return ERR_UNICODE_FILE_NOT_FOUND;
				}
				if (ctx->xlate.from != (my_iconv_t)-1)
					my_iconv_close(ctx->xlate.from);
				ctx->xlate.from = f;
				if (ctx->xlate.to != (my_iconv_t)-1)
					my_iconv_close(ctx->xlate.to);
				ctx->xlate.to = t;
				if (ctx->dck.local_charset)
					free(ctx->dck.local_charset);
				ctx->dck.local_charset = strdup(name);
				ctx->dck.flags |= DCV_XLATE_STRINGS;
				return 0;
			}
	}
	return ERR_BAD_KEY;
}

NWDSCCODE NWDSGetContext2(NWDSContextHandle ctx, int key, void* ptr, size_t maxlen) {
	NWDSCCODE err;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	switch (key) {
		case DCK_FLAGS:
			if (maxlen < sizeof(u_int32_t))
				return NWE_BUFFER_OVERFLOW;
			*(u_int32_t*)ptr = ctx->dck.flags;
			return 0;
		case DCK_CONFIDENCE:
			if (maxlen < sizeof(u_int32_t))
				return NWE_BUFFER_OVERFLOW;
			*(u_int32_t*)ptr = ctx->dck.confidence;
			return 0;
		case DCK_NAME_CONTEXT:
			err = NWDSXlateToCtx(ctx, ptr, maxlen, ctx->dck.namectx ? : L"[Root]", NULL);
			return err;
		case DCK_RDN:
			if (maxlen < sizeof(ctx->dck.rdn))
				return NWE_BUFFER_OVERFLOW;
			memcpy(ptr, &ctx->dck.rdn, sizeof(ctx->dck.rdn));
			return 0;
		case DCK_LAST_CONNECTION:
			if (maxlen < sizeof(NWCONN_HANDLE))
				return NWE_BUFFER_OVERFLOW;
			*(NWCONN_HANDLE*)ptr = ctx->dck.last_connection.conn;
			return 0;
		case DCK_TREE_NAME:
			{
				static const wchar_t null = 0;
				const wchar_t* src = ctx->ds_connection->dck.tree_name;
				
				if (!src) src = &null;
				err = NWDSXlateToCtx(ctx, ptr, maxlen, src, NULL);
			}
			return err;
		case DCK_DSI_FLAGS:
			if (maxlen < sizeof(nuint32))
				return NWE_BUFFER_OVERFLOW;
			*(u_int32_t*)ptr = ctx->dck.dsi_flags;
			return 0;
		case DCK_NAME_FORM:
			if (maxlen < sizeof(nuint32))
				return NWE_BUFFER_OVERFLOW;
			{
				nuint32* nf = ptr;

				if (ctx->dck.name_form == 4)
					*nf = DCV_NF_FULL_DOT;
				else if (ctx->dck.name_form == 2)
					*nf = DCV_NF_SLASH;
				else
					*nf = DCV_NF_PARTIAL_DOT;
			}
			return 0;
	}
	return ERR_BAD_KEY;
}
	
NWDSCCODE NWDSGetContext(NWDSContextHandle ctx, int key, void* ptr) {
	static size_t optlen[] =
		{		      0, 
		      sizeof(u_int32_t),	/* 1, DCK_FLAGS */
		      sizeof(u_int32_t),	/* 2, DCK_CONFIDENCE */      
		      	   MAX_DN_BYTES,	/* 3, DCK_NAME_CONTEXT */
				      0,	/* 4, N/A */
				      0,	/* 5, N/A */
		 sizeof(struct RDNInfo),	/* 6, DCK_RDN */		 
		 		      0,	/* 7, N/A */
		  sizeof(NWCONN_HANDLE),	/* 8, DCK_LAST_CONNECTION */
		  		      0,	/* 9, N/A */
		  		      0,	/* 10, N/A */
	  4 * (MAX_TREE_NAME_CHARS + 1),	/* 11, DCK_TREE_NAME */
		      sizeof(u_int32_t),	/* 12, DCK_DSI_FLAGS */
		      sizeof(u_int32_t),	/* 13, DCK_NAME_FORM */
				      0,	/* 14, N/A */
				      0};	/* 15, N/A */

	if ((key >= DCK_FLAGS) && (key <= DCK_NAME_FORM))
		return NWDSGetContext2(ctx, key, ptr, optlen[key]);
	else
		return NWDSGetContext2(ctx, key, ptr, ~0);	/* unlimited buffer if size not known */
	
}

void NWDSSetupBuf(Buf_T* buf, void* ptr, size_t len) {
	buf->bufFlags = 0;
	buf->cmdFlags = 0;
	buf->dsiFlags = 0;
	buf->data = buf->curPos = ptr;
	buf->dataend = buf->allocend = ptr + len;
	buf->operation = 0;
	buf->attrCountPtr = NULL;
	buf->valCountPtr = NULL;
}

NWDSCCODE NWDSCreateBuf(Buf_T** buff, void* ptr, size_t len) {
	Buf_T* buf;
	
	*buff = NULL;
	buf = (Buf_T*)malloc(sizeof(*buf));
	if (!buf)
		return ERR_NOT_ENOUGH_MEMORY;
	NWDSSetupBuf(buf, ptr, len);
	*buff = buf;
	return 0;
}

NWDSCCODE NWDSAllocBuf(size_t len, pBuf_T* buff) {
	NWDSCCODE err;
	nuint8* buffer;
	Buf_T* buf;
	
	*buff = NULL;
	len = ROUNDBUFF(len);
	buffer = (nuint8*)malloc(len);
	if (!buffer)
		return ERR_NOT_ENOUGH_MEMORY;
	/* if (buffer & 3) { panic("Sorry, malloc must return dword aligned value"}; */
	err = NWDSCreateBuf(&buf, buffer, len);
	if (err)
		free(buffer);
	else {
		buf->bufFlags |= NWDSBUFT_ALLOCATED;
		*buff = buf;
	}
	return err;
}

NWDSCCODE NWDSFreeBuf(Buf_T* buf) {
	if (buf) {
		if (buf->bufFlags & NWDSBUFT_ALLOCATED) {
			free(buf->data);
			buf->data = NULL;
		}
		free(buf);
	}
	return 0;	/* easy... */
}

NWDSCCODE NWDSClearFreeBuf(Buf_T* buf) {
	if (buf && buf->data) {
		memset(buf->data, 0, buf->allocend - buf->data);
	}
	return NWDSFreeBuf(buf);
}

#define NWDSFINDCONN_NOCREATE		0x0000
#define NWDSFINDCONN_CREATEALLOWED	0x0001
#define NWDSFINDCONN_DSREADBUF		0x0002
static NWDSCCODE NWDXFindConnection(NWDS_HANDLE dsh, NWCONN_HANDLE* pconn, u_int32_t saddr, Buf_T* addr, int flags) {
	struct list_head* stop = &dsh->conns;
	u_int32_t saddr2;
	void* p;
	NWDSCCODE err = 0;
	
	saddr2 = saddr;
	p = NWDSBufPeekPtr(addr);
	while (saddr--) {
		struct list_head* current;
		nuint32 addrtype;
		nuint32 len;
		void* data;
		
		if (flags & NWDSFINDCONN_DSREADBUF) {
			nuint32 tlen;

			err = NWDSBufGetLE32(addr, &tlen);
			if (err)
				break;
			if (tlen < 8) {
				err = ERR_INVALID_SERVER_RESPONSE;
				break;
			}
			err = NWDSBufGetLE32(addr, &addrtype);
			if (err)
				break;
			err = NWDSBufGetLE32(addr, &len);
			if (err)
				break;
			if (len > tlen - 8) {
				err = ERR_INVALID_SERVER_RESPONSE;
				break;
			}
			data = NWDSBufGetPtr(addr, tlen - 8);
		} else {
			err = NWDSBufGetLE32(addr, &addrtype);
			if (err)
				break;
			err = NWDSBufGetLE32(addr, &len);
			if (err)
				break;
			data = NWDSBufGetPtr(addr, len);
		}
		if (!data) {
			err = ERR_BUFFER_EMPTY;
			break;
		}
		for (current = stop->next; current != stop; current = current->next) {
			NWCONN_HANDLE conn = list_entry(current, struct ncp_conn, nds_ring);
			NWObjectCount connaddresses;
			nuint8* conndata;
			/* compare addresses */
			
			if (__NWCCGetServerAddressPtr(conn, &connaddresses, &conndata))
				continue;
			while (connaddresses--) {
				if ((DVAL_LH(conndata, 4) == addrtype) &&
				    (DVAL_LH(conndata, 8) == len) &&
				    !memcmp(conndata + 12, data, len)) {
					ncp_conn_use(conn);
					*pconn = conn;
					return 0;
				}
				conndata += 4 + ROUNDPKT(DVAL_LH(conndata, 0));
			}
			
		}
	}
	if (flags & NWDSFINDCONN_CREATEALLOWED) {
		saddr = saddr2;
		NWDSBufSeek(addr, p);
		while (saddr--) {
			nuint32 addrtype;
			nuint32 len;
			void* data;
			NWCCTranAddr tran;
		
			if (flags & NWDSFINDCONN_DSREADBUF) {
				nuint32 tlen;

				err = NWDSBufGetLE32(addr, &tlen);
				if (err)
					break;
				if (tlen < 8) {
					err = ERR_INVALID_SERVER_RESPONSE;
					break;
				}
				err = NWDSBufGetLE32(addr, &addrtype);
				if (err)
					break;
				err = NWDSBufGetLE32(addr, &len);
				if (err)
					break;
				if (len > tlen - 8) {
					err = ERR_INVALID_SERVER_RESPONSE;
					break;
				}
				data = NWDSBufGetPtr(addr, tlen - 8);
			} else {
				err = NWDSBufGetLE32(addr, &addrtype);
				if (err)
					break;
				err = NWDSBufGetLE32(addr, &len);
				if (err)
					break;
				data = NWDSBufGetPtr(addr, len);
			}
			if (!data) {
				err = ERR_BUFFER_EMPTY;
				break;
			}
			switch (addrtype) {
				case NT_IPX:	tran.type = NWCC_TRAN_TYPE_IPX; break;
				case NT_UDP:	tran.type = NWCC_TRAN_TYPE_UDP; break;
				default:	goto nextAddr;
			}
			tran.buffer = data;
			tran.len = len;
			err = NWCCOpenConnByAddr(&tran, 0, NWCC_RESERVED, pconn);
			if (!err)
				return 0;
nextAddr:;
		}
	}
	if (err)
		return err;
	return ERR_NO_CONNECTION;	
}

static NWDSCCODE NWDSTryAuthenticateConn(NWDSContextHandle ctx, NWCONN_HANDLE conn) {
	/* FIXME: Check for ctx->ds_connection->authinfo... */
	return NWDSAuthenticateConn(ctx, conn);
}

static NWDSCCODE NWDSFindConnection(NWDSContextHandle ctx, NWCONN_HANDLE* pconn, u_int32_t cnt, Buf_T* addr, int flags) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
		
	err = NWDXFindConnection(ctx->ds_connection, &conn, cnt, addr, flags);
	if (err)
		return err;
	if (!(conn->connState & CONNECTION_AUTHENTICATED) && 
	    !(ctx->priv_flags & DCV_PRIV_AUTHENTICATING))
		NWDSTryAuthenticateConn(ctx, conn);
	err = NWDSSetLastConnection(ctx, conn);
	if (err) {
		NWCCCloseConn(conn);
		return err;
	}
	*pconn = conn;
	return 0;
}

static NWDSCCODE NWDSFindConnection2(NWDSContextHandle ctx, NWCONN_HANDLE* pconn, Buf_T* addr, int flags) {
	u_int32_t cnt;
	NWDSCCODE err;
	
	err = NWDSBufGetLE32(addr, &cnt);
	if (!err)
		err = NWDSFindConnection(ctx, pconn, cnt, addr, flags);
	return err;
}

static inline NWDSCCODE NWDSGetDSIRaw(NWCONN_HANDLE conn, nuint32 flags, 
		nuint32 DNstyle, NWObjectID id, Buf_T* reply) {
	unsigned char x[16];
	unsigned char rpl[4096];
	size_t rpl_len;
	NWDSCCODE err;
	
	DSET_LH(x, 0, 2);	/* version */
	DSET_LH(x, 4, DNstyle);
	DSET_LH(x, 8, flags);
	DSET_HL(x, 12, id);
	err = ncp_send_nds_frag(conn, DSV_READ_ENTRY_INFO, x, 16, rpl,
		sizeof(rpl), &rpl_len);
	if (!err) {
		dfprintf(stderr, "Reply len: %u\n", rpl_len);
		NWDSBufStartPut(reply, DSV_READ_ENTRY_INFO);
		NWDSBufSetDSIFlags(reply, flags);
		err = NWDSBufPut(reply, rpl, rpl_len);
		NWDSBufFinishPut(reply);
	}
	return err;
}

static NWDSCCODE NWDSXlateUniToCtx(NWDSContextHandle ctx, void* data,
		size_t *maxlen, const unicode* src, size_t ln) {
	NWDSCCODE err;
	nuint32 val;
	const unicode* srcEnd = src + ln / sizeof(*src);
	char* dst = data;
	char* dstEnd = dst + *maxlen;

	err = NWDSGetContext(ctx, DCK_FLAGS, &val);
	if (err)
		return err;
	if (val & DCV_XLATE_STRINGS) {
		NWDSCCODE err2;

		ncpt_mutex_lock(&ctx->xlate.tolock);
		__NWULocalInit(ctx->xlate.to);
		do {
			wchar_t tbuff[128];
			wchar_t* tEnd;

			err2 = __NWUUnicodeToInternal(tbuff, tbuff+128, src, srcEnd, NULL, &tEnd, &src);
			if (data)
				err = __NWUInternalToLocal(ctx->xlate.to, dst, dstEnd, tbuff, tEnd, NULL, &dst, NULL);
			else {
				char tmpChar[1024];
				char* tChar;

				err = __NWUInternalToLocal(ctx->xlate.to, tmpChar, tmpChar+1024, tbuff, tEnd, NULL, &tChar, NULL);
				dst += tChar - tmpChar;
			}
		} while ((!err) && err2 == E2BIG);
		ncpt_mutex_unlock(&ctx->xlate.tolock);
		if (err)
			return err;
		if (err2)
			return err2;
	} else {
		size_t slen = (srcEnd - src) * sizeof(*src);
		if (data) {
			if ((size_t)(dstEnd - dst) < slen)
				return E2BIG;
			memcpy(dst, src, slen);
		}
		dst += slen;
	}
	*maxlen = dst - (char*)data;
	return 0;
}

static NWDSCCODE NWDSXlateCtxToUni(NWDSContextHandle ctx, unicode* data,
		size_t *maxlen, const void* src, size_t ln) {
	NWDSCCODE err;
	nuint32 val;
	unicode* dst = data;
	unicode* dstEnd = dst + *maxlen / sizeof(unicode);

	err = NWDSGetContext(ctx, DCK_FLAGS, &val);
	if (err)
		return err;
	if (val & DCV_XLATE_STRINGS) {
		NWDSCCODE err2;
		const nuint8* srcEnd;

		ncpt_mutex_lock(&ctx->xlate.fromlock);
		__NWULocalInit(ctx->xlate.from);
		if (!ln) {
			if (my_iconv_is_wchar(ctx->xlate.from))
				ln = (wcslen((const wchar_t*)src) + 1) * sizeof(wchar_t);
			else
				ln = strlen(src) + 1;
		}
		srcEnd = src + ln;
		do {
			wchar_t tbuff[128];
			wchar_t* tEnd;

			err2 = __NWULocalToInternal(ctx->xlate.from, tbuff, tbuff+128, src, srcEnd, NULL, &tEnd, (const char**)&src);
			err = __NWUInternalToUnicode(dst, dstEnd, tbuff, tEnd, NULL, &dst, NULL);
		} while ((!err) && err2 == E2BIG);
		ncpt_mutex_unlock(&ctx->xlate.fromlock);
		if (err)
			return err;
		if (err2)
			return err2;
	} else {
		if (!ln)
			ln = (unilen(src) + 1) * sizeof(unicode);
		if (*maxlen < ln)
			return E2BIG;
		memcpy(dst, src, ln);
		dst += ln / sizeof(unicode);
	}
	*maxlen = (dst - data) * sizeof(unicode);
	return 0;
}

static NWDSCCODE NWDSXlateToCtx(NWDSContextHandle ctx, void* data, 
		size_t maxlen, const wchar_t* src, size_t* ln) {
	NWDSCCODE err;
	nuint32 val;
	
	err = NWDSGetContext(ctx, DCK_FLAGS, &val);
	if (err)
		return err;
	if (val & DCV_XLATE_STRINGS) {
		char* ep;
		ncpt_mutex_lock(&ctx->xlate.tolock);
		__NWULocalInit(ctx->xlate.to);
		err = __NWUInternalToLocal(ctx->xlate.to, data, (char*)data+maxlen, 
				src, NULL, NULL, &ep, NULL);
		ncpt_mutex_unlock(&ctx->xlate.tolock);
		if (err)
			return ERR_DN_TOO_LONG; /* or INVALID MULTIBYTE */
		if (ln)
			*ln = ep - (char*)data;
	} else {
		unicode* ep;
		err = __NWUInternalToUnicode(data, (unicode*)data + maxlen / sizeof(unicode), 
				src, NULL, NULL, &ep, NULL);
		if (ln)
			*ln = (ep - (unicode*)data) * sizeof(*ep);
	}
	return err;
}

static NWDSCCODE NWDSXlateFromCtx(NWDSContextHandle ctx, wchar_t* dst,
		size_t maxlen, const void* src) {
	NWDSCCODE err;
	nuint32 val;
	
	err = NWDSGetContext(ctx, DCK_FLAGS, &val);
	if (err)
		return err;
	if (val & DCV_XLATE_STRINGS) {
		ncpt_mutex_lock(&ctx->xlate.fromlock);
		__NWULocalInit(ctx->xlate.from);
		err = __NWULocalToInternal(ctx->xlate.from, dst, dst + maxlen / sizeof(*dst), 
				src, NULL, NULL, NULL, NULL);
		ncpt_mutex_unlock(&ctx->xlate.fromlock);
		if (err)
			return ERR_DN_TOO_LONG;	/* or INVALID MULTIBYTE */
	} else {
		err = __NWUUnicodeToInternal(dst, dst + maxlen / sizeof(*dst), 
				src, NULL, NULL, NULL, NULL);
	}
	return err;
}

static NWDSCCODE NWDSPtrCtxString(NWDSContextHandle ctx,
		const unicode* ptr, size_t len, 
		void* chrs, size_t maxlen, size_t* realLen) {
	NWDSCCODE err;
	static const unicode uz = 0;

	if (!ptr)
		return ERR_BUFFER_EMPTY;
	if (len & 1)
		return ERR_INVALID_OBJECT_NAME;
	if (!len) {
		ptr = &uz;
		len = sizeof(uz);
	}
	if (ptr[(len >> 1) - 1])
		return ERR_INVALID_OBJECT_NAME;
	err = NWDSXlateUniToCtx(ctx, chrs, &maxlen, ptr, len);
	if (realLen)
		*realLen = maxlen;
	return err;
}

NWDSCCODE NWDSBufCtxString(NWDSContextHandle ctx, Buf_T* buffer, 
		void* chrs, size_t maxlen, size_t *realLen) {
	NWDSCCODE err;
	nuint32 len32;
	const unicode* ptr;
	
	err = NWDSBufGetLE32(buffer, &len32);
	if (err)
		return err;
	ptr = NWDSBufGetPtr(buffer, len32);
	return NWDSPtrCtxString(ctx, ptr, len32, chrs, maxlen, realLen);
}

static NWDSCCODE NWDSCtxPtrString(NWDSContextHandle ctx, void* ptr, size_t maxlen,
		const NWDSChar* string, size_t* realLen) {
	NWDSCCODE err;
	
	err = NWDSXlateCtxToUni(ctx, ptr, &maxlen, string, 0);
	if (realLen)
		*realLen = maxlen;
	return err;
}
	
NWDSCCODE NWDSCtxBufString(NWDSContextHandle ctx, Buf_T* buffer,
		const NWDSChar* string) {
	NWDSCCODE err;
	nuint8* p;
	nuint8* strdata;
	size_t strlen;
	
	p = NWDSBufPutPtr(buffer, 4);
	if (!p)
		return ERR_BUFFER_FULL;
	strdata = NWDSBufPutPtrLen(buffer, &strlen);
	if (!strdata)
		return ERR_BUFFER_FULL;
	if (string) {
		err = NWDSCtxPtrString(ctx, strdata, strlen, string, &strlen);
		if (err)
			return err;
	}
	DSET_LH(p, 0, strlen);
	NWDSBufPutSkip(buffer, strlen);
	return 0;
}

NWDSCCODE NWDSPtrDN(const unicode* ptr, size_t len, 
		wchar_t* data, size_t maxlen) {
	NWDSCCODE err;

	if (!ptr)
		return ERR_BUFFER_EMPTY;
	if (len & 1)
		return ERR_INVALID_OBJECT_NAME;	
	if (!len) {
		if (data)
			*data = 0;
		return 0;
	}
	if (ptr[(len >> 1)-1])
		return ERR_INVALID_OBJECT_NAME;
	if (data) {
		err = __NWUUnicodeToInternal(data, data + maxlen / sizeof(*data), 
				ptr, ptr + len / sizeof(*ptr), NULL, NULL, NULL);
		if (err)
			return ERR_DN_TOO_LONG;
	}
	return 0;
}

NWDSCCODE NWDSBufDN(Buf_T* buffer, wchar_t* data, size_t maxlen) {
	NWDSCCODE err;
	nuint32 len32;
	const unicode* ptr;
	
	err = NWDSBufGetLE32(buffer, &len32);
	if (err)
		return err;
	ptr = NWDSBufGetPtr(buffer, len32);
	return NWDSPtrDN(ptr, len32, data, maxlen);
}

static inline NWDSCCODE NWDSBufSkipCIString(Buf_T* buffer) {
	return NWDSBufDN(buffer, NULL, 0);
}

static NWDSCCODE NWDSPtrCtxDN(NWDSContextHandle ctx,
		const unicode* p, size_t len,
		void* ret, size_t* ln) {
	NWDSCCODE err;
	nuint32 v;
	wchar_t tmpb[MAX_DN_CHARS+1];

	err = NWDSGetContext(ctx, DCK_FLAGS, &v);
	if (err)
		return err;
	err = NWDSPtrDN(p, len, tmpb, sizeof(tmpb));
	if (err)
		return err;
	if ((v & DCV_CANONICALIZE_NAMES) && (ctx->dck.name_form == 0)) {
		wchar_t abbrev[MAX_DN_CHARS+1];
		
		err = NWDSAbbreviateNameW(ctx, tmpb, abbrev);
		if (!err)
			err = NWDSXlateToCtx(ctx, ret, MAX_DN_BYTES, abbrev, ln);
	} else {
		err = NWDSXlateToCtx(ctx, ret, MAX_DN_BYTES, tmpb, ln);
	}
	return err;
}

NWDSCCODE NWDSBufCtxDN(NWDSContextHandle ctx, Buf_T* buffer,
		void* ret, size_t* ln) {
	NWDSCCODE err;
	nuint32 len32;
	const unicode* ptr;
	
	err = NWDSBufGetLE32(buffer, &len32);
	if (err)
		return err;
	ptr = NWDSBufGetPtr(buffer, len32);
	return NWDSPtrCtxDN(ctx, ptr, len32, ret, ln);
}

static NWDSCCODE NWDSBufPutCIStringLen(Buf_T* buffer, size_t len, const wchar_t* string) {
	NWDSCCODE err;
	size_t maxlen;
	void* lenspace;
	unicode* strspace;
	unicode* ep;
	
	lenspace = NWDSBufPutPtr(buffer, 4);
	if (!lenspace)
		return ERR_BUFFER_FULL;
	strspace = NWDSBufPutPtrLen(buffer, &maxlen);
	err = __NWUInternalToUnicode(strspace, strspace + maxlen / sizeof(*strspace), 
		string, string + len, NULL, &ep, NULL);
	if (err) {
		buffer->curPos = lenspace;
		return ERR_BUFFER_FULL;
	}
	maxlen = (ep - strspace) * sizeof(*ep);
	DSET_LH(lenspace, 0, maxlen);
	NWDSBufPutSkip(buffer, maxlen);
	return 0;
}

static inline NWDSCCODE NWDSBufPutCIString(Buf_T* buffer, const wchar_t* string) {
	return NWDSBufPutCIStringLen(buffer, wcslen(string)+1, string);
}

NWDSCCODE NWDSBufSkipBuffer(Buf_T* buffer) {
	NWDSCCODE err;
	nuint32 len;
	
	err = NWDSBufGetLE32(buffer, &len);
	if (err)
		return err;
	NWDSBufGetSkip(buffer, len);
	return 0;
}
	
NWDSCCODE NWDSBufPutBuffer(Buf_T* buffer, const void* data, size_t len) {
	nuint8* lenspace;
	
	lenspace = NWDSBufPutPtr(buffer, 4 + len);
	if (!lenspace)
		return ERR_BUFFER_FULL;
	DSET_LH(lenspace, 0, len);
	memcpy(lenspace + 4, data, len);
	return 0;
}

static inline NWDSCCODE NWDSBufPutUnicodeString(Buf_T* buffer, const unicode* string) {
	return NWDSBufPutBuffer(buffer, string, (unilen(string) + 1) * sizeof(unicode));
}

NWDSCCODE NWDSGetCanonicalizedName(NWDSContextHandle ctx, const NWDSChar* src,
		wchar_t* dst) {
	NWDSCCODE err;

	if (src) {
		nuint32 v;
		
		err = NWDSGetContext(ctx, DCK_FLAGS, &v);
		if (err)
			return err;
		if ((v & DCV_CANONICALIZE_NAMES) && (ctx->dck.name_form == 0)) {
			wchar_t tmp[MAX_DN_CHARS+1];
			
			err = NWDSXlateFromCtx(ctx, tmp, sizeof(tmp), src);
			if (!err)
				err = NWDSCanonicalizeNameW(ctx, tmp, dst);
		} else
			err = NWDSXlateFromCtx(ctx, dst, (MAX_DN_CHARS + 1) * sizeof(wchar_t), src);
	} else {
		*dst = 0;
		err = 0;
	}
	return err;
}

NWDSCCODE NWDSCtxBufDN(NWDSContextHandle ctx, Buf_T* buffer,
		const NWDSChar* name) {
	NWDSCCODE err;
		
	if (name) {
		wchar_t tmpb[MAX_DN_CHARS+1];
		
		err = NWDSGetCanonicalizedName(ctx, name, tmpb);
		if (!err)
			err = NWDSBufPutCIString(buffer, tmpb);
	} else
		err = NWDSBufPutLE32(buffer, 0); /* NULL -> zero length... */
	return err;
}

static inline NWDSCCODE NWDSPutAttrVal_DIST_NAME(NWDSContextHandle ctx, Buf_T* buffer,
		const NWDSChar* name) {
	return NWDSCtxBufDN(ctx, buffer, name);
}

/* Attribute name */
static inline NWDSCCODE NWDSGetAttrVal_XX_STRING(NWDSContextHandle ctx, Buf_T* buffer,
		NWDSChar* name) {
	return NWDSBufCtxString(ctx, buffer, name, 999999, NULL);
}

/* Object name (not in attribute) */
static inline NWDSCCODE NWDSGetAttrVal_DIST_NAME(NWDSContextHandle ctx, Buf_T* buffer,
		void* ret) {
	return NWDSBufCtxDN(ctx, buffer, ret, NULL);
}

/* Any string... */
static inline NWDSCCODE NWDSGetAttrSize_XX_STRING2(NWDSContextHandle ctx,
		const unicode* ptr, size_t len, size_t* strLen) {
	return NWDSPtrCtxString(ctx, ptr, len, NULL, 0, strLen);
}

static inline NWDSCCODE NWDSGetAttrVal_XX_STRING2(NWDSContextHandle ctx, 
		const unicode* ptr, size_t len, NWDSChar* name) {
	return NWDSPtrCtxString(ctx, ptr, len, name, 9999999, NULL);
}

/* Dist Name */
static inline NWDSCCODE NWDSGetAttrSize_DIST_NAME2(NWDSContextHandle ctx,
		const unicode* p, size_t len, size_t* strLen) {
	char tmp[MAX_DN_BYTES+1];
	return NWDSPtrCtxDN(ctx, p, len, tmp, strLen);
}

static inline NWDSCCODE NWDSGetAttrVal_DIST_NAME2(NWDSContextHandle ctx, 
		const unicode* p, size_t len, void* ret) {
	return NWDSPtrCtxDN(ctx, p, len, ret, NULL);
}

/* CI List */
static NWDSCCODE NWDSGetAttrSize_CI_LIST(NWDSContextHandle ctx, Buf_T* buffer,
		size_t* size) {
	NWDSCCODE err;
	nuint32 cnt;
	size_t cs = 0;
	
	err = NWDSBufGetLE32(buffer, &cnt);
	if (err)
		return err;
	while (cnt--) {
		size_t ln;
		
		err = NWDSBufCtxString(ctx, buffer, NULL, 0, &ln);
		if (err)
			return err;
		cs += ROUNDBUFF(ln) + sizeof(CI_List_T);
	}
	if (!cs)
		cs = sizeof(CI_List_T);
	*size = cs;
	return 0;
}

static NWDSCCODE NWDSGetAttrVal_CI_LIST(NWDSContextHandle ctx, Buf_T* buffer,
		CI_List_T* ptr) {
	NWDSCCODE err;
	nuint32 cnt;
	u_int8_t* spareSpace;
	
	err = NWDSBufGetLE32(buffer, &cnt);
	if (err)
		return err;
	ptr->s = NULL;
	spareSpace = (u_int8_t*)ptr;
	while (cnt--) {
		size_t ln;
		
		ptr->next = (CI_List_T*)spareSpace;
		ptr = (CI_List_T*)spareSpace;
		spareSpace = (u_int8_t*)(ptr + 1);
		ptr->s = (NWDSChar*)spareSpace;
		err = NWDSBufCtxString(ctx, buffer, (NWDSChar*)spareSpace, 999999, &ln);
		if (err)
			return err;
		spareSpace += ROUNDBUFF(ln);
	}
	ptr->next = NULL;
	return 0;
}

static NWDSCCODE NWDSPutAttrVal_CI_LIST(NWDSContextHandle ctx, Buf_T* buffer,
		const CI_List_T* ptr) {
	NWDSCCODE err;
	nuint32 cnt = 0;
	nuint8* tlen;
	
	tlen = NWDSBufPutPtr(buffer, 8);
	if (!tlen)
		return ERR_BUFFER_FULL;
	while (ptr) {
		err = NWDSCtxBufString(ctx, buffer, ptr->s);
		if (err)
			return err;
		cnt++;
		ptr = ptr->next;
	}
	DSET_LH(tlen, 0, buffer->curPos - tlen - 4);
	DSET_LH(tlen, 4, cnt);
	return 0;
}
	
/* Octet List */
static NWDSCCODE NWDSGetAttrSize_OCTET_LIST(NWDSContextHandle ctx, 
		Buf_T* buffer, size_t* size) {
	NWDSCCODE err;
	nuint32 cnt;
	size_t cs = 0;
	
	err = NWDSBufGetLE32(buffer, &cnt);
	if (err)
		return err;
	while (cnt--) {
		nuint32 ln;
		
		err = NWDSBufGetLE32(buffer, &ln);
		if (err)
			return err;
		if (!NWDSBufGetPtr(buffer, ln))
			return ERR_BUFFER_EMPTY;
		cs += ROUNDBUFF(ln) + sizeof(Octet_List_T);
	}
	if (!cs)
		cs = sizeof(Octet_List_T);
	*size = cs;
	return 0;
}

static NWDSCCODE NWDSGetAttrVal_OCTET_LIST(NWDSContextHandle ctx, Buf_T* buffer,
		Octet_List_T* ptr) {
	NWDSCCODE err;
	nuint32 cnt;
	u_int8_t* spareSpace;
	
	err = NWDSBufGetLE32(buffer, &cnt);
	if (err)
		return err;
	ptr->data = NULL;
	spareSpace = (u_int8_t*)ptr;
	while (cnt--) {
		nuint32 ln;
		
		ptr->next = (Octet_List_T*)spareSpace;
		ptr = (Octet_List_T*)spareSpace;
		spareSpace = (u_int8_t*)(ptr + 1);
		ptr->data = spareSpace;
		err = NWDSBufGetLE32(buffer, &ln);
		if (err)
			return err;
		ptr->length = ln;
		err = NWDSBufGet(buffer, spareSpace, ln);
		if (err)
			return err;
		spareSpace += ROUNDBUFF(ln);
	}
	ptr->next = NULL;
	return 0;
}

static NWDSCCODE NWDSPutAttrVal_OCTET_LIST(NWDSContextHandle ctx, Buf_T* buffer,
		const Octet_List_T* ptr) {
	NWDSCCODE err;
	nuint32 cnt = 0;
	nuint8* tlen;
	
	tlen = NWDSBufPutPtr(buffer, 8);
	if (!tlen)
		return ERR_BUFFER_FULL;
	while (ptr) {
		if (ptr->length && !ptr->data)
			return ERR_NULL_POINTER;
		err = NWDSBufPutBuffer(buffer, ptr->data, ptr->length);
		if (err)
			return err;
		cnt++;
		ptr = ptr->next;
	}
	DSET_LH(tlen, 0, buffer->curPos - tlen - 4);
	DSET_LH(tlen, 4, cnt);
	return 0;
}

/* Replica Pointer */
static NWDSCCODE NWDSGetAttrSize_REPLICA_POINTER(NWDSContextHandle ctx, 
		Buf_T* buffer, size_t* size) {
	NWDSCCODE err;
	char tmpb[MAX_DN_BYTES+1];
	nuint32 le32;
	nuint32 cnt;
	size_t cs;
	
	err = NWDSBufCtxDN(ctx, buffer, tmpb, &cs);
	if (err)
		return err;
	/* DN is at end of buffer, so no rounding here... */
	err = NWDSBufGetLE32(buffer, &le32);
	if (err)
		return err;
	err = NWDSBufGetLE32(buffer, &le32);
	if (err)
		return err;
	err = NWDSBufGetLE32(buffer, &cnt);
	if (err)
		return err;
	cs += sizeof(Net_Address_T) * cnt + 
	      offsetof(Replica_Pointer_T,replicaAddressHint);
	while (cnt--) {
		err = NWDSBufGetLE32(buffer, &le32);
		if (err)
			return err;
		err = NWDSBufGetLE32(buffer, &le32);
		if (err)
			return err;
		if (!NWDSBufGetPtr(buffer, le32))
			return ERR_BUFFER_EMPTY;
		cs += ROUNDBUFF(le32);
	}
	*size = cs;
	return err;
}

static NWDSCCODE NWDSGetAttrVal_REPLICA_POINTER(NWDSContextHandle ctx, Buf_T* buffer,
		Replica_Pointer_T* ptr) {
	NWDSCCODE err;
	nuint32 v;
	wchar_t tmpb[MAX_DN_CHARS+1];
	nuint32 le32;
	nuint32 cnt;
	Net_Address_T* nat;
	u_int8_t* spareSpace;
	
	err = NWDSGetContext(ctx, DCK_FLAGS, &v);
	if (err)
		return err;
	err = NWDSBufDN(buffer, tmpb, sizeof(tmpb));
	if (err)
		return err;
	err = NWDSBufGetLE32(buffer, &le32);
	if (err)
		return err;
	ptr->replicaType = le32;
	err = NWDSBufGetLE32(buffer, &le32);
	if (err)
		return err;
	ptr->replicaNumber = le32;
	err = NWDSBufGetLE32(buffer, &cnt);
	if (err)
		return err;
	ptr->count = cnt;
	nat = ptr->replicaAddressHint;
	spareSpace = (u_int8_t*)(nat + cnt);
	while (cnt--) {
		err = NWDSBufGetLE32(buffer, &le32);
		if (err)
			return err;
		nat->addressType = le32;
		err = NWDSBufGetLE32(buffer, &le32);
		if (err)
			return err;
		nat->addressLength = le32;
		err = NWDSBufGet(buffer, spareSpace, le32);
		if (err)
			return err;
		nat->address = spareSpace;
		spareSpace += ROUNDBUFF(le32);
		nat++;
	}
	ptr->serverName = (NWDSChar*)spareSpace;
	if ((v & DCV_CANONICALIZE_NAMES) && (ctx->dck.name_form == 0)) {
		wchar_t abbrev[MAX_DN_CHARS+1];
		
		err = NWDSAbbreviateNameW(ctx, tmpb, abbrev);
		if (!err)
			err = NWDSXlateToCtx(ctx, spareSpace, MAX_DN_BYTES, abbrev, NULL);
	} else {
		err = NWDSXlateToCtx(ctx, spareSpace, MAX_DN_BYTES, tmpb, NULL);
	}
	return err;
}

static NWDSCCODE NWDSPutAttrVal_REPLICA_POINTER(NWDSContextHandle ctx, Buf_T* buffer,
		const Replica_Pointer_T* ptr) {
	NWDSCCODE err;
	nuint32 cnt;
	const Net_Address_T* nat;
	nuint8* p;
	
	p = NWDSBufPutPtr(buffer, 4);
	if (!p)
		return ERR_BUFFER_FULL;
	err = NWDSCtxBufDN(ctx, buffer, ptr->serverName);
	if (err)
		return err;
	err = NWDSBufPutLE32(buffer, ptr->replicaType);
	if (err)
		return err;
	err = NWDSBufPutLE32(buffer, ptr->replicaNumber);
	if (err)
		return err;
	err = NWDSBufPutLE32(buffer, cnt = ptr->count);
	if (err)
		return err;
	nat = ptr->replicaAddressHint;
	while (cnt--) {
		if (nat->addressLength && !nat->address)
			return ERR_NULL_POINTER;
		err = NWDSBufPutLE32(buffer, nat->addressType);
		if (err)
			return err;
		err = NWDSBufPutBuffer(buffer, nat->address, nat->addressLength);
		if (err)
			return err;
		nat++;
	}
	DSET_LH(p, 0, buffer->curPos - p - 4);
	return 0;
}

/* Net address... size is easy */
static NWDSCCODE NWDSGetAttrVal_NET_ADDRESS(NWDSContextHandle ctx, Buf_T* buffer,
		Net_Address_T* nt) {
	nuint32 type, len;
	NWDSCCODE err;
	void* p;
	
	err = NWDSBufGetLE32(buffer, &type);
	if (err)
		return err;
	err = NWDSBufGetLE32(buffer, &len);
	if (err)
		return err;
	p = NWDSBufGetPtr(buffer, len);
	if (!p)
		return ERR_BUFFER_EMPTY;
	if (nt) {
		nt->addressType = type;
		nt->addressLength = len;
		memcpy(nt->address = (void*)(nt + 1), p, len);
	}
	return err;
}

static NWDSCCODE NWDSPutAttrVal_NET_ADDRESS(NWDSContextHandle ctx, Buf_T* buffer,
		const Net_Address_T* nt) {
	nuint8* ptr;
	
	if (nt->addressLength && !nt->address)
		return ERR_NULL_POINTER;
	ptr = NWDSBufPutPtr(buffer, 4 + 4 + 4 + nt->addressLength);
	if (!ptr)
		return ERR_BUFFER_FULL;
	DSET_LH(ptr, 0, 4 + 4 + nt->addressLength);
	DSET_LH(ptr, 4, nt->addressType);
	DSET_LH(ptr, 8, nt->addressLength);
	memcpy(ptr + 12, nt->address, nt->addressLength);
	return 0;
}
	
/* Octet string... size is easy */
static inline NWDSCCODE NWDSGetAttrVal_OCTET_STRING(NWDSContextHandle ctx, 
		void* p, size_t len, Octet_String_T* os) {
	os->length = len;
	memcpy(os->data = (void*)(os + 1), p, len);
	return 0;
}

static NWDSCCODE NWDSPutAttrVal_OCTET_STRING(NWDSContextHandle ctx,
		Buf_T* buffer, const Octet_String_T* os) {
	if (os->length && !os->data)
		return ERR_NULL_POINTER;
	return NWDSBufPutBuffer(buffer, os->data, os->length);
}

/* Integer... size is easy */
static NWDSCCODE NWDSBufPutSizedLE32(Buf_T* buffer, nuint32 le) {
	NWDSCCODE err;

	err = NWDSBufPutLE32(buffer, 4);
	if (err)
		return err;
	return NWDSBufPutLE32(buffer, le);
}

static NWDSCCODE NWDSGetAttrVal_INTEGER(NWDSContextHandle ctx, 
		void* p, size_t len, Integer_T* i) {
	if (len != 4)
		return NWE_BUFFER_OVERFLOW;
	*i = DVAL_LH(p, 0);
	return 0;
}

static inline NWDSCCODE NWDSPutAttrVal_INTEGER(NWDSContextHandle ctx,
		Buf_T* buffer, const Integer_T* i) {
	return NWDSBufPutSizedLE32(buffer, *i);
}

/* Boolean... size is easy */
static NWDSCCODE NWDSGetAttrVal_BOOLEAN(NWDSContextHandle ctx,
		void* p, size_t len, Boolean_T* b) {
	if (len != 1)
		return NWE_BUFFER_OVERFLOW;
	*b = BVAL(p, 0);
	return 0;
}

static inline NWDSCCODE NWDSPutAttrVal_BOOLEAN(NWDSContextHandle ctx,
		Buf_T* buffer, const Boolean_T* b) {
	return NWDSBufPutSizedLE32(buffer, *b);
}

/* Time... size is easy */
static NWDSCCODE NWDSGetAttrVal_TIME(NWDSContextHandle ctx,
		void* p, size_t len, Time_T* t) {
	if (len != 4)
		return NWE_BUFFER_OVERFLOW;
	*t = DVAL_LH(p, 0);
	return 0;
}

static inline NWDSCCODE NWDSPutAttrVal_TIME(NWDSContextHandle ctx,
		Buf_T* buffer, const Time_T* t) {
	return NWDSBufPutSizedLE32(buffer, *t);
}

/* TimeStamp... size is easy */
static NWDSCCODE NWDSGetAttrVal_TIMESTAMP(NWDSContextHandle ctx,
		void* p, size_t len, TimeStamp_T* ts) {
	if (len != 8)
		return NWE_BUFFER_OVERFLOW;
	ts->wholeSeconds = DVAL_LH(p, 0);
	ts->replicaNum = WVAL_LH(p, 4);
	ts->eventID = WVAL_LH(p, 6);
	return 0;
}

NWDSCCODE NWDSPutAttrVal_TIMESTAMP(NWDSContextHandle ctx,
		Buf_T* buffer, const TimeStamp_T* ts) {
	NWDSCCODE err;
	void* p;

	err = NWDSBufPutLE32(buffer, 8);
	if (err)
		return err;
	p = NWDSBufPutPtr(buffer, 8);
	if (!p)
		return ERR_BUFFER_FULL;
	DSET_LH(p, 0, ts->wholeSeconds);
	WSET_LH(p, 4, ts->replicaNum);
	WSET_LH(p, 6, ts->eventID);
	return 0;
}

/* Object ACL */
static NWDSCCODE NWDSGetAttrSize_OBJECT_ACL(NWDSContextHandle ctx, Buf_T* buffer,
		size_t* size) {
	NWDSCCODE err;
	size_t ln;
	char tmp[MAX_DN_BYTES+1];
	size_t cs;
	
	err = NWDSBufCtxString(ctx, buffer, NULL, 0, &ln);
	if (err)
		return err;
	cs = ROUNDBUFF(ln) + sizeof(Object_ACL_T);
	
	err = NWDSBufCtxDN(ctx, buffer, tmp, &ln);
	if (err)
		return err;
	*size = cs + ln;
	return err;
}

static NWDSCCODE NWDSGetAttrVal_OBJECT_ACL(NWDSContextHandle ctx, Buf_T* buffer,
		Object_ACL_T* oacl) {
	u_int8_t* spareSpace;
	NWDSCCODE err;
	size_t ln;
	nuint32 priv;
	
	spareSpace = (u_int8_t*)(oacl + 1);
	oacl->protectedAttrName = (NWDSChar*)spareSpace;
	
	err = NWDSBufCtxString(ctx, buffer, (NWDSChar*)spareSpace, 999999, &ln);
	if (err)
		return err;
	spareSpace += ROUNDBUFF(ln);
	
	oacl->subjectName = (NWDSChar*)spareSpace;
	err = NWDSBufCtxDN(ctx, buffer, (NWDSChar*)spareSpace, NULL);
	if (err)
		return err;
	err = NWDSBufGetLE32(buffer, &priv);
	oacl->privileges = priv;
	return err;
}

static NWDSCCODE NWDSPutAttrVal_OBJECT_ACL(NWDSContextHandle ctx, Buf_T* buffer,
		const Object_ACL_T* oacl) {
	NWDSCCODE err;
	nuint8* p = NWDSBufPutPtr(buffer, 4);

	if (!p)
		return ERR_BUFFER_FULL;
	err = NWDSCtxBufString(ctx, buffer, oacl->protectedAttrName);
	if (err)
		return err;
	err = NWDSCtxBufDN(ctx, buffer, oacl->subjectName);
	if (err)
		return err;
	err = NWDSBufPutLE32(buffer, oacl->privileges);
	if (err)
		return err;
	DSET_LH(p, 0, buffer->curPos - p - 4);
	return 0;
}

/* Path */
static NWDSCCODE NWDSGetAttrSize_PATH(NWDSContextHandle ctx, Buf_T* buffer,
		size_t* size) {
	NWDSCCODE err;
	size_t ln;
	nuint32 nstype;
	size_t cs;
	char tmp[MAX_DN_BYTES+1];
	
	err = NWDSBufGetLE32(buffer, &nstype);
	if (err)
		return err;
	
	err = NWDSBufCtxDN(ctx, buffer, tmp, &ln);
	if (err)
		return err;
	cs = ROUNDBUFF(ln) + sizeof(Path_T);
	
	err = NWDSBufCtxString(ctx, buffer, NULL, 0, &ln);
	if (err)
		return err;
	*size = cs + ln;
	return 0;
}

static NWDSCCODE NWDSGetAttrVal_PATH(NWDSContextHandle ctx, Buf_T* buffer,
		Path_T* p) {
	u_int8_t* spareSpace;
	NWDSCCODE err;
	size_t ln;
	nuint32 nstype;
	
	spareSpace = (u_int8_t*)(p + 1);
	err = NWDSBufGetLE32(buffer, &nstype);
	if (err)
		return err;
	p->nameSpaceType = nstype;
	p->volumeName = (NWDSChar*)spareSpace;
	
	err = NWDSBufCtxDN(ctx, buffer, (NWDSChar*)spareSpace, &ln);
	if (err)
		return err;
	spareSpace += ROUNDBUFF(ln);
	
	p->path = (NWDSChar*)spareSpace;
	err = NWDSBufCtxString(ctx, buffer, (NWDSChar*)spareSpace, 999999, &ln);
	return err;
}

static NWDSCCODE NWDSPutAttrVal_PATH(NWDSContextHandle ctx, Buf_T* buffer,
		const Path_T* p) {
	NWDSCCODE err;
	nuint8* bp = NWDSBufPutPtr(buffer, 8); 
	
	DSET_LH(bp, 4, p->nameSpaceType);
	err = NWDSCtxBufDN(ctx, buffer, p->volumeName);
	if (err)
		return err;
	err = NWDSCtxBufString(ctx, buffer, p->path);
	if (err)
		return err;
	DSET_LH(bp, 0, buffer->curPos - bp - 4);
	return 0;
}

/* Typed Name */
static NWDSCCODE NWDSGetAttrSize_TYPED_NAME(NWDSContextHandle ctx, Buf_T* buffer,
		size_t* size) {
	NWDSCCODE err;
	size_t ln;
	nuint32 v;
	char tmp[MAX_DN_BYTES+1];
	
	err = NWDSBufGetLE32(buffer, &v);
	if (err)
		return err;

	err = NWDSBufGetLE32(buffer, &v);
	if (err)
		return err;

	err = NWDSBufCtxDN(ctx, buffer, tmp, &ln);
	if (err)
		return err;
	*size = ln + sizeof(Typed_Name_T);
	return 0;
}

static NWDSCCODE NWDSGetAttrVal_TYPED_NAME(NWDSContextHandle ctx, Buf_T* buffer,
		Typed_Name_T* p) {
	NWDSChar* spareSpace;
	NWDSCCODE err;
	size_t ln;
	nuint32 v;
	
	spareSpace = (NWDSChar*)(p + 1);
	err = NWDSBufGetLE32(buffer, &v);
	if (err)
		return err;
	p->level = v;

	err = NWDSBufGetLE32(buffer, &v);
	if (err)
		return err;
	p->interval = v;

	p->objectName = spareSpace;
	err = NWDSBufCtxDN(ctx, buffer, spareSpace, &ln);
	return err;
}

static NWDSCCODE NWDSPutAttrVal_TYPED_NAME(NWDSContextHandle ctx, Buf_T* buffer,
		const Typed_Name_T* tn) {
	NWDSCCODE err;
	nuint8* p = NWDSBufPutPtr(buffer, 12);
	
	if (!p)
		return ERR_BUFFER_FULL;
	DSET_LH(p, 4, tn->level);
	DSET_LH(p, 8, tn->interval);
	err = NWDSCtxBufDN(ctx, buffer, tn->objectName);
	if (err)
		return err;
	DSET_LH(p, 0, buffer->curPos - p - 4);
	return 0;
}

/* BackLink */
static NWDSCCODE NWDSGetAttrSize_BACK_LINK(NWDSContextHandle ctx, Buf_T* buffer,
		size_t* size) {
	NWDSCCODE err;
	size_t ln;
	NWObjectID id;
	char tmp[MAX_DN_BYTES+1];
	
	err = NWDSBufGetID(buffer, &id);
	if (err)
		return err;

	err = NWDSBufCtxDN(ctx, buffer, tmp, &ln);
	if (err)
		return err;
	*size = ln + sizeof(Back_Link_T);
	return 0;
}

static NWDSCCODE NWDSGetAttrVal_BACK_LINK(NWDSContextHandle ctx, Buf_T* buffer,
		Back_Link_T* p) {
	NWDSChar* spareSpace;
	NWDSCCODE err;
	size_t ln;
	NWObjectID id;
	
	spareSpace = (NWDSChar*)(p + 1);
	err = NWDSBufGetID(buffer, &id);
	if (err)
		return err;
	p->remoteID = id;

	p->objectName = spareSpace;
	err = NWDSBufCtxDN(ctx, buffer, spareSpace, &ln);
	return err;
}

static NWDSCCODE NWDSPutAttrVal_BACK_LINK(NWDSContextHandle ctx, Buf_T* buffer,
		const Back_Link_T* bl) {
	NWDSCCODE err;
	nuint8* p = NWDSBufPutPtr(buffer, 8);
	
	if (!bl->objectName)
		return ERR_NULL_POINTER;
	if (!p)
		return ERR_BUFFER_FULL;
	DSET_HL(p, 4, bl->remoteID);
	err = NWDSCtxBufDN(ctx, buffer, bl->objectName);
	if (err)
		return err;
	DSET_LH(p, 0, buffer->curPos - p - 4);
	return 0;
}

/* Hold */
static NWDSCCODE NWDSGetAttrSize_HOLD(NWDSContextHandle ctx, Buf_T* buffer,
		size_t* size) {
	NWDSCCODE err;
	size_t ln;
	nuint32 amount;
	char tmp[MAX_DN_BYTES+1];
	
	err = NWDSBufGetLE32(buffer, &amount);
	if (err)
		return err;

	err = NWDSBufCtxDN(ctx, buffer, tmp, &ln);
	if (err)
		return err;
	*size = ln + sizeof(Hold_T);
	return 0;
}

static NWDSCCODE NWDSGetAttrVal_HOLD(NWDSContextHandle ctx, Buf_T* buffer,
		Hold_T* p) {
	NWDSChar* spareSpace;
	NWDSCCODE err;
	size_t ln;
	nuint32 amount;
	
	spareSpace = (NWDSChar*)(p + 1);
	err = NWDSBufGetLE32(buffer, &amount);
	if (err)
		return err;
	p->amount = amount;

	p->objectName = spareSpace;
	err = NWDSBufCtxDN(ctx, buffer, spareSpace, &ln);
	return err;
}

static NWDSCCODE NWDSPutAttrVal_HOLD(NWDSContextHandle ctx, Buf_T* buffer,
		const Hold_T* h) {
	NWDSCCODE err;
	nuint8* p = NWDSBufPutPtr(buffer, 8);
	
	if (!h->objectName)
		return ERR_NULL_POINTER;
	if (!p)
		return ERR_BUFFER_FULL;
	DSET_LH(p, 4, h->amount);
	err = NWDSCtxBufDN(ctx, buffer, h->objectName);
	if (err)
		return err;
	DSET_LH(p, 0, buffer->curPos - p - 4);
	return 0;
}

/* Fax Number */
static NWDSCCODE NWDSGetAttrSize_FAX_NUMBER(NWDSContextHandle ctx, Buf_T* buffer,
		size_t* size) {
	NWDSCCODE err;
	size_t ln;
	nuint32 v;
	nuint32 l;
	
	err = NWDSBufCtxString(ctx, buffer, NULL, 0, &ln);
	if (err)
		return err;

	err = NWDSBufGetLE32(buffer, &v);
	if (err)
		return err;

	err = NWDSBufGetLE32(buffer, &l);
	if (err)
		return err;
	if (v > l * 8)
		return ERR_INVALID_SERVER_RESPONSE;
	*size = sizeof(Fax_Number_T) + ROUNDBUFF(ln) + l;
	return 0;
}

static NWDSCCODE NWDSGetAttrVal_FAX_NUMBER(NWDSContextHandle ctx, Buf_T* buffer,
		Fax_Number_T* p) {
	u_int8_t* spareSpace;
	NWDSCCODE err;
	size_t ln;
	nuint32 v;
	nuint32 l;
	
	spareSpace = (u_int8_t*)(p + 1);
	p->telephoneNumber = (NWDSChar*)spareSpace;
	err = NWDSBufCtxString(ctx, buffer, (NWDSChar*)spareSpace, 999999, &ln);
	if (err)
		return err;
	spareSpace += ROUNDBUFF(ln);

	err = NWDSBufGetLE32(buffer, &v);
	if (err)
		return err;
	p->parameters.numOfBits = v;

	err = NWDSBufGetLE32(buffer, &l);
	if (err)
		return err;
	if (v > l * 8)
		return ERR_INVALID_SERVER_RESPONSE;
	p->parameters.data = spareSpace;
	err = NWDSBufGet(buffer, spareSpace, l);
	return err;
}

static NWDSCCODE NWDSPutAttrVal_FAX_NUMBER(NWDSContextHandle ctx, Buf_T* buffer,
		const Fax_Number_T* fn) {
	NWDSCCODE err;
	nuint8* p;
	
	if (fn->parameters.numOfBits && !fn->parameters.data)
		return ERR_NULL_POINTER;
	p = NWDSBufPutPtr(buffer, 4);
	if (!p)
		return ERR_BUFFER_FULL;
	err = NWDSCtxBufString(ctx, buffer, fn->telephoneNumber);
	if (err)
		return err;
	err = NWDSBufPutLE32(buffer, fn->parameters.numOfBits);
	if (err)
		return err;
	err = NWDSBufPutBuffer(buffer, fn->parameters.data, (fn->parameters.numOfBits + 7) >> 3);
	if (err)
		return err;
	DSET_LH(p, 0, buffer->curPos - p - 4);
	return 0;
}

/* Email Address */
static NWDSCCODE NWDSGetAttrSize_EMAIL_ADDRESS(NWDSContextHandle ctx, Buf_T* buffer,
		size_t* size) {
	NWDSCCODE err;
	size_t ln;
	nuint32 v;
	
	err = NWDSBufGetLE32(buffer, &v);
	if (err)
		return err;

	err = NWDSBufCtxString(ctx, buffer, NULL, 0, &ln);
	if (err)
		return err;
	*size = ln + sizeof(EMail_Address_T);
	return 0;
}

static NWDSCCODE NWDSGetAttrVal_EMAIL_ADDRESS(NWDSContextHandle ctx, Buf_T* buffer,
		EMail_Address_T* p) {
	NWDSChar* spareSpace;
	NWDSCCODE err;
	size_t ln;
	nuint32 v;
	
	spareSpace = (NWDSChar*)(p + 1);
	err = NWDSBufGetLE32(buffer, &v);
	if (err)
		return err;
	p->type = v;

	p->address = spareSpace;
	err = NWDSBufCtxString(ctx, buffer, spareSpace, 999999, &ln);
	return err;
}

static NWDSCCODE NWDSPutAttrVal_EMAIL_ADDRESS(NWDSContextHandle ctx, Buf_T* buffer,
		const EMail_Address_T* ea) {
	NWDSCCODE err;
	nuint8* p = NWDSBufPutPtr(buffer, 8);
	
	if (!p)
		return ERR_BUFFER_FULL;
	DSET_LH(p, 4, ea->type);
	err = NWDSCtxBufString(ctx, buffer, ea->address);
	if (err)
		return err;
	DSET_LH(p, 0, buffer->curPos - p - 4);
	return 0;
}

/* PO Address */
static NWDSCCODE NWDSGetAttrSize_PO_ADDRESS(NWDSContextHandle ctx, Buf_T* buffer,
		size_t* size) {
	NWDSCCODE err;
	size_t ln;
	nuint32 v;
	nuint32 cnt;
	size_t cs = sizeof(Postal_Address_T);
	
	err = NWDSBufGetLE32(buffer, &v);
	if (err)
		return err;
	if (v > 6)
		v = 6;
	cnt = 6 - v;
	while (v--) {
		err = NWDSBufCtxString(ctx, buffer, NULL, 0, &ln);
		if (err)
			return err;
		cs += ROUNDBUFF(ln);
	}
	*size = cs;
	return 0;
}

static NWDSCCODE NWDSGetAttrVal_PO_ADDRESS(NWDSContextHandle ctx, Buf_T* buffer,
		Postal_Address_T* p) {
	u_int8_t* spareSpace;
	NWDSCCODE err;
	size_t ln;
	nuint32 v;
	nuint32 cnt;
	NWDSChar** q;
	
	spareSpace = (u_int8_t*)(p + 1);
	err = NWDSBufGetLE32(buffer, &v);
	if (err)
		return err;
	if (v > 6)
		v = 6;
	cnt = 6 - v;
	q = *p;
	while (v--) {
		*q++ = (NWDSChar*)spareSpace;
		err = NWDSBufCtxString(ctx, buffer, (NWDSChar*)spareSpace, 9999999, &ln);
		if (err)
			return err;
		spareSpace += ROUNDBUFF(ln);
	}
	while (cnt--) {
		*q++ = NULL;
	}
	return 0;
}

static NWDSCCODE NWDSPutAttrVal_PO_ADDRESS(NWDSContextHandle ctx, Buf_T* buffer,
		const NWDSChar* const* pa) {
	NWDSCCODE err;
	const NWDSChar* const* q;
	nuint32 cnt;
	nuint8* p = NWDSBufPutPtr(buffer, 8);
	
	q = pa;
	for (cnt = 6; cnt > 0; cnt--) {
		if (q[cnt])
			break;
	}
	DSET_LH(p, 4, cnt);
	while (cnt--) {
		err = NWDSCtxBufString(ctx, buffer, *(q++));
		if (err)
			return err;
	}
	DSET_LH(p, 0, buffer->curPos - p -4);
	return 0;
}

/* ... */

static NWDSCCODE NWDSSplitName(
		NWDSContextHandle ctx,
		const NWDSChar* objectName,
		wchar_t* parentName,
		wchar_t* childName) {
	NWDSCCODE err;
	nuint32 v;
	wchar_t tmpb[MAX_DN_CHARS+1];
	wchar_t* c;
		
	err = NWDSGetContext(ctx, DCK_FLAGS, &v);
	if (err)
		return err;
	if ((v & DCV_CANONICALIZE_NAMES) && (ctx->dck.name_form == 0)) {
		err = NWDSXlateFromCtx(ctx, parentName, MAX_DN_BYTES, objectName);
		if (err)
			return err;
		err = NWDSCanonicalizeNameW(ctx, parentName, tmpb);
	} else {
		err = NWDSXlateFromCtx(ctx, tmpb, sizeof(tmpb), objectName);
	}
	if (err)
		return err;
	c = tmpb;
	if ((tmpb[0] == '\\') && ((tmpb[1] != '.') && (tmpb[1] != '+') && (tmpb[1] != '=') && (tmpb[1] != '\\'))) {
		wchar_t* lastSlash;
		
		lastSlash = c++;
		while (*c) {
			if (*c == '\\')
				lastSlash = c;
			c++;
		}
		*lastSlash++ = 0;
		memcpy(parentName, tmpb, (lastSlash - tmpb) * sizeof(wchar_t));
		memcpy(childName, lastSlash, (c - lastSlash + 1) * sizeof(wchar_t));
	} else {
		while (*c) {
			if (*c == '.')
				break;
			if (*c++ == '\\') {
				if (*c)
					c++;
			}
		}
		wcscpy(parentName, (*c)?c+1:L"Root");
		*c = 0;
		memcpy(childName, tmpb, (c - tmpb + 1) * sizeof(wchar_t));
	}
	return 0;
}

NWDSCCODE NWDSPutAttrVal_XX_STRING(NWDSContextHandle ctx, Buf_T* buffer,
		const NWDSChar* name) {
	return NWDSCtxBufString(ctx, buffer, name);
}

NWDSCCODE NWDSComputeAttrValSize(NWDSContextHandle ctx, Buf_T* buffer,
		enum SYNTAX syntaxID, size_t* valsize) {
	NWDSCCODE err = 0;
	nuint32 le32;
	size_t size = 0;
	nuint32 tlen;
	void* p;
	Buf_T buf;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_INPUT)
		return ERR_BAD_VERB;
	switch (buffer->operation) {
		case DSV_READ_CLASS_DEF:;
			if (syntaxID != SYN_OBJECT_ACL)
				return ERR_BAD_VERB;
			p = NWDSBufRetrievePtrAndLen(buffer, &tlen);
			NWDSSetupBuf(&buf, p, tlen);
			err = NWDSGetAttrSize_OBJECT_ACL(ctx, &buf, &size);
			goto quit;
		default:;
			break;
	}
	err = NWDSBufPeekLE32(buffer, 0, &tlen);
	if (err)
		return err;
	p = NWDSBufPeekPtrLen(buffer, 4, tlen);
	if (!p)
		return ERR_BUFFER_EMPTY;
	NWDSSetupBuf(&buf, p, tlen);
	switch (syntaxID) {
		case SYN_DIST_NAME:
			err = NWDSGetAttrSize_DIST_NAME2(ctx, p, tlen, &size);
			break;
		case SYN_CE_STRING:
		case SYN_CI_STRING:
		case SYN_PR_STRING:
		case SYN_NU_STRING:
		case SYN_CLASS_NAME:
		case SYN_TEL_NUMBER:
			err = NWDSGetAttrSize_XX_STRING2(ctx, p, tlen, &size);
			break;
		case SYN_OCTET_STRING:
		case SYN_STREAM:
			err = NWDSBufPeekLE32(buffer, 0, &le32);
			if (!err)
				size = le32 + sizeof(Octet_String_T);
			break;
		case SYN_NET_ADDRESS:
			err = NWDSBufPeekLE32(buffer, 8, &le32);
			if (!err)
				size = le32 + sizeof(Net_Address_T);
			break;
		case SYN_BOOLEAN:
			size = sizeof(Boolean_T);
			break;
		case SYN_COUNTER:
		case SYN_INTEGER:
		case SYN_INTERVAL:
			size = sizeof(Integer_T);
			break;
		case SYN_TIMESTAMP:
			size = sizeof(TimeStamp_T);
			break;
		case SYN_TIME:
			size = sizeof(Time_T);
			break;
		case SYN_REPLICA_POINTER:
			err = NWDSGetAttrSize_REPLICA_POINTER(ctx, &buf, &size);
			break;
		case SYN_OBJECT_ACL:
			err = NWDSGetAttrSize_OBJECT_ACL(ctx, &buf, &size);
			break;
		case SYN_PATH:
			err = NWDSGetAttrSize_PATH(ctx, &buf, &size);
			break;
		case SYN_CI_LIST:
			err = NWDSGetAttrSize_CI_LIST(ctx, &buf, &size);
			break;
		case SYN_TYPED_NAME:
			err = NWDSGetAttrSize_TYPED_NAME(ctx, &buf, &size);
			break;
		case SYN_BACK_LINK:
			err = NWDSGetAttrSize_BACK_LINK(ctx, &buf, &size);
			break;
		case SYN_FAX_NUMBER:
			err = NWDSGetAttrSize_FAX_NUMBER(ctx, &buf, &size);
			break;
		case SYN_EMAIL_ADDRESS:
			err = NWDSGetAttrSize_EMAIL_ADDRESS(ctx, &buf, &size);
			break;
		case SYN_PO_ADDRESS:
			err = NWDSGetAttrSize_PO_ADDRESS(ctx, &buf, &size);
			break;
		case SYN_OCTET_LIST:
			err = NWDSGetAttrSize_OCTET_LIST(ctx, &buf, &size);
			break;
		case SYN_HOLD:
			err = NWDSGetAttrSize_HOLD(ctx, &buf, &size);
			break;
		/* FIXME: other syntaxes... */
		default:
			err = ERR_NO_SUCH_SYNTAX;
			break;
	}
quit:;
	if (!err && valsize)
		*valsize = size;
	return err;
}

NWDSCCODE NWDSGetAttrVal(NWDSContextHandle ctx, Buf_T* buffer,
		enum SYNTAX syntaxID, void* ptr) {
	NWDSCCODE err;
	nuint32 tlen;
	void* p;
	Buf_T buf;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_INPUT)
		return ERR_BAD_VERB;
	switch (buffer->operation) {
		case DSV_READ_CLASS_DEF:;
			if (syntaxID != SYN_OBJECT_ACL)
				return ERR_BAD_VERB;
			p = NWDSBufRetrievePtrAndLen(buffer, &tlen);
			NWDSSetupBuf(&buf, p, tlen);
			err = NWDSGetAttrVal_OBJECT_ACL(ctx, &buf, ptr);
			if (err)
				return err;
			buffer->curPos = buf.curPos;
			return 0;
		default:;
			break;
	}
	err = NWDSBufPeekLE32(buffer, 0, &tlen);
	if (err)
		return err;
	p = NWDSBufPeekPtrLen(buffer, 4, tlen);
	if (!p)
		return ERR_BUFFER_EMPTY;
	if (ptr) {
		NWDSSetupBuf(&buf, p, tlen);
		switch (syntaxID) {
			case SYN_DIST_NAME:
				err = NWDSGetAttrVal_DIST_NAME2(ctx, p, tlen, ptr);
				break;
			case SYN_NET_ADDRESS:
				err = NWDSGetAttrVal_NET_ADDRESS(ctx, &buf, ptr);
				break;
			case SYN_CI_STRING:
			case SYN_CE_STRING:
			case SYN_NU_STRING:
			case SYN_PR_STRING:
			case SYN_CLASS_NAME:
			case SYN_TEL_NUMBER:
				err = NWDSGetAttrVal_XX_STRING2(ctx, p, tlen, ptr);
				break;
			case SYN_OCTET_STRING:
			case SYN_STREAM:
				err = NWDSGetAttrVal_OCTET_STRING(ctx, p, tlen, ptr);
				break;
			case SYN_BOOLEAN:
				err = NWDSGetAttrVal_BOOLEAN(ctx, p, tlen, ptr);
				break;
			case SYN_COUNTER:
			case SYN_INTEGER:
			case SYN_INTERVAL:
				err = NWDSGetAttrVal_INTEGER(ctx, p, tlen, ptr);
				break;
			case SYN_TIMESTAMP:
				err = NWDSGetAttrVal_TIMESTAMP(ctx, p, tlen, ptr);
				break;
			case SYN_REPLICA_POINTER:
				err = NWDSGetAttrVal_REPLICA_POINTER(ctx, &buf, ptr);
				break;
			case SYN_OBJECT_ACL:
				err = NWDSGetAttrVal_OBJECT_ACL(ctx, &buf, ptr);
				break;
			case SYN_PATH:
				err = NWDSGetAttrVal_PATH(ctx, &buf, ptr);
				break;
			case SYN_TIME:
				err = NWDSGetAttrVal_TIME(ctx, p, tlen, ptr);
				break;
			case SYN_CI_LIST:
				err = NWDSGetAttrVal_CI_LIST(ctx, &buf, ptr);
				break;
			case SYN_TYPED_NAME:
				err = NWDSGetAttrVal_TYPED_NAME(ctx, &buf, ptr);
				break;
			case SYN_BACK_LINK:
				err = NWDSGetAttrVal_BACK_LINK(ctx, &buf, ptr);
				break;
			case SYN_FAX_NUMBER:
				err = NWDSGetAttrVal_FAX_NUMBER(ctx, &buf, ptr);
				break;
			case SYN_EMAIL_ADDRESS:
				err = NWDSGetAttrVal_EMAIL_ADDRESS(ctx, &buf, ptr);
				break;
			case SYN_PO_ADDRESS:
				err = NWDSGetAttrVal_PO_ADDRESS(ctx, &buf, ptr);
				break;
			case SYN_OCTET_LIST:
				err = NWDSGetAttrVal_OCTET_LIST(ctx, &buf, ptr);
				break;
			case SYN_HOLD:
				err = NWDSGetAttrVal_HOLD(ctx, &buf, ptr);
				break;
			default:
				err = ERR_NO_SUCH_SYNTAX;
				break;
		}
	}
	if (err)
		return err;
	NWDSBufGetSkip(buffer, 4 + tlen);
	return 0;
}

NWDSCCODE NWDSGetAttrCount(NWDSContextHandle ctx, Buf_T* buffer,
		NWObjectCount* count) {
	NWDSCCODE err;
	nuint32 le32;

	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_INPUT)
		return ERR_BAD_VERB;
	err = NWDSBufGetLE32(buffer, &le32);
	if (err)
		return err;
	if (count)
		*count = le32;
	return 0;
}

NWDSCCODE NWDSGetObjectCount(NWDSContextHandle ctx, Buf_T* buffer,
		NWObjectCount* count) {
	NWDSCCODE err;
	nuint32 le32;

	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_INPUT)
		return ERR_BAD_VERB;
	switch (buffer->operation) {
		case DSV_LIST:
		case DSV_SEARCH:
			break;
		default:
			return ERR_BAD_VERB;
	}
	err = NWDSBufGetLE32(buffer, &le32);
	if (err)
		return err;
	if (count)
		*count = le32;
	return 0;	
}

#define CMDFLAGS_FLAGS		0x01
#define CMDFLAGS_STAMP		0x02
#define CMDFLAGS_VALUELEN	0x04
#define CMDFLAGS_VALUEDATA	0x08
#define CMDFLAGS_SYNTAXID	0x10
#define CMDFLAGS_VALCOUNT	0x20
NWDSCCODE NWDSBufSetInfoType(Buf_T* buffer, nuint32 infoType) {
#define STD	CMDFLAGS_SYNTAXID | CMDFLAGS_VALCOUNT | CMDFLAGS_VALUELEN
	static const nuint32 replytype[] = {	0,
				STD | CMDFLAGS_VALUEDATA,
				STD | CMDFLAGS_VALUEDATA,
				STD | CMDFLAGS_FLAGS | CMDFLAGS_STAMP | CMDFLAGS_VALUEDATA,
				STD | CMDFLAGS_FLAGS | CMDFLAGS_STAMP };
#undef STD

	if (infoType > 4)
		return ERR_INVALID_REQUEST;
	buffer->cmdFlags = replytype[infoType];
	return 0;		
}

NWDSCCODE NWDSGetObjectName(NWDSContextHandle ctx, Buf_T* buffer,
		NWDSChar* objectName, NWObjectCount* attrCount, 
		Object_Info_T* oi) {
	NWDSCCODE err;
	nuint32 le32;
	nuint32 dsiflags;
	
	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_INPUT)
		return ERR_BAD_VERB;
	switch (buffer->operation) {
		case DSV_LIST:
		case DSV_READ_ENTRY_INFO:
		case DSV_SEARCH:
			break;
		default:
			return ERR_BAD_VERB;
	}
	if (oi)
		memset(oi, 0, sizeof(*oi));
	dsiflags = buffer->dsiFlags;
	if (dsiflags & DSI_OUTPUT_FIELDS) {
		err = NWDSBufGetLE32(buffer, &dsiflags);
		if (err)
			return err;
	}
	if (dsiflags & DSI_ENTRY_ID)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_ENTRY_FLAGS) {
		err = NWDSBufGetLE32(buffer, &le32);
		if (err)
			return err;
		if (oi)
			oi->objectFlags = le32;
	}
	if (dsiflags & DSI_SUBORDINATE_COUNT) {
		err = NWDSBufGetLE32(buffer, &le32);
		if (err)
			return err;
		if (oi)
			oi->subordinateCount = le32;
	}
	if (dsiflags & DSI_MODIFICATION_TIME) {
		err = NWDSBufGetLE32(buffer, &le32);
		if (err)
			return err;
		if (oi)
			oi->modificationTime = le32;
	}
	if (dsiflags & DSI_MODIFICATION_TIMESTAMP)
		NWDSBufGetSkip(buffer, 8);
	if (dsiflags & DSI_CREATION_TIMESTAMP)
		NWDSBufGetSkip(buffer, 8);
	if (dsiflags & DSI_PARTITION_ROOT_ID)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_PARENT_ID)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_REVISION_COUNT)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_REPLICA_TYPE)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_BASE_CLASS) {
		err = NWDSBufCtxString(ctx, buffer, oi?oi->baseClass:NULL, MAX_SCHEMA_NAME_BYTES, NULL);
		if (err)
			return err;
	}
	if (dsiflags & DSI_ENTRY_RDN) {
		err = NWDSBufSkipCIString(buffer);
		if (err)
			return err;
	}
	if (dsiflags & DSI_ENTRY_DN) {
		err = NWDSBufCtxDN(ctx, buffer, objectName, NULL);
		if (err)
			return err;
	}
	if (dsiflags & DSI_PARTITION_ROOT_DN) {
		err = NWDSBufSkipCIString(buffer);
		if (err)
			return err;
	}
	if (dsiflags & DSI_PARENT_DN) {
		err = NWDSBufSkipCIString(buffer);
		if (err)
			return err;
	}
	if (dsiflags & DSI_PURGE_TIME)
		NWDSBufGetSkip(buffer, 4);
	/* DSI_DEREFERENCE_BASE_CLASSES */
	if (dsiflags & DSI_REPLICA_NUMBER)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_REPLICA_STATE)
		NWDSBufGetSkip(buffer, 4);
	if (buffer->operation == DSV_SEARCH) {
		err = NWDSBufGetLE32(buffer, &le32);
		if (err)
			return err;
		/* Set which fields are valid for THIS object.
		   Maybe it can differ from one object to another? */
		err = NWDSBufSetInfoType(buffer, le32);
		if (err)
			return err;
		err = NWDSBufGetLE32(buffer, &le32);
		if (err)
			return err;
	} else
		le32 = 0;
	if (attrCount)
		*attrCount = le32;
	return 0;
}

NWDSCCODE NWDSGetObjectNameAndInfo(
		NWDSContextHandle ctx, 
		Buf_T* buffer,
		NWDSChar* objectName, 
		NWObjectCount* attrCount, 
		char** info) {
	NWDSCCODE err;
	nuint32 dsiflags;
	nuint32 le32;
	
	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_INPUT)
		return ERR_BAD_VERB;
	switch (buffer->operation) {
		case DSV_LIST:
		case DSV_READ_ENTRY_INFO:
		case DSV_SEARCH:
			break;
		default:
			return ERR_BAD_VERB;
	}
	if (info)
		*info = buffer->curPos;
	dsiflags = buffer->dsiFlags;
	if (dsiflags & DSI_OUTPUT_FIELDS) {
		err = NWDSBufGetLE32(buffer, &dsiflags);
		if (err)
			return err;
	}
	if (dsiflags & DSI_ENTRY_ID)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_ENTRY_FLAGS)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_SUBORDINATE_COUNT)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_MODIFICATION_TIME)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_MODIFICATION_TIMESTAMP)
		NWDSBufGetSkip(buffer, 8);
	if (dsiflags & DSI_CREATION_TIMESTAMP)
		NWDSBufGetSkip(buffer, 8);
	if (dsiflags & DSI_PARTITION_ROOT_ID)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_PARENT_ID)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_REVISION_COUNT)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_REPLICA_TYPE)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_BASE_CLASS) {
		err = NWDSBufSkipCIString(buffer);
		if (err)
			return err;
	}
	if (dsiflags & DSI_ENTRY_RDN) {
		err = NWDSBufSkipCIString(buffer);
		if (err)
			return err;
	}
	if (dsiflags & DSI_ENTRY_DN) {
		if (objectName)
			err = NWDSBufCtxDN(ctx, buffer, objectName, NULL);
		else
			err = NWDSBufSkipCIString(buffer);
		if (err)
			return err;
	}
	if (dsiflags & DSI_PARTITION_ROOT_DN) {
		err = NWDSBufSkipCIString(buffer);
		if (err)
			return err;
	}
	if (dsiflags & DSI_PARENT_DN) {
		err = NWDSBufSkipCIString(buffer);
		if (err)
			return err;
	}
	if (dsiflags & DSI_PURGE_TIME)
		NWDSBufGetSkip(buffer, 4);
	/* DSI_DEREFERENCE_BASE_CLASSES */
	if (dsiflags & DSI_REPLICA_NUMBER)
		NWDSBufGetSkip(buffer, 4);
	if (dsiflags & DSI_REPLICA_STATE)
		NWDSBufGetSkip(buffer, 4);
	if (buffer->operation == DSV_SEARCH) {
		err = NWDSBufGetLE32(buffer, &le32);
		if (err)
			return err;
		err = NWDSBufSetInfoType(buffer, le32);
		if (err)
			return err;
		err = NWDSBufGetLE32(buffer, &le32);
		if (err)
			return err;
	} else
		le32 = 0;
	if (attrCount)
		*attrCount = le32;
	return 0;
}

NWDSCCODE NWDSGetDSIInfo(
		NWDSContextHandle ctx,
		void* buffer,
		size_t buflen,
		nuint32 dsiget,
		void* dst) {
	Buf_T buff;
	nuint32 le32;
	nuint32 dsiflags;
	NWDSCCODE err;
	
	if (dsiget & (dsiget - 1))
		return ERR_BAD_KEY;
	if (!buffer)
		return ERR_NULL_POINTER;
	NWDSSetupBuf(&buff, buffer, buflen);
	err = NWDSBufGetLE32(&buff, &dsiflags);
	if (err)
		return err;
	if ((dsiflags & dsiget) != dsiget)
		return ERR_BAD_KEY;
	if (!dst)
		return ERR_NULL_POINTER;
	if (dsiget & DSI_OUTPUT_FIELDS) {
		*(nuint32*)dst = dsiflags;
		return 0;
	}
	if (dsiget & DSI_ENTRY_ID) {
		err = NWDSBufGetID(&buff, dst);
		return err;
	}
	if (dsiflags & DSI_ENTRY_ID)
		NWDSBufGetSkip(&buff, 4);
		
	if (dsiget & DSI_ENTRY_FLAGS) {
		err = NWDSBufGetLE32(&buff, dst);
		return err;
	}
	if (dsiflags & DSI_ENTRY_FLAGS)
		NWDSBufGetSkip(&buff, 4);

	if (dsiget & DSI_SUBORDINATE_COUNT) {
		err = NWDSBufGetLE32(&buff, dst);
		return err;
	}
	if (dsiflags & DSI_SUBORDINATE_COUNT)
		NWDSBufGetSkip(&buff, 4);

	if (dsiget & DSI_MODIFICATION_TIME) {
		err = NWDSBufGetLE32(&buff, &le32);
		if (err)
			return err;
		*(time_t*)dst = le32;
		return 0;
	}
	if (dsiflags & DSI_MODIFICATION_TIME)
		NWDSBufGetSkip(&buff, 4);

	if (dsiget & DSI_MODIFICATION_TIMESTAMP) {
		void* p;
		
		p = NWDSBufGetPtr(&buff, 8);
		if (err)
			return err;
		((TimeStamp_T*)dst)->wholeSeconds = DVAL_LH(p, 0);
		((TimeStamp_T*)dst)->replicaNum = WVAL_LH(p, 4);
		((TimeStamp_T*)dst)->eventID = WVAL_LH(p, 6);
		return 0;
	}
	if (dsiflags & DSI_MODIFICATION_TIMESTAMP)
		NWDSBufGetSkip(&buff, 8);

	if (dsiget & DSI_CREATION_TIMESTAMP) {
		void* p;
		
		p = NWDSBufGetPtr(&buff, 8);
		if (err)
			return err;
		((TimeStamp_T*)dst)->wholeSeconds = DVAL_LH(p, 0);
		((TimeStamp_T*)dst)->replicaNum = WVAL_LH(p, 4);
		((TimeStamp_T*)dst)->eventID = WVAL_LH(p, 6);
		return 0;
	}
	if (dsiflags & DSI_CREATION_TIMESTAMP)
		NWDSBufGetSkip(&buff, 8);
	
	if (dsiget & DSI_PARTITION_ROOT_ID) {
		err = NWDSBufGetID(&buff, dst);
		return err;
	}
	if (dsiflags & DSI_PARTITION_ROOT_ID)
		NWDSBufGetSkip(&buff, 4);

	if (dsiget & DSI_PARENT_ID) {
		err = NWDSBufGetID(&buff, dst);
		return err;
	}
	if (dsiflags & DSI_PARENT_ID)
		NWDSBufGetSkip(&buff, 4);

	if (dsiget & DSI_REVISION_COUNT) {
		err = NWDSBufGetLE32(&buff, dst);
		return err;
	}
	if (dsiflags & DSI_REVISION_COUNT)
		NWDSBufGetSkip(&buff, 4);

	if (dsiget & DSI_REPLICA_TYPE) {
		err = NWDSBufGetLE32(&buff, dst);
		return err;
	}
	if (dsiflags & DSI_REPLICA_TYPE)
		NWDSBufGetSkip(&buff, 4);

	if (dsiget & DSI_BASE_CLASS) {
		err = NWDSBufCtxString(ctx, &buff, dst, MAX_SCHEMA_NAME_BYTES, NULL);
		return err;
	}
	if (dsiflags & DSI_BASE_CLASS) {
		err = NWDSBufSkipCIString(&buff);
		if (err)
			return err;
	}

	if (dsiget & DSI_ENTRY_RDN) {
		err = NWDSBufCtxString(ctx, &buff, dst, MAX_RDN_BYTES, NULL);
		return err;
	}
	if (dsiflags & DSI_ENTRY_RDN) {
		err = NWDSBufSkipCIString(&buff);
		if (err)
			return err;
	}
	
	if (dsiget & DSI_ENTRY_DN) {
		err = NWDSBufCtxDN(ctx, &buff, dst, NULL);
		return err;
	}
	if (dsiflags & DSI_ENTRY_DN) {
		err = NWDSBufSkipCIString(&buff);
		if (err)
			return err;
	}

	if (dsiget & DSI_PARTITION_ROOT_DN) {
		err = NWDSBufCtxDN(ctx, &buff, dst, NULL);
		return err;
	}
	if (dsiflags & DSI_PARTITION_ROOT_DN) {
		err = NWDSBufSkipCIString(buffer);
		if (err)
			return err;
	}

	if (dsiget & DSI_PARENT_DN) {
		err = NWDSBufCtxDN(ctx, &buff, dst, NULL);
		return err;
	}
	if (dsiflags & DSI_PARENT_DN) {
		err = NWDSBufSkipCIString(buffer);
		if (err)
			return err;
	}

	if (dsiget & DSI_PURGE_TIME) {
		err = NWDSBufGetLE32(&buff, &le32);
		if (err)
			return err;
		*(time_t*)dst = le32;
		return 0;
	}
	if (dsiflags & DSI_PURGE_TIME)
		NWDSBufGetSkip(buffer, 4);

	/* DSI_DEREFERENCE_BASE_CLASSES */

	if (dsiget & DSI_REPLICA_NUMBER) {
		err = NWDSBufGetLE32(&buff, &le32);
		return err;
	}
	if (dsiflags & DSI_REPLICA_NUMBER)
		NWDSBufGetSkip(buffer, 4);

	if (dsiget & DSI_REPLICA_STATE) {
		err = NWDSBufGetLE32(&buff, &le32);
		return err;
	}
	if (dsiflags & DSI_REPLICA_STATE)
		NWDSBufGetSkip(buffer, 4);
	return ERR_BUFFER_EMPTY; /* we did not find requested item... */
}

NWDSCCODE NWDSGetAttrName(NWDSContextHandle ctx, Buf_T* buffer,
		NWDSChar* name, NWObjectCount* valcount, 
		enum SYNTAX *syntaxID) {
	NWDSCCODE err;
	nuint32 le32;
	
	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_INPUT)
		return ERR_BAD_VERB;
	switch (buffer->operation) {
		case DSV_READ:
		case DSV_SEARCH:
			break;
		default:
			return ERR_BAD_VERB;
	}
	if (buffer->cmdFlags & CMDFLAGS_SYNTAXID) {
		err = NWDSBufGetLE32(buffer, &le32);
		if (err)
			return err;
	} else
		le32 = SYN_UNKNOWN;
	if (syntaxID)
		*syntaxID = le32;
	err = NWDSGetAttrVal_XX_STRING(ctx, buffer, name);
	if (err)
		return err;
	if (buffer->cmdFlags & CMDFLAGS_VALCOUNT) {
		err = NWDSBufGetLE32(buffer, &le32);
		if (err)
			return err;
	} else
		le32 = 0;
	if (valcount)
		*valcount = le32;
	return 0;
}

NWDSCCODE NWDSGetAttrValFlags(NWDSContextHandle ctx, Buf_T* buffer,
		nuint32* flags) {
	NWDSCCODE err;
	nuint32 le32;
	
	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_INPUT)
		return ERR_BAD_VERB;
	switch (buffer->operation) {
		case DSV_READ:
		case DSV_SEARCH:
			break;
		default:
			return ERR_BAD_VERB;
	}
	if (!(buffer->cmdFlags & CMDFLAGS_FLAGS))
		return ERR_BAD_VERB;
	err = NWDSBufGetLE32(buffer, &le32);
	if (err)
		return err;
	if (flags)
		*flags = le32;
	return 0;
}

NWDSCCODE NWDSGetAttrValModTime(NWDSContextHandle ctx, Buf_T* buffer,
		TimeStamp_T* stamp) {
	NWDSCCODE err;
	nuint32 le32;
	
	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_INPUT)
		return ERR_BAD_VERB;
	switch (buffer->operation) {
		case DSV_READ:
		case DSV_SEARCH:
			break;
		default:
			return ERR_BAD_VERB;
	}
	if (!(buffer->cmdFlags & CMDFLAGS_STAMP))
		return ERR_BAD_VERB;
	err = NWDSBufGetLE32(buffer, &le32);
	if (err)
		return err;
	if (stamp)
		stamp->wholeSeconds = le32;
	err = NWDSBufGetLE32(buffer, &le32);
	if (err)
		return err;
	if (stamp) {
		stamp->replicaNum = le32 & 0xFFFF;
		stamp->eventID = le32 >> 16;
	}
	return 0;
}

NWDSCCODE NWDSGetServerName(NWDSContextHandle ctx, Buf_T* buffer,
		NWDSChar* name, NWObjectCount* partcount) {
	NWDSCCODE err;
	nuint32 le32;
	
	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_INPUT)
		return ERR_BAD_VERB;
	switch (buffer->operation) {
		case DSV_LIST_PARTITIONS:
			break;
		default:
			return ERR_BAD_VERB;
	}
	err = NWDSBufCtxDN(ctx, buffer, name, NULL);
	if (err)
		return err;
	err = NWDSBufGetLE32(buffer, &le32);
	if (partcount)
		*partcount = le32;
	return 0;
}

static nuint32 tmp32;

NWDSCCODE NWDSInitBuf(NWDSContextHandle ctx, nuint operation,
		Buf_T* buffer) {
	NWDSCCODE err;

	NWDSBufStartPut(buffer, operation);
	buffer->bufFlags |= NWDSBUFT_INPUT;
	buffer->bufFlags &= ~NWDSBUFT_OUTPUT;
	switch (operation) {
		case DSV_READ:
		case DSV_SEARCH:
		case DSV_COMPARE:
		case DSV_ADD_ENTRY:
		case DSV_MODIFY_ENTRY:
		case DSV_READ_ATTR_DEF:
		case DSV_READ_CLASS_DEF:
		case DSV_MODIFY_CLASS_DEF:
			buffer->attrCountPtr = buffer->curPos;
			err = NWDSBufPutLE32(buffer, 0);
			if (err)
				return err;
			break;
		case DSV_SEARCH_FILTER:
			buffer->attrCountPtr = (nuint8*)&tmp32;
			break;
		case DSV_DEFINE_CLASS:
			break;
	}
	return 0;
}

NWDSCCODE NWDSPutAttrName(NWDSContextHandle ctx, Buf_T* buffer,
		const NWDSChar* name) {
	NWDSCCODE err;
	void* bpos;

	if (!buffer || !name)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_OUTPUT)
		return ERR_BAD_VERB;
	switch (buffer->operation) {
		case DSV_READ:
		case DSV_SEARCH:
		case DSV_COMPARE:
		case DSV_ADD_ENTRY:
		case DSV_SEARCH_FILTER:
			break;
		default:
			return ERR_BAD_VERB;
	}
	if (!buffer->attrCountPtr)
		return ERR_BAD_VERB;
	bpos = buffer->curPos;
	err = NWDSPutAttrVal_XX_STRING(ctx, buffer, name);
	if (err)
		return err;
	if ((buffer->operation == DSV_COMPARE) ||
	    (buffer->operation == DSV_ADD_ENTRY)) {
		void *vcp = buffer->curPos;
		err = NWDSBufPutLE32(buffer, 0);
		if (err) {
			buffer->curPos = bpos;
			return err;
		}
		buffer->valCountPtr = vcp;
	} else if (buffer->operation == DSV_SEARCH_FILTER) {
		buffer->valCountPtr = (nuint8*)&tmp32;
	} else
		buffer->valCountPtr = NULL;
	DSET_LH(buffer->attrCountPtr, 0, DVAL_LH(buffer->attrCountPtr, 0) + 1);
	return 0;
}

NWDSCCODE NWDSPutChange(NWDSContextHandle ctx, Buf_T* buffer,
		nuint changeType, const NWDSChar* name) {
	NWDSCCODE err;
	void* bpos;
	
	if (!buffer || !name)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_OUTPUT)
		return ERR_BAD_VERB;
	if ((buffer->operation != DSV_MODIFY_ENTRY))
		return ERR_BAD_VERB;
	if (!buffer->attrCountPtr)
		return ERR_BAD_VERB;
	bpos = buffer->curPos;
	err = NWDSBufPutLE32(buffer, changeType);
	if (err)
		goto errq;
	err = NWDSPutAttrVal_XX_STRING(ctx, buffer, name);
	if (err)
		goto errq;
	if ((changeType != DS_REMOVE_ATTRIBUTE) &&
	    (changeType != DS_CLEAR_ATTRIBUTE)) {
		void* vcp = buffer->curPos;
		err = NWDSBufPutLE32(buffer, 0);
		if (err)
			goto errq;
		buffer->valCountPtr = vcp;
	} else
		buffer->valCountPtr = NULL;
	DSET_LH(buffer->attrCountPtr, 0, DVAL_LH(buffer->attrCountPtr, 0) + 1);
	return 0;
errq:;
	buffer->curPos = bpos;
	return err;
}

NWDSCCODE NWDSPutAttrVal(NWDSContextHandle ctx, Buf_T* buffer,
		enum SYNTAX syntaxID, const void* attrVal) {
	NWDSCCODE err;
	
	if (!buffer || !attrVal)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_OUTPUT)
		return ERR_BAD_VERB;
	if (!buffer->valCountPtr)
		return ERR_BAD_VERB;
	switch (syntaxID) {
		case SYN_DIST_NAME:
			err = NWDSPutAttrVal_DIST_NAME(ctx, buffer, attrVal);
			break;
		case SYN_NET_ADDRESS:
			err = NWDSPutAttrVal_NET_ADDRESS(ctx, buffer, attrVal);
			break;
		case SYN_CI_STRING:
		case SYN_CE_STRING:
		case SYN_NU_STRING:
		case SYN_PR_STRING:
		case SYN_CLASS_NAME:
		case SYN_TEL_NUMBER:
			err = NWDSPutAttrVal_XX_STRING(ctx, buffer, attrVal);
			break;
		case SYN_OCTET_STRING:
		case SYN_STREAM:
			err = NWDSPutAttrVal_OCTET_STRING(ctx, buffer, attrVal);
			break;
		case SYN_BOOLEAN:
			err = NWDSPutAttrVal_BOOLEAN(ctx, buffer, attrVal);
			break;
		case SYN_COUNTER:
		case SYN_INTEGER:
		case SYN_INTERVAL:
			err = NWDSPutAttrVal_INTEGER(ctx, buffer, attrVal);
			break;
		case SYN_TIMESTAMP:
			err = NWDSPutAttrVal_TIMESTAMP(ctx, buffer, attrVal);
			break;
		case SYN_REPLICA_POINTER:
			err = NWDSPutAttrVal_REPLICA_POINTER(ctx, buffer, attrVal);
			break;
		case SYN_OBJECT_ACL:
			err = NWDSPutAttrVal_OBJECT_ACL(ctx, buffer, attrVal);
			break;
		case SYN_PATH:
			err = NWDSPutAttrVal_PATH(ctx, buffer, attrVal);
			break;
		case SYN_TIME:
			err = NWDSPutAttrVal_TIME(ctx, buffer, attrVal);
			break;
		case SYN_CI_LIST:
			err = NWDSPutAttrVal_CI_LIST(ctx, buffer, attrVal);
			break;
		case SYN_TYPED_NAME:
			err = NWDSPutAttrVal_TYPED_NAME(ctx, buffer, attrVal);
			break;
		case SYN_BACK_LINK:
			err = NWDSPutAttrVal_BACK_LINK(ctx, buffer, attrVal);
			break;
		case SYN_FAX_NUMBER:
			err = NWDSPutAttrVal_FAX_NUMBER(ctx, buffer, attrVal);
			break;
		case SYN_EMAIL_ADDRESS:
			err = NWDSPutAttrVal_EMAIL_ADDRESS(ctx, buffer, attrVal);
			break;
		case SYN_PO_ADDRESS:
			err = NWDSPutAttrVal_PO_ADDRESS(ctx, buffer, attrVal);
			break;
		case SYN_OCTET_LIST:
			err = NWDSPutAttrVal_OCTET_LIST(ctx, buffer, attrVal);
			break;
		case SYN_HOLD:
			err = NWDSPutAttrVal_HOLD(ctx, buffer, attrVal);
			break;
		default:
			err = ERR_NO_SUCH_SYNTAX;
			break;
	}
	if (err)
		return err;
	DSET_LH(buffer->valCountPtr, 0, DVAL_LH(buffer->valCountPtr, 0) + 1);
	return 0;
}

NWDSCCODE NWDSPutAttrNameAndVal(NWDSContextHandle ctx, Buf_T* buffer,
		const NWDSChar* name, enum SYNTAX syntaxID, 
		const void* attrVal) {
	NWDSCCODE err;
	void* curpos;
	void* vcp;
	nuint32 ac;
	
	if (!buffer)
		return ERR_NULL_POINTER;
	if (!buffer->attrCountPtr)
		return ERR_BAD_VERB;
	ac = DVAL_LH(buffer->attrCountPtr, 0);
	curpos = buffer->curPos;
	vcp = buffer->valCountPtr;
	err = NWDSPutAttrName(ctx, buffer, name);
	if (err)
		return err;
	err = NWDSPutAttrVal(ctx, buffer, syntaxID, attrVal);
	if (err) {
		buffer->curPos = curpos;
		buffer->valCountPtr = vcp;
		DSET_LH(buffer->attrCountPtr, 0, ac);
	}
	return err;
}

NWDSCCODE NWDSPutChangeAndVal(NWDSContextHandle ctx, Buf_T* buffer,
		nuint changeType, const NWDSChar* name,
		enum SYNTAX syntaxID, const void* attrVal) {
	NWDSCCODE err;
	void* curpos;
	void* vcp;
	nuint32 ac;
	
	if (!buffer)
		return ERR_NULL_POINTER;
	if (!buffer->attrCountPtr)
		return ERR_BAD_VERB;
	ac = DVAL_LH(buffer->attrCountPtr, 0);
	curpos = buffer->curPos;
	vcp = buffer->valCountPtr;
	err = NWDSPutChange(ctx, buffer, changeType, name);
	if (err)
		return err;
	err = NWDSPutAttrVal(ctx, buffer, syntaxID, attrVal);
	if (err) {
		buffer->curPos = curpos;
		buffer->valCountPtr = vcp;
		DSET_LH(buffer->attrCountPtr, 0, ac);
	}
	return err;
}

NWDSCCODE NWDSCanonicalizeName(NWDSContextHandle ctx, const NWDSChar* src,
		NWDSChar* dst) {
	wchar_t srcW[MAX_DN_CHARS+1];
	wchar_t dstW[MAX_DN_CHARS+1];
	NWDSCCODE err;
	
	err = NWDSXlateFromCtx(ctx, srcW, sizeof(srcW), src);
	if (err)
		return err;
	err = NWDSCanonicalizeNameW(ctx, srcW, dstW);
	if (err)
		return err;
	return NWDSXlateToCtx(ctx, dst, MAX_DN_BYTES, dstW, NULL);
}

NWDSCCODE NWDSAbbreviateName(NWDSContextHandle ctx, const NWDSChar* src,
		NWDSChar* dst) {
	wchar_t srcW[MAX_DN_CHARS+1];
	wchar_t dstW[MAX_DN_CHARS+1];
	NWDSCCODE err;
	
	err = NWDSXlateFromCtx(ctx, srcW, sizeof(srcW), src);
	if (err)
		return err;
	err = NWDSAbbreviateNameW(ctx, srcW, dstW);
	if (err)
		return err;
	return NWDSXlateToCtx(ctx, dst, MAX_DN_BYTES, dstW, NULL);
}

NWDSCCODE NWDSRemoveAllTypes(NWDSContextHandle ctx, const NWDSChar* src,
		NWDSChar* dst) {
	wchar_t srcW[MAX_DN_CHARS+1];
	wchar_t dstW[MAX_DN_CHARS+1];
	NWDSCCODE err;
	
	err = NWDSXlateFromCtx(ctx, srcW, sizeof(srcW), src);
	if (err)
		return err;
	err = NWDSRemoveAllTypesW(ctx, srcW, dstW);
	if (err)
		return err;
	return NWDSXlateToCtx(ctx, dst, MAX_DN_BYTES, dstW, NULL);
}

static NWDSCCODE __NWDSResolvePrep(
		NWDSContextHandle ctx,
		u_int32_t version,
		u_int32_t flag,
		const NWDSChar* name,
		Buf_T* rq,
		int wchar_name) {
	NWDSCCODE err;
	void* p;
	
	p = NWDSBufPutPtr(rq, 12);
	if (!p)
		return ERR_BUFFER_FULL;
	DSET_LH(p, 0, version);
	DSET_LH(p, 4, flag);
	DSET_LH(p, 8, ctx->dck.confidence);
	switch (wchar_name) {
		case 1:
			err = NWDSBufPutCIString(rq, (const wchar_t*)name);
			break;
		case 2:
			err = NWDSBufPutUnicodeString(rq, (const unicode*)name);
			break;
		default:
			err = NWDSPutAttrVal_DIST_NAME(ctx, rq, name);
			break;
	}
	if (err)
		return err;
	err = NWDSBufPutLE32(rq, ctx->dck.transports);
	if (err)
		return err;
	err = NWDSBufPut(rq, ctx->dck.transport_types, ctx->dck.transports * sizeof(nuint32));
	if (err)
		return err;
	err = NWDSBufPutLE32(rq, ctx->dck.transports);
	if (err)
		return err;
	err = NWDSBufPut(rq, ctx->dck.transport_types, ctx->dck.transports * sizeof(nuint32));
	if (err)
		return err;
	return 0;
}

static inline NWDSCCODE __NWDSResolveNameInt(
		NWDSContextHandle ctx, 
		NWCONN_HANDLE conn, 
		Buf_T* reply, 
		Buf_T* rq) {
	NWDSCCODE err;
	void* p;
	size_t rpl_len;

	NWDSBufStartPut(reply, DSV_RESOLVE_NAME);
	p = NWDSBufPutPtrLen(reply, &rpl_len);
	err = ncp_send_nds_frag(conn, DSV_RESOLVE_NAME, 
			rq->data, rq->curPos-rq->data, 
			p, rpl_len, &rpl_len);
	if (!err) {
		if (rpl_len < 8)
			err = ERR_INVALID_SERVER_RESPONSE;
		else {
			NWDSBufPutSkip(reply, rpl_len);
		}
	}
	NWDSBufFinishPut(reply);
	return err;
}

NWDSCCODE NWDSResolveNameInt(
		NWDSContextHandle ctx, 
		NWCONN_HANDLE conn, 
		u_int32_t version, 
		u_int32_t flag, 
		const NWDSChar* name, 
		Buf_T* reply) {
	NWDSCCODE err;
	Buf_T* rq;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	err = NWDSAllocBuf(DEFAULT_MESSAGE_LEN, &rq);
	if (err)
		return err;
	err = __NWDSResolvePrep(ctx, version, flag, name, rq, 0);	
	if (err)
		return err;
	err = __NWDSResolveNameInt(ctx, conn, reply, rq);
	NWDSFreeBuf(rq);
	return err;
}
		
/* server oriented NDS calls */

NWDSCCODE NWDSMapNameToID(NWDSContextHandle ctx, NWCONN_HANDLE conn,
		const NWDSChar* name, NWObjectID* ID) {
	NWDSCCODE err;
	Buf_T* rp;
	
	err = NWDSAllocBuf(DEFAULT_MESSAGE_LEN, &rp);
	if (err)
		return err;
	err = NWDSResolveNameInt(ctx, conn, DS_RESOLVE_V0, DS_RESOLVE_CREATE_ID, name, rp);
	if (!err) {
		nuint32 replytype;
		
		err = NWDSBufGetLE32(rp, &replytype);
		if (!err) {
			/* we requested local entry */
			if (replytype != DS_RESOLVE_REPLY_LOCAL_ENTRY)
				err = ERR_INVALID_SERVER_RESPONSE;
			else {
				err = NWDSBufGetID(rp, ID);
			}
		}
	}
	NWDSFreeBuf(rp);
	return err;
}

NWDSCCODE NWDSMapIDToName(NWDSContextHandle ctx, NWCONN_HANDLE conn,
		NWObjectID ID, NWDSChar* name) {
	NWDSCCODE err;
	Buf_T* b;
	nuint32 replyFmt = 0;
	nuint32 dsiFmt = DSI_ENTRY_DN;
	nuint32 val;
	
	err = NWDSGetContext(ctx, DCK_FLAGS, &val);
	if (err)
		return err;
	if (val & DCV_TYPELESS_NAMES)
		replyFmt |= 1;
	if (val & DCV_DEREF_BASE_CLASS)
		dsiFmt |= DSI_DEREFERENCE_BASE_CLASS;
	/* DCV_DEREF_ALIASES does not have DSI_ counterpart */
	/* DCV_DISALLOW_REFERRALS is N/A for MapIDToName (server centric) */
	/* DCV_ASYNC_MODE: TODO (if someone knows...) */
	replyFmt |= ctx->dck.name_form;
	err = NWDSAllocBuf(MAX_DN_BYTES, &b);
	if (err)
		return err;
	err = NWDSGetDSIRaw(conn, dsiFmt, replyFmt, ID, b);
	if (!err) {
		/* Netware DS 6.02, 7.26, 7.28 and 7.30 (all I tested)
		 * has problem that [Self] maps to 0x040000FF, but
		 * 0x040000FF maps to [Self][Inheritance Mask]. Do you
		 * think that I should add hack to map [Self][Inheritance Mask]
		 * here? Or is it meant in another way and I oversight
		 * something? */
		err = NWDSGetAttrVal_DIST_NAME(ctx, b, name);
	}
	NWDSFreeBuf(b);
	return err;
}

static NWDSCCODE NWDSResolveName99(
		NWDSContextHandle ctx, 
		NWCONN_HANDLE conn,
		Buf_T* rq,
		NWCONN_HANDLE* resconn, 
		NWObjectID* ID, 
		Buf_T** rplbuf, 
		void*** array, 
		unsigned int* saddr, 
		int* idepth) {
	NWDSCCODE err;
	Buf_T* rp;
	
	*rplbuf = NULL;
	*array = NULL;
	*saddr = 0;
	*resconn = NULL;
	err = NWDSAllocBuf(DEFAULT_MESSAGE_LEN, &rp);
	if (err)
		return err;
	err = __NWDSResolveNameInt(ctx, conn, rp, rq);
	if (!err) {
		nuint32 replytype;
		nuint32 dummy;
			
		err = NWDSBufGetLE32(rp, &replytype);
		if (!err) switch (replytype) {
			case DS_RESOLVE_REPLY_LOCAL_ENTRY:
				dfprintf(stderr, "Local Entry\n");
				err = NWDSBufGetID(rp, ID);
				if (err)
					break;
				ncp_conn_use(conn);
				*resconn = conn;
				break;
			case DS_RESOLVE_REPLY_REMOTE_ENTRY:
				dfprintf(stderr, "Remote Entry\n");
				err = NWDSBufGetID(rp, ID);
				if (err)
					break;
				err = NWDSBufGetLE32(rp, &dummy);
				if (err)
					break;
				err = NWDSFindConnection2(ctx, resconn, rp, NWDSFINDCONN_CREATEALLOWED);
				break;
			case DS_RESOLVE_REPLY_REFERRAL_AND_ENTRY:
			case DS_RESOLVE_REPLY_REFERRAL:
				{
					u_int32_t depth;
					u_int32_t servercount;
					u_int32_t parsedaddresses;
					void** addresses;
					unsigned int pos;
							
					if (replytype == DS_RESOLVE_REPLY_REFERRAL) {
						dfprintf(stderr, "Referrals\n");
						err = NWDSBufGetLE32(rp, &depth);
						if (err)
							break;
					} else {
						u_int32_t dummy;
						NWObjectID objid;
								
						dfprintf(stderr, "Referrals + ID\n");
						err = NWDSBufGetLE32(rp, &dummy);
						if (err)
							break;
						err = NWDSBufGetID(rp, &objid);
						if (err)
							break;
						if (objid != (NWObjectID)~0) {
							*ID = objid;
							ncp_conn_use(conn);
							*resconn = conn;
							dfprintf(stderr, "Got ID\n");
							break;
						}
						depth = 0xFFFF0000; /* ignore all referrals */
					}
					err = NWDSBufGetLE32(rp, &servercount);
					if (err)
						break;
					if (servercount > 1024) { /* some unreasonable value */
						/* return ERR_TOO_MANY_REFERRALS; ??? */
						servercount = 1024;
					}
					if (!servercount) {
						err = ERR_NO_REFERRALS;
						break;
					}
					dfprintf(stderr, "%d servers\n", servercount);
					addresses = (void**)malloc(sizeof(void*) * servercount);
					if (!addresses) {
						err = ERR_NOT_ENOUGH_MEMORY;
						break;
					}
					parsedaddresses = 0;
					for (pos = 0; pos < servercount; pos++) {
						u_int32_t saddr;
						u_int32_t dummy;
						u_int32_t len;
						void* data;
						u_int32_t i;
									
						addresses[parsedaddresses] = NWDSBufPeekPtr(rp);
						err = NWDSBufGetLE32(rp, &saddr);
						if (err)
							break;
						for (i = 0; i < saddr; i++) {
							err = NWDSBufGetLE32(rp, &dummy);
							if (err)
								goto packetEOF;
							err = NWDSBufGetLE32(rp, &len);
							if (err)
								goto packetEOF;
							data = NWDSBufGetPtr(rp, len);
							if (!data) {
								err = ERR_BUFFER_EMPTY;
								goto packetEOF;
							}
						}
						parsedaddresses++;
					}
				packetEOF:;
					if (err) {
						free(addresses);
						break;
					}
					if (!parsedaddresses) {
						free(addresses);
						err = ERR_NO_REFERRALS;
						break;
					}
					*rplbuf = rp;
					*array = addresses;
					*saddr = parsedaddresses;
					*idepth = depth;
				}
				return 0;
			default:
				err = ERR_INVALID_SERVER_RESPONSE;
				dfprintf(stderr, "Invalid server response\n");
				break;
		}
	}
	NWDSFreeBuf(rp);
	return err;
}

static NWDSCCODE __NWDSResolveName2(
		NWDSContextHandle ctx, 
		Buf_T* rq,
		NWCONN_HANDLE* resconn, 
		NWObjectID* ID) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
	Buf_T* rp;
	void** addresses;
	unsigned int servers;
	int depth;
		
	err = __NWDSGetConnection(ctx, &conn);
	if (err)
		return err;
	err = NWDSResolveName99(ctx, conn, rq, resconn, ID, &rp, &addresses, &servers, &depth);
	if (!err && rp) {
		unsigned int pos;
		unsigned int crmode;
		NWDSCCODE dserr = ERR_ALL_REFERRALS_FAILED;
again:;
		dfprintf(stderr, "Received referrals\n");
		crmode = NWDSFINDCONN_NOCREATE;
again2:;
		for (pos = 0; pos < servers; pos++) {
			NWCONN_HANDLE conn2;
			Buf_T* buf;
			Buf_T* b2;
			void** addr2;
			unsigned int serv2;
			int depth2;
									
			if (!addresses[pos])
				continue;
			err = NWDSCreateBuf(&buf, addresses[pos], 0x7FFFFFF);
			if (err)
				continue;
			err = NWDSFindConnection2(ctx, &conn2, buf, crmode);
			NWDSFreeBuf(buf);
			if (err)
				continue;
			dfprintf(stderr, "Processing referrals\n");
			addresses[pos] = NULL;
			err = NWDSResolveName99(ctx, conn2, rq, resconn, ID, &b2, &addr2, &serv2, &depth2);
			if (!err) {
				if (!b2) {
					NWCCCloseConn(conn2);
					goto freeExit;
				}
				if (depth2 < depth) {
					free(addresses);
					addresses = addr2;
					NWDSFreeBuf(rp);
					rp = b2;
					depth = depth2;
					servers = serv2;
					NWCCCloseConn(conn);
					conn = conn2;
					goto again;
				}
				dfprintf(stderr, "Referral ignored; %d >= %d\n", depth2, depth);
				free(addr2);
				NWDSFreeBuf(b2);
				err = dserr;
			} else
				dserr = err;
			NWCCCloseConn(conn2);
		}
		if (crmode == NWDSFINDCONN_NOCREATE) {
			dfprintf(stderr, "Connection not found, making new\n");
			crmode = NWDSFINDCONN_CREATEALLOWED;
			goto again2;
		}
freeExit:;
		free(addresses);
		NWDSFreeBuf(rp);
	}
	NWDSConnectionFinished(ctx, conn);
	return err;
}

NWDSCCODE NWDSResolveName2DR(
		NWDSContextHandle ctx,
		const NWDSChar* name,
		u_int32_t flag,
		NWCONN_HANDLE* resconn,
		NWObjectID* ID) {
	NWDSCCODE err;
	char rq_b[DEFAULT_MESSAGE_LEN];
	Buf_T rq;
		
	NWDSSetupBuf(&rq, rq_b, sizeof(rq_b));
	err = __NWDSResolvePrep(ctx, DS_RESOLVE_V0, flag, name, &rq, 0);
	if (err)
		return err;
	return __NWDSResolveName2(ctx, &rq, resconn, ID);
}

NWDSCCODE __NWDSResolveName2u(
		NWDSContextHandle ctx,
		const unicode* name,
		u_int32_t flag,
		NWCONN_HANDLE* resconn,
		NWObjectID* ID) {
	NWDSCCODE err;
	char rq_b[DEFAULT_MESSAGE_LEN];
	Buf_T rq;

	flag &= ~(DS_RESOLVE_DEREF_ALIASES);	
	if (ctx->dck.flags & DCV_DEREF_ALIASES)
		flag |= DS_RESOLVE_DEREF_ALIASES;
	NWDSSetupBuf(&rq, rq_b, sizeof(rq_b));
	err = __NWDSResolvePrep(ctx, DS_RESOLVE_V0, flag, (const NWDSChar*)name, &rq, 2);
	if (err)
		return err;
	return __NWDSResolveName2(ctx, &rq, resconn, ID);
}

NWDSCCODE __NWDSResolveName2w(
		NWDSContextHandle ctx,
		const wchar_t* name,
		u_int32_t flag,
		NWCONN_HANDLE* resconn,
		NWObjectID* ID
) {
	NWDSCCODE err;
	char rq_b[DEFAULT_MESSAGE_LEN];
	Buf_T rq;

	flag &= ~(DS_RESOLVE_DEREF_ALIASES);	
	if (ctx->dck.flags & DCV_DEREF_ALIASES)
		flag |= DS_RESOLVE_DEREF_ALIASES;
	NWDSSetupBuf(&rq, rq_b, sizeof(rq_b));
	err = __NWDSResolvePrep(ctx, DS_RESOLVE_V0, flag, (const NWDSChar*)name, &rq, 1);
	if (err)
		return err;
	return __NWDSResolveName2(ctx, &rq, resconn, ID);
}

NWDSCCODE NWDSResolveName2(
		NWDSContextHandle ctx, 
		const NWDSChar* name, 
		u_int32_t flag,
		NWCONN_HANDLE* resconn, 
		NWObjectID* ID) {
	flag &= ~(DS_RESOLVE_DEREF_ALIASES);
	if (ctx->dck.flags & DCV_DEREF_ALIASES)
		flag |= DS_RESOLVE_DEREF_ALIASES;
	return NWDSResolveName2DR(ctx, name, flag, resconn, ID);
}

NWDSCCODE __NWDSResolveName2p(
		NWDSContextHandle ctx,
		const NWDSChar* name,
		u_int32_t flag,
		NWCONN_HANDLE* resconn,
		NWObjectID* ID,
		wchar_t* childName) {
	NWDSCCODE err;
	wchar_t parentName[MAX_DN_CHARS+1];
	char rq_b[DEFAULT_MESSAGE_LEN];
	Buf_T rq;
	
	err = NWDSSplitName(ctx, name, parentName, childName);
	if (err)
		return err;
	if (ctx->dck.flags & DCV_DEREF_ALIASES)
		flag |= DS_RESOLVE_DEREF_ALIASES;
	NWDSSetupBuf(&rq, rq_b, sizeof(rq_b));
	err = __NWDSResolvePrep(ctx, DS_RESOLVE_V0, flag, (NWDSChar*)parentName, &rq, 1);
	if (err)
		return err;
	return __NWDSResolveName2(ctx, &rq, resconn, ID);
}

NWDSCCODE NWDSResolveName(NWDSContextHandle ctx, const NWDSChar* name,
		NWCONN_HANDLE* resconn, NWObjectID* ID) {
	return NWDSResolveName2(ctx, name, DS_RESOLVE_WRITEABLE,
		resconn, ID);
}

static NWDSCCODE NWDSGetServerNameAddress(NWCONN_HANDLE conn, u_int32_t version,
	u_int32_t nameform, Buf_T* reply) {
	NWDSCCODE err;
	unsigned char rq[8];
	size_t rpl_len;
	void* rpl;
	
	DSET_LH(rq, 0, version);
	DSET_LH(rq, 4, nameform);
	NWDSBufStartPut(reply, DSV_GET_SERVER_ADDRESS);
	rpl = NWDSBufPutPtrLen(reply, &rpl_len);
	err = ncp_send_nds_frag(conn, DSV_GET_SERVER_ADDRESS, rq, sizeof(rq), rpl,
		rpl_len, &rpl_len);
	if (!err) {
		if (rpl_len < 8) {
			/* at least 0 bytes name and 0 network addresses */
			err = ERR_INVALID_SERVER_RESPONSE;
		} else {
			NWDSBufPutSkip(reply, rpl_len);
			NWDSBufFinishPut(reply);
		}
	}
	return err;
}

/* NWSDK returns abbreviated typeless name of server ... */
NWDSCCODE NWDSGetServerDN(NWDSContextHandle ctx, NWCONN_HANDLE conn, NWDSChar* name) {
	NWDSCCODE err;
	Buf_T* reply;
	u_int32_t replyFmt;
	
	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	err = NWDSAllocBuf(DEFAULT_MESSAGE_LEN, &reply);
	if (err)
		return err;
	replyFmt = ctx->dck.name_form;
	if (ctx->dck.flags & DCV_TYPELESS_NAMES)
		replyFmt |= 1;
	/* COMPATIBILITY
	replyFmt |= 1;
	*/
	err = NWDSGetServerNameAddress(conn, 0, replyFmt, reply);
	if (!err) {
		err = NWDSGetAttrVal_DIST_NAME(ctx, reply, name);
		/* COMPATIBILITY
		if (!err)
			err = NWDSAbbreviateName(ctx, name, name);
		*/
	}
	NWDSFreeBuf(reply);
	return err;
}

static NWDSCCODE NWDSGetServerAddressInt(NWCONN_HANDLE conn, 
		NWObjectCount* addrcnt,
		Buf_T* naddrs) {
	NWDSCCODE err;
	Buf_T* reply;
	
	err = NWDSAllocBuf(DEFAULT_MESSAGE_LEN, &reply);
	if (err)
		return err;
	/* version: 0, format: partial dot, typeless */
	err = NWDSGetServerNameAddress(conn, 0, 1, reply);
	if (!err) {
		err = NWDSBufSkipCIString(reply);
		if (!err) {
			nuint32 cnt;
			
			err = NWDSBufGetLE32(reply, &cnt);
			if (!err) {
				if (addrcnt)
					*addrcnt = cnt;
				if (naddrs) {
					NWDSBufStartPut(naddrs, DSV_GET_SERVER_ADDRESS);
					/* copy addresses... maybe we can 
					   copy whole rest of buffer
					   as unstructured data block */
					while (cnt--) {
						nuint32 val;
						nuint32 size;
						void* ptr;

						/* address type */
						err = NWDSBufGetLE32(reply, &val);
						if (err) break;
						/* address length */
						err = NWDSBufGetLE32(reply, &size);
						if (err) break;

						err = NWDSBufPutLE32(naddrs, size + 8);
						if (err) break;
						err = NWDSBufPutLE32(naddrs, val);
						if (err) break;
						err = NWDSBufPutLE32(naddrs, size);
						if (err) break;

						/* address value */
						ptr = NWDSBufGetPtr(reply, size);
						if (!ptr) {
							err = ERR_BUFFER_EMPTY;
							break;
						}
						err = NWDSBufPut(naddrs, ptr, size);
						if (err)
							break;
					}
					NWDSBufFinishPut(naddrs);
				}
			}
		}
	}
	NWDSFreeBuf(reply);
	return err;
}

NWDSCCODE NWDSGetServerAddress(NWDSContextHandle ctx, NWCONN_HANDLE conn, 
		NWObjectCount* addrcnt,
		Buf_T* naddrs) {
	NWDSCCODE err;

	err = NWDSIsContextValid(ctx);
	if (err)
		return err;
	return NWDSGetServerAddressInt(conn, addrcnt, naddrs);
}

/* difference? */
NWDSCCODE NWDSGetServerAddress2(NWDSContextHandle ctx, NWCONN_HANDLE conn, 
		NWObjectCount* addrcnt,
		Buf_T* naddrs) {
	return NWDSGetServerAddress(ctx, conn, addrcnt, naddrs);
}

static NWDSCCODE __NWCCGetServerAddressPtr(NWCONN_HANDLE conn, 
		NWObjectCount* count, nuint8** data) {
	if (!conn->serverAddresses.buffer) {
		NWDSCCODE err;
		Buf_T* bb;
		NWObjectCount cnt;
		
		err = NWDSAllocBuf(DEFAULT_MESSAGE_LEN, &bb);
		if (err)
			return err;
		err = NWDSGetServerAddressInt(conn, &cnt, bb);
		if (err)
			return err;
		/* TODO: lock connection */
		if (conn->serverAddresses.buffer) {
			NWDSFreeBuf(bb);
		} else {
			conn->serverAddresses.buffer = bb;
			conn->serverAddresses.count = cnt;
		}
		/* unlock connection */
	}
	*count = conn->serverAddresses.count;
	*data = NWDSBufPeekPtr(conn->serverAddresses.buffer);
	return 0;
}

NWDSCCODE __NWDSCompare(
		NWDSContextHandle ctx,
		NWCONN_HANDLE conn,
		NWObjectID objectID,
		Buf_T* buffer,
		nbool8* matched) {
	NWDSCCODE err;
	unsigned char rq[8192];
	size_t rpl_len;
	unsigned char rpl[4];
	size_t pos;
	nuint32 ctxflags;
	
	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->operation != DSV_COMPARE)
		return ERR_BAD_VERB;
	err = NWDSGetContext(ctx, DCK_FLAGS, &ctxflags);
	if (err)
		return err;
	DSET_LH(rq, 0, 0); 		/* version */
	DSET_HL(rq, 4, objectID);
	pos = 8;
	{
		size_t len = buffer->curPos - buffer->data;
		memcpy(rq+pos, buffer->data, len);
		pos += ROUNDPKT(len);
	}
	err = ncp_send_nds_frag(conn, DSV_COMPARE, rq, pos, rpl, sizeof(rpl), &rpl_len);
	if (!err) {
		if (matched) {
			*matched = BVAL(rpl, 0) != 0;
		}
	}
	return err;
}

NWDSCCODE NWDSCompare(
		NWDSContextHandle ctx,
		const NWDSChar* objectName,
		Buf_T* buffer,
		nbool8* matched) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
	NWObjectID objID;
	
	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->operation != DSV_COMPARE)
		return ERR_BAD_VERB;
	err = NWDSResolveName2(ctx, objectName, DS_RESOLVE_READABLE | DS_RESOLVE_WALK_TREE, 
		&conn, &objID);
	if (err)
		return err;
	err = __NWDSCompare(ctx, conn, objID, buffer, matched);
	NWCCCloseConn(conn);
	return err;
}

/* qflags:
   0x00000001 -> return typeless names
   0x00000002 -> return containers only
   0x00000004 \  return name format
   0x00000008 /
   0x00000010
   0x00000020
   0x00000040 -> deref base class
*/
NWDSCCODE __NWDSReadObjectDSIInfo(
		NWDSContextHandle ctx,
		NWCONN_HANDLE conn,
		NWObjectID id,
		nuint32 dsiFmt,
		Buf_T* b) {
	NWDSCCODE err;
	nuint32 replyFmt = 0;
	nuint32 val;
	
	err = NWDSGetContext(ctx, DCK_FLAGS, &val);
	if (err)
		return err;
	if (val & DCV_TYPELESS_NAMES)
		replyFmt |= 1;
	if (val & DCV_DEREF_BASE_CLASS)
		dsiFmt |= DSI_DEREFERENCE_BASE_CLASS;
	replyFmt |= ctx->dck.name_form;
	return NWDSGetDSIRaw(conn, dsiFmt, replyFmt, id, b);
}

NWDSCCODE __NWDSReadObjectInfo(
		NWDSContextHandle ctx,
		NWCONN_HANDLE conn,
		NWObjectID id,
		NWDSChar* distname,
		Object_Info_T* oi) {
	NWDSCCODE err;
	char buffer[8192];
	Buf_T b;
	
	NWDSSetupBuf(&b, buffer, sizeof(buffer));
	err = __NWDSReadObjectDSIInfo(ctx, conn, id, DSI_OUTPUT_FIELDS | 
		DSI_ENTRY_FLAGS | DSI_SUBORDINATE_COUNT | 
		DSI_MODIFICATION_TIME | DSI_BASE_CLASS | DSI_ENTRY_DN, &b);
	if (err)
		return err;
	err = NWDSGetObjectName(ctx, &b, distname, NULL, oi);
	return err;
}

NWDSCCODE NWDSReadObjectInfo(NWDSContextHandle ctx, const NWDSChar* name,
		NWDSChar* distname, Object_Info_T* oi) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
	NWObjectID objID;
	
	err = NWDSResolveName2(ctx, name, DS_RESOLVE_READABLE | DS_RESOLVE_WALK_TREE,
		&conn, &objID);
	if (err)
		return err;
	err = __NWDSReadObjectInfo(ctx, conn, objID, distname, oi);
	NWCCCloseConn(conn);
	return err;
}

NWDSCCODE NWDSReadObjectDSIInfo(NWDSContextHandle ctx, const NWDSChar* name,
		size_t len, void* buffer) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
	NWObjectID objID;
	Buf_T b;
	
	if (!buffer)
		return ERR_NULL_POINTER;
	err = NWDSResolveName2(ctx, name, DS_RESOLVE_READABLE | DS_RESOLVE_WALK_TREE,
		&conn, &objID);
	if (err)
		return err;
	NWDSSetupBuf(&b, buffer, len);
	err = __NWDSReadObjectDSIInfo(ctx, conn, objID, ctx->dck.dsi_flags, &b);
	NWCCCloseConn(conn);
	return err;
}

NWDSCCODE __NWDSCloseIterationV0(
		NWCONN_HANDLE	conn,
		nuint32		iterHandle,
		nuint32		verb) {
	unsigned char rq[12];
	size_t rpl_len;
	unsigned char rpl[512];
	
	DSET_LH(rq, 0, 0);		/* version */
	DSET_LH(rq, 4, iterHandle);	/* iterHandle */
	DSET_LH(rq, 8, verb);		/* verb */
	return ncp_send_nds_frag(conn, DSV_CLOSE_ITERATION, rq, sizeof(rq), rpl, sizeof(rpl), &rpl_len);
}

NWDSCCODE NWDSCloseIteration(NWDSContextHandle ctx, nuint32 iterHandle,
		nuint32 verb) {
	NWCONN_HANDLE conn;
	NWDSCCODE err;
	struct wrappedIterationHandle* ih;
	
	switch (verb) {
		case DSV_ADD_ENTRY:
			err = NWDSGetContext(ctx, DCK_LAST_CONNECTION, &conn);
			if (err)
				return err;
			return __NWDSCloseIterationV0(conn, iterHandle, verb);
	}
	ih = __NWDSIHLookup(iterHandle, verb);
	if (!ih)
		return ERR_INVALID_HANDLE;
	err = __NWDSIHDelete(ih);
	free(ih);
	return err;
}

static NWDSCCODE __NWDSAddObjectV0(
		NWCONN_HANDLE conn,
		nuint32 flags,
		NWObjectID objID,
		const wchar_t* childName,
		Buf_T* buffer) {
	NWDSCCODE err;
	char rq_b[DEFAULT_MESSAGE_LEN];
	Buf_T rq;
	void* p;
	size_t l;
	char rp_b[16];
	
	NWDSSetupBuf(&rq, rq_b, sizeof(rq_b));
	NWDSBufPutPtr(&rq, 12);		
	DSET_LH(rq_b, 0, 0); 		/* version */
	DSET_LH(rq_b, 4, flags);	/* flags */
	DSET_HL(rq_b, 8, objID);	/* parentID */
	err = NWDSBufPutCIString(&rq, childName);
	if (err)
		return err;
	p = NWDSBufRetrieve(buffer, &l);
	err = NWDSBufPut(&rq, p, l);
	if (err)
		return err;
	err = ncp_send_nds_frag(conn, DSV_ADD_ENTRY, rq.data, rq.curPos - rq.data,
		rp_b, sizeof(rp_b), &l);
	return err;
}

static NWDSCCODE __NWDSAddObjectV2(
		NWCONN_HANDLE conn,
		nuint32 flags,
		nuint32* iterHandle,
		NWObjectID objID,
		const wchar_t* childName,
		Buf_T* buffer) {
	NWDSCCODE err;
	char rq_b[DEFAULT_MESSAGE_LEN];
	Buf_T rq;
	void* p;
	size_t l;
	char rp_b[16];
	
	NWDSSetupBuf(&rq, rq_b, sizeof(rq_b));
	NWDSBufPutPtr(&rq, 16);		
	DSET_LH(rq_b, 0, 2); 		/* version */
	DSET_LH(rq_b, 4, flags);	/* flags */
	DSET_LH(rq_b, 8, iterHandle?*iterHandle:-1);
	DSET_HL(rq_b, 12, objID);	/* parentID */
	err = NWDSBufPutCIString(&rq, childName);
	if (err)
		return err;
	p = NWDSBufRetrieve(buffer, &l);
	err = NWDSBufPut(&rq, p, l);
	if (err)
		return err;
	err = ncp_send_nds_frag(conn, DSV_ADD_ENTRY, rq.data, rq.curPos - rq.data,
		rp_b, sizeof(rp_b), &l);
	if (err)
		return err;
	if (l < 4) {
		if (iterHandle)
			*iterHandle = NO_MORE_ITERATIONS;
	} else {
		if (iterHandle)
			*iterHandle = DVAL_LH(rp_b, 0);
	}
	return 0;
}

NWDSCCODE NWDSAddObject(
		NWDSContextHandle ctx, 
		const NWDSChar* objectName,
		nuint32* iterHandle, 
		nbool8 moreIter, 
		Buf_T* buffer) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
	NWObjectID objID;
	wchar_t childName[MAX_DN_CHARS+1];
	
	if (moreIter && !iterHandle)
		return ERR_NULL_POINTER;
	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_OUTPUT)
		return ERR_BAD_VERB;
	if (buffer->operation != DSV_ADD_ENTRY)
		return ERR_BAD_VERB;
	err = __NWDSResolveName2p(ctx, objectName, DS_RESOLVE_WRITEABLE | DS_RESOLVE_WALK_TREE,
		&conn, &objID, childName);
	if (err)
		return err;
	err = __NWDSAddObjectV2(conn, moreIter?1:0, iterHandle, objID, childName, buffer);
	if ((err == ERR_INVALID_API_VERSION) && !moreIter && (!iterHandle || *iterHandle == NO_MORE_ITERATIONS))
		err = __NWDSAddObjectV0(conn, 0, objID, childName, buffer);
	NWCCCloseConn(conn);
	return err;
}

static NWDSCCODE __NWDSRemoveObjectV0(
		NWCONN_HANDLE conn,
		NWObjectID objID) {
	NWDSCCODE err;
	char rq_b[8];
	size_t l;
	char rp_b[16];
	
	DSET_LH(rq_b, 0, 0); 		/* version */
	DSET_HL(rq_b, 4, objID);	/* object ID */
	err = ncp_send_nds_frag(conn, DSV_REMOVE_ENTRY, rq_b, sizeof(rq_b),
		rp_b, sizeof(rp_b), &l);
	return err;
}

NWDSCCODE NWDSRemoveObject(
		NWDSContextHandle ctx,
		const NWDSChar* objectName) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
	NWObjectID objID;
	
	err = NWDSResolveName2DR(ctx, objectName, DS_RESOLVE_WRITEABLE | DS_RESOLVE_WALK_TREE,
		&conn, &objID);
	if (err)
		return err;
	err = __NWDSRemoveObjectV0(conn, objID);
	NWCCCloseConn(conn);
	return err;
}

static NWDSCCODE __NWDSModifyObjectV0(
		NWCONN_HANDLE conn,
		nuint32 flags,
		NWObjectID objID,
		Buf_T* buffer) {
	NWDSCCODE err;
	char rq_b[DEFAULT_MESSAGE_LEN];
	Buf_T rq;
	void* p;
	size_t l;
	char rp_b[16];
	
	NWDSSetupBuf(&rq, rq_b, sizeof(rq_b));
	NWDSBufPutPtr(&rq, 12);		
	DSET_LH(rq_b, 0, 0); 		/* version */
	DSET_LH(rq_b, 4, flags);	/* flags */
	DSET_HL(rq_b, 8, objID);	/* parentID */
	p = NWDSBufRetrieve(buffer, &l);
	err = NWDSBufPut(&rq, p, l);
	if (err)
		return err;
	err = ncp_send_nds_frag(conn, DSV_MODIFY_ENTRY, rq.data, rq.curPos - rq.data,
		rp_b, sizeof(rp_b), &l);
	return err;
}

static NWDSCCODE __NWDSModifyObjectV2(
		NWCONN_HANDLE conn,
		nuint32 flags,
		nuint32* iterHandle,
		NWObjectID objID,
		Buf_T* buffer) {
	NWDSCCODE err;
	char rq_b[DEFAULT_MESSAGE_LEN];
	Buf_T rq;
	void* p;
	size_t l;
	char rp_b[16];
	
	NWDSSetupBuf(&rq, rq_b, sizeof(rq_b));
	NWDSBufPutPtr(&rq, 16);		
	DSET_LH(rq_b, 0, 2); 		/* version */
	DSET_LH(rq_b, 4, flags);	/* flags */
	DSET_LH(rq_b, 8, iterHandle?*iterHandle:-1);
	DSET_HL(rq_b, 12, objID);	/* parentID */
	p = NWDSBufRetrieve(buffer, &l);
	err = NWDSBufPut(&rq, p, l);
	if (err)
		return err;
	err = ncp_send_nds_frag(conn, DSV_MODIFY_ENTRY, rq.data, rq.curPos - rq.data,
		rp_b, sizeof(rp_b), &l);
	if (err)
		return err;
	if (l < 4) {
		if (iterHandle)
			*iterHandle = NO_MORE_ITERATIONS;
	} else {
		if (iterHandle)
			*iterHandle = DVAL_LH(rp_b, 0);
	}
	return 0;
}

NWDSCCODE NWDSModifyObject(
		NWDSContextHandle ctx, 
		const NWDSChar* objectName,
		nuint32* iterHandle, 
		nbool8 moreIter, 
		Buf_T* buffer) {
	NWDSCCODE err;
	NWCONN_HANDLE conn;
	NWObjectID objID;
	nuint32 lh;
	struct wrappedIterationHandle* ih = NULL;

	if (moreIter && !iterHandle)
		return ERR_NULL_POINTER;
	if (!buffer)
		return ERR_NULL_POINTER;
	if (buffer->bufFlags & NWDSBUFT_OUTPUT)
		return ERR_BAD_VERB;
	if (buffer->operation != DSV_MODIFY_ENTRY)
		return ERR_BAD_VERB;
	if (iterHandle && *iterHandle != NO_MORE_ITERATIONS) {
		ih = __NWDSIHLookup(*iterHandle, DSV_MODIFY_ENTRY);
		if (!ih)
			return ERR_INVALID_HANDLE;
		conn = ih->conn;
		objID = ih->objectID;
		lh = ih->iterHandle;
	} else {
		err = NWDSResolveName2DR(ctx, objectName, DS_RESOLVE_WRITEABLE | DS_RESOLVE_WALK_TREE,
			&conn, &objID);
		if (err)
			return err;
		lh = NO_MORE_ITERATIONS;
	}
	err = __NWDSModifyObjectV2(conn, moreIter?1:0, &lh, objID, buffer);
	if ((err == ERR_INVALID_API_VERSION) && !moreIter && (!iterHandle || *iterHandle == NO_MORE_ITERATIONS)) {
		lh = NO_MORE_ITERATIONS;
		err = __NWDSModifyObjectV0(conn, 0, objID, buffer);
	}
	if (ih)
		return __NWDSIHUpdate(ih, err, lh, iterHandle);
	return __NWDSIHCreate(err, conn, objID, lh, DSV_MODIFY_ENTRY, iterHandle);
}

static NWDSCCODE __NWDSModifyRDNV0(
		NWCONN_HANDLE conn,
		NWObjectID objID,
		nuint32 deleteOldRDN,
		const wchar_t* newRDN) {
	NWDSCCODE err;
	char rq_b[DEFAULT_MESSAGE_LEN];
	Buf_T rq;
	size_t l;
	char rp_b[16];
	
	NWDSSetupBuf(&rq, rq_b, sizeof(rq_b));
	NWDSBufPutPtr(&rq, 12);		
	DSET_LH(rq_b, 0, 0); 		/* version */
	DSET_HL(rq_b, 4, objID);	/* object ID */
	DSET_LH(rq_b, 8, deleteOldRDN);	/* delete old RDN */
	err = NWDSBufPutCIString(&rq, newRDN);
	if (err)
		return err;
	err = ncp_send_nds_frag(conn, DSV_MODIFY_RDN, rq.data, rq.curPos - rq.data,
		rp_b, sizeof(rp_b), &l);
	return err;
}

NWDSCCODE NWDSModifyRDN(
		NWDSContextHandle ctx,
		const NWDSChar* objectName,
		const NWDSChar* newName,
		nuint deleteOldRDN) {
	wchar_t newParent[MAX_DN_CHARS+1];
	wchar_t newRDN[MAX_DN_CHARS+1];
	NWDSCCODE err;
	NWCONN_HANDLE conn;
	NWObjectID objID;
	
	if (!objectName || !newName)
		return ERR_NULL_POINTER;
	err = NWDSSplitName(ctx, newName, newParent, newRDN);
	if (err)
		return err;
	err = NWDSResolveName2DR(ctx, objectName, DS_RESOLVE_WRITEABLE | DS_RESOLVE_WALK_TREE,
		&conn, &objID);
	if (err)
		return err;
	err = __NWDSModifyRDNV0(conn, objID, deleteOldRDN, newRDN);
	NWCCCloseConn(conn);
	return err;
}

NWDSCCODE __NWDSGetServerDN(
		NWCONN_HANDLE conn,
		wchar_t* name,
		size_t maxlen) {
	char rp_b[DEFAULT_MESSAGE_LEN];
	Buf_T rp;
	NWDSCCODE err;
	
	NWDSSetupBuf(&rp, rp_b, sizeof(rp_b));
	err = NWDSGetServerNameAddress(conn, 0, 0, &rp);
	if (err)
		return err;
	return NWDSBufDN(&rp, name, maxlen);
}

NWDSCCODE __NWDSGetObjectDN(
		NWCONN_HANDLE conn,
		NWObjectID id,
		wchar_t* name,
		size_t maxlen) {
	char rp_b[DEFAULT_MESSAGE_LEN];
	Buf_T rp;
	NWDSCCODE err;
	
	NWDSSetupBuf(&rp, rp_b, sizeof(rp_b));
	err = NWDSGetDSIRaw(conn, DSI_ENTRY_DN, 0, id, &rp);
	if (err)
		return err;
	return NWDSBufDN(&rp, name, maxlen);
}

NWDSCCODE __NWDSGetObjectDNUnicode(
		NWCONN_HANDLE conn,
		NWObjectID id,
		unicode* name,
		size_t* len) {
	char rp_b[DEFAULT_MESSAGE_LEN];
	Buf_T rp;
	NWDSCCODE err;
	size_t rlen;
	
	NWDSSetupBuf(&rp, rp_b, sizeof(rp_b));
	err = NWDSGetDSIRaw(conn, DSI_ENTRY_DN, 0, id, &rp);
	if (err)
		return err;
	err = NWDSBufGetLE32(&rp, &rlen);
	if (err)
		return err;
	if (rlen > *len)
		return NWE_BUFFER_OVERFLOW;
	err = NWDSBufGet(&rp, name, rlen);
	if (err)
		return err;
	*len = rlen;
	return 0;
}

static NWDSCCODE __NWDSBeginMoveEntryV0(
		NWCONN_HANDLE dstConn,
		nuint32 flags,
		NWObjectID dstParentID,
		const wchar_t* newRDN,
		const wchar_t* srcServer) {
	NWDSCCODE err;
	char rq_b[DEFAULT_MESSAGE_LEN];
	Buf_T rq;
	size_t l;
	char rp_b[16];
	
	NWDSSetupBuf(&rq, rq_b, sizeof(rq_b));
	NWDSBufPutPtr(&rq, 12);
	DSET_LH(rq_b, 0, 0); 		/* version */
	DSET_LH(rq_b, 4, flags);	/* flags */
	DSET_HL(rq_b, 8, dstParentID);	/* dst parent ID */
	err = NWDSBufPutCIString(&rq, newRDN);
	if (err)
		return err;
	err = NWDSBufPutCIString(&rq, srcServer);
	if (err)
		return err;
	err = ncp_send_nds_frag(dstConn, DSV_BEGIN_MOVE_ENTRY, rq.data, rq.curPos - rq.data,
		rp_b, sizeof(rp_b), &l);
	return err;
}

static NWDSCCODE __NWDSFinishMoveEntryV0(
		NWCONN_HANDLE srcConn,
		nuint32 flags,
		NWObjectID srcID,
		NWObjectID dstParentID,
		const wchar_t* newRDN,
		const wchar_t* dstServer) {
	NWDSCCODE err;
	char rq_b[DEFAULT_MESSAGE_LEN];
	Buf_T rq;
	size_t l;
	char rp_b[16];
	
	NWDSSetupBuf(&rq, rq_b, sizeof(rq_b));
	NWDSBufPutPtr(&rq, 16);
	DSET_LH(rq_b, 0, 0); 		/* version */
	DSET_LH(rq_b, 4, flags);	/* flags */
	DSET_HL(rq_b, 8, srcID);	/* source ID */
	DSET_HL(rq_b, 12, dstParentID);	/* dst parent ID */
	err = NWDSBufPutCIString(&rq, newRDN);
	if (err)
		return err;
	err = NWDSBufPutCIString(&rq, dstServer);
	if (err)
		return err;
	err = ncp_send_nds_frag(srcConn, DSV_FINISH_MOVE_ENTRY, rq.data, rq.curPos - rq.data,
		rp_b, sizeof(rp_b), &l);
	return err;
}

static const wchar_t* findDelim(const wchar_t* str, wint_t delim) {
	wint_t tmp;
	
	while ((tmp = *str++) != 0) {
		if (tmp == delim)
			return str;
		if (tmp == '\\') {
			if (!*str++)
				return NULL;
		}
	}
	return NULL;		
}

NWDSCCODE NWDSMoveObject(
		NWDSContextHandle ctx,
		const NWDSChar* srcObjectName,
		const NWDSChar* dstParentName,
		const NWDSChar* dstRDN) {
	NWCONN_HANDLE srcConn;
	NWObjectID srcObjID;
	NWCONN_HANDLE dstConn;
	NWObjectID dstObjID;
	NWDSCCODE err;
	wchar_t srcDN[MAX_DN_CHARS+1];
	wchar_t dstDN[MAX_DN_CHARS+1];
	wchar_t wRDN[MAX_RDN_CHARS+1];
	const wchar_t* srcParentDN;
	
	if (!srcObjectName || !dstParentName || !dstRDN)
		return ERR_NULL_POINTER;
	err = NWDSXlateFromCtx(ctx, wRDN, sizeof(wRDN), dstRDN);
	if (err)
		return err;
	err = NWDSResolveName2DR(ctx, srcObjectName, DS_RESOLVE_WRITEABLE | DS_RESOLVE_WALK_TREE,
		&srcConn, &srcObjID);
	if (err)
		return err;
	err = NWDSResolveName2(ctx, dstParentName, DS_RESOLVE_WRITEABLE | DS_RESOLVE_WALK_TREE,
		&dstConn, &dstObjID);
	if (err)
		goto err1;
	err = __NWDSGetObjectDN(srcConn, srcObjID, srcDN, sizeof(srcDN));
	if (err)
		goto error;
	err = __NWDSGetObjectDN(dstConn, dstObjID, dstDN, sizeof(dstDN));
	if (err)
		goto error;
	srcParentDN = findDelim(srcDN, '.');
	if (!srcParentDN)
		srcParentDN = L"[Root]";
	if (!wcscasecmp(srcParentDN, dstDN)) {
		err = ERR_RENAME_NOT_ALLOWED;
		goto error;
	}
	err = __NWDSGetServerDN(srcConn, srcDN, sizeof(srcDN));
	if (err)
		goto error;
	err = __NWDSGetServerDN(dstConn, dstDN, sizeof(dstDN));
	if (err)
		goto error;
	err = __NWDSBeginMoveEntryV0(dstConn, 0, dstObjID, wRDN, srcDN);
	if (err)
		goto error;
	err = __NWDSFinishMoveEntryV0(srcConn, 1, srcObjID, dstObjID, wRDN, dstDN);
error:;
	NWCCCloseConn(dstConn);
err1:;
	NWCCCloseConn(srcConn);
	return err;
}

NWDSCCODE NWDSModifyDN(
		NWDSContextHandle ctx,
		const NWDSChar* srcObjectName,
		const NWDSChar* dstObjectName,
		nuint deleteOldRDN) {
	NWCONN_HANDLE srcConn;
	NWObjectID srcObjID;
	NWCONN_HANDLE dstConn;
	NWObjectID dstObjID;
	NWDSCCODE err;
	wchar_t srcDN[MAX_DN_CHARS+1];
	wchar_t dstDN[MAX_DN_CHARS+1];
	wchar_t wRDN[MAX_DN_CHARS+1];
	const wchar_t* srcParentDN;
	
	
	if (!srcObjectName || !dstObjectName)
		return ERR_NULL_POINTER;
	if (deleteOldRDN)
		deleteOldRDN = 1;
	err = NWDSResolveName2DR(ctx, srcObjectName, DS_RESOLVE_WRITEABLE | DS_RESOLVE_WALK_TREE,
		&srcConn, &srcObjID);
	if (err)
		return err;
	err = __NWDSResolveName2p(ctx, dstObjectName, DS_RESOLVE_WRITEABLE | DS_RESOLVE_WALK_TREE,
		&dstConn, &dstObjID, wRDN);
	if (err)
		goto err1;
	err = __NWDSGetObjectDN(srcConn, srcObjID, srcDN, sizeof(srcDN));
	if (err)
		goto error;
	err = __NWDSGetObjectDN(dstConn, dstObjID, dstDN, sizeof(dstDN));
	if (err)
		goto error;
	srcParentDN = findDelim(srcDN, '.');
	if (!srcParentDN)
		srcParentDN = L"[Root]";
	if (!wcscasecmp(srcParentDN, dstDN)) {
		err = __NWDSModifyRDNV0(srcConn, srcObjID, deleteOldRDN, wRDN);
		goto error;
	}
	err = __NWDSGetServerDN(srcConn, srcDN, sizeof(srcDN));
	if (err)
		goto error;
	err = __NWDSGetServerDN(dstConn, dstDN, sizeof(dstDN));
	if (err)
		goto error;
	err = __NWDSBeginMoveEntryV0(dstConn, 0, dstObjID, wRDN, srcDN);
	if (err)
		goto error;
	err = __NWDSFinishMoveEntryV0(srcConn, deleteOldRDN, srcObjID, dstObjID, wRDN, dstDN);
error:;
	NWCCCloseConn(dstConn);
err1:;
	NWCCCloseConn(srcConn);
	return err;
}

NWDSCCODE __NWDSBeginLoginV0(
		NWCONN_HANDLE conn,
		NWObjectID objID,
		NWObjectID *p1,
		void *p2) {
	NWDSCCODE err;
	char rq_b[8];
	size_t l;
	char rp_b[16];
	
	DSET_LH(rq_b, 0, 0); 		/* version */
	DSET_HL(rq_b, 4, objID);	/* object ID */
	err = ncp_send_nds_frag(conn, DSV_BEGIN_LOGIN, rq_b, 8,
		rp_b, sizeof(rp_b), &l);
	if (err)
		return err;
	if (l < 8)
		return ERR_INVALID_SERVER_RESPONSE;
	if (p1)
		*p1 = DVAL_HL(rp_b, 0);
	if (p2)
		memcpy(p2, rp_b+4, 4);
	return 0;
}

NWDSCCODE __NWDSFinishLoginV2(
		NWCONN_HANDLE conn,
		nuint32 flag,
		NWObjectID objID,
		Buf_T* rqb,
		nuint8 p1[8],
		Buf_T* rpb) {
	NWDSCCODE err;
	char rq_b[DEFAULT_MESSAGE_LEN];
	Buf_T rq;
	void* p;
	void* q;
	size_t ln;
	
	NWDSSetupBuf(&rq, rq_b, sizeof(rq_b));
	q = NWDSBufRetrieve(rqb, &ln);
	p = NWDSBufPutPtr(&rq, 16);
	DSET_LH(p, 0, 2);	/* version */
	DSET_LH(p, 4, flag);	/* flags */
	DSET_HL(p, 8, objID);	/* object ID */
	DSET_LH(p, 12, ln);
	err = NWDSBufPut(&rq, q, ln);
	if (err)
		return err;
	NWDSBufStartPut(rpb, DSV_FINISH_LOGIN);
	q = NWDSBufPutPtrLen(rpb, &ln);
	err = ncp_send_nds_frag(conn, DSV_FINISH_LOGIN, rq_b, rq.curPos - rq.data, q, ln, &ln);
	memset(rq_b, 0, sizeof(rq_b));
	if (!err || (err == NWE_PASSWORD_EXPIRED)) {
		NWDSBufPutSkip(rpb, ln);
		NWDSBufFinishPut(rpb);
		err = NWDSBufGet(rpb, p1, 8);
	}
	return err;
}

NWDSCCODE __NWDSBeginAuthenticationV0(
		NWCONN_HANDLE conn,
		NWObjectID objID,
		const nuint8 seed[4],
		nuint8 authid[4],
		Buf_T* rpb) {
	NWDSCCODE err;
	char rq_b[12];
	void* q;
	size_t ln;
	
	DSET_LH(rq_b, 0, 0);		/* version */
	DSET_HL(rq_b, 4, objID);	/* object ID */
	memcpy(rq_b + 8, seed, 4);	/* seed */
	NWDSBufStartPut(rpb, DSV_BEGIN_AUTHENTICATION);
	q = NWDSBufPutPtrLen(rpb, &ln);
	err = ncp_send_nds_frag(conn, DSV_BEGIN_AUTHENTICATION, rq_b, sizeof(rq_b), q, ln, &ln);
	if (!err) {
		NWDSBufPutSkip(rpb, ln);
		NWDSBufFinishPut(rpb);
		err = NWDSBufGet(rpb, authid, 4);
		if (!err) {
			nuint32 le32;
			
			err = NWDSBufGetLE32(rpb, &le32);
			if (!err) {
				if (!NWDSBufPeekPtrLen(rpb, 0, le32))
					err = ERR_INVALID_SERVER_RESPONSE;
				else
					rpb->dataend = rpb->curPos + le32;
			}
		}
	}
	return err;
}

NWDSCCODE __NWDSFinishAuthenticationV0(
		NWCONN_HANDLE	conn,
		Buf_T*		md5_key,
		const void*	login_identity,
		size_t		login_identity_len,
		Buf_T*		auth_key) {
	NWDSCCODE err;
	char rq_b[DEFAULT_MESSAGE_LEN];
	char rp_b[16];		/* dummy, no data expected */
	Buf_T rq;
	void* p;
	void* q;
	size_t ln;
	
	NWDSSetupBuf(&rq, rq_b, sizeof(rq_b));
	q = NWDSBufRetrieve(md5_key, &ln);
	p = NWDSBufPutPtr(&rq, 8);
	DSET_LH(p, 0, 0);	/* version */
	DSET_LH(p, 4, ln);
	if (ln) {
		err = NWDSBufPut(&rq, q, ln);
		if (err)
			return err;
	}
	err = NWDSBufPutLE32(&rq, login_identity_len);
	if (err)
		return err;
	if (login_identity_len) {
		err = NWDSBufPut(&rq, login_identity, login_identity_len);
		if (err)
			return err;
	}
	q = NWDSBufRetrieve(auth_key, &ln);
	err = NWDSBufPutLE32(&rq, ln);
	if (err)
		return err;
	if (ln) {
		err = NWDSBufPut(&rq, q, ln);
		if (err)
			return err;
	}
	err = ncp_send_nds_frag(conn, DSV_FINISH_AUTHENTICATION, rq_b, rq.curPos - rq.data, rp_b, sizeof(rp_b), &ln);
	memset(rq_b, 0, sizeof(rq_b));
	return err;
}

NWDSCCODE NWDSGetObjectHostServerAddress(
		NWDSContextHandle ctx,
		const NWDSChar* objectName,
		NWDSChar* serverName,
		Buf_T* netAddresses) {
	NWDSContextHandle tmp;
	NWDSCCODE err;
	Buf_T attrname;
	char attrname_b[DEFAULT_MESSAGE_LEN];
	Buf_T hostname;
	char hostname_b[DEFAULT_MESSAGE_LEN];
	wchar_t rattrname[MAX_DN_CHARS+1];
	nuint32 ih = NO_MORE_ITERATIONS;
	NWObjectCount cnt, valcnt;
	enum SYNTAX synt;
	
	err = NWDSDuplicateContextHandleInt(ctx, &tmp);
	if (err)
		return err;
	NWDSSetupBuf(&attrname, attrname_b, sizeof(attrname_b));
	NWDSSetupBuf(&hostname, hostname_b, sizeof(hostname_b));
	err = NWDSInitBuf(tmp, DSV_READ, &attrname);
	if (err)
		goto freectx;
	err = NWDSPutAttrName(tmp, &attrname, (const NWDSChar*)L"Host Server");
	if (err)
		goto freectx;
	/* use input context... */
	err = NWDSRead(ctx, objectName, DS_ATTRIBUTE_VALUES, 0, &attrname, &ih, &hostname);
	if (err)
		goto freectx;
	if (ih != NO_MORE_ITERATIONS)
		NWDSCloseIteration(ctx, ih, DSV_READ);
	err = NWDSGetAttrCount(ctx, &hostname, &cnt);
	if (err)
		goto freectx;
	if (cnt < 1) {
		err = ERR_BUFFER_EMPTY;
		goto freectx;
	}
	err = NWDSGetAttrName(tmp, &hostname, (NWDSChar*)rattrname, &valcnt, &synt);
	if (err)
		goto freectx;
	if (wcscmp(rattrname, L"Host Server") || (synt != SYN_DIST_NAME) || (valcnt < 1)) {
		err = ERR_SYSTEM_ERROR;
		goto freectx;
	}
	if (serverName) {
		void* cur = NWDSBufPeekPtr(&hostname);
		
		err = NWDSGetAttrVal(ctx, &hostname, synt, serverName);
		if (err)
			goto freectx;
		NWDSBufSeek(&hostname, cur);
	}
	if (netAddresses) {
		err = NWDSGetAttrVal(tmp, &hostname, synt, rattrname);
		if (err)
			goto freectx;
		err = NWDSInitBuf(tmp, DSV_READ, &attrname);
		if (err)
			goto freectx;
		err = NWDSPutAttrName(tmp, &attrname, (const NWDSChar*)L"Network Address");
		if (err)
			goto freectx;
		ih = NO_MORE_ITERATIONS;
		err = NWDSRead(tmp, (NWDSChar*)rattrname, DS_ATTRIBUTE_VALUES, 0, &attrname, &ih, netAddresses);
		if (err)
			goto freectx;
		if (ih != NO_MORE_ITERATIONS) {
			NWDSCloseIteration(ctx, ih, DSV_READ);
			err = ERR_BUFFER_FULL;
			goto freectx;
		}
	}
	err = 0;
freectx:
	NWDSFreeContext(tmp);
	return err;
}

NWDSCCODE NWDSOpenConnToNDSServer(
		NWDSContextHandle ctx,
		const NWDSChar* serverName,
		NWCONN_HANDLE* pconn) {
	NWDSContextHandle tmp;
	NWDSCCODE err;
	Buf_T attrname;
	char attrname_b[DEFAULT_MESSAGE_LEN];
	Buf_T hostaddr;
	char hostaddr_b[DEFAULT_MESSAGE_LEN];
	wchar_t rattrname[MAX_DN_CHARS+1];
	nuint32 ih = NO_MORE_ITERATIONS;
	NWObjectCount cnt, valcnt;
	enum SYNTAX synt;
	
	err = NWDSDuplicateContextHandleInt(ctx, &tmp);
	if (err)
		return err;
	NWDSSetupBuf(&attrname, attrname_b, sizeof(attrname_b));
	NWDSSetupBuf(&hostaddr, hostaddr_b, sizeof(hostaddr_b));
	err = NWDSInitBuf(tmp, DSV_READ, &attrname);
	if (err)
		goto freectx;
	err = NWDSPutAttrName(tmp, &attrname, (const NWDSChar*)L"Network Address");
	if (err)
		goto freectx;
	/* use input context... */
	err = NWDSRead(ctx, serverName, DS_ATTRIBUTE_VALUES, 0, &attrname, &ih, &hostaddr);
	if (err)
		goto freectx;
	if (ih != NO_MORE_ITERATIONS)
		NWDSCloseIteration(ctx, ih, DSV_READ);
	err = NWDSGetAttrCount(ctx, &hostaddr, &cnt);
	if (err)
		goto freectx;
	if (cnt < 1) {
		err = ERR_BUFFER_EMPTY;
		goto freectx;
	}
	err = NWDSGetAttrName(tmp, &hostaddr, (NWDSChar*)rattrname, &valcnt, &synt);
	if (err)
		goto freectx;
	if (wcscmp(rattrname, L"Network Address") || (synt != SYN_NET_ADDRESS) || (valcnt < 1)) {
		err = ERR_SYSTEM_ERROR;
		goto freectx;
	}
	err = NWDSFindConnection(ctx, pconn, valcnt, &hostaddr, NWDSFINDCONN_CREATEALLOWED | NWDSFINDCONN_DSREADBUF);
freectx:
	NWDSFreeContext(tmp);
	return err;
}
