// scroller.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include <InterViews/adjuster.h>
#include <InterViews/border.h>
#include <InterViews/box.h>
#include <InterViews/glue.h>
#include <InterViews/message.h>
#include <InterViews/painter.h>
#include <InterViews/pattern.h>
#include <InterViews/perspective.h>
#include <InterViews/rubrect.h>
#include <InterViews/scrollbar.h>
#include <InterViews/sensor.h>
#include <InterViews/shape.h>
#include <math.h>
#include <string.h>
#include "scroller.h"

XEnlarger::XEnlarger (Interactor* i) : Enlarger(i) {
    Init();
}

XEnlarger::XEnlarger (const char* name, Interactor* i) : Enlarger(name, i) {
    Init();
}

void
XEnlarger::Init() { SetClassName("XEnlarger"); }

void XEnlarger::AdjustView (Event&) {
    register Perspective* s = shown;
    Coord cx;

    *s = *view->GetPerspective();
    cx = s->curx + s->curwidth/2;
    s->curwidth = iround(float(s->curwidth) / 2.0);
    s->curx = cx - s->curwidth/2;
    view->Adjust(*s);    
}

///////////////////

YEnlarger::YEnlarger (Interactor* i) : Enlarger(i) {
    Init();
}

YEnlarger::YEnlarger (const char* name, Interactor* i) : Enlarger(name, i) {
    Init();
}

void
YEnlarger::Init() { SetClassName("YEnlarger"); }

void
YEnlarger::AdjustView (Event&) {
    register Perspective* s = shown;
    Coord cy;

    *s = *view->GetPerspective();
    cy = s->cury + s->curheight/2;
    s->curheight = iround(float(s->curheight) / 2.0);
    s->cury = cy - s->curheight/2;
    view->Adjust(*s);    
}

///////////////////

XReducer::XReducer (Interactor* i) : Reducer(i) {
    Init();
}

XReducer::XReducer (const char* name, Interactor* i) : Reducer(name, i) {
    Init();
}

void
XReducer::Init() { SetClassName("XReducer"); }

void XReducer::AdjustView (Event&) {
    register Perspective* s = shown;
    Coord cx;

    *s = *view->GetPerspective();
    cx = s->curx + s->curwidth/2;
    s->curwidth = iround(float(s->curwidth) / 0.5);
    s->curx = cx - s->curwidth/2;
    view->Adjust(*s);    
}

///////////////////

YReducer::YReducer (Interactor* i) : Reducer(i) {
    Init();
}

YReducer::YReducer (const char* name, Interactor* i) : Reducer(name, i) {
    Init();
}

void
YReducer::Init() { SetClassName("YReducer"); }

void
YReducer::AdjustView (Event&) {
    register Perspective* s = shown;
    Coord cy;

    *s = *view->GetPerspective();
    cy = s->cury + s->curheight/2;
    s->curheight = iround(float(s->curheight) / 0.5);
    s->cury = cy - s->curheight/2;
    view->Adjust(*s);    
}

///////////////////

ViewScaler::ViewScaler (Interactor* i, int size) {
    Init(i, size);
}

ViewScaler::ViewScaler (const char* name, Interactor* i, int size) {
    SetInstance(name);
    Init(i, size);
}

ViewScaler::ViewScaler (Interactor* i, int size, Painter* p) {
    output = p;
    output->Reference();
    Init(i, size);
}

/* 0.3 second delay for auto-repeat */
static int DELAY = 3;

void ViewScaler::Init (Interactor* i, int n) {
    SetClassName("ViewScaler");
    size = n;
    adjusters = new HBox(
        new HGlue(),
		new VBox(
			new VGlue(2, 0, 0),
			new HBox(
				new Message("Horizontal Zoom In:  "),
				new HGlue(2, 0, 0),
				new XEnlarger(i)
			),
			new HBox(
				new Message("Horizontal Zoom Out: "),
				new HGlue(2, 0, 0),
				new XReducer(i)
			),
			new VGlue(2, 0, 0)
		),
		new VBox(
			new VGlue(2, 0, 0),
			new HBox(
				new Message("Vertical Zoom In:  "),
				new HGlue(2, 0, 0),
				new YEnlarger(i)
			),
			new HBox(
				new Message("Vertical Zoom Out: "),
				new HGlue(2, 0, 0),
				new YReducer(i)
			),
			new VGlue(2, 0, 0)
		),
        new HGlue()
    );
    Insert(adjusters);
}

void ViewScaler::Reconfig () {
    MonoScene::Reconfig();
    Shape a = *adjusters->GetShape();
    if (a.vstretch != 0 || a.vshrink != a.height / 3) {
        if (size != 0) {
            a.width = size;
            a.hshrink = a.hstretch = 0;
        }
        a.vstretch = 0;
        a.vshrink = a.height/3;
        adjusters->Reshape(a);
    }
}

////////////////////////////

ScrollerBar::ScrollerBar (Interactor* i, int ht) {
    Init(i, ht);
}

ScrollerBar::ScrollerBar (const char* name, Interactor* i, int ht) {
    SetInstance(name);
    Init(i, ht);
}

void ScrollerBar::Init (Interactor* i, int ht) {
    SetClassName("ScrollerBar");
    interactor = i;
    view = i->GetPerspective();
    view->Attach(this);
    shown = new Perspective;
    *shown = *view;
    shape->vstretch = shape->vshrink = 0;
    shape->height = ht;
    prevl = prevb = prevr = prevt = 0;
    left = bottom = right = top = 0;
    input = new Sensor(updownEvents);
}

void ScrollerBar::Reconfig () {
    Painter* p = new Painter(output);
    p->Reference();
    Resource::unref(output);
    output = p;

    syncScroll = AttributeIsSet("syncScroll");
}

void ScrollerBar::Reshape (Shape& ns) {
    if (shown->width == 0) {
		*shape = ns;
    } else {
		shape->width = ((canvas == nil) ? ns.width : xmax + 1);
    }
}

void ScrollerBar::Draw () {
    if (canvas != nil) {
		output->SetPattern(new Pattern(Pattern::lightgray));
		output->FillRect(canvas, 0, 0, xmax, ymax);
		output->SetPattern(new Pattern(Pattern::clear));
		output->FillRect(canvas, left, bottom, right, top);
		output->SetPattern(new Pattern(Pattern::solid));
		output->Rect(canvas, left, bottom, right, top);
		output->Line(canvas, left+1, bottom-1, right+1, bottom-1);
		output->Line(canvas, right+1, bottom-1, right+1, top-1);
	
		prevl = left; prevb = bottom;
		prevr = right; prevt = top;
    }
}

void ScrollerBar::Redraw (Coord left, Coord bottom, Coord right, Coord top) {
    output->Clip(canvas, left, bottom, right, top);
    Draw();
    output->NoClip();
}

inline Coord ScrollerBar::ViewX (Coord x) {
    return iround(float(x) * float(shown->width) / float(xmax));
}

inline Coord ScrollerBar::ViewY (Coord y) {
    return y;
}

inline Coord ScrollerBar::ScrollerBarX (Coord x) {
    return iround(float(x) * float(xmax) / float(shown->width));
}

inline Coord ScrollerBar::ScrollerBarY (Coord y) {
    return y;
}

void ScrollerBar::Move (Coord dx, Coord dy) {
    shown->curx += dx;
}

boolean ScrollerBar::Inside (Event& e) {
    return e.x > left && e.x < right && e.y > bottom && e.y < top;
}

void ScrollerBar::CalcLimits (Event& e) {
    llim = e.x - max(0, left);
    blim = e.y - max(0, bottom);
    rlim = e.x + max(0, xmax - right);
    tlim = e.y + max(0, ymax - top);
    origx = e.x;
    origy = e.y;
}

void ScrollerBar::Constrain (Event& e) {
    e.x = min(max(e.x, llim), rlim);
    e.y = min(max(e.y, blim), tlim);
}

void ScrollerBar::Slide (Event& e) {
    Coord newleft, newbot, dummy;
    boolean control = e.control;

    Listen(allEvents);
    SlidingRect r(output, canvas, left, bottom, right, top, e.x, e.y);
    CalcLimits(e);
    do {
		switch (e.eventType) {
			case MotionEvent:
				e.target->GetRelative(e.x, e.y, this);
				Constrain(e);
				r.Track(e.x, e.y);
				if ((syncScroll && !control) || (!syncScroll && control)) {
					r.Erase();
					r.GetCurrent(newleft, newbot, dummy, dummy);
					Move(ViewX(newleft - left), ViewY(newbot - bottom));
					makeAdjust();
				}
				break;
			default:
				break;
		}
		Read(e);
    } while (e.eventType != UpEvent);

    r.GetCurrent(newleft, newbot, dummy, dummy);
    Move(ViewX(newleft - left), ViewY(newbot - bottom));
    Listen(input);
}

void
ScrollerBar::makeAdjust() {
	interactor->Adjust(*shown);
}

void ScrollerBar::Jump (Event& e) {
    register Perspective* s = shown;
    Coord dx, dy = 0;
    
    if (e.button == RIGHTMOUSE) {
	dx = ViewX(e.x) - s->curx - s->curwidth/2;
    } else {
	if (e.button == LEFTMOUSE) {
	    dx = s->sx;
	} else {
	    dx = s->lx;
	}

	if (e.x < left) {
	    dx = -dx;
	} else if (e.x < right) {
	    dx = 0;
	}
    }
    dx = min(
		max(s->x0 - s->curx, dx), s->x0 + s->width - s->curx - s->curwidth
    );
    Move(dx, dy);
}	

void ScrollerBar::Handle (Event& e) {
    if (e.eventType == DownEvent) {
		if (Inside(e)) {
			Slide(e);
		} else {
			Jump(e);
		}
		makeAdjust();
    }
}
    
static const int MIN_SIZE = 8;

void ScrollerBar::SizeKnob () {
    register Perspective* s = shown;
    
    if (canvas != nil) {
		left = ScrollerBarX(s->curx - s->x0);
		bottom = 0;
		right = left + max(ScrollerBarX(s->curwidth), MIN_SIZE);
		top = ymax;
    }
}    

void ScrollerBar::Update () {
    register Perspective* p = shown;
    int h, oldwidth, oldheight;
    Scene* s;
    Shape ns;

    oldwidth = p->width;
    oldheight = p->height;
    *p = *view;

    SizeKnob();
    if (p->width != oldwidth || p->height != oldheight) {
    	Draw();
    }
    else if ( prevl != left  || prevr != right) {
		Draw();
    }
}

void ScrollerBar::Resize () {
    int w = xmax + 1;
    if (shape->width != w) {
		Shape ns = *shape;
		ns.width = w;
		Reshape(ns);
    }
    SizeKnob();
}

ScrollerBar::~ScrollerBar () {
    view->Detach(this);
    Resource::unref(shown);
}

///////////////////

HorizontalViewScroller::HorizontalViewScroller (Interactor* i, int height) {
    Init(i, height);
}

HorizontalViewScroller::HorizontalViewScroller (const char* name, Interactor* i, int height) {
    SetInstance(name);
    Init(i, height);
}

HorizontalViewScroller::HorizontalViewScroller (Interactor* i, int height, Painter* p) {
    output = p;
    output->Reference();
    Init(i, height);
}

void
HorizontalViewScroller::Init (Interactor* i, int height) {
    SetClassName("HorizontalViewScroller");
    scrollbar = new ScrollerBar(i, height);
    Insert(
		new HBox(
		    new LeftMover(i, DELAY),
		    scrollbar,
		    new RightMover(i, DELAY)
	    	)
    );
}

void
HorizontalViewScroller::Update() {
	scrollbar->Update();
}

///////////////////

VerticalViewScroller::VerticalViewScroller (Interactor* i, int width) {
    Init(i, width);
}

VerticalViewScroller::VerticalViewScroller (const char* name, Interactor* i, int width) {
    SetInstance(name);
    Init(i, width);
}

VerticalViewScroller::VerticalViewScroller (Interactor* i, int width, Painter* p) {
    output = p;
    output->Reference();
    Init(i, width);
}

void
VerticalViewScroller::Init (Interactor* i, int width) {
    SetClassName("VerticalViewScroller");
    scrollbar = new VScrollBar(i, width);
    Insert(
		new VBox(
		    new UpMover(i, DELAY),
		    scrollbar,
		    new DownMover(i, DELAY)
	    	)
    );
}

void
VerticalViewScroller::Update() {
	scrollbar->Update();
}

