/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: generate.c,v 1.23 2002/01/03 12:17:10 ostborn Exp $
*/

#include <libintl.h>
#include <locale.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <assert.h>

/* ./src/ */

#include "mconfig.h"
#include "mstate.h"
#include "mlocale.h"
#include "mhash.h"
#include "mlist.h"
#include "mdatatypes.h"
#include "mplugins.h"
#include "misc.h"

#include "datatypes/brokenlink/datatype.h"

/* ./src/output/template */

#include "plugin_config.h"
#include "template.h"
#include "generate.h"
#include "web.h"
#include "mail.h"
#include "pictures.h"
#include "mtree.h"

typedef struct {
	char *key;
	char *subst;
} tmpl_vars;

/* tla -> Three Letter Abbreviation */
char *get_month_string(int m, int tla) {
	static char monthname[255];
	
	struct tm *tm;
	time_t t = time(NULL);
	
	tm = localtime(&t);
	
	tm->tm_mon = m > 0 ? m - 1 : 11;
	
	strftime(monthname, sizeof(monthname)-1, tla ? "%b" : "%B", tm);
	return monthname;
}

char * generate_output_link (mconfig *ext_conf, int year, int month, const char *current) {
	config_output *conf = ext_conf->plugin_conf;
	tmpl_main *tmpl;
	char date[7], *fn;
	
	/* huge overkill, but dawm nice code */
	
	tmpl = tmpl_init();
	sprintf(date, "%04d%02d", 
		year,
		month);
	
	tmpl_load_string(tmpl,  conf->filename_pattern);
	tmpl_set_var(tmpl, "NAME", current);
	tmpl_set_var(tmpl, "DATE", date);
	
	fn = tmpl_replace(tmpl);
	
	tmpl_free(tmpl);
	
	return fn;
}

char * generate_output_filename (mconfig *ext_conf, mstate *state, const char *subpath, const char *current) {
	char *lnk = generate_output_link(ext_conf, state->year, state->month, current);
	char *fn;
	
	fn = malloc(strlen(ext_conf->outputdir) + (subpath ? strlen(subpath) : 0) + strlen(lnk) + 1 + 2);
	
	sprintf(fn, "%s/%s/%s", 
		ext_conf->outputdir,
		subpath ? subpath : "",
		lnk);
	
	free(lnk);
	
	return fn;
}

char * generate_template_filename(mconfig *ext_conf, int type) {
	char *fn, *t = NULL;
	config_output *conf = ext_conf->plugin_conf;
	
	switch(type) {
	case M_TMPL_TABLE:
		t = conf->tmpl_table;
		break;
	case M_TMPL_OUTER:
		t = conf->tmpl_outer;
		break;
	case M_TMPL_MENU:
		t = conf->tmpl_menu;
		break;
	case M_TMPL_INDEX:
		t = conf->tmpl_index;
		break;
	}
	
	if (!t || !conf->template_path || !conf->template_name) {
		M_DEBUG3(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "something is NULL: t = %p, tmpl-path: %p, tmpl-name: %p\n",
			 t, conf->template_path, conf->template_name);
		return NULL;
	}
	
	fn = malloc(strlen(conf->template_path) + 
		    strlen(conf->template_name) + 
		    strlen(t) + 
		    2 + 1);
	
	sprintf(fn, "%s/%s/%s",
		conf->template_path,
		conf->template_name,
		t);
	
	return fn;
}

int gen_menu_tree(mconfig *ext_conf, mstate *state, tmpl_main *tmpl, const mtree *menu, const char *current, int level) {
	char *parent, *fn;
	int i;
	config_output *conf = ext_conf->plugin_conf;
	
	if (!menu) return -1;
	if (!menu->data) return -1;
	
	parent = menu->data->key;
	
	for (i = 0; i < level; i++) {
		tmpl_set_current_block(tmpl, "menurowspacer");
		tmpl_parse_current_block(tmpl);
	}
	
	tmpl_set_current_block(tmpl, "menuentry");
	tmpl_set_var(tmpl, "MENU_CLASS", 
		     (0 == strcmp(parent, current)) 
		     ? "active"
		     : "menu");
	
	if (level != 0) {
		fn = generate_output_link(ext_conf, state->year, state->month, parent);
		tmpl_set_var(tmpl, "MENU_URL", fn);
		free(fn);
	} else {
		tmpl_set_var(tmpl, "MENU_URL", conf->index_filename);
	}
	
	tmpl_set_var(tmpl, "MENU_NAME", 
		     menu->data->data.brokenlink.referrer ? 
		     menu->data->data.brokenlink.referrer : 
		     parent);
	tmpl_parse_current_block(tmpl);
	tmpl_clear_block(tmpl, "menurowspacer");
	
	if (mtree_is_child(menu, current)) {
		int i;
		
		for (i = 0; i < menu->num_childs; i++) {
			gen_menu_tree(ext_conf, state, tmpl, menu->childs[i], current, level + 1);
		}
	}
	return 0;
}

char * generate_menu (mconfig *ext_conf, mstate *state, const char *current) {
	char *s, *fn;
	config_output *conf = ext_conf->plugin_conf;
	tmpl_main * tmpl;
	
	tmpl = tmpl_init();
	assert(tmpl);
	
	if ((fn = generate_template_filename(ext_conf, M_TMPL_MENU)) == NULL) {
		tmpl_free(tmpl);
		return NULL;
	}
	
	if (tmpl_load_template(tmpl, fn) != 0) {
		free(fn);
		tmpl_free(tmpl);
		return NULL;
	}
	free(fn);
	
/*	mtree_print(conf->menu);*/
	
	gen_menu_tree(ext_conf, state, tmpl, conf->menu, current, 0);
	
	s = tmpl_replace(tmpl);
	
	tmpl_free(tmpl);
	
	return s;
}

char * generate_report (mconfig *ext_conf, mstate *state, tmpl_reports *reports,  const char *current, const char *subpath) {
	char *s = NULL;
	int i;

	/* search for a title */
	for (i = 0; reports[i].key; i++) {
		if (0 == strcmp(reports[i].key, current)) {
			/* found */
			break;
		}
	}
		
	if (reports[i].key == NULL)
		return NULL;
	
	if (reports[i].func != NULL) {
		if ((s = reports[i].func(ext_conf, state, current, 30 /* max-value */, subpath)) == NULL) {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "no chance for %s\n", current);
		}
	} 

	return s;
}

/**
 * unfold the tree to an array
 * 
 * @return last used index in the array
 */

int mtree_to_marray (mtree *t, mdata **d, int ndx) {
	int i;
	
	if (!t) return ndx;
	
	d[ndx++] = t->data;
	
	for (i = 0; i < t->num_childs; i++) {
		ndx = mtree_to_marray(t->childs[i], d, ndx);
	}
	
	return ndx;
}

int mplugins_output_generate_monthly_output(mconfig *ext_conf, mstate *state, const char *subpath) {
#define NUM_TIMER 14
	int ret;
	char *s, *fn;
	config_output *conf = ext_conf->plugin_conf;
	tmpl_main * tmpl;
	mlist *l;
	int i, cnt, j;
	mdata **d;
	
	char buf[255];
	struct tm *_tm;
	time_t t;
	
	mtimer timer[NUM_TIMER];
	
	tmpl_vars vars[] = {
		/* generated by this plugin */
		{ "LASTRECORD", NULL }, /* hard coded to 0 */
		{ "GENERATEDAT", NULL }, /* hard coded to 1 */
		
		/* hard values */
		{ "MLA_URL", "http://jan.kneschke.de/projects/modlogan/" },
		{ "MLA_PACKAGE", PACKAGE },
		{ "MLA_VERSION", VERSION },
		
		/* translated strings */
		{ "TXT_STATISTICS", _("Statistics for")},
		{ "TXT_LASTRECORD", _("Last record")},
		{ "TXT_GENERATEDAT", _("Generated at")},
		{ "TXT_OUTPUTBY", _("Output generated by")},
		{ "TITLE", _("Statistics") },
		
		/* link handlers */
		{ "LNK_HOME", NULL },
		{ "LNK_UP", NULL }, /* 11 - hard coded !! */
		{ "LNK_PREV", NULL },
		{ "LNK_NEXT", NULL },
		{ "LNK_CURRENT", NULL },
		
		
		/* internal placeholders */
		{ "MENU", NULL },
		{ "REPORT", NULL },
		
		{ NULL,  NULL }
	};
	
	if (!conf) {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "conf is NULL\n");
		return -1;
	}
	
	vars[11].subst = conf->index_filename;
	
	for (i = 0; i < NUM_TIMER; i++) {
		MTIMER_RESET(timer[i]);
	}
	
	MTIMER_START(timer[0]);
	
	/* init template handle */
	tmpl = tmpl_init();
	assert(tmpl);
	
	tmpl->debug_level = ext_conf->debug_level;
	
	if ((fn = generate_template_filename(ext_conf, M_TMPL_OUTER)) == NULL) {
		M_DEBUG0(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "can't generate filename of the template\n");
		tmpl_free(tmpl);
		return -1;
	}

	MTIMER_START(timer[1]);
	/* load and parse template */
	if ((ret = tmpl_load_template(tmpl, fn)) != 0) {
		M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "parsing template %s failed\n",
			 fn);
		free(fn);
		tmpl_free(tmpl);
		return ret;
	}
	free(fn);
	MTIMER_STOP(timer[1]);
	MTIMER_CALC(timer[1]);
	
	MTIMER_START(timer[2]);
	/* set external variables from config-file */
	for (l = conf->variables; l; l = l->next) {
		if (l->data) {
			char *k, *v;
			k = l->data->key;
			
			if (k) {
				if ((v = strchr(k, ',')) != NULL) {
					v++;
		
					/* check if the variable is used internally */
					for (i = 0; vars[i].key; i++) {
						if (0 == strncmp(k, vars[i].key, v-k-1)) 
							break;
					} 
					
					/* not found */
					if (vars[i].key == NULL) {
						char *key = malloc(v-k);
						strncpy(key, k, v-k-1);
						key[v-k-1] = '\0';
						tmpl_set_var(tmpl, key, v);
						free(key);
					}
				} else {
					M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
						 "no ',' found in %s\n", k);
				}
			}
		}
	}
	MTIMER_STOP(timer[2]);
	MTIMER_CALC(timer[2]);
	
	MTIMER_START(timer[3]);
	
	/* create current timestamp */
	t = time(NULL);
	_tm = localtime(&t);
	strftime(buf, sizeof(buf), "%X %x", _tm);
	tmpl_set_var(tmpl, "GENERATEDAT", buf);
	
	/* create timestamp of the last record */
	if ((t = state->timestamp)) {
		_tm = localtime(&t);
		strftime(buf, sizeof(buf), "%X %x", _tm);
		tmpl_set_var(tmpl, "LASTRECORD", buf);
	} else {
		tmpl_set_var(tmpl, "LASTRECORD", _("(not set)"));
	}
	
	tmpl_set_var(tmpl, "LNK_HOME", conf->index_filename);
	
	if (conf->menu) {
		cnt = mtree_num_elements(conf->menu) + 1;
		d = malloc(sizeof(mdata *) * cnt);
	
		mtree_to_marray(conf->menu, d, 0);
	} else {
		cnt = 0;
		d = NULL;
	}
	
	MTIMER_STOP(timer[3]);
	MTIMER_CALC(timer[3]);
	
	MTIMER_START(timer[4]);
	
	/* generate reports */
	for (l = conf->reports; l; l = l->next) {
		MTIMER_START(timer[7]);
		for (i = 0; conf->avail_reports[i].key; i++) {
			if (0 == strcmp(conf->avail_reports[i].key, l->data->key)) {
				/* found */
				break;
			}
		}
		
		if (conf->avail_reports[i].key == NULL) {
			/* report is unknown */
			M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_WARNINGS,
					 "report-type '%s' is unknown - skipped\n",
					 l->data->key);
			continue;
		}
		
		/* set template variables */
		for (i = 0; vars[i].key; i++) {
			if (i != 0 && i != 1 && /* ignore GENERATEDAT and LASTRECORD as they are already set */
			    vars[i].subst != NULL) { 
				if (0 != tmpl_set_var(tmpl, vars[i].key, vars[i].subst)) {
					M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_WARNINGS,
						 "substituing key %s failed\n",
						 vars[i].key);
				} else {
					/* everything ok */
				}
			} else if (vars[i].subst == NULL) {
				/* perhaps this variable is handled by a function */
				char *s = NULL;
				
				if (0 == strcmp("MENU", vars[i].key)) {
					MTIMER_START(timer[9]);
					if ((s = generate_menu(ext_conf, state, l->data->key)) == NULL) {
						M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
							 "generating %s failed\n",
							 vars[i].key);
						return -1;
					} else {
						tmpl_set_var(tmpl, vars[i].key, s);
						free(s);
					}
					MTIMER_STOP(timer[9]);
					MTIMER_CALC(timer[9]);
				} else if (0 == strcmp("REPORT", vars[i].key)) {
					MTIMER_START(timer[10]);
					if ((s = generate_report(ext_conf, state, conf->avail_reports, l->data->key, subpath)) == NULL) {
						M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
							 "generating %s failed\n",
							 vars[i].key);
						return -1;
					} else {
						tmpl_set_var(tmpl, vars[i].key, s);
						free(s);
					}
					MTIMER_STOP(timer[10]);
					MTIMER_CALC(timer[10]);
				} else if (0 == strcmp("LNK_CURRENT", vars[i].key)) {
					char *fn;
					MTIMER_START(timer[11]);
					
					fn = generate_output_link(ext_conf, state->year, state->month, l->data->key);
					
					tmpl_set_var(tmpl, vars[i].key, fn);
					free(fn);
					MTIMER_STOP(timer[11]);
					MTIMER_CALC(timer[11]);
				} else if (0 == strcmp("LNK_PREV", vars[i].key)) {
					/* locate entry */
					MTIMER_START(timer[12]);
					for (j = 0; j < cnt; j++) 
						if (0 == strcmp(l->data->key, d[j]->key)) 
							break;
					
					/* found */
					if (j != cnt && j > 0) {
						fn= generate_output_link(ext_conf, state->year, state->month, d[j-1]->key);
						tmpl_set_var(tmpl, vars[i].key, fn);
						free(fn);
					} else {
						tmpl_set_var(tmpl, vars[i].key, conf->index_filename);
					}
					MTIMER_STOP(timer[12]);
					MTIMER_CALC(timer[12]);
				} else if (0 == strcmp("LNK_NEXT", vars[i].key)) {
					/* locate entry */
					MTIMER_START(timer[13]);
					for (j = 0; j < cnt; j++) 
						if (0 == strcmp(l->data->key, d[j]->key)) 
							break;
					
					/* found */
					if (j != cnt && j < cnt - 1) {
						fn= generate_output_link(ext_conf, state->year, state->month, d[j+1]->key);
						tmpl_set_var(tmpl, vars[i].key, fn);
						free(fn);
					} else {
						tmpl_set_var(tmpl, vars[i].key, conf->index_filename);
					}
					MTIMER_STOP(timer[13]);
					MTIMER_CALC(timer[13]);
				}
			}
		}
		
		MTIMER_STOP(timer[7]);
		MTIMER_CALC(timer[7]);
		
		MTIMER_START(timer[8]);
	
		/* replace tags by the actual output */
		if ((s = tmpl_replace(tmpl))) {
			/* index-file */
			char *fn = generate_output_filename(ext_conf, state, subpath, l->data->key);
			FILE *f;
			
			if (!fn) {
				M_DEBUG0(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
					 "no filename for output");
				return -1;
			}
			
			if ((f = fopen(fn, "wb")) == NULL) {
				M_DEBUG0(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
					 "opening failed");
				return -1;
			}
			fputs(s, f);
			fclose(f);
			
			free(fn);
			free(s);
		}
		MTIMER_STOP(timer[8]);
		MTIMER_CALC(timer[8]);
	}
	
	MTIMER_STOP(timer[4]);
	MTIMER_CALC(timer[4]);
	
	/* generate pages which are not related to a menu */
	
	MTIMER_START(timer[5]);
	
	for (i = 0; i < cnt; i++) {
		/* check if we have already generated the page */
		for (l = conf->reports; l; l = l->next) {
			if (0 == strcmp(d[i]->key, l->data->key))
				break;
		}
		
		if (l) {
			for (j = 0; conf->avail_reports[j].key; j++) {
				if (0 == strcmp(conf->avail_reports[j].key, d[i]->key)) {
					/* found */
					break;
				}
			}
			
			if (conf->avail_reports[j].key != NULL) {
				/* report is known and is generated */
				continue;
			} 
		}
		
		/* set template variables */
		for (j = 0; vars[j].key; j++) {
			if (j != 0 && j != 1 && /* ignore GENERATEDAT and LASTRECORD as they are already set */
			    vars[j].subst != NULL) { 
				if (0 != tmpl_set_var(tmpl, vars[j].key, vars[j].subst)) {
					M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_WARNINGS,
						 "substituing key %s failed\n",
						 vars[j].key);
				} else {
					/* everything ok */
				}
			} else if (vars[j].subst == NULL) {
				/* perhaps this variable is handled by a function */
				char *s = NULL;
				
				if (0 == strcmp("MENU", vars[j].key)) {
					if ((s = generate_menu(ext_conf, state, d[i]->key)) == NULL) {
						M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
							 "generating %s failed\n",
							 vars[j].key);
						return -1;
					} else {
						tmpl_set_var(tmpl, vars[j].key, s);
						free(s);
					}
				} else if (0 == strcmp("REPORT", vars[j].key)) {
					/* no report */
					tmpl_clear_var(tmpl, vars[j].key);
				} else if (0 == strcmp("LNK_CURRENT", vars[j].key)) {
					char *fn;
					
					fn = generate_output_link(ext_conf, state->year, state->month, d[i]->key);
					
					tmpl_set_var(tmpl, vars[j].key, fn);
					free(fn);
				} else if (0 == strcmp("LNK_PREV", vars[j].key)) {
					
					if (i > 0) {
						fn = generate_output_link(ext_conf, state->year, state->month, d[i-1]->key);
						tmpl_set_var(tmpl, vars[j].key, fn);
						free(fn);
					} else {
						tmpl_set_var(tmpl, vars[j].key, conf->index_filename);
					}
				} else if (0 == strcmp("LNK_NEXT", vars[j].key)) {
					if (i < cnt - 1) {
						fn = generate_output_link(ext_conf, state->year, state->month, d[i+1]->key);
						tmpl_set_var(tmpl, vars[j].key, fn);
						free(fn);
					} else {
						tmpl_set_var(tmpl, vars[j].key, conf->index_filename);
					}
				}
				
			}
		}
		
		/* replace tags by the actual output */
		if ((s = tmpl_replace(tmpl))) {
			/* index-file */
			char *fn = generate_output_filename(ext_conf, state, subpath, d[i]->key);
			
			FILE *f;
			
			f = fopen(fn, "w");
			fputs(s, f);
			fclose(f);
			
			free(fn);
			free(s);
		}
	}
	
	free(d);
	d = NULL;
	
	MTIMER_STOP(timer[5]);
	MTIMER_CALC(timer[5]);
	
	MTIMER_START(timer[6]);
	
	/* copy files */
	for (l = conf->files; l; l = l->next) {
		int fd1, fd2, n;
		char buf[256];
		char *fn1, *fn2, *p, *f, *dir;
		
		fn1 = malloc(strlen(conf->template_path) + strlen(conf->template_name) + strlen(l->data->key) + 1 + 2);
		sprintf(fn1, "%s/%s/%s",
			conf->template_path,
			conf->template_name,
			l->data->key);
		
		fn2 = malloc(strlen(ext_conf->outputdir) + (subpath ? strlen(subpath) : 0) + strlen(l->data->key) + 1 + 2);
		dir = malloc(strlen(ext_conf->outputdir) + (subpath ? strlen(subpath) : 0) + strlen(l->data->key) + 1 + 2);
		sprintf(fn2, "%s/%s/%s",
			ext_conf->outputdir,
			subpath ? subpath : "",
			l->data->key);
		
		if (strlen(l->data->key) > sizeof(buf) - 10) {
			M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "filename '%s' is too long\n",
				 l->data->key
				 );
			return -1;
		}
		
		
		sprintf(buf, "%s/%s", subpath ? subpath : ".", l->data->key);
		strcpy(dir, ext_conf->outputdir);
		f = buf;
		while ((p = strchr(f, '/')) != NULL) {
			struct stat st;
			*p = '\0';
			strcat(dir, "/");
			strcat(dir, f);
			
			if (stat(dir, &st) == 0) {
				if (S_ISDIR(st.st_mode)) {
					f = p + 1;
					continue;
				} else {
					M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
						 "'%s' exists, but is not a diretory\n",
						 dir);
					free(dir);
					return -1;
				}
			} else {
			
				if (mkdir(dir, 
					  S_IROTH | S_IXOTH |
					  S_IRGRP | S_IXGRP |
					  S_IRUSR | S_IXUSR | S_IWUSR) != 0) /* 0755 */ { 
						  M_DEBUG2(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
							   "makeing directory %s failed: %s\n",
							   dir,
							   strerror(errno));
						  free(dir);
						  return -1;
					  }
			}
			
			f = p + 1;
		}
		free(dir);
		
		if ((fd1 = open(fn1, O_RDONLY)) == -1) {
			M_DEBUG2(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "opening source %s failed: %s\n",
				 fn1,
				 strerror(errno));
			
			free(fn1);
			free(fn2);
			return -1;
		}
		if ((fd2 = open(fn2, O_CREAT | O_TRUNC | O_WRONLY, S_IROTH | S_IRGRP | S_IWUSR | S_IRUSR)) == -1) {
			M_DEBUG2(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "opening dest file '%s' failed: %s\n",
				 fn2,
				 strerror(errno));
			free(fn1);
			free(fn2);
			return -1;
		}
		
		while( (n = read(fd1, buf, sizeof(buf)) )) {
			write(fd2, buf, n);
		}
		close(fd1);
		close(fd2);
		
		free(fn1);
		free(fn2);
	}
	
	tmpl_free(tmpl);
	
	MTIMER_STOP(timer[6]);
	MTIMER_CALC(timer[6]);
	
	MTIMER_STOP(timer[0]);
	MTIMER_CALC(timer[0]);
	
/*	for (i = 0; i < NUM_TIMER; i++) {
		fprintf(stderr, "timer %d: %ld msec\n",
			i, timer[i].span);
	}
 */
	
	return 0;
}

int set_line(tmpl_main *tmpl, const char *desc, 
	     long hits, long pages, long files, long visits, double traffic,
	     int days_passed
	     ) {
	
	char buf[255];
	char c;
	double tr;
	
	tmpl_set_current_block(tmpl, "row");
	
	tmpl_set_var(tmpl, "DESC", desc);
	
	sprintf(buf, "%ld", hits / days_passed);
	tmpl_set_var(tmpl, "AVG_HITS", buf);
	sprintf(buf, "%ld", pages / days_passed);
	tmpl_set_var(tmpl, "AVG_PAGES", buf);
	sprintf(buf, "%ld", files / days_passed);
	tmpl_set_var(tmpl, "AVG_FILES", buf);
	sprintf(buf, "%ld", visits / days_passed);
	tmpl_set_var(tmpl, "AVG_VISITS", buf);
	c = ' ';
	tr = traffic / days_passed;
	if (tr > 1024) {tr /= 1024; c = 'k';}
	if (tr > 1024) {tr /= 1024; c = 'M';}
	if (tr > 1024) {tr /= 1024; c = 'G';}
	sprintf(buf,"%.2f&nbsp;%cB", tr, c);
	tmpl_set_var(tmpl, "AVG_TRAFFIC", buf);
	
	sprintf(buf, "%ld", hits);
	tmpl_set_var(tmpl, "TOT_HITS", buf);
	sprintf(buf, "%ld", pages);
	tmpl_set_var(tmpl, "TOT_PAGES", buf);
	sprintf(buf, "%ld", files);
	tmpl_set_var(tmpl, "TOT_FILES", buf);
	sprintf(buf, "%ld", visits);
	tmpl_set_var(tmpl, "TOT_VISITS", buf);
	c = ' ';
	tr = traffic;
	if (tr > 1024) {tr /= 1024; c = 'k';}
	if (tr > 1024) {tr /= 1024; c = 'M';}
	if (tr > 1024) {tr /= 1024; c = 'G';}
	sprintf(buf,"%.2f&nbsp;%cB", tr, c);
	tmpl_set_var(tmpl, "TOT_TRAFFIC", buf);
	
	tmpl_parse_current_block(tmpl);
	
	return 0;
}

int mplugins_output_generate_history_output(mconfig *ext_conf, mlist *history, const char *subpath) {
	mlist *l = history;
	FILE *f;
	char filename[255];
	config_output *conf = ext_conf->plugin_conf;
	data_WebHistory hist, yearly;
	tmpl_main * tmpl;
	char *fn;
	int ret;
	char *s, *first_report;
	int i;
	
	char buf[255];
	struct tm *_tm;
	time_t t;
	
	tmpl_vars vars[] = {
		/* generated by this plugin */
		{ "LASTRECORD", NULL },
		{ "GENERATEDAT", NULL },
		
		/* hard values */
		{ "MLA_URL", "http://jan.kneschke.de/projects/modlogan/" },
		{ "MLA_PACKAGE", PACKAGE },
		{ "MLA_VERSION", VERSION },
		
		/* translated strings */
		{ "TXT_STATISTICS", _("Statistics for")},
		{ "TXT_LASTRECORD", _("Last record")},
		{ "TXT_GENERATEDAT", _("Generated at")},
		{ "TXT_HISTORY", _("History")},
		{ "TXT_HITS", _("Hits")},
		{ "TXT_PAGES", _("Pages")},
		{ "TXT_FILES", _("Files")},
		{ "TXT_VISITS", _("Visits")},
		{ "TXT_TRAFFIC", _("Traffic")},
		{ "TXT_MONTH", _("Month")},
		{ "TXT_AVERAGE_DAY", _("Average/Day")},
		{ "TXT_TOTALS", _("Totals")},
		{ "TXT_OUTPUTBY", _("Output generated by")},
		
		{ "TITLE", _("Statistics") },
		
		{ NULL,  NULL }
	};
	
	if (history == NULL) return -1;
	
	/* set the correct link */
	if (conf->menu && conf->menu->data && conf->menu->data->key) {
		first_report = conf->menu->data->key;
	} else {
		first_report = conf->reports->data->key;
	}
	
	/* suffix for the generated statistics page files */
	sprintf(filename, "%s%s%s/%s", 
		ext_conf->outputdir ? ext_conf->outputdir : ".",
		subpath ? "/" : "",
		subpath ? subpath : "",
		conf->index_filename);
	
	if (!(f = fopen(filename, "w"))) {
		return -1;
	}
	
	tmpl = tmpl_init();
	assert(tmpl);
	
	tmpl->debug_level = ext_conf->debug_level;
	
	if ((fn = generate_template_filename(ext_conf, M_TMPL_INDEX)) == NULL) {
		M_DEBUG0(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "can't generate filename of the template\n");
		tmpl_free(tmpl);
		return -1;
	}

	/* load and parse template */
	if ((ret = tmpl_load_template(tmpl, fn)) != 0) {
		M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "parsing template %s failed\n",
			 fn);
		tmpl_free(tmpl);
		free(fn);
		return ret;
	}
	free(fn);

	if (/*conf->show_monthly_graph*/ 1) {
		char * ref = create_pic_12_month(ext_conf, history, subpath);
	
		if (ref && *ref) {
			tmpl_set_var(tmpl, "IMAGE", ref);
		}
	}
	/* header vars */
	
#define HIST(x) \
	hist.x = 0;
	
	HIST(hits);
	HIST(files);
	HIST(pages);
	HIST(visits);
	HIST(xfersize);
	HIST(days_passed);
#undef HIST
	
#define HIST(x) \
	yearly.x = 0;
	
	HIST(year);
	HIST(hits);
	HIST(files);
	HIST(pages);
	HIST(visits);
	HIST(xfersize);
	HIST(days_passed);
#undef HIST
	
	/* go to the last element */
	while (l->next) l = l->next;
	
	while (l) {
		mdata *data = l->data;
		
		if (!data) break;
		
		if (data->data.webhist.days_passed != 0) {
			char *lnk;
			if (yearly.year > data->data.webhist.year) {
				sprintf(buf, "%04d",
					yearly.year
					);
				
				set_line(tmpl, 
					 buf,
					 yearly.hits,
					 yearly.files,
					 yearly.pages,
					 yearly.visits,
					 yearly.xfersize,
					 yearly.days_passed
					 );
			}
			
			lnk = generate_output_link(ext_conf, 
						   data->data.webhist.year, 
						   data->data.webhist.month,
						   first_report);
			
			sprintf(buf, "<a href=\"%s\">%s&nbsp;%04d</a>",
				lnk,
				get_month_string(data->data.webhist.month, 1),
				data->data.webhist.year
				);
			free(lnk);
			
			set_line(tmpl, 
				 buf,
				 data->data.webhist.hits,
				 data->data.webhist.files,
				 data->data.webhist.pages,
				 data->data.webhist.visits,
				 data->data.webhist.xfersize,
				 data->data.webhist.days_passed
				 );

			if (yearly.year > data->data.webhist.year) {
				
#define HIST(x) \
	yearly.x = data->data.webhist.x;
				HIST(year);
				HIST(hits);
				HIST(files);
				HIST(pages);
				HIST(visits);
				HIST(xfersize);
				HIST(days_passed);
#undef HIST
	
			} else {
#define HIST(x) \
	yearly.x += data->data.webhist.x
				yearly.year = data->data.webhist.year;
				HIST(hits);
				HIST(files);
				HIST(pages);
				HIST(visits);
				HIST(xfersize);
				HIST(days_passed);
#undef HIST
			}
#define HIST(x) \
	hist.x += data->data.webhist.x
			
			HIST(hits);
			HIST(files);
			HIST(pages);
			HIST(visits);
			HIST(xfersize);
			HIST(days_passed);
#undef HIST
		} else {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_OUTPUT, M_DEBUG_LEVEL_WARNINGS,
				 "count == 0, is this ok ?? splitby for '%s' without an entry ??\n",
				 data->key);
		}
		l = l->prev;
	}
	
	if (yearly.year && yearly.days_passed) {
		sprintf(buf, "%04d",
			yearly.year
			);
		
		set_line(tmpl, 
			 buf,
			 yearly.hits,
			 yearly.files,
			 yearly.pages,
			 yearly.visits,
			 yearly.xfersize,
			 yearly.days_passed
			 );
	}
	
	if (hist.days_passed) {
		set_line(tmpl, 
			 _("totals"),
			 hist.hits,
			 hist.files,
			 hist.pages,
			 hist.visits,
			 hist.xfersize,
			 hist.days_passed
			);
	}
	
	/* set external variables from config-file */
	for (l = conf->variables; l; l = l->next) {
		if (l->data) {
			char *k, *v;
			k = l->data->key;
			
			if (k) {
				if ((v = strchr(k, ',')) != NULL) {
					v++;
		
					/* check if the variable is used internally */
					for (i = 0; vars[i].key; i++) {
						if (0 == strncmp(k, vars[i].key, v-k-1)) 
							break;
					} 
					
					/* not found */
					if (vars[i].key == NULL) {
						char *key = malloc(v-k);
						strncpy(key, k, v-k-1);
						key[v-k-1] = '\0';
						tmpl_set_var(tmpl, key, v);
						free(key);
					}
				} else {
					M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
						 "no ',' found in %s\n", k);
				}
			}
		}
	}
	
	/* create current timestamp */
	t = time(NULL);
	_tm = localtime(&t);
	strftime(buf, sizeof(buf), "%X %x", _tm);
	tmpl_set_var(tmpl, "GENERATEDAT", buf);
	
	/* set template variables */
	for (i = 0; vars[i].key; i++) {
		if (i != 0 && i != 1 && /* ignore GENERATEDAT and LASTRECORD as they are already set */
		    vars[i].subst != NULL && 
		    0 != tmpl_set_var(tmpl, vars[i].key, vars[i].subst)) {
			M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_WARNINGS,
				 "substituing key %s failed\n",
				 vars[i].key);
		}
	}
	
	/* replace tags by the actual output */
	if ((s = tmpl_replace(tmpl))) {
		/* index-file */
		FILE *f;
		
		f = fopen(filename, "wb");
		fputs(s, f);
		fclose(f);
		
		free(s);
	} else {
		tmpl_free(tmpl);
		return -1;
	}
	
	tmpl_free(tmpl);
		
	return 0;
}
