/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <cmath>
using namespace std;

#include "common/SoundFile.h"
#include "common/Average.h"
#include "common/FFT.h"
#include "common/Exception.h"
#include "common/Hamming.h"

#include "AudioData.h"

int main(int argc, char **argv)
{
	if (argc < 4) {
		::fprintf(stderr, "Usage: AudioData <freq res> <sample rate> <filename>\n");
		return 1;
	}

	double    freqRes = ::atof(argv[1]);
	int    sampleRate = ::atoi(argv[2]);
	wxString fileName = wxString(argv[3]);

	try {
		CAudioData speed(freqRes, sampleRate, fileName);
		speed.run();
	}
	catch (CException& ex) {
		fprintf(stderr, "AudioData: %s\n", ex.getMessage().c_str());
		return 1;
	}

	return 0;
}

CAudioData::CAudioData(double freqRes, int sampleRate, const wxString& fileName) :
m_freqRes(freqRes),
m_sampleRate(sampleRate),
m_fileName(fileName)
{
}

CAudioData::~CAudioData()
{
}

void CAudioData::run()
{
	int fftLength = int(double(m_sampleRate) / m_freqRes + 0.5);
	int minBin    = (300 * fftLength) / m_sampleRate;
	int maxBin    = (2500 * fftLength) / m_sampleRate;

	CAverage amplitude;
	CAverage* binValues = new CAverage[maxBin];

	CFFT fft(fftLength);
	CHamming hamming(fftLength);

	CSoundDev* file = new CSoundFile;

	file->openRead(m_fileName, m_sampleRate, 16);

	double* in = new double[fftLength];
	double* out = new double[fftLength];

	for (;;) {
		int len = fftLength;
		if (!file->read(in, len))
			break;

		for (int i = 0; i < len; i++)
			amplitude.addValue(in[i] * in[i]);

		hamming.filter(in, out);
		double* bins = fft.process(out);

		for (int i = 0; i < maxBin; i++)
			binValues[i].addValue(bins[i]);
	}

	file->close();

	printf("Audio Amplitude: %d samples, min/ave/max: %.2f/%.2f/%.2f\n",
		amplitude.getCount(),
		::sqrt(amplitude.getMinimum()),
		::sqrt(amplitude.getAverage()),
		::sqrt(amplitude.getMaximum()));

	int highestBin    = 0;
	double highestVal = 0.0;
	int lowestBin     = 0;
	double lowestVal  = 999.0;

	for (int i = minBin; i < maxBin; i++) {
		if (binValues[i].getMaximum() > highestVal) {
			highestBin = i;
			highestVal = binValues[i].getMaximum();
		}

		if (binValues[i].getMinimum() < lowestVal) {
			lowestBin = i;
			lowestVal = binValues[i].getMinimum();
		}
	}

	double highestFreq = double(m_sampleRate * highestBin) / double(fftLength);
	double lowestFreq  = double(m_sampleRate * lowestBin) / double(fftLength);

	printf("Bin Values: highest: %d/%.1fHz/%.2f lowest: %d/%.1fHz/%.2f\n",
		highestBin, highestFreq, highestVal,
		lowestBin, lowestFreq, lowestVal);

	for (int i = minBin; i < highestBin; i++) {
		double val = binValues[i].getMaximum();

		for (double dB = -1.0; dB >= -7.0; dB -= 1.0) {
			if (nearTo(val, dB, highestVal)) {
				double freq = double(m_sampleRate * i) / double(fftLength);
				printf("%.0fdB at %d/%.1fHz\n", dB, i, freq);
				break;
			}
		}
	}

	printf("0dB at %d/%.1fHz\n", highestBin, highestFreq);

	for (int i = highestBin + 1; i < maxBin; i++) {
		double val = binValues[i].getMaximum();

		for (double dB = -1.0; dB >= -7.0; dB -= 1.0) {
			if (nearTo(val, dB, highestVal)) {
				double freq = double(m_sampleRate * i) / double(fftLength);
				printf("%.0fdB at %d/%.1fHz\n", dB, i, freq);
				break;
			}
		}
	}

	delete file;
	delete[] in;
	delete[] out;
	delete[] binValues;
}

bool CAudioData::nearTo(double val, double dB, double maxVal) const
{
	double ratio = ::pow(10.0, dB / 10.0);

	double wantedVal = maxVal * ratio;

	return val >= wantedVal;
}
