/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "JT44Preferences.h"
#include "JT44Frame.h"
#include "JT44App.h"

#include "jt44/JT44Defs.h"

#include <wx/tglbtn.h>

const int BORDER_SIZE = 5;

enum {
	Menu_File_Open = 100,
	Menu_File_Saving,
	Menu_File_Delete,
	Menu_File_Quit,
	Menu_Edit_Preferences,
	Menu_Help_About
};

BEGIN_DECLARE_EVENT_TYPES()
	DECLARE_EVENT_TYPE(CARD_LEVELS,  2561)
	DECLARE_EVENT_TYPE(FILE_LEVELS,  2562)
	DECLARE_EVENT_TYPE(CARD_MESSAGE, 2563)
	DECLARE_EVENT_TYPE(FILE_MESSAGE, 2564)
        DECLARE_EVENT_TYPE(ERROR_EVENT,  2565)
END_DECLARE_EVENT_TYPES()

DEFINE_EVENT_TYPE(CARD_LEVELS)
DEFINE_EVENT_TYPE(FILE_LEVELS)
DEFINE_EVENT_TYPE(CARD_MESSAGE)
DEFINE_EVENT_TYPE(FILE_MESSAGE)
DEFINE_EVENT_TYPE(ERROR_EVENT)

BEGIN_EVENT_TABLE(CJT44Frame, wxFrame)
	EVT_MENU(Menu_File_Open, CJT44Frame::onOpen)
	EVT_MENU(Menu_File_Saving, CJT44Frame::onSaving)
	EVT_MENU(Menu_File_Delete, CJT44Frame::onDelete)
	EVT_MENU(Menu_File_Quit, CJT44Frame::onQuit)
	EVT_MENU(Menu_Edit_Preferences, CJT44Frame::onPreferences)
	EVT_MENU(Menu_Help_About, CJT44Frame::onAbout)

	EVT_TOGGLEBUTTON(Button_Input_Record, CJT44Frame::onRecord)
	EVT_CHOICE(Choice_TX_Order, CJT44Frame::onTXOrder)
	EVT_CHECKBOX(Button_TX_Enable, CJT44Frame::onTXEnable)
	EVT_SPINCTRL(Spin_Minimum_Sync, CJT44Frame::onSyncLevel)
	EVT_BUTTON(Button_Remote, CJT44Frame::onUpdate)
	EVT_CHOICE(Choice_Band, CJT44Frame::onBand)

	EVT_TEXT(Text_Messages_Begin + 0, CJT44Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 1, CJT44Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 2, CJT44Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 3, CJT44Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 4, CJT44Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 5, CJT44Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 6, CJT44Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 7, CJT44Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 8, CJT44Frame::onText)
	EVT_TEXT(Text_Messages_Begin + 9, CJT44Frame::onText)

	EVT_RADIOBUTTON(Button_Messages_Begin + 0, CJT44Frame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 1, CJT44Frame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 2, CJT44Frame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 3, CJT44Frame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 4, CJT44Frame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 5, CJT44Frame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 6, CJT44Frame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 7, CJT44Frame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 8, CJT44Frame::onMessage)
	EVT_RADIOBUTTON(Button_Messages_Begin + 9, CJT44Frame::onMessage)

	EVT_CUSTOM(CARD_LEVELS,  -1, CJT44Frame::onCardLevels)
	EVT_CUSTOM(FILE_LEVELS,  -1, CJT44Frame::onFileLevels)
	EVT_CUSTOM(CARD_MESSAGE, -1, CJT44Frame::onCardMessage)
	EVT_CUSTOM(FILE_MESSAGE, -1, CJT44Frame::onFileMessage)
	EVT_CUSTOM(ERROR_EVENT,  -1, CJT44Frame::onError)
END_EVENT_TABLE()

CJT44Frame::CJT44Frame() :
wxFrame(NULL, -1, title),
m_cardDisplay(NULL),
m_fileDisplay(NULL),
m_sunMoonDisplay(NULL),
m_record(NULL),
m_transmit(NULL),
m_remote(NULL),
m_clock(NULL),
m_message(NULL),
m_settings(NULL),
m_messageNo(-1)
{
	SetMenuBar(createMenuBar());

	CreateStatusBar(2);

	// The top level sizer, graph, list box and then controls
	wxBoxSizer* mainSizer = new wxBoxSizer(wxVERTICAL);

	// Spectrum graphs
	wxNotebook* noteBook = new wxNotebook(this, -1);
	wxNotebookSizer* noteSizer = new wxNotebookSizer(noteBook);

	m_cardDisplay = new CJT44Display(noteBook);
	noteBook->AddPage(m_cardDisplay, wxT("Sound Card"), true);

	m_fileDisplay = new CJT44Display(noteBook);
	noteBook->AddPage(m_fileDisplay, wxT("WAV Files"), false);

	wxString callsign, locator;
	int frequency = ::wxGetApp().getBand();
	::wxGetApp().getPersonal(callsign, locator);
	m_sunMoonDisplay = new CJT44SunMoonDisplay(noteBook, -1, locator, frequency);
	noteBook->AddPage(m_sunMoonDisplay, wxT("Sun & Moon Tracking"), false);

	mainSizer->Add(noteSizer, 0, wxBOTTOM, BORDER_SIZE);

	// The three input sections, left two are control, right is messages
	wxBoxSizer* inputSizer = new wxBoxSizer(wxHORIZONTAL);

	// Handle the recording and transmit control boxes
	wxBoxSizer* leftSizer = new wxBoxSizer(wxVERTICAL);

	m_record = new CRecordControl(this);
	leftSizer->Add(m_record);

	bool txFirst = ::wxGetApp().getTXFirst();
	bool txEnable = ::wxGetApp().getTXEnable();
	m_transmit = new CTransmitControl(this, -1, txFirst, txEnable);
	leftSizer->Add(m_transmit);

	int minSync = ::wxGetApp().getMinSync();
	m_cardDisplay->setMinSync(minSync);
	m_fileDisplay->setMinSync(minSync);
	m_settings = new CJT44SettingsControl(this, -1, minSync);
	leftSizer->Add(m_settings);

	inputSizer->Add(leftSizer);

	wxBoxSizer* middleSizer = new wxBoxSizer(wxVERTICAL);

	m_remote = new CRemoteControl(this);
	middleSizer->Add(m_remote);

	m_clock = new CClockPanel(this, -1, wxDefaultPosition, wxSize(145, 80));
	middleSizer->Add(m_clock);

	inputSizer->Add(middleSizer);

	wxString* messages = new wxString[JT44_MAX_MESSAGES];
	for (int i = 0; i < JT44_MAX_MESSAGES; i++)
		messages[i] = ::wxGetApp().getReplacedMessage(i);
	m_message = new CMessageControl(this, -1, messages, JT44_MAX_MESSAGES, JT44_MESSAGE_LENGTH);
	inputSizer->Add(m_message);
	delete[] messages;

	m_messageNo = 0;
	wxString message = m_message->getMessage(m_messageNo);
	::wxGetApp().sendMessage(message);

	mainSizer->Add(inputSizer);

	SetAutoLayout(true);

	mainSizer->Fit(this);
	mainSizer->SetSizeHints(this);

	SetSizer(mainSizer);
}

CJT44Frame::~CJT44Frame()
{
}

wxMenuBar* CJT44Frame::createMenuBar()
{
	wxMenu* fileMenu = new wxMenu();
	fileMenu->Append(Menu_File_Open, wxT("&Open sound file\tCtrl-O"));
	fileMenu->AppendSeparator();
	fileMenu->Append(Menu_File_Saving, wxT("&Save received text\tCtrl-A"), wxT(""), true);
	fileMenu->Append(Menu_File_Delete, wxT("&Remove received text\tCtrl-R"));
	fileMenu->AppendSeparator();
	fileMenu->Append(Menu_File_Quit, wxT("E&xit\tAlt-X"));

	wxMenu* editMenu = new wxMenu();
	editMenu->Append(Menu_Edit_Preferences, wxT("&Preferences...\tCtrl-P"));

	wxMenu* helpMenu = new wxMenu();
	helpMenu->Append(Menu_Help_About, wxT("&About JT44\tF1"));

	wxMenuBar* menuBar = new wxMenuBar();
	menuBar->Append(fileMenu, wxT("&File"));
	menuBar->Append(editMenu, wxT("&Edit"));
	menuBar->Append(helpMenu, wxT("&Help"));

	return menuBar;
}

void CJT44Frame::showCardLevels(CJT44Levels* levels)
{
	wxASSERT(levels != NULL);

	CJT44LevelEvent event(levels, CARD_LEVELS);

	AddPendingEvent(event);
}

void CJT44Frame::showFileLevels(CJT44Levels* levels)
{
	wxASSERT(levels != NULL);

	CJT44LevelEvent event(levels, FILE_LEVELS);

	AddPendingEvent(event);
}

void CJT44Frame::showCardMessage(CJT44Message* message)
{
	wxASSERT(message != NULL);

	CJT44MessageEvent event(message, CARD_MESSAGE);

	AddPendingEvent(event);
}

void CJT44Frame::showFileMessage(CJT44Message* message)
{
	wxASSERT(message != NULL);

	CJT44MessageEvent event(message, FILE_MESSAGE);

	AddPendingEvent(event);
}

void CJT44Frame::error(const wxString& text)
{
	CErrorEvent event(text, ERROR_EVENT);

	AddPendingEvent(event);
}

void CJT44Frame::onOpen(const wxCommandEvent& event)
{
	static wxString s_dir;
	static wxString s_file;

	wxFileDialog dialog(this, wxT("Open sound file"), s_dir, s_file, wxT("WAV files (*.wav;*.WAV)|*.wav;*.WAV"));

	if (dialog.ShowModal() != wxID_OK)
		return;

	s_dir  = dialog.GetDirectory();
	s_file = dialog.GetFilename();
	wxString fileName = dialog.GetPath();

	wxGetApp().readFile(fileName);
}

void CJT44Frame::onSaving(const wxCommandEvent& event)
{
	bool saving = event.IsChecked();

	::wxGetApp().setSaving(saving);
}

void CJT44Frame::onDelete(const wxCommandEvent& event)
{
	::wxGetApp().deleteMessages();
}

void CJT44Frame::onQuit(const wxCommandEvent& event)
{
	Close(true);
}

void CJT44Frame::onPreferences(const wxCommandEvent& event)
{
	CJT44Preferences dialog(this);

	if (dialog.ShowModal() != wxID_OK)
		return;

	for (int i = 0; i < JT44_MAX_MESSAGES; i++)
		m_message->setMessage(i, ::wxGetApp().getReplacedMessage(i));
}

void CJT44Frame::onAbout(const wxCommandEvent& event)
{
	::wxMessageBox(title + wxT("\n\nWritten by Jonathan Naylor  HB9DRD\n\nLots of advice from Tomi Manninen  OH2BNS"), wxT("About JT44"), wxICON_INFORMATION);
}

void CJT44Frame::onRecord(const wxCommandEvent& event)
{
	bool recording = event.IsChecked();

	::wxGetApp().setRecording(recording);
}

void CJT44Frame::onTXOrder(const wxCommandEvent& event)
{
	bool txFirst = int(event.GetClientData()) == Transmit_First;

	::wxGetApp().setTXFirst(txFirst);
}

void CJT44Frame::onTXEnable(const wxCommandEvent& event)
{
	bool txEnabled = event.IsChecked();

	::wxGetApp().setTXEnable(txEnabled);
}

void CJT44Frame::onSyncLevel(const wxSpinEvent& event)
{
	wxASSERT(m_cardDisplay != NULL);
	wxASSERT(m_fileDisplay != NULL);

	int level = event.GetPosition();

	::wxGetApp().setMinSync(level);

	m_cardDisplay->setMinSync(level);
	m_fileDisplay->setMinSync(level);
}

void CJT44Frame::onUpdate(const wxCommandEvent& event)
{
        wxASSERT(m_remote != NULL);

	wxString callsign = m_remote->getCallsign();
	wxString report   = m_remote->getReport();

	::wxGetApp().setRemote(callsign, report);

	for (int i = 0; i < JT44_MAX_MESSAGES; i++)
		m_message->setMessage(i, ::wxGetApp().getReplacedMessage(i));
}

void CJT44Frame::onBand(const wxCommandEvent& event)
{
	wxASSERT(m_sunMoonDisplay != NULL);

	int band = m_sunMoonDisplay->getBand();

	::wxGetApp().setBand(band);
}

void CJT44Frame::onText(const wxCommandEvent& event)
{
	int messageNo = event.GetId() - Text_Messages_Begin;

	wxASSERT(messageNo >= 0 && messageNo < JT44_MAX_MESSAGES);

	if (messageNo == m_messageNo) {
		wxString message = m_message->getMessage(m_messageNo);
		::wxGetApp().sendMessage(message);
	}
}

void CJT44Frame::onMessage(const wxCommandEvent& event)
{
	m_messageNo = event.GetId() - Button_Messages_Begin;

	wxASSERT(m_messageNo >= 0 && m_messageNo < JT44_MAX_MESSAGES);

	wxString message = m_message->getMessage(m_messageNo);

	::wxGetApp().sendMessage(message);
}

void CJT44Frame::onCardLevels(const CJT44LevelEvent& event)
{
	wxASSERT(m_cardDisplay != NULL);

	CJT44Levels* levels = event.getLevels();

	int level = levels->getLevel();

	wxString text;
	switch (level) {
		case Audio_High:
			text = wxT("Sound card level is too high");
			break;
		case Audio_OK:
			text = wxT("Sound card level is OK");
			break;
		case Audio_Low:
			text = wxT("Sound card level is too low");
			break;
	}

	SetStatusText(text, 0);

	m_cardDisplay->showLevels(levels);

	delete levels;
}

void CJT44Frame::onFileLevels(const CJT44LevelEvent& event)
{
	wxASSERT(m_fileDisplay != NULL);

	CJT44Levels* levels = event.getLevels();

	int level = levels->getLevel();

	wxString text;
	switch (level) {
		case Audio_High:
			text = wxT("WAV file audio level is too high");
			break;
		case Audio_OK:
			text = wxT("WAV file audio level is OK");
			break;
		case Audio_Low:
			text = wxT("WAV file audio level is too low");
			break;
	}

	SetStatusText(text, 1);

	m_fileDisplay->showLevels(levels);

	delete levels;
}

void CJT44Frame::onCardMessage(const CJT44MessageEvent& event)
{
	wxASSERT(m_cardDisplay != NULL);

	CJT44Message* message = event.getMessage();

	m_cardDisplay->showMessage(message);
}

void CJT44Frame::onFileMessage(const CJT44MessageEvent& event)
{
	wxASSERT(m_fileDisplay != NULL);

	CJT44Message* message = event.getMessage();

	m_fileDisplay->showMessage(message);
}

void CJT44Frame::onError(const CErrorEvent& event)
{
	wxString text = event.getText();

	wxMessageDialog dialog(this, text, wxT("JT44 Error"), wxICON_ERROR);
	dialog.ShowModal();
}
