/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "SoundFile.h"
#include "Exception.h"

#include <wx/debug.h>
#include <wx/log.h>

typedef unsigned char uint8;
typedef signed short  sint16;

enum {
	WAV_READ,
	WAV_WRITE
};

CSoundFile::CSoundFile(const wxString& fileName, int sampleRate, int sampleWidth) :
CSoundDev(),
m_file(NULL),
m_fileName(fileName),
m_sampleRate(sampleRate),
m_sampleWidth(sampleWidth),
m_mode(-1)
{
}

CSoundFile::CSoundFile() :
CSoundDev(),
m_file(NULL),
m_fileName(wxEmptyString),
m_sampleRate(11025),
m_sampleWidth(16),
m_mode(-1)
{
}

CSoundFile::~CSoundFile()
{
}

void CSoundFile::openRead(const wxString& fileName, int sampleRate, int sampleWidth)
{
	m_fileName    = fileName;
	m_sampleRate  = sampleRate;
	m_sampleWidth = sampleWidth;

	openRead();
}

void CSoundFile::openRead()
{
	wxASSERT(m_file == NULL);
	wxASSERT(m_sampleRate > 0);
	wxASSERT(m_sampleWidth == 8 || m_sampleWidth == 16);

	SF_INFO info;
	info.format = 0;
	m_file = ::sf_open(m_fileName.mb_str(), SFM_READ, &info);

	if (m_file == NULL)
		throw CException(wxT("Cannot open the WAV file for reading"));

	if (info.samplerate != m_sampleRate) {
		wxString text;
		text.Printf(wxT("Invalid sample rate in the WAV file, found %d samples/s"), info.samplerate);
		throw CException(text);
	}

	if (info.channels != 1)
		throw CException(wxT("More than one channel in the WAV file"));

	m_mode = WAV_READ;
}

void CSoundFile::openWrite(const wxString& fileName, int sampleRate, int sampleWidth)
{
	m_fileName    = fileName;
	m_sampleRate  = sampleRate;
	m_sampleWidth = sampleWidth;

	openWrite();
}

void CSoundFile::openWrite()
{
	wxASSERT(m_file == NULL);
	wxASSERT(m_sampleRate > 0);
	wxASSERT(m_sampleWidth == 8 || m_sampleWidth == 16);

	SF_INFO info;
	info.samplerate = m_sampleRate;
	info.channels   = 1;
	info.format     = SF_FORMAT_WAV;

	if (m_sampleWidth == 8)
		info.format |= SF_FORMAT_PCM_U8;
	else
		info.format |= SF_FORMAT_PCM_16;

	int ret = ::sf_format_check(&info);

	if (!ret)
		throw CException(wxT("Mistake in setting up SF_INFO for write"));

	m_file = ::sf_open(m_fileName.mb_str(), SFM_WRITE, &info);

	if (m_file == NULL)
		throw CException(wxT("Cannot open the WAV file for writing"));

	m_mode = WAV_WRITE;
}

bool CSoundFile::read(double* sample, int& len)
{
	wxASSERT(m_file != NULL);
	wxASSERT(sample != NULL);
	wxASSERT(len > 0);

	len = ::sf_read_double(m_file, sample, len);

	if (len <= 0)
		return false;

	return true;
}

void CSoundFile::write(double* sample, int len, double volume)
{
	wxASSERT(m_file != NULL);
	wxASSERT(sample != NULL);
	wxASSERT(len > 0);

	for (int i = 0; i < len; i++)
		sample[i] *= volume;

	int n = ::sf_write_double(m_file, sample, len);

	if (n != len)
		throw CException(wxT("Error writing to the sound file"));
}

void CSoundFile::close()
{
	wxASSERT(m_file != NULL);

	::sf_close(m_file);
	m_file = NULL;
}
