/*
 *   Copyright (C) 2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "Hilbert.h"

#include <wx/log.h>
#include <wx/debug.h>

#include <cmath>
using namespace std;

const int FILTER_LEN = 37;
const int BUFFER_LEN = 1024;

/*
 * Create a band pass Hilbert transformer / filter with 6 dB corner
 * frequencies of 'f1' and 'f2'. (0 <= f1 < f2 <= 0.5)
 */
CHilbert::CHilbert(double f1, double f2) :
m_iFilter(NULL),
m_qFilter(NULL),
m_iBuffer(NULL),
m_qBuffer(NULL),
m_ptr(FILTER_LEN)
{
	m_iFilter = new double[FILTER_LEN];
	m_qFilter = new double[FILTER_LEN];
	m_iBuffer = new double[BUFFER_LEN];
	m_qBuffer = new double[BUFFER_LEN];

	for (int i = 0; i < FILTER_LEN; i++) {
		double t = double(i) - double(FILTER_LEN - 1) / 2.0;
		double h = double(i) * (1.0 / double(FILTER_LEN - 1));

		double x = (2.0 * f2 * sinc(2.0 * f2 * t) - 2.0 * f1 * sinc(2.0 * f1 * t)) * hamming(h);
		m_iFilter[i] = x;

		/*
		 * The actual filter code assumes the impulse response
		 * is in time reversed order. This will be anti-
		 * symmetric so the minus sign handles that for us.
		 */
		x = (2.0 * f2 * cosc(2.0 * f2 * t) - 2.0 * f1 * cosc(2.0 * f1 * t)) * hamming(h);
		m_qFilter[i] = -x;
	}
}

CHilbert::~CHilbert()
{
	delete[] m_iFilter;
	delete[] m_qFilter;
	delete[] m_iBuffer;
	delete[] m_qBuffer;
}

void CHilbert::transform(complex<double>* input, complex<double>* output, int length)
{
	wxASSERT(input != NULL);
	wxASSERT(output != NULL);
	wxASSERT(length > 0);

	for (int i = 0; i < length; i++) {
		m_iBuffer[m_ptr] = input[i].real();
		m_qBuffer[m_ptr] = input[i].imag();

		double real = mac(m_iBuffer, m_ptr - FILTER_LEN, m_iFilter, 0);
		double imag = mac(m_qBuffer, m_ptr - FILTER_LEN, m_qFilter, 0);

		m_ptr++;
		if (m_ptr == BUFFER_LEN) {
			int ptr = BUFFER_LEN - FILTER_LEN;

			for (int j = 0; j < FILTER_LEN; j++) {
				m_iBuffer[j] = m_iBuffer[j + ptr];
				m_qBuffer[j] = m_qBuffer[j + ptr];
			}

			m_ptr = FILTER_LEN;
		}

		output[i] = complex<double>(real, imag);
	}
}

void CHilbert::transform(double* input, complex<double>* output, int length)
{
	wxASSERT(input != NULL);
	wxASSERT(output != NULL);
	wxASSERT(length > 0);

	for (int i = 0; i < length; i++) {
		m_iBuffer[m_ptr] = input[i];
		m_qBuffer[m_ptr] = input[i];

		double real = mac(m_iBuffer, m_ptr - FILTER_LEN, m_iFilter, 0);
		double imag = mac(m_qBuffer, m_ptr - FILTER_LEN, m_qFilter, 0);

		m_ptr++;
		if (m_ptr == BUFFER_LEN) {
			int ptr = BUFFER_LEN - FILTER_LEN;

			for (int j = 0; j < FILTER_LEN; j++) {
				m_iBuffer[j] = m_iBuffer[j + ptr];
				m_qBuffer[j] = m_qBuffer[j + ptr];
			}

			m_ptr = FILTER_LEN;
		}

		output[i] = complex<double>(real, imag);
	}
}

double CHilbert::sinc(double x) const
{
	if (::fabs(x) < 1e-10)
		return 1.0;
	else
		return ::sin(M_PI * x) / (M_PI * x);
}

double CHilbert::cosc(double x) const
{
	if (::fabs(x) < 1e-10)
		return 0.0;
	else
		return (1.0 - ::cos(M_PI * x)) / (M_PI * x);
}

double CHilbert::hamming(double x) const
{
	return 0.54 - 0.46 * ::cos(2 * M_PI * x);
}

double CHilbert::mac(double* a, int n1, double* b, int n2) const
{
	wxASSERT(a != NULL);
	wxASSERT(b != NULL);

	double sum = 0.0;

	for (int i = 0; i < FILTER_LEN; i++)
		sum += a[i + n1] * b[i + n2];

	return sum;
}
