/*
 *   Copyright (C) 2002,2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "Controller.h"
#include "Exception.h"

#include <wx/datetime.h>
#include <wx/debug.h>
#include <wx/log.h>

CController::CController(CSoundDev* soundDev, CPTTPort* pttPort) :
wxThread(),
m_soundDev(soundDev),
m_pttPort(pttPort),
m_receive1(NULL),
m_receive2(NULL),
m_send(NULL),
m_message(wxEmptyString),
m_txFirst(true),
m_txEnable(false),
m_record(false),
m_mutex()
{
	wxASSERT(soundDev != NULL);
	wxASSERT(pttPort != NULL);
}

CController::~CController()
{
}

void CController::setTXEnable(bool enable)
{
	wxASSERT(m_send != NULL);

	if (!enable && m_txEnable) {
		if (!m_send->IsPaused())
			m_send->stop();
	}

	m_mutex.Lock();
	m_txEnable = enable;
	m_mutex.Unlock();
}

void CController::setTXFirst(bool txFirst)
{
	m_mutex.Lock();
	m_txFirst = txFirst;
	m_mutex.Unlock();
}

void CController::setRXRecording(bool record)
{
	m_mutex.Lock();
	m_record = record;
	m_mutex.Unlock();
}

void CController::sendMessage(const wxString& message)
{
	m_mutex.Lock();
	m_message = message;
	m_mutex.Unlock();
}

void* CController::Entry()
{
	::wxLogInfo(wxT("%ld: Controller thread started"), GetId());

	m_receive1 = createReceiveThread(m_soundDev);
	m_receive2 = createReceiveThread(m_soundDev);
	m_send     = createSendThread(m_soundDev, m_pttPort);

	int secs = waitForNextPeriod();

	while (!TestDestroy()) {
		m_mutex.Lock();
		wxString message = m_message;
		bool txEnable    = m_txEnable;
		bool txFirst     = m_txFirst;
		bool record      = m_record;
		setLocal();
		m_mutex.Unlock();

		CThread* thread;
		if (secs == 0 && txFirst && txEnable && !message.IsEmpty()) {
			m_send->setMessage(message);
			thread = m_send;
		} else if (secs == 30 && !txFirst && txEnable && !message.IsEmpty()) {
			m_send->setMessage(message);
			thread = m_send;
		} else if (secs == 0) {
			m_receive1->setRecord(record);
			thread = m_receive1;
		} else {
			m_receive2->setRecord(record);
			thread = m_receive2;
		}

		// Start the thread off and immediately post a Pause
		thread->Resume();
		thread->Pause();

		secs = waitForNextPeriod();

		if (TestDestroy()) {
			thread->stop();
			::wxLogInfo(wxT("%ld: Killing the running thread"), GetId());
			break;
		}

		if (!thread->IsPaused()) {
			thread->stop();
			::wxLogInfo(wxT("%ld: Stopping the running thread"), GetId());
		}
	}

	m_receive1->Delete();
	m_receive2->Delete();
	m_send->Delete();

	::wxLogInfo(wxT("%ld: Controller thread ended"), GetId());

	return NULL;
}

int CController::waitForNextPeriod()
{
	wxDateTime now = wxDateTime::Now();

	int secs = now.GetSecond();

	if (secs < 30) {
		waitTime(30);
		return 30;
	} else {
		waitTime(0);
		return 0;
	}
}

void CController::waitTime(int seconds)
{
	wxASSERT(seconds >= 0 && seconds <= 59);

	wxDateTime now = wxDateTime::Now();

	while (!TestDestroy() && now.GetSecond() != seconds) {
		Sleep(250);

		now = wxDateTime::Now();
	}
}

CReceive* CController::getReceive1()
{
	return m_receive1;
}

CReceive* CController::getReceive2()
{
	return m_receive2;
}

CSend* CController::getSend()
{
	return m_send;
}

wxMutex& CController::getMutex()
{
	return m_mutex;
}

void CController::setLocal()
{
}
