/*
 * $Id: kl_dump_4_1.c,v 1.2 2005/02/23 01:09:12 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides functions for compatibility with lkcd 4.1 
 * dump versions (version <= 7).
 *
 * Created by IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * Copyright (C) 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright (C) 2004 - 2005 Silicon Graphics, Inc. All rights reserved.
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <klib.h>
#include <stdint.h>
#include <sys/time.h>

/* Definition of the old dump header structures:
 */

/*  S390:
 */

typedef struct _dump_header_s390_s {
        uint64_t             dh_magic_number;
        uint32_t             dh_version;
        uint32_t             dh_header_size;
        uint32_t             dh_dump_level;
        uint32_t             dh_dump_page_size;
        uint64_t             dh_memory_size;
        uint64_t             dh_memory_start;
        uint64_t             dh_memory_end;
#if DUMP_DEBUG >= 6
        uint64_t             dh_num_bytes;
#endif
        uint32_t             dh_num_dump_pages;
        char                 dh_panic_string[KL_DUMP_PANIC_LEN];
        struct {uint32_t tv_sec;
                uint32_t tv_usec;
        } dh_time; /* the time of the system crash */
        char                 dh_utsname_sysname[65];
        char                 dh_utsname_nodename[65];
        char                 dh_utsname_release[65];
        char                 dh_utsname_version[65];
        char                 dh_utsname_machine[65];
        char                 dh_utsname_domainname[65];
	char                 pad1[2]; 
        uint32_t             dh_current_task;
        uint32_t             dh_dump_compress;
        uint32_t             dh_dump_flags;
        uint32_t             dh_dump_device;
} __attribute__((packed)) kl_dh_reg_s390_t;

/* I386:
 */

typedef struct _dump_header_i386_s {
        uint64_t             dh_magic_number;
        uint32_t             dh_version;
        uint32_t             dh_header_size;
        uint32_t             dh_dump_level;
        uint32_t             dh_dump_page_size;
        uint64_t             dh_memory_size;
        uint64_t             dh_memory_start;
        uint64_t             dh_memory_end;
#if DUMP_DEBUG >= 6
        uint64_t             dh_num_bytes;
#endif
        uint32_t             dh_num_dump_pages;
        char                 dh_panic_string[KL_DUMP_PANIC_LEN];
        struct {uint32_t tv_sec;
                uint32_t tv_usec;
        } dh_time; /* the time of the system crash */
        char                 dh_utsname_sysname[65];
        char                 dh_utsname_nodename[65];
        char                 dh_utsname_release[65];
        char                 dh_utsname_version[65];
        char                 dh_utsname_machine[65];
        char                 dh_utsname_domainname[65];
        char                 pad1[2];
        uint32_t             dh_current_task;
        uint32_t             dh_dump_compress;
        uint32_t             dh_dump_flags;
        uint32_t             dh_dump_device;
} __attribute__((packed)) kl_dh_reg_i386_t;

/* S390X:
 */

typedef struct _dump_header_s390x_s {
        uint64_t             dh_magic_number;
        uint32_t             dh_version;
        uint32_t             dh_header_size;
        uint32_t             dh_dump_level;
        uint32_t             dh_dump_page_size;
        uint64_t             dh_memory_size;
        uint64_t             dh_memory_start;
        uint64_t             dh_memory_end;
#if DUMP_DEBUG >= 6
        uint64_t             dh_num_bytes;
#endif
        uint32_t             dh_num_dump_pages;
        char                 dh_panic_string[KL_DUMP_PANIC_LEN];
        char                 pad0[4];
        struct {uint64_t tv_sec;
                uint64_t tv_usec;
        } dh_time; /* the time of the system crash */
        char                 dh_utsname_sysname[65];
        char                 dh_utsname_nodename[65];
        char                 dh_utsname_release[65];
        char                 dh_utsname_version[65];
        char                 dh_utsname_machine[65];
        char                 dh_utsname_domainname[65];
        char                 pad1[2];
        uint64_t             dh_current_task;
        uint32_t             dh_dump_compress;
        uint32_t             dh_dump_flags;
        uint32_t             dh_dump_device;
        char                 pad2[4];
} __attribute__((packed)) kl_dh_reg_s390x_t;

typedef kl_dh_reg_s390x_t kl_dh_reg_ia64_t;
typedef kl_dh_reg_s390x_t kl_dh_reg_alpha_t;

/* __convert_4_1_s390_dump_header_to_new()
 */

int __convert_4_1_s390_dump_header_to_new(kl_dh_reg_s390_t* old, kl_dump_header_t* new)
{
	new->magic_number  = old->dh_magic_number;
	new->version       = old->dh_version;
	new->header_size   = old->dh_header_size;
	new->dump_level    = old->dh_dump_level;
	new->page_size     = old->dh_dump_page_size;
	new->memory_size   = old->dh_memory_size;
	new->memory_start  = old->dh_memory_start;
	new->memory_end    = old->dh_memory_end;
	new->num_dump_pages   = old->dh_num_dump_pages;
	memcpy(new->panic_string,old->dh_panic_string,KL_DUMP_PANIC_LEN);
	new->time.tv_sec   = old->dh_time.tv_sec;
	new->time.tv_usec  = old->dh_time.tv_usec;
	memcpy(new->utsname_sysname,old->dh_utsname_sysname,KL_UTS_LEN);
	memcpy(new->utsname_nodename,old->dh_utsname_nodename,KL_UTS_LEN);
	memcpy(new->utsname_release,old->dh_utsname_release,KL_UTS_LEN);
	memcpy(new->utsname_version,old->dh_utsname_version,KL_UTS_LEN);
	memcpy(new->utsname_machine,old->dh_utsname_machine,KL_UTS_LEN);
	memcpy(new->utsname_domainname,old->dh_utsname_domainname,KL_UTS_LEN);
	new->current_task  = (uint32_t)old->dh_current_task;
	new->dump_compress = old->dh_dump_compress;
	new->dump_flags    = old->dh_dump_flags;
	new->dump_device   = old->dh_dump_device;
	return(0);
}

/* __convert_4_1_s390x_dump_header_to_new()
 */

int __convert_4_1_s390x_dump_header_to_new(kl_dh_reg_s390x_t* old, kl_dump_header_t* new)
{
        new->magic_number  = old->dh_magic_number;
        new->version       = old->dh_version;
        new->header_size   = old->dh_header_size;
        new->dump_level    = old->dh_dump_level;
        new->page_size     = old->dh_dump_page_size;
        new->memory_size   = old->dh_memory_size;
        new->memory_start  = old->dh_memory_start;
        new->memory_end    = old->dh_memory_end;
        new->num_dump_pages     = old->dh_num_dump_pages;
        memcpy(new->panic_string,old->dh_panic_string,KL_DUMP_PANIC_LEN);
        new->time.tv_sec   = old->dh_time.tv_sec;
        new->time.tv_usec  = old->dh_time.tv_usec;
        memcpy(new->utsname_sysname,old->dh_utsname_sysname,KL_UTS_LEN);
        memcpy(new->utsname_nodename,old->dh_utsname_nodename,KL_UTS_LEN);
        memcpy(new->utsname_release,old->dh_utsname_release,KL_UTS_LEN);
        memcpy(new->utsname_version,old->dh_utsname_version,KL_UTS_LEN);
        memcpy(new->utsname_machine,old->dh_utsname_machine,KL_UTS_LEN);
        memcpy(new->utsname_domainname,old->dh_utsname_domainname,KL_UTS_LEN);
        new->current_task  = (uint64_t)old->dh_current_task;
        new->dump_compress = old->dh_dump_compress;
        new->dump_flags    = old->dh_dump_flags;
        new->dump_device   = old->dh_dump_device;
        return(0);
}

/* __convert_4_1_i386_dump_header_to_new()
 */

int __convert_4_1_i386_dump_header_to_new(kl_dh_reg_i386_t* old, kl_dump_header_t* new)
{
        new->magic_number  = old->dh_magic_number;
        new->version       = old->dh_version;
        new->header_size   = old->dh_header_size;
        new->dump_level    = old->dh_dump_level;
        new->page_size     = old->dh_dump_page_size;
        new->memory_size   = old->dh_memory_size;
        new->memory_start  = old->dh_memory_start;
        new->memory_end    = old->dh_memory_end;
        new->num_dump_pages = old->dh_num_dump_pages;
        memcpy(new->panic_string,old->dh_panic_string,KL_DUMP_PANIC_LEN);
        new->time.tv_sec   = old->dh_time.tv_sec;
        new->time.tv_usec  = old->dh_time.tv_usec;
        memcpy(new->utsname_sysname,old->dh_utsname_sysname,KL_UTS_LEN);
        memcpy(new->utsname_nodename,old->dh_utsname_nodename,KL_UTS_LEN);
        memcpy(new->utsname_release,old->dh_utsname_release,KL_UTS_LEN);
        memcpy(new->utsname_version,old->dh_utsname_version,KL_UTS_LEN);
        memcpy(new->utsname_machine,old->dh_utsname_machine,KL_UTS_LEN);
        memcpy(new->utsname_domainname,old->dh_utsname_domainname,KL_UTS_LEN);
        new->current_task  = (uint64_t)old->dh_current_task;
        new->dump_compress = old->dh_dump_compress;
        new->dump_flags    = old->dh_dump_flags;
        new->dump_device   = old->dh_dump_device;
        return(0);
}

/* 
 * kl_dump_arch_4_1()
 */
int 
kl_dump_arch_4_1(void *dh)
{
	int dumparch = KL_ARCH_UNKNOWN;

	if(strncmp( ((kl_dh_reg_s390_t*)dh)->dh_utsname_machine, 
			KL_ARCH_STR_S390, strlen(KL_ARCH_STR_S390)) == 0){
		dumparch = KL_ARCH_S390;
	} else if(strncmp( ((kl_dh_reg_s390x_t*)dh)->dh_utsname_machine, 
			KL_ARCH_STR_S390X, strlen(KL_ARCH_STR_S390X)) == 0){
		dumparch = KL_ARCH_S390X;
	} else if( (((kl_dh_reg_i386_t*)dh)->dh_utsname_machine[0] == 'i') &&
		   (((kl_dh_reg_i386_t*)dh)->dh_utsname_machine[2] == '8') &&
		   (((kl_dh_reg_i386_t*)dh)->dh_utsname_machine[3] == '6')){
                dumparch = KL_ARCH_I386;
	}
	return (dumparch);
}

/*
 * get_dump_header_4_1()
 */
kl_dump_header_t *
get_dump_header_4_1(void *rdh)
{
	int dump_arch, err = 0;
	kl_dump_header_t *dh;

	if (!rdh) {
		return((kl_dump_header_t *)NULL);
	}
	dh = (kl_dump_header_t *)malloc(sizeof(kl_dump_header_t));

	dump_arch = kl_dump_arch_4_1(rdh);
        switch(dump_arch){
#ifdef DUMP_ARCH_ALPHA
		case KL_ARCH_ALPHA:
			err = 1;
			break;
#endif
#ifdef DUMP_ARCH_I386
		case KL_ARCH_I386: {
			kl_dh_reg_i386_t old_dh;

			__convert_4_1_i386_dump_header_to_new(&old_dh,dh);
			fprintf(KL_ERRORFP,"\n\tWARNING: This is an old dump "
				"format - asm header will not be converted\n");
			break;
		}
#endif
#ifdef DUMP_ARCH_IA64
		case KL_ARCH_IA64:
			err = 1;
			break;
#endif
#ifdef DUMP_ARCH_S390
		case KL_ARCH_S390: {
			kl_dh_reg_s390_t old_dh;

			__convert_4_1_s390_dump_header_to_new(&old_dh,dh);
                	break;
		}
#endif
#ifdef DUMP_ARCH_S390X
		case KL_ARCH_S390X: {
			kl_dh_reg_s390x_t old_dh;

			__convert_4_1_s390x_dump_header_to_new(&old_dh,dh);
			break;
		}
#endif
		default:
			err = 1;
        }
	if (err) {
		free(dh);
		return((kl_dump_header_t *)NULL);
	}
	return(dh);
}
