/*
 * $Id: ia64_cpu.c,v 1.1 2004/12/21 23:26:20 tjm Exp $
 *
 * This file is part of libhwconfig.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 *
 * Copyright (C) 2003-2004 Silicon Graphics, Inc. All rights reserved.
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */
#include <klib.h>
#include <kl_cpu.h>

int cpu_count = 0;
cpuinfo_t *cpu_list = (cpuinfo_t *)NULL;

int CPUINFO_IA64_SZ;

#define MAX_CPU_COUNT	512

/*
 * get_ia64_cpuinfo()
 */
static cpuinfo_t *
get_ia64_cpuinfo(kaddr_t addr, int flg)
{
	cpuinfo_t *cpu;
	void *ptr;
	
	cpu = (cpuinfo_t *)kl_alloc_block(sizeof(cpuinfo_t), flg);
	if (!cpu) {
		return((cpuinfo_t *)NULL);
	}
	ptr = kl_alloc_block(CPUINFO_IA64_SZ, flg);
	GET_BLOCK(addr, CPUINFO_IA64_SZ, ptr);
	if (KL_ERROR) {
		kl_free_block(cpu);
		return((cpuinfo_t *)NULL);
	}
	cpu->addr = addr;
	cpu->ptr = ptr;
	return(cpu);
}

static int
cpu_online(int64_t *map, int map_size, int id)
{
	int idx;

	idx = (id/64);
	if (idx >= map_size) {
		return(0);
	}
	return((map[idx] >> id) & 1);
}

/*
 * init_ia64_cpuinfo()
 */
int
init_ia64_cpuinfo(void)
{
	int i, cpu_map_size;
	syment_t *sp;
	kaddr_t addr, per_cpu_offset;
	int64_t offset_val, cpu_map[MAX_CPU_COUNT/64];
	void *ptr;
	cpuinfo_t *cpu;

	if (cpu_list) {
		/* We have already gathered up the cpu info. Return with
		 * no error.
		 */
		return(0);
	}
	CPUINFO_IA64_SZ = kl_struct_len("cpuinfo_ia64");

	/* Get the offset to the cpuinfo_ia64 struct...note
	 * that the "address" for this symbol is not really and
	 * address. Rather, it is an offset in the __per_cpu
	 * area to the struct we are interested in.
	 */
	if (!(sp = kl_lkup_symname("per_cpu__cpu_info"))) {
		return(1);
	}
	offset_val = (int64_t)sp->s_addr;

	if (!(sp = kl_lkup_symname("__per_cpu_offset"))) {
		return(1);
	}
	per_cpu_offset = sp->s_addr;

	if (!(sp = kl_lkup_symname("cpu_online_map"))) {
		return(1);
	}
	cpu_map_size = MAX_CPU_COUNT/64;
	if (kl_get_block(sp->s_addr, (cpu_map_size*8), (void*)&cpu_map, 0)) {
		return(1);
	}

	/* Walk through the cpuinfo table and get the pointer for the
	 * info struct for each possible cpu in the system.
	 */
	for (i = 0; i < MAX_CPU_COUNT; i++) {

		addr = KL_VREAD_PTR((per_cpu_offset + (i * KL_NBPW)));
		if (addr == 0) {
			return(1);
		}

		/* adjust for the cpu offset value...
		 */
		addr += offset_val;

		/* Get the cpu entry
		 */
		if (!(cpu = get_ia64_cpuinfo(addr, K_PERM))) {
			return(1);
		}
		
		if (!cpu_online(cpu_map, cpu_map_size, i)) {
			cpu->id = -1;	
		} else {
			cpu->id = KL_UINT(cpu->ptr, "cpuinfo_ia64", "cpu");
			ptr = (void*)(unsigned long)K_PTR(cpu->ptr, 
				"cpuinfo_ia64", "node_data");
			cpu->node = KL_UINT(cpu->ptr, "ia64_node_data", "node");
			cpu_count++;
		}
		if (cpu_list) {
			list_add(&cpu_list->list, &cpu->list);
		} else {
			cpu_list = cpu;
		}
	}
	return(0);
}
