/*
 * Copyright (C) 1999-2003 Hewlett-Packard Co
 *      David Mosberger-Tang <davidm@hpl.hp.com>
 *
 * Copyright (C) 2004 - 2005 Silicon Graphics, Inc. All rights reserved.
 */

/* This file has been adapted from kernel source module 
 *
 *   arch/ia64/kernel/unwind.c 
 * 
 * It provides call frame unwind support for the Linux kernel for
 * a user level application such as lcrash.
 *
 * Parsing and processing the unwind information is time-consuming, 
 * so this implementation translates the unwind descriptors into unwind 
 * scripts.  These scripts are very simple (basically a sequence of 
 * assignments) and efficient to execute.  They are cached for later 
 * re-use.  Each script is specific for a given instruction pointer 
 * address and the set of predicate values that the script depends on 
 * (most unwind descriptors are unconditional and scripts often do 
 * not depend on predicates at all).  This code is based on the unwind 
 * conventions described in the "IA-64 Software Conventions and Runtime 
 * Architecture" manual.
 *
 */
#include <klib.h>
#include "lc_unwind.h"
#include "unwind_i.h"

#define UNW_LOG_CACHE_SIZE      7 /* each unw_script is ~256 bytes in size */
#define UNW_CACHE_SIZE          (1 << UNW_LOG_CACHE_SIZE)

#define UNW_LOG_HASH_SIZE       (UNW_LOG_CACHE_SIZE + 1)
#define UNW_HASH_SIZE           (1 << UNW_LOG_HASH_SIZE)

#define p5              5

/*
 * Preserved registers that are shared between code in ivt.S and entry.S.
 * borrowed from arch/ia64/entry.h
 */
#define pNonSys    p5      /* from arch/ia64/entry.h */

int unw_debug_level = -1; /* set to -1 for no debug */

#define alloc_reg_state() \
	kl_alloc_block(sizeof(struct unw_state_record), K_PERM)
#define free_reg_state(usr)     kl_free_block(usr)
#define alloc_labeled_state() \
	kl_alloc_block(sizeof(struct unw_labeled_state), K_PERM)
#define free_labeled_state(usr) kl_free_block(usr)

/*
 * Main kernel unwind data struct which is defined explicitly within kernel
 * source. We have to declare it here because we do not have access to the
 * type information for this structure.  Note that we only use the part
 * of the structure from kernel memory that relates to the unwind tables. 
 * We reinitialize everything else. The structure also does not contain 
 * the UMW_DEBUG and UNW_STATS portions defined in the kernel as there is 
 * no way to know if they exist or not (they are kernel dependent) and
 * we aren't going to be using them.
 */
typedef struct unw_s {         
        uint32_t lock;                      /* spinlock for unwind data */

	uint32_t dummy;	/* We need this so 32-bit and 64-bit line up */

        /* list of unwind tables (one per load-module) */
        uint64_t tables; /* ptr to unw_table list */

        uint64_t r0;          /* constant 0 for r0 */

        /* table of registers that prologues can save (and order in which 
	 * they're saved): 
	 */
        unsigned char save_order[8];

        /* maps a preserved register index (preg_index) to corresponding 
	 * switch_stack offset: 
	 */
        uint16_t sw_off[sizeof(struct unw_frame_info) / 8];

        uint16_t lru_head;   /* index of lead-recently used script */
        uint16_t lru_tail;   /* index of most-recently used script */

        /* index into unw_frame_info for preserved register i */
        uint16_t preg_index[UNW_NUM_REGS];

        int16_t pt_regs_offsets[32];

        /* unwind table for the kernel: */
        struct unw_table kernel_table;

        /* unwind table describing the gate page (kernel code that is 
	 * mapped into user space): 
	 */
        uint64_t gate_table_size;
        uint64_t gate_table;

        /* hash table that maps instruction pointer to script index: */
        uint16_t hash[UNW_HASH_SIZE];

        /* script cache: */                                                                               
        struct unw_script cache[UNW_CACHE_SIZE]; 

	/* Rest of structure definition from kernel deleted... */

} unw_t;

/*
 * The struct definition for unw_s is slightly different for certain older
 * dumps. In order for the unwind code to work there needs to be a consistent
 * layout (since we are reading it the struct from kernel memory). In order
 * to ensuer this is the case, we have a separate definition for the older
 * version. If needed, we copy the old struct to the new format, field by
 * field.
 */
typedef struct unw24_s {

	/* place holder for kernel unwind data spinlock */
	unsigned int lock;

	/* list of unwind tables (one per load-module) */
	uint64_t tables;

	/* table of registers that prologues can save (and order in
	 * which they're saved): */
	const unsigned char save_order[8];

	/* maps a preserved register index (preg_index) to corresponding
	 * switch_stack offset: */
	unsigned short sw_off[sizeof(struct unw_frame_info) / 8];

	unsigned short lru_head; /* index of lead-recently used script */
	unsigned short lru_tail; /* index of most-recently used script */

	/* index into unw_frame_info for preserved register i */
	unsigned short preg_index[UNW_NUM_REGS];

	/* unwind table for the kernel: */
	struct unw_table kernel_table;

	/* unwind table describing the gate page (kernel code that is
	 * mapped into user space): */
	size_t gate_table_size;
	uint64_t gate_table;

	/* hash table that maps instruction pointer to script index: */
	unsigned short hash[UNW_HASH_SIZE];

	/* script cache: */
	struct unw_script cache[UNW_CACHE_SIZE];

	/* Rest of structure definition from kernel deleted... */

} unw24_t;

unw_t *unw = (unw_t *)NULL;
unw24_t *unw24 = (unw24_t *)NULL;
uint64_t unw_addr = 0;

/****
 **** Utility functions                          
 ****/

/*
 * get_loc_value()
 */
static uint64_t
get_loc_value(uint64_t ptr)
{
	uint64_t val;

	/* Check to see if this is a kernel address or an
	 * application address.
	 */
	if (kl_is_valid_kaddr(ptr, 0, 0)) {
		val = (uint64_t)KL_VREAD_UINT64(ptr);
	} else {
#if __WORDSIZE == 32
		uint32_t ptr32 = (uint32_t)ptr;
		val = *(uint64_t*)ptr32;
#else
		val = *(uint64_t*)ptr;
#endif
	}
	return(val);
}

#define LOC_VAL(v)      ((uint64_t*)(v))
#define GET_LOC_VAL(p)  get_loc_value((uint64_t)p)

/****
 **** Access KDB running_process cache data
 ****/

void *kdb_running_process = NULL;
int smp_num_cpus = -1;
kaddr_t CPU_DATA = 0;

/*
 * cpu_data()
 */
static uint64_t
cpu_data(int cpuid)
{
	uint64_t cpudata;

	cpudata = KL_VREAD_UINT64(CPU_DATA + (cpuid * KL_NBPW));
	return(cpudata);
}

/*
 * kdb_cpu_init()
 */
static int
kdb_cpu_init(void)
{
	syment_t *sp = (syment_t *)NULL;

	/* See if we need to initialize the array of kdb cpu cache
	 * information.
	 */
	if (!kdb_running_process) {
		if (smp_num_cpus == -1) {
			if (!(sp = kl_lkup_symname("smp_num_cpus"))) {
				return(1);
			}
			GET_BLOCK(sp->s_addr, 4, &smp_num_cpus);
		}
		if (!(sp = kl_lkup_symname("kdb_running_process"))) {
			return(1);
		}
		kdb_running_process = kl_alloc_block(
			(smp_num_cpus * KDB_RUNNING_PROCESS_SZ), K_PERM);
		GET_BLOCK(sp->s_addr, (smp_num_cpus * KDB_RUNNING_PROCESS_SZ),
			kdb_running_process);
	}
	return(0);
}

/*
 * kdb_cpu_ptr()
 */
void *
kdb_cpu_ptr(uint64_t task)
{
	int i;
	void *ptr = NULL;
	uint64_t t;

	/* Make sure this is not a live system or a core dump generated
	 * via the live_dump command.
	 */
	if (CORE_IS_KMEM) {
		return((void *)NULL);
	}

	kdb_cpu_init();

	/* Now see if the process is running on a cpu. If it is, then
	 * return the index to the kdb_running_process entry.
	 */
	ptr = kdb_running_process;
	for (i = 0; i < smp_num_cpus; i++) {
		t = *(uint64_t*)ptr;
		if (t == task) {
			return(ptr);
		}
		ptr = NEXT_KDB_CPU(ptr);
	}
	return((void *)NULL);
}

/*
 * kdb_cpu_id()
 */
static int
kdb_cpu_id(uint64_t task)
{
	int i;
	void *ptr = NULL;
	uint64_t t;

	/* Make sure this is not a live system or a core dump generated
	 * via the live_dump command.
	 */
	if (CORE_IS_KMEM) {
		return(-1);
	}

	kdb_cpu_init();

	/* Now see if the process is running on a cpu. If it is, then
	 * return the index to the kdb_running_process entry.
	 */
	ptr = kdb_running_process;
	for (i = 0; i < smp_num_cpus; i++) {
		t = *(uint64_t *)ptr;
		if (t == task) {
			return(i);
		}
		ptr = NEXT_KDB_CPU(ptr);
	}
	return(-1);
}

/****
 **** Routines for accessing kernel unwind data
 ****/

/*
 * get_unw_table()
 *
 */
int
get_unw_table(uint64_t addr, struct unw_table *tbl)
{
	GET_BLOCK(addr, sizeof(*tbl), tbl);
	return (KL_ERROR);
}

/*
 * get_unw_table_entry()
 *
 */
int
get_unw_table_entry(uint64_t addr, struct unw_table_entry *e)
{
	GET_BLOCK(addr, sizeof(*e), e);
	return (KL_ERROR);
}

/*
 * lookup_unw_table()
 */
struct unw_table *
lookup_unw_table(uint64_t ip)
{
	uint64_t taddr;
	struct unw_table *table;

	taddr = (uint64_t)unw->tables;
	table = (struct unw_table *)
		kl_alloc_block(sizeof(struct unw_table), K_TEMP);

	while (taddr) {
		if (get_unw_table(taddr, table)) {
			kl_free_block(table);
			table = NULL;
			break;
		}
		if ((ip >= table->start) && (ip < table->end)) {
			break;
		}
		taddr = (uint64_t)table->next;
	}
	return(table);
}

/*
 * lookup_unw_table_entry()
 */
struct unw_table_entry *
lookup_unw_table_entry(uint64_t ip)
{
	uint64_t lo, hi, mid, ip_off;
	struct unw_table_entry *e;
	struct unw_table *table;
	uint64_t eaddr;

	if (!(table = lookup_unw_table(ip))) {
		return(NULL);
	}

	/* do a binary search for the right entry
	 */
	e = (struct unw_table_entry *)
		kl_alloc_block(sizeof(struct unw_table_entry), K_TEMP);
	ip_off = (ip - table->segment_base);
	for (lo = 0, hi = table->length; lo < hi; ) {
		mid = (lo + hi) / 2;
		eaddr = table->array + 
			((sizeof(struct unw_table_entry) * mid));
		if (get_unw_table_entry(eaddr, e)) {
			kl_free_block((void*)e);
			return(NULL);
		}
		if (ip_off < e->start_offset) {
			hi = mid;
		} else if (ip_off >= e->end_offset) {
			lo = mid + 1;
		} else {
			break;
		}
	}
	if ((ip_off < e->start_offset) || (ip_off > e->end_offset)) {
		kl_free_block(e);
		return(NULL);
	}
	return(e);
}

#ifdef UNW_DEBUG
void
walk_unw_tables(void)
{
	int i;
	uint64_t taddr, eaddr;
	char name[256];
	struct unw_table tbl;
	struct unw_table_entry e;

	if (!unw) {
		fprintf(stderr, "unw struct not initialized!\n");
		return;
	}
	taddr = (uint64_t)unw->tables;

	while (taddr) {
		if (get_unw_table(taddr, &tbl)) {
			fprintf(stderr,
				"Error reading unw_table from 0x%"FMTPTR"x\n", taddr);
			break;
		}
		GET_BLOCK((uint64_t)tbl.name, 256, name);
		fprintf(stdout, "0x%"FMTPTR"x: Name=%s, Start=0x%"FMTPTR"x, "
			"End=0x%"FMTPTR"x, gp=0x%"FMTPTR"x\n",
			taddr, name, tbl.start, tbl.end, tbl.gp);
		for (i = 0; i < tbl.length; i++) {
			eaddr = (uint64_t)
				((uint64_t)tbl.array + (i * sizeof(e)));
			if (get_unw_table_entry(eaddr, &e)) {
				fprintf(stderr, "Error reading "
					"unw_table_entry from 0x%"FMTPTR"x\n", eaddr);
				break;
			}
			fprintf(stdout, "%5d: start_offset=0x%"FMTPTR"x, "
				"end_offset=0x%"FMTPTR"x, info_offset=0x%"FMTPTR"x\n", i,
				e.start_offset, e.end_offset, e.info_offset);
		}
		taddr = (uint64_t)tbl.next;
	}
}
#endif

/**
 ** unwind support functions
 **/

/*
 * rse_slot_num()
 */
static int64_t
rse_slot_num(uint64_t addr)
{
	return((((uint64_t) addr) >> 3) & 0x3f);
}

#ifdef NOTUSED
/*
 * rse_rnat_addr()
 */
static uint64_t
rse_rnat_addr(uint64_t slot_addr)
{
	return ((uint64_t)((uint64_t)slot_addr | (0x3f << 3)));
}
#endif

/*
 * rse_skip_regs()
 */
static uint64_t
rse_skip_regs(uint64_t addr, int num_regs)
{
	int64_t delta = rse_slot_num(addr) + num_regs;

	if (num_regs < 0) {
		delta -= 0x3e;
	}
	return(addr + ((num_regs + delta/0x3f) * 8));
}

/*
 * pt_regs_off()
 *
 * Returns offset of reg in struct pt_regs.
 */
static inline uint64_t
pt_regs_off(uint64_t reg)
{
	uint64_t off =0;

	if (reg >= 1 && reg <= 3)
		off = kl_member_offset("pt_regs", "r1") + 8*(reg - 1);
	else if (reg <= 11)
		off = kl_member_offset("pt_regs", "r8") + 8*(reg - 8);
	else if (reg <= 15)
		off = kl_member_offset("pt_regs", "r12") + 8*(reg - 12);
	else if (reg <= 31)
		off = kl_member_offset("pt_regs", "r16") + 8*(reg - 16);
	else
		UNW_DPRINT(0, "unwind.%s: bad scratch reg r%"FMTPTR"u\n",
			__FUNCTION__, reg);
	return off;
}

/*
 * unw_access_pr()
 */
int
unw_access_pr(struct unw_frame_info *info, uint64_t *val, int write)
{
	void *ptr;
	uaddr_t a;

	if (write) {
		UNW_DPRINT(0, "unwind.%s: trying to write to pr!",
			 __FUNCTION__);
		return(1);
	}
	if (!info->pr_loc) {
		a = (uaddr_t)info->sw + kl_member_offset("switch_stack", "pr");
		ptr = (void*)a;
		*val = *LOC_VAL(ptr);
	} else {
		*val = GET_LOC_VAL(info->pr_loc);
	}
	return(0);
}

/*
 * hash()
 */
static unw_hash_index_t
hash(uint64_t ip)
{
#define magic    0x9e3779b97f4a7c16      /* based on (sqrt(5)/2-1)*2^64 */

	return((ip >> 4)*magic >> (64 - UNW_LOG_HASH_SIZE));
}

/*
 * cache_match()
 */
static int64_t
cache_match(struct unw_script *script, uint64_t ip, uint64_t pr)
{
	if ((ip == script->ip) &&
			(((pr ^ script->pr_val) & script->pr_mask) == 0)) {
		return(1);
	}
	return(0);
}

/*
 * get_scratch_regs()
 */
static uint64_t
get_scratch_regs(struct unw_frame_info *info)
{
	if (!info->pt) {
		/* This should not happen with valid unwind info.
		 */
		UNW_DPRINT(0, "unwind.%s: bad unwind info: "
			"resetting info->pt\n", __FUNCTION__);
		if (info->flags & UNW_FLAG_INTERRUPT_FRAME) {
			info->pt = (info->psp - PT_REGS_SZ);
		} else {
			info->pt = info->sp - 16;
		}
	}
	UNW_DPRINT(3, "unwind.%s: sp 0x%"FMTPTR"x pt 0x%"FMTPTR"x\n", 
		__FUNCTION__, info->sp, info->pt);
	return(info->pt);
}

/****
 **** Routines to manipulate the state stack
 ****/

/*
 * push()
 */
static inline void
push(struct unw_state_record *sr)
{
	struct unw_reg_state *rs;

	rs = alloc_reg_state();
	if (!rs) {
		fprintf(KL_ERRORFP, "unwind: cannot stack reg state!\n");
		return;
	}
	memcpy(rs, &sr->curr, sizeof(*rs));
	sr->curr.next = rs;
}

/*
 * pop()
 */
static void
pop(struct unw_state_record *sr)
{
	struct unw_reg_state *rs = sr->curr.next;

	if (!rs) {
		fprintf(KL_ERRORFP, "unwind: stack underflow!\n");
		return;
	}
	memcpy(&sr->curr, rs, sizeof(*rs));
	free_reg_state(rs);
}

/*
 * dup_state_stack()
 *
 * Make a copy of the state stack.  Non-recursive to avoid stack overflows.
 */
static struct unw_reg_state *
dup_state_stack (struct unw_reg_state *rs)
{
	struct unw_reg_state *copy, *prev = NULL, *first = NULL;

	while (rs) {
		copy = alloc_reg_state();
		if (!copy) {
			fprintf(KL_ERRORFP, "unwind.dup_state_stack: "
				"out of memory\n");
			return NULL;
		}
		memcpy(copy, rs, sizeof(*copy));
		if (first)
			prev->next = copy;
		else
			first = copy;
		rs = rs->next;
		prev = copy;
	}
	return first;
}

/*
 * free_state_stack()
 *
 * Free all stacked register states (but not RS itself).
 */
static void
free_state_stack (struct unw_reg_state *rs)
{
	struct unw_reg_state *p, *next;

	for (p = rs->next; p != NULL; p = next) {
		next = p->next;
		free_reg_state(p);
	}
	rs->next = NULL;
}

/* 
 * decode_abreg()
 */
static enum unw_register_index __attribute__((const))
decode_abreg (unsigned char abreg, int memory)
{       
	switch (abreg) {
		case 0x04 ... 0x07: return UNW_REG_R4 + (abreg - 0x04);
		case 0x22 ... 0x25: return UNW_REG_F2 + (abreg - 0x22);
		case 0x30 ... 0x3f: return UNW_REG_F16 + (abreg - 0x30);
		case 0x41 ... 0x45: return UNW_REG_B1 + (abreg - 0x41);
		case 0x60: return (UNW_REG_PR);
		case 0x61: return (UNW_REG_PSP);
		case 0x62: return (memory ? 
			UNW_REG_PRI_UNAT_MEM : UNW_REG_PRI_UNAT_GR);
		case 0x63: return UNW_REG_RP;
		case 0x64: return UNW_REG_BSP;
		case 0x65: return UNW_REG_BSPSTORE;
		case 0x66: return UNW_REG_RNAT;
		case 0x67: return UNW_REG_UNAT;
		case 0x68: return UNW_REG_FPSR;
		case 0x69: return UNW_REG_PFS;
		case 0x6a: return UNW_REG_LC;
		default:
		break;
	}
	UNW_DPRINT(0, "unwind.%s: bad abreg=0x%x\n", __FUNCTION__, abreg);
	return UNW_REG_LC;
}

/*
 * set_reg()
 */
static void
set_reg(
	struct unw_reg_info *reg,
	enum unw_where where,
	int when,
	uint64_t val)
{
	reg->val = val;
	reg->where = where;
	if (reg->when == UNW_WHEN_NEVER) {
		reg->when = when;
	}
}

/*
 * alloc_spill_area()
 */
static void
alloc_spill_area(
	uint64_t *offp,
	uint64_t regsize,
	struct unw_reg_info *lo,
	struct unw_reg_info *hi)
{
	struct unw_reg_info *reg;

	for (reg = hi; reg >= lo; --reg) {
		if (reg->where == UNW_WHERE_SPILL_HOME) {
			reg->where = UNW_WHERE_PSPREL;
			*offp -= regsize;
			reg->val = *offp;
		}
	}
}

/*
 * spill_next_when()
 */
static inline void
spill_next_when(
	struct unw_reg_info **regp,
	struct unw_reg_info *lim,
	unw_word t)
{
	struct unw_reg_info *reg;

	for (reg = *regp; reg <= lim; ++reg) {
		if (reg->where == UNW_WHERE_SPILL_HOME) {
			reg->when = t;
			*regp = reg + 1;
			return;
		}
	}
	UNW_DPRINT(0, "unwind.%s: excess spill!\n",  __FUNCTION__);
}

/*
 * finish_prologue()
 */
static inline void
finish_prologue(struct unw_state_record *sr)
{
	struct unw_reg_info *reg;
	uint64_t off;
	int i;

	/* First, resolve implicit register save locations (see Section
	 * "11.4.2.3 Rules for Using Unwind Descriptors", rule 3):
	 */
	for (i = 0; i < (int)
		sizeof(unw->save_order)/sizeof(unw->save_order[0]); ++i) {

		reg = sr->curr.reg + unw->save_order[i];
		if (reg->where == UNW_WHERE_GR_SAVE) {
			reg->where = UNW_WHERE_GR;
			reg->val = sr->gr_save_loc++;
		}
	}

	/* Next, compute when the fp, general, and branch registers get
	 * saved.  This must come before alloc_spill_area() because
	 * we need to know which registers are spilled to their home
	 * locations.
	 */
	if (sr->imask) {
		unsigned char kind, mask = 0;
		kaddr_t cp = sr->imask;
		uint64_t t;
		static const unsigned char limit[3] = {
			UNW_REG_F31, UNW_REG_R7, UNW_REG_B5
		};
		struct unw_reg_info *(regs[3]);

		regs[0] = sr->curr.reg + UNW_REG_F2;
		regs[1] = sr->curr.reg + UNW_REG_R4;
		regs[2] = sr->curr.reg + UNW_REG_B1;

		for (t = 0; t < sr->region_len; ++t) {
			if ((t & 3) == 0)
				mask = (unw_word)KL_VREAD_UINT8((kaddr_t)cp);
				cp += 1;

			kind = (mask >> 2*(3-(t & 3))) & 3;
			if (kind > 0)
				spill_next_when(&regs[kind - 1],
					sr->curr.reg + limit[kind - 1],
					sr->region_start + t);
		}
	}

	/* Next, lay out the memory stack spill area:
	 */
	if (sr->any_spills) {
		off = sr->spill_offset;
		alloc_spill_area(&off, 16,
			sr->curr.reg + UNW_REG_F2, sr->curr.reg + UNW_REG_F31);
		alloc_spill_area(&off,  8,
			sr->curr.reg + UNW_REG_B1, sr->curr.reg + UNW_REG_B5);
		alloc_spill_area(&off,  8,
			sr->curr.reg + UNW_REG_R4, sr->curr.reg + UNW_REG_R7);
	}
}

/****
 **** Region header descriptors.
 ****/

/*
 * desc_prologue()
 */
static void
desc_prologue(
	int body,
	unw_word rlen,
	unsigned char mask,
	unsigned char grsave,
	struct unw_state_record *sr)
{
	int i, region_start;

	if (!(sr->in_body || sr->first_region)) {
		finish_prologue(sr);
	}
	sr->first_region = 0;

	/* check if we're done:
	 */
	if (sr->when_target < (sr->region_start + sr->region_len)) {
		sr->done = 1;
		return;
	}

	region_start = sr->region_start + sr->region_len;

	for (i = 0; i < sr->epilogue_count; ++i) {
		pop(sr);
	}
	sr->epilogue_count = 0;
	sr->epilogue_start = UNW_WHEN_NEVER;

	sr->region_start = region_start;
	sr->region_len = rlen;
	sr->in_body = body;

	if (!body) {

		push(sr);

		for (i = 0; i < 4; ++i) {
			if (mask & 0x8) {
				set_reg(sr->curr.reg + unw->save_order[i],
					UNW_WHERE_GR,
					sr->region_start + sr->region_len - 1,
					grsave++);
			}
			mask <<= 1;
		}
		sr->gr_save_loc = grsave;
		sr->any_spills = 0;
		sr->imask = 0;
		sr->spill_offset = 0x10;        /* default to psp+16 */
	}
}

/****
 **** Prologue descriptors.
 ****/

/*
 * desc_abi()
 */
static inline void
desc_abi(unsigned char abi, unsigned char context, struct unw_state_record *sr)
{
	if (((abi == 0) || (abi == 3)) && context == 'i') {
		sr->flags |= UNW_FLAG_INTERRUPT_FRAME;
		UNW_DPRINT(3, "unwind.%s: interrupt frame\n",  __FUNCTION__);
	} else {
		UNW_DPRINT(0, "unwind%s: ignoring unwabi(abi=0x%x,"
			"context=0x%x)\n", __FUNCTION__, abi, context);
	}
}

/*
 * desc_br_gr()
 */
static inline void
desc_br_gr(unsigned char brmask, unsigned char gr, struct unw_state_record *sr)
{
	int i;

	for (i = 0; i < 5; ++i) {
		if (brmask & 1) {
			set_reg(sr->curr.reg + UNW_REG_B1 + i, UNW_WHERE_GR,
				sr->region_start + sr->region_len - 1, gr++);
		}
		brmask >>= 1;
	}
}

/*
 * desc_br_mem()
 */
static inline void
desc_br_mem(unsigned char brmask, struct unw_state_record *sr)
{
	int i;

	for (i = 0; i < 5; ++i) {
		if (brmask & 1) {
			set_reg(sr->curr.reg + UNW_REG_B1 + i,
				UNW_WHERE_SPILL_HOME,
				sr->region_start + sr->region_len - 1, 0);
			sr->any_spills = 1;
		}
		brmask >>= 1;
	}
}

/*
 * desc_frgr_mem()
 */
static inline void
desc_frgr_mem(
	unsigned char grmask,
	unw_word frmask,
	struct unw_state_record *sr)
{
	int i;

	for (i = 0; i < 4; ++i) {
		if ((grmask & 1) != 0) {
			set_reg(sr->curr.reg + UNW_REG_R4 + i,
				UNW_WHERE_SPILL_HOME,
				sr->region_start + sr->region_len - 1, 0);
			sr->any_spills = 1;
		}
		grmask >>= 1;
	}
	for (i = 0; i < 20; ++i) {
		if ((frmask & 1) != 0) {
			int base = (i < 4) ? UNW_REG_F2 : UNW_REG_F16 - 4;
			set_reg(sr->curr.reg + base + i, UNW_WHERE_SPILL_HOME,
				sr->region_start + sr->region_len - 1, 0);
			sr->any_spills = 1;
		}
		frmask >>= 1;
	}
}

/*
 * desc_fr_mem()
 */
static inline void
desc_fr_mem(unsigned char frmask, struct unw_state_record *sr)
{
	int i;

	for (i = 0; i < 4; ++i) {
		if ((frmask & 1) != 0) {
			set_reg(sr->curr.reg + UNW_REG_F2 + i,
				UNW_WHERE_SPILL_HOME,
				sr->region_start + sr->region_len - 1, 0);
			sr->any_spills = 1;
		}
		frmask >>= 1;
	}
}

/*
 * desc_gr_gr()
 */
static inline void
desc_gr_gr(unsigned char grmask, unsigned char gr, struct unw_state_record *sr)
{
	int i;

	for (i = 0; i < 4; ++i) {
		if ((grmask & 1) != 0)
			set_reg(sr->curr.reg + UNW_REG_R4 + i, UNW_WHERE_GR,
				sr->region_start + sr->region_len - 1, gr++);
		grmask >>= 1;
	}
}

/*
 * desc_gr_mem()
 */
static inline void
desc_gr_mem(unsigned char grmask, struct unw_state_record *sr)
{
	int i;

	for (i = 0; i < 4; ++i) {
		if ((grmask & 1) != 0) {
			set_reg(sr->curr.reg + UNW_REG_R4 + i,
				UNW_WHERE_SPILL_HOME,
				sr->region_start + sr->region_len - 1, 0);
			sr->any_spills = 1;
		}
		grmask >>= 1;
	}
}

/*
 * desc_mem_stack_f()
 */
static inline void
desc_mem_stack_f(unw_word t, unw_word size, struct unw_state_record *sr)
{
	set_reg(sr->curr.reg + UNW_REG_PSP, UNW_WHERE_NONE,
		sr->region_start + MIN((int)t, sr->region_len - 1), 16*size);
}

/*
 * desc_mem_stack_v()
 */
static inline void
desc_mem_stack_v(unw_word t, struct unw_state_record *sr)
{
	sr->curr.reg[UNW_REG_PSP].when =
		sr->region_start + MIN((int)t, sr->region_len - 1);
}

/*
 * desc_reg_gr()
 */
static inline void
desc_reg_gr(unsigned char reg, unsigned char dst, struct unw_state_record *sr)
{
	set_reg(sr->curr.reg + reg, UNW_WHERE_GR,
		sr->region_start + sr->region_len - 1, dst);
}

/*
 * desc_reg_psprel()
 */
static inline void
desc_reg_psprel(unsigned char reg, unw_word pspoff, struct unw_state_record *sr)
{
	set_reg(sr->curr.reg + reg, UNW_WHERE_PSPREL,
		sr->region_start + sr->region_len - 1, 0x10 - 4*pspoff);
}

/*
 * desc_reg_sprel()
 */
static inline void
desc_reg_sprel(unsigned char reg, unw_word spoff, struct unw_state_record *sr)
{
	set_reg(sr->curr.reg + reg, UNW_WHERE_SPREL,
		sr->region_start + sr->region_len - 1, 4*spoff);
}

/*
 * desc_rp_br()
 */
static inline void
desc_rp_br(unsigned char dst, struct unw_state_record *sr)
{
	sr->return_link_reg = dst;
}

/*
 * desc_reg_when()
 */
static inline void
desc_reg_when(unsigned char regnum, unw_word t, struct unw_state_record *sr)
{
	struct unw_reg_info *reg = sr->curr.reg + regnum;

	if (reg->where == UNW_WHERE_NONE)
		reg->where = UNW_WHERE_GR_SAVE;
	reg->when = sr->region_start + MIN((int)t, sr->region_len - 1);
}

/*
 * desc_spill_base()
 */
static inline void
desc_spill_base(unw_word pspoff, struct unw_state_record *sr)
{
	sr->spill_offset = 0x10 - 4*pspoff;
}

/*
 * desc_spill_mask()
 */
static inline kaddr_t
desc_spill_mask(kaddr_t imaskp, struct unw_state_record *sr)
{
	sr->imask = imaskp;
	return imaskp + (2*sr->region_len + 7)/8;
}

/****
 **** Body descriptors.
 ****/

/*
 * desc_epilogue()
 */
static inline void
desc_epilogue(unw_word t, unw_word ecount, struct unw_state_record *sr)
{
	sr->epilogue_start = sr->region_start + sr->region_len - 1 - t;
	sr->epilogue_count = ecount + 1;
}

/*
 * desc_epilogue()
 */
static inline void
desc_copy_state(unw_word label, struct unw_state_record *sr)
{
	struct unw_labeled_state *ls;

	for (ls = sr->labeled_states; ls; ls = ls->next) {
		if (ls->label == label) {
			free_state_stack(&sr->curr);
			memcpy(&sr->curr, &ls->saved_state, sizeof(sr->curr));
			sr->curr.next = dup_state_stack(ls->saved_state.next);
			return;
		}
	}
	fprintf(KL_ERRORFP, "unwind: failed to find state labeled 0x%"FMTPTR"x\n",
		label);
}

/*
 * desc_label_state()
 */
static inline void
desc_label_state(unw_word label, struct unw_state_record *sr)
{
	struct unw_labeled_state *ls;

	ls = alloc_labeled_state();
	if (!ls) {
		fprintf(KL_ERRORFP, "unwind.desc_label_state(): "
			"out of memory\n");
		return;
	}
	ls->label = label;
	memcpy(&ls->saved_state, &sr->curr, sizeof(ls->saved_state));
	ls->saved_state.next = dup_state_stack(sr->curr.next);

	/* insert into list of labeled states: */
	ls->next = sr->labeled_states;
	sr->labeled_states = ls;
}

/****
 **** General descriptors.
 ****/

/*
 * desc_is_active()
 */
static inline int
desc_is_active(unsigned char qp, unw_word t, struct unw_state_record *sr)
{
	if (sr->when_target <= sr->region_start +
			MIN((int)t, sr->region_len - 1)) {
		return 0;
	}
	if (qp > 0) {
		if ((sr->pr_val & (1ULL << qp)) == 0)
			return 0;
		sr->pr_mask |= (1ULL << qp);
	}
	return 1;
}

/*
 * desc_restore_p()
 */
static inline void
desc_restore_p(
	unsigned char qp,
	unw_word t,
	unsigned char abreg,
	struct unw_state_record *sr)
{
	struct unw_reg_info *r;

	if (!desc_is_active(qp, t, sr))
		return;

	r = sr->curr.reg + decode_abreg(abreg, 0);
	r->where = UNW_WHERE_NONE;
	r->when = UNW_WHEN_NEVER;
	r->val = 0;
}

/*
 * desc_spill_reg_p()
 */
static inline void
desc_spill_reg_p(
	unsigned char qp,
	unw_word t,
	unsigned char abreg,
	unsigned char x,
	unsigned char ytreg,
	struct unw_state_record *sr)
{
	enum unw_where where = UNW_WHERE_GR;
	struct unw_reg_info *r;

	if (!desc_is_active(qp, t, sr))
		return;

	if (x)
		where = UNW_WHERE_BR;
	else if (ytreg & 0x80)
		where = UNW_WHERE_FR;

	r = sr->curr.reg + decode_abreg(abreg, 0);
	r->where = where;
	r->when = sr->region_start + MIN((int)t, sr->region_len - 1);
	r->val = (ytreg & 0x7f);
}

/*
 * desc_spill_psprel_p()
 */
static inline void
desc_spill_psprel_p(
	unsigned char qp,
	unw_word t,
	unsigned char abreg,
	unw_word pspoff,
	struct unw_state_record *sr)
{
	struct unw_reg_info *r;

	if (!desc_is_active(qp, t, sr))
		return;

	r = sr->curr.reg + decode_abreg(abreg, 1);
	r->where = UNW_WHERE_PSPREL;
	r->when = sr->region_start + MIN((int)t, sr->region_len - 1);
	r->val = 0x10 - 4*pspoff;
}

/*
 * desc_spill_sprel_p()
 */
static inline void
desc_spill_sprel_p(
	unsigned char qp,
	unw_word t,
	unsigned char abreg,
	unw_word spoff,
	struct unw_state_record *sr)
{
	struct unw_reg_info *r;

	if (!desc_is_active(qp, t, sr))
		return;

	r = sr->curr.reg + decode_abreg(abreg, 1);
	r->where = UNW_WHERE_SPREL;
	r->when = sr->region_start + MIN((int)t, sr->region_len - 1);
	r->val = 4*spoff;
}

#define UNW_DEC_BAD_CODE(code)  \
	fprintf(KL_ERRORFP, "unwind: unknown code 0x%02x\n", code);

/*
 * region headers:
 */
#define UNW_DEC_PROLOGUE_GR(fmt,r,m,gr,arg)     desc_prologue(0,r,m,gr,arg)
#define UNW_DEC_PROLOGUE(fmt,b,r,arg)           desc_prologue(b,r,0,32,arg)
/*
 * prologue descriptors:
 */
#define UNW_DEC_ABI(fmt,a,c,arg)                desc_abi(a,c,arg)
#define UNW_DEC_BR_GR(fmt,b,g,arg)              desc_br_gr(b,g,arg)
#define UNW_DEC_BR_MEM(fmt,b,arg)               desc_br_mem(b,arg)
#define UNW_DEC_FRGR_MEM(fmt,g,f,arg)           desc_frgr_mem(g,f,arg)
#define UNW_DEC_FR_MEM(fmt,f,arg)               desc_fr_mem(f,arg)
#define UNW_DEC_GR_GR(fmt,m,g,arg)              desc_gr_gr(m,g,arg)
#define UNW_DEC_GR_MEM(fmt,m,arg)               desc_gr_mem(m,arg)
#define UNW_DEC_MEM_STACK_F(fmt,t,s,arg)        desc_mem_stack_f(t,s,arg)
#define UNW_DEC_MEM_STACK_V(fmt,t,arg)          desc_mem_stack_v(t,arg)
#define UNW_DEC_REG_GR(fmt,r,d,arg)             desc_reg_gr(r,d,arg)
#define UNW_DEC_REG_PSPREL(fmt,r,o,arg)         desc_reg_psprel(r,o,arg)
#define UNW_DEC_REG_SPREL(fmt,r,o,arg)          desc_reg_sprel(r,o,arg)
#define UNW_DEC_REG_WHEN(fmt,r,t,arg)           desc_reg_when(r,t,arg)
#define UNW_DEC_PRIUNAT_WHEN_GR(fmt,t,arg) \
	desc_reg_when(UNW_REG_PRI_UNAT_GR,t,arg)
#define UNW_DEC_PRIUNAT_WHEN_MEM(fmt,t,arg) \
    	desc_reg_when(UNW_REG_PRI_UNAT_MEM,t,arg)
#define UNW_DEC_PRIUNAT_GR(fmt,r,arg) \
	desc_reg_gr(UNW_REG_PRI_UNAT_GR,r,arg)
#define UNW_DEC_PRIUNAT_PSPREL(fmt,o,arg) \
	desc_reg_psprel(UNW_REG_PRI_UNAT_MEM,o,arg)
#define UNW_DEC_PRIUNAT_SPREL(fmt,o,arg) \
	desc_reg_sprel(UNW_REG_PRI_UNAT_MEM,o,arg)
#define UNW_DEC_RP_BR(fmt,d,arg)                desc_rp_br(d,arg)
#define UNW_DEC_SPILL_BASE(fmt,o,arg)           desc_spill_base(o,arg)
#define UNW_DEC_SPILL_MASK(fmt,m,arg)           (m = desc_spill_mask(m,arg))
/*
 * body descriptors:
 */
#define UNW_DEC_EPILOGUE(fmt,t,c,arg)           desc_epilogue(t,c,arg)
#define UNW_DEC_COPY_STATE(fmt,l,arg)           desc_copy_state(l,arg)
#define UNW_DEC_LABEL_STATE(fmt,l,arg)          desc_label_state(l,arg)
/*
 * general unwind descriptors:
 */
#define UNW_DEC_SPILL_REG_P(f,p,t,a,x,y,arg)    desc_spill_reg_p(p,t,a,x,y,arg)
#define UNW_DEC_SPILL_REG(f,t,a,x,y,arg)        desc_spill_reg_p(0,t,a,x,y,arg)
#define UNW_DEC_SPILL_PSPREL_P(f,p,t,a,o,arg)   desc_spill_psprel_p(p,t,a,o,arg)
#define UNW_DEC_SPILL_PSPREL(f,t,a,o,arg)       desc_spill_psprel_p(0,t,a,o,arg)
#define UNW_DEC_SPILL_SPREL_P(f,p,t,a,o,arg)    desc_spill_sprel_p(p,t,a,o,arg)
#define UNW_DEC_SPILL_SPREL(f,t,a,o,arg)        desc_spill_sprel_p(0,t,a,o,arg)
#define UNW_DEC_RESTORE_P(f,p,t,a,arg)          desc_restore_p(p,t,a,arg)
#define UNW_DEC_RESTORE(f,t,a,arg)              desc_restore_p(0,t,a,arg)

#include "unwind_decoder.c"

/**** 
 **** unwind script routines
 ****/

/*
 * script_new()
 *
 */
static struct unw_script *
script_new(uint64_t ip)
{
	struct unw_script *script, *prev, *tmp;
	unw_hash_index_t index;
	uint16_t head;

	head = unw->lru_head;
	script = unw->cache + head;
	unw->lru_head = script->lru_chain;

	/* re-insert script at the tail of the LRU chain:
	 */
	unw->cache[unw->lru_tail].lru_chain = head;
	unw->lru_tail = head;

	/* remove the old script from the hash table (if it's there):
	 */
	if (script->ip) {
		index = hash(script->ip);
		tmp = unw->cache + unw->hash[index];
		prev = 0;
		while (1) {
			if (tmp == script) {
				if (prev) {
					prev->coll_chain = tmp->coll_chain;
				} else {
					unw->hash[index] = tmp->coll_chain;
				}
				break;
			} else {
				prev = tmp;
			}
			if (tmp->coll_chain >= UNW_CACHE_SIZE) {
				/* old script wasn't in the hash-table
				 */
				break;
			}
			tmp = unw->cache + tmp->coll_chain;
		}
	}

	/* enter new script in the hash table
	 */
	index = hash(ip);
	script->coll_chain = unw->hash[index];
	unw->hash[index] = script - unw->cache;

	script->ip = ip;
	script->flags = 0;
	script->hint = 0;
	script->count = 0;
	return(script);
}

/*
 * script_finalize()
 */
static void
script_finalize(struct unw_script *script, struct unw_state_record *sr)
{
	script->pr_mask = sr->pr_mask;
	script->pr_val = sr->pr_val;
	/*
	 * We could down-grade our write-lock on script->lock here but
	 * the rwlock API doesn't offer atomic lock downgrading, so
	 * we'll just keep the write-lock and release it later when
	 * we're done using the script.
	 */
}

/*
 * script_emit()
 */
static inline void
script_emit(struct unw_script *script, struct unw_insn insn)
{
	if (script->count >= UNW_MAX_SCRIPT_LEN) {
		UNW_DPRINT(0, "unwind.%s: script exceeds maximum size of "
			"%u instructions!\n",
			__FUNCTION__, UNW_MAX_SCRIPT_LEN);
		return;
	}
	script->insn[script->count++] = insn;
}

/*
 * emit_nat_info()
 */
static inline void
emit_nat_info(struct unw_state_record *sr, int i, struct unw_script *script)
{
	struct unw_reg_info *r = sr->curr.reg + i;
	enum unw_insn_opcode opc;
	struct unw_insn insn;
	uint64_t val = 0;

	switch (r->where) {
		case UNW_WHERE_GR:
			if (r->val >= 32) {
				/* register got spilled to a stacked
				 * register
				 */
				opc = UNW_INSN_SETNAT_TYPE;
				val = UNW_NAT_REGSTK;
			} else
				/* register got spilled to a scratch
				 * register
				 */
				opc = UNW_INSN_SETNAT_MEMSTK;
			break;

		case UNW_WHERE_FR:
			opc = UNW_INSN_SETNAT_TYPE;
			val = UNW_NAT_VAL;
			break;

		case UNW_WHERE_BR:
			opc = UNW_INSN_SETNAT_TYPE;
			val = UNW_NAT_NONE;
			break;

		case UNW_WHERE_PSPREL:
		case UNW_WHERE_SPREL:
			opc = UNW_INSN_SETNAT_MEMSTK;
			break;

		default:
			UNW_DPRINT(0, "unwind.%s: don't know how to emit nat "
				"info for where = %u\n",
				__FUNCTION__, r->where);
			return;
	}
	insn.opc = opc;
	insn.dst = unw->preg_index[i];
	insn.val = val;
	script_emit(script, insn);
}

/*
 * compile_reg()
 */
static void
compile_reg(struct unw_state_record *sr, int i, struct unw_script *script)
{
	struct unw_reg_info *r = sr->curr.reg + i;
	enum unw_insn_opcode opc;
	uint64_t val, rval;
	struct unw_insn insn;
	int64_t need_nat_info;

	if (r->where == UNW_WHERE_NONE || r->when >= sr->when_target) {
		return;
	}

	opc = UNW_INSN_MOVE;
	val = rval = r->val;
	need_nat_info = (i >= UNW_REG_R4 && i <= UNW_REG_R7);

	switch (r->where) {
		case UNW_WHERE_GR:
			if (rval >= 32) {
				opc = UNW_INSN_MOVE_STACKED;
				val = rval - 32;
			} else if (rval >= 4 && rval <= 7) {
				if (need_nat_info) {
					opc = UNW_INSN_MOVE2;
					need_nat_info = 0;
				}
				val = unw->preg_index[UNW_REG_R4 + (rval - 4)];
			} else {
				/* register got spilled to a scratch register */
				opc = UNW_INSN_MOVE_SCRATCH;
				val = pt_regs_off(rval);
			}
			break;

		case UNW_WHERE_FR:
			if (rval <= 5)
				val = unw->preg_index[UNW_REG_F2  + 
					(rval -  1)];
			else if (rval >= 16 && rval <= 31)
				val = unw->preg_index[UNW_REG_F16 + 
					(rval - 16)];
			else {
				opc = UNW_INSN_MOVE_SCRATCH;
				if (rval <= 9)
					val = kl_member_offset("pt_regs", "f6") 
						+ 16*(rval - 6);
				else
					UNW_DPRINT(0, "unwind.%s: kernel may "
						"not touch f%"FMTPTR"u\n",
						__FUNCTION__, rval);
			}
			break;


		case UNW_WHERE_BR:
			if (rval >= 1 && rval <= 5)
				val = unw->preg_index[UNW_REG_B1 + (rval - 1)];
			else {
				opc = UNW_INSN_MOVE_SCRATCH;
				if (rval == 0)
					val = kl_member_offset("pt_regs", "b0");
				else if (rval == 6)
					val = kl_member_offset("pt_regs", "b6");
				else
					val = kl_member_offset("pt_regs", "b7");
			}
			break;

		case UNW_WHERE_SPREL:
			opc = UNW_INSN_ADD_SP;
			break;

		case UNW_WHERE_PSPREL:
			opc = UNW_INSN_ADD_PSP;
			break;

		default:
			UNW_DPRINT(0, "unwind%s: register %u has unexpected "
				"`where' value of %u\n",
				__FUNCTION__, i, r->where);
			break;
	}
	insn.opc = opc;
	insn.dst = unw->preg_index[i];
	insn.val = val;
	script_emit(script, insn);
	if (need_nat_info)
		emit_nat_info(sr, i, script);

	if (i == UNW_REG_PSP) {
		/*
		 * info->psp must contain the _value_ of the previous
		 * sp, not it's save location.  We get this by
		 * dereferencing the value we just stored in
		 * info->psp:
		 */
		insn.opc = UNW_INSN_LOAD;
		insn.dst = insn.val = unw->preg_index[UNW_REG_PSP];
		script_emit(script, insn);
	}
}

/*
 * build_script()
 *
 * Build an unwind script that unwinds from state OLD_STATE to the
 * entrypoint of the function that called OLD_STATE.
 */
static struct unw_script *
build_script(struct unw_frame_info *info)
{
	const struct unw_table_entry *e = 0;
	struct unw_script *script = 0;
	struct unw_labeled_state *ls, *next;
	uint64_t ip = info->ip;
	struct unw_state_record sr;
	struct unw_table *table;
	struct unw_reg_info *r;
	struct unw_insn insn;
	kaddr_t dp, desc_end;
	u64 hdr;
	int i;

	/* build state record
	 */
	memset(&sr, 0, sizeof(sr));
	for (r = sr.curr.reg; r < sr.curr.reg + UNW_NUM_REGS; ++r) {
		r->when = UNW_WHEN_NEVER;
	}
	sr.pr_val = info->pr;

	UNW_DPRINT(3, "unwind.%s: ip 0x%"FMTPTR"x\n", __FUNCTION__, ip);
	script = script_new(ip);
	if (!script) {
		UNW_DPRINT(0, "unwind.%s: failed to create unwind script\n",
			__FUNCTION__);
		return(0);
	}

	unw->cache[info->prev_script].hint = script - unw->cache;

	/* search the kernels and the modules' unwind tables for IP
	 */
	e = lookup_unw_table_entry(ip);
	if (!e) {
		/* no info, return default unwinder (leaf proc, no mem stack,
		 * no saved regs)
		 */
		UNW_DPRINT(1, "unwind.%s: no unwind info for ip=0x%"FMTPTR"x "
			"(prev ip=0x%"FMTPTR"x)\n",
			__FUNCTION__, ip, unw->cache[info->prev_script].ip);
		sr.curr.reg[UNW_REG_RP].where = UNW_WHERE_BR;
		sr.curr.reg[UNW_REG_RP].when = -1;
		sr.curr.reg[UNW_REG_RP].val = 0;
		compile_reg(&sr, UNW_REG_RP, script);
		script_finalize(script, &sr);
		kl_free_block((void*)e);
		return(script);
	}

	table = lookup_unw_table(ip);
	sr.when_target = (3*((ip & ~0xfULL) -
		(table->segment_base + e->start_offset))/16
			  + (ip & 0xfULL));
	hdr = KL_VREAD_UINT64((table->segment_base + e->info_offset));
	dp = (kaddr_t)(table->segment_base + e->info_offset + 8);
	desc_end = dp + 8*UNW_LENGTH(hdr);

	while (!sr.done && dp < desc_end) {
		dp = unw_decode(dp, sr.in_body, &sr);
	}

	if (sr.when_target > sr.epilogue_start) {
		/* sp has been restored and all values on the memory
		 * stack below psp also have been restored.
		 */
		sr.curr.reg[UNW_REG_PSP].val = 0;
		sr.curr.reg[UNW_REG_PSP].where = UNW_WHERE_NONE;
		sr.curr.reg[UNW_REG_PSP].when = UNW_WHEN_NEVER;
		for (r = sr.curr.reg; r < sr.curr.reg + UNW_NUM_REGS; ++r)
			if ((r->where == UNW_WHERE_PSPREL && r->val <= 0x10)
			    || r->where == UNW_WHERE_SPREL)
			{
				r->val = 0;
				r->where = UNW_WHERE_NONE;
				r->when = UNW_WHEN_NEVER;
			}
	}
	script->flags = sr.flags;

	/*
	 * If RP did't get saved, generate entry for the return link
	 * register.
	 */
	if (sr.curr.reg[UNW_REG_RP].when >= sr.when_target) {
		sr.curr.reg[UNW_REG_RP].where = UNW_WHERE_BR;
		sr.curr.reg[UNW_REG_RP].when = -1;
		sr.curr.reg[UNW_REG_RP].val = sr.return_link_reg;
		UNW_DPRINT(1, "unwind.%s: using default for rp at ip=0x%"FMTPTR"x "
			"where=%d val=0x%"FMTPTR"x\n",
			__FUNCTION__, ip, sr.curr.reg[UNW_REG_RP].where,
			sr.curr.reg[UNW_REG_RP].val);
	}

#ifdef UNW_DEBUG
	UNW_DPRINT(1, "unwind.%s: state record for func 0x%"FMTPTR"x, t=%u:\n",
		__FUNCTION__, table->segment_base + e->start_offset,
		sr.when_target);
	for (r = sr.curr.reg; r < sr.curr.reg + UNW_NUM_REGS; ++r) {
		if (r->where != UNW_WHERE_NONE || r->when != UNW_WHEN_NEVER) {
			UNW_DPRINT(1, "  %s <- ",
				unw.preg_name[r - sr.curr.reg]);
			switch (r->where) {
				case UNW_WHERE_GR:
					UNW_DPRINT(1, "r%"FMTPTR"u", r->val);
					break;
				case UNW_WHERE_FR:
					UNW_DPRINT(1, "f%"FMTPTR"u", r->val);
					break;
				case UNW_WHERE_BR:
					UNW_DPRINT(1, "b%"FMTPTR"u", r->val);
					break;
				case UNW_WHERE_SPREL:
					UNW_DPRINT(1, "[sp+0x%"FMTPTR"x]", r->val);
					break;
				case UNW_WHERE_PSPREL:
					UNW_DPRINT(1, "[psp+0x%"FMTPTR"x]", r->val);
					break;
				case UNW_WHERE_NONE:
					UNW_DPRINT(1, "%s+0x%"FMTPTR"x",
						unw.preg_name[r - sr.curr.reg],
						r->val);
					break;

				default:
					UNW_DPRINT(1, "BADWHERE(%d)", r->where);
					break;
			}
			UNW_DPRINT(1, "\t\t%d\n", r->when);
		}
	}
#endif
	/* translate state record into unwinder instructions: */

	/*
	 * First, set psp if we're dealing with a fixed-size frame;
	 * subsequent instructions may depend on this value.
	 */
	if (sr.when_target > sr.curr.reg[UNW_REG_PSP].when
	    && (sr.curr.reg[UNW_REG_PSP].where == UNW_WHERE_NONE)
	    && sr.curr.reg[UNW_REG_PSP].val != 0) {
		/* new psp is sp plus frame size */
		insn.opc = UNW_INSN_ADD;
		insn.dst = kl_member_offset("unw_frame_info", "psp")/8;
		insn.val = sr.curr.reg[UNW_REG_PSP].val;        /* frame size */
		script_emit(script, insn);
	}

	/* determine where the primary UNaT is: */
	if (sr.when_target < sr.curr.reg[UNW_REG_PRI_UNAT_GR].when)
		i = UNW_REG_PRI_UNAT_MEM;
	else if (sr.when_target < sr.curr.reg[UNW_REG_PRI_UNAT_MEM].when)
		i = UNW_REG_PRI_UNAT_GR;
	else if (sr.curr.reg[UNW_REG_PRI_UNAT_MEM].when >
			sr.curr.reg[UNW_REG_PRI_UNAT_GR].when)
		i = UNW_REG_PRI_UNAT_MEM;
	else
		i = UNW_REG_PRI_UNAT_GR;

	compile_reg(&sr, i, script);

	for (i = UNW_REG_BSP; i < UNW_NUM_REGS; ++i)
		compile_reg(&sr, i, script);

	/* free labeled register states & stack: */

	for (ls = sr.labeled_states; ls; ls = next) {
		next = ls->next;
		free_state_stack(&ls->saved_state);
		free_labeled_state(ls);
	}
	free_state_stack(&sr.curr);

	script_finalize(script, &sr);
	return(script);
}

/*
 * script_lookup()
 */
static struct unw_script *
script_lookup(struct unw_frame_info *info)
{
	struct unw_script *script = unw->cache + info->hint;
	uint16_t index;
	uint64_t ip, pr;

	ip = info->ip;
	pr = info->pr;

	for (index = 0; index < UNW_CACHE_SIZE; index++) {
		script = unw->cache + index;
		if (cache_match(script, ip, pr)) {
			break;
		}
	}
	UNW_DPRINT(1, "unwind.%s: index=%d, script=0x%lx, "
		"ip=0x%"FMTPTR"x, pr=0x%"FMTPTR"x\n",
		__FUNCTION__, index, (uaddr_t)script, ip, pr);
	if (index == UNW_CACHE_SIZE) {
		return((struct unw_script *)NULL);
	}
	return(script);
}

/*
 * run_script()
 *
 * Apply the unwinding actions represented by OPS and update SR to
 * reflect the state that existed upon entry to the function that this
 * unwinder represents.
 */
static void
run_script(struct unw_script *script, struct unw_frame_info *state)
{
	struct unw_insn *ip, *limit, next_insn;
	uint64_t opc, dst, val, off;
	uint64_t *s = (uint64_t *) state;
	uaddr_t a;

	state->flags = script->flags;
	ip = script->insn;
	limit = script->insn + script->count;
	next_insn = *ip;

	while (ip++ < limit) {
		opc = next_insn.opc;
		dst = next_insn.dst;
		val = next_insn.val;
		next_insn = *ip;

redo:
		switch (opc) {
			case UNW_INSN_ADD:
				s[dst] += val;
				break;

			case UNW_INSN_MOVE2:
				if (!s[val]) {
					goto lazy_init;
				}
				s[dst+1] = s[val+1];
				s[dst] = s[val];
				break;

			case UNW_INSN_MOVE:
				if (!s[val]) {
					goto lazy_init;
				}
				s[dst] = s[val];
				break;

			case UNW_INSN_MOVE_SCRATCH:
				if (state->pt) {
					s[dst] = (get_scratch_regs(state)
						+ val);
				} else {
					s[dst] = 0;
					UNW_DPRINT(0, "unwind.%s: no "
						"state->pt, dst=%"FMTPTR"d, "
						"val=%"FMTPTR"d\n",
						__FUNCTION__, dst, val);
				}
				break;

			case UNW_INSN_MOVE_STACKED:
				s[dst] = (uint64_t)
					rse_skip_regs(state->bsp, val);
				break;

			case UNW_INSN_ADD_PSP:
				s[dst] = state->psp + val;
				break;

			case UNW_INSN_ADD_SP:
				s[dst] = state->sp + val;
				break;

			case UNW_INSN_SETNAT_MEMSTK:
				if (!state->pri_unat_loc) {
					state->pri_unat_loc = (state->sw + 
					kl_member_offset("switch_stack", "ar_unat"));
				}

				/* register off. is a multiple of 8, so the
				 * least 3 bits (type) are 0
				 */
				a = (uaddr_t)state->pri_unat_loc;
				s[dst+1] = (*(uint64_t*)a - s[dst]) 
					| UNW_NAT_MEMSTK;
				break;

			case UNW_INSN_SETNAT_TYPE:
				s[dst+1] = val;
				break;

			case UNW_INSN_LOAD:
#ifdef UNW_DEBUG
				if ((s[val] &
					(local_cpu_data->unimpl_va_mask | 0x7))
						!= 0 || s[val] < TASK_SIZE) {
					UNW_DPRINT(0, "unwind.%s: rejecting "
						"bad psp=0x%"FMTPTR"x\n",
						__FUNCTION__, s[val]);
					break;
				}
#endif
				a = (uaddr_t)s[val];
				s[dst] = *(uint64_t *)a;
				break;
		}
	}
	return;

lazy_init:
	off = unw->sw_off[val];
	s[val] = (uint64_t) state->sw + off;
	if (off >= kl_member_offset("switch_stack", "r4")
			&& off <= kl_member_offset("switch_stack", "r7")) {

		/* We're initializing a general register: init NaT info, too.
		 * Note that the offset is a multiple of 8 which gives us
		 * the 3 bits needed for the type field.
		 */
		s[val+1] = (kl_member_offset("switch_stack", "ar_unat") - off)
			| UNW_NAT_MEMSTK;
	}
	goto redo;
}

/*                                      
 * find_save_locs()             
 */
static int                      
find_save_locs(struct unw_frame_info *info)
{               
	struct unw_script *scr;

	uint64_t cpudata;

	/* XXX -- This section of code assumes that there is a saved 
	 * cpuinfo_ia64 struct that was saved by kdb when the system crash
	 * occurred. That functionality does not currently exist for 
	 * this version of lcrash. Leaving the code block here in case we
	 * come up with a similar solution in the future (it reflects the
	 * modification of a block of code in the origianl kernel unwind.c
	 * module.
	 */
	if (SN2_24X && 
		(cpudata = cpu_data(kdb_cpu_id((uint64_t)info->task)))) {
		uint64_t unimpl_va_mask;
	
		unimpl_va_mask = KL_VREAD_PTR(cpudata + 
			kl_member_offset("cpuinfo_ia64", "unimpl_va_mask"));
		if (KL_ERROR) {
			return(-1);
		}
		if ((info->ip & (unimpl_va_mask | 0xf)) || 
			(info->ip < kl_task_size((uint64_t)info->task))) {
		 
			/* don't let obviously bad addresses pollute the  
			 * cache */
	
			/* FIXME: should really be level 0 but it occurs too
			 * often. KAO
			 */
			UNW_DPRINT(1, "unwind.%s: rejecting bad ip=0x%"FMTPTR"x\n",
				__FUNCTION__, info->ip);
			info->rp_loc = 0;
			return(-1);
		}
	}

	scr = script_lookup(info);
	if (!scr) {
		scr = build_script(info);
		if (!scr) {
			UNW_DPRINT(0, "unwind.%s: failed to locate/build "
				"unwind script for ip 0x%"FMTPTR"x\n",
				   __FUNCTION__, info->ip);
			return(-1);
		}
	}
	info->hint = scr->hint;
	info->prev_script = scr - unw->cache;
	run_script(scr, info);
	return(0);
}

/****
 **** Initialize unwind information
 ****/

/*
 * unw24_to_unw()
 */
static void
unw24_to_unw(unw24_t *unw24, unw_t *unw)
{
	int i, sz;

	unw->lock = unw24->lock;
	unw->tables = (uint64_t)unw24->tables;
        unw->r0 = 0;
	for (i = 0; i < 8; i++) {
		unw->save_order[i] = unw24->save_order[i];
	}	

	sz = sizeof(struct unw_frame_info) / 8;
	for (i = 0; i < sz; i++) {
		unw->sw_off[i] = unw24->sw_off[i];
	}

	unw->lru_head = unw24->lru_head;
	unw->lru_tail = unw24->lru_tail;

	for (i = 0; i < UNW_NUM_REGS; i++) {
		unw->preg_index[i] = unw24->preg_index[i];
	}

	for (i = 0; i < 32; i++) {
		unw->pt_regs_offsets[i] = 0;
	}	

	memcpy(&unw->kernel_table, &unw24->kernel_table, 
		sizeof(unw->kernel_table));

        unw->gate_table_size = (uint64_t)unw24->gate_table_size;
        unw->gate_table = (uint64_t)unw24->gate_table;

	for (i = 0; i < UNW_HASH_SIZE; i++) {
		unw->hash[i] = unw24->hash[i];
	}

	for (i = 0; i < UNW_CACHE_SIZE; i++) {
		unw->cache[i] = unw24->cache[i];
	}
}

/*
 * init_unw()
 */
void
init_unw(void)
{
	int i;
	syment_t *sp = (syment_t *)NULL;

	/* Make sure we capture the kernel unwind cache data. We only
	 * need to do this once, as we maintain the table after that
	 * for both dumps and live systems.
	 */
	if (unw) {
		return;
	}

	/* Get the address of the save area for running cpus (for older
	 * SGI dumps only).
	 */
	if (SN2_24X) {
		sp = kl_lkup_symname("_cpu_data");
		CPU_DATA = sp->s_addr;
	}

	unw = (unw_t*)kl_alloc_block(sizeof(*unw), K_PERM);
	sp = kl_lkup_symname("unw");
	unw_addr = sp->s_addr;

	/* If this is an older dump, convert the unw struct to the new
	 * format now (for those fields we care about). We do this here
	 * so that the rest of the unwind code does not need to be touched.
	 */
	if (SN2_24X) {
		unw24 = (unw24_t*)kl_alloc_block(sizeof(*unw24), K_PERM);
		GET_BLOCK(unw_addr, sizeof(*unw24), unw24);
		unw24_to_unw(unw24, unw);
	} else {
		GET_BLOCK(unw_addr, sizeof(*unw), unw);
	}

	/* Initialize the unw_script cache. Since we copied in the
	 * one from the kernel, we have to zero it out and start
	 * fresh.
	 */
	memset(unw->cache, 0, sizeof(unw->cache));
	for (i = 0; i < UNW_CACHE_SIZE; ++i) {
		if (i > 0) {
			unw->cache[i].lru_chain = (i - 1);
		}
		unw->cache[i].coll_chain = -1;
	}
	unw->lru_head = UNW_CACHE_SIZE - 1;
	unw->lru_tail = 0;
}

/****
 **** Initialize task stack unwind
 ****/

/*
 * init_frame_info()
 */
static void
init_frame_info(
	struct unw_frame_info *info,
	uint64_t task,
	switch_stack_t *s)
{
	uint64_t sol;
	uint64_t rbslimit, rbstop, stklimit, stktop;
	void *sw = (void*)s->sw;

	init_unw();

	rbslimit = (uint64_t)task + KL_IA64_RBS_OFFSET;
	rbstop = *LOC_VAL((uaddr_t)sw +
                kl_member_offset("switch_stack", "ar_bspstore"));
	if ((rbstop - (uint64_t)task) >= KL_STACK_OFFSET) {
		rbstop = rbslimit;
	}

	stklimit = (uint64_t)task + KL_STACK_OFFSET;
	stktop = ((uint64_t)(s->addr + SWITCH_STACK_SZ) - 16);
	if (stktop <= rbstop) {
		stktop = rbstop;
	}

	info->task = task;
	info->regstk.limit = rbslimit;
	info->regstk.top = rbstop;
	info->memstk.limit = stklimit;
	info->memstk.top = stktop;
	info->sp = info->psp = stktop;
	info->pr = *LOC_VAL((uaddr_t)sw +
                kl_member_offset("switch_stack", "pr"));

	UNW_DPRINT(3, "unwind.%s:\n"
		"  task   0x%"FMTPTR"x\n"
		"  rbs = [0x%"FMTPTR"x-0x%"FMTPTR"x)\n"
		"  stk = [0x%"FMTPTR"x-0x%"FMTPTR"x)\n"
		"  pr     0x%"FMTPTR"x\n"
		"  sw     0x%"FMTPTR"x\n"
		"  sp     0x%"FMTPTR"x\n",
		__FUNCTION__, (uint64_t) task, rbslimit, rbstop, stktop, 
		stklimit, info->pr, (uint64_t) s->addr, info->sp);

	info->cfm_loc = (uint64_t)((uaddr_t)sw +
			kl_member_offset("switch_stack", "ar_pfs"));
	/* Get size of locals */
	sol = (*(uint64_t*)((uaddr_t)info->cfm_loc) >> 7) & 0x7f; 
	info->bsp = (uint64_t)rse_skip_regs(info->regstk.top, -sol);
	info->ip = *(uint64_t*)LOC_VAL(((uaddr_t)sw +
                kl_member_offset("switch_stack", "b0")));
	info->sw = (uaddr_t)sw;
	find_save_locs(info);
	UNW_DPRINT(3, "unwind.%s:\n"
		"  bsp    0x%"FMTPTR"x\n"
		"  sol    0x%"FMTPTR"x\n"
		"  ip     0x%"FMTPTR"x\n",
		__FUNCTION__, info->bsp, sol, info->ip);

#ifdef UNW_DEBUG
	struct unw_script *scr;
	scr = build_script(info);
	walk_unw_tables();
#endif
}

/*
 * unwind()
 */
static int
unwind(struct unw_frame_info *info)
{
	uint64_t prev_ip, prev_sp, prev_bsp;
	uint64_t ip, pr, num_regs;
	int retval;
	uaddr_t a;

	prev_ip = info->ip;
	prev_sp = info->sp;
	prev_bsp = info->bsp;

	/* restore the ip
	 */
	if (!info->rp_loc) {
		/* FIXME: should really be level 0 but it occurs too
		 * often.  KAO
		 */
		UNW_DPRINT(1, "unwind.%s: failed to locate return link "
			"(ip=0x%"FMTPTR"x)!\n", __FUNCTION__, info->ip);
		return(-1);
	}
	ip = info->ip = KL_VREAD_UINT64((uint64_t)info->rp_loc);

	if (ip < (GATE_ADDR + KL_PAGE_SIZE)) {
		/*
		 * We don't have unwind info for the gate page, so we
		 * consider that part of user-space for the purpose of
		 * unwinding.
		 */
		UNW_DPRINT(2, "unwind.%s: reached user-space (ip=0x%"FMTPTR"x)\n",
			__FUNCTION__, ip);
		return(-1);
	}

	/* restore the cfm:
	 */
	if (!info->pfs_loc) {
		UNW_DPRINT(0, "unwind.%s: failed to locate ar.pfs!\n",
			__FUNCTION__);
		return(-1);
	}
	info->cfm_loc = info->pfs_loc;

	/* restore the bsp:
	 */
	pr = info->pr;
	num_regs = 0;
	if ((info->flags & UNW_FLAG_INTERRUPT_FRAME)) {
		info->pt = (info->sp + 16);
		UNW_DPRINT(3, "unwind.%s: interrupt_frame pt 0x%"FMTPTR"x\n",
			__FUNCTION__, info->pt);
		if ((pr & (1ULL << pNonSys)) != 0) {
			/* get the size of the frame
			 */
			num_regs = KL_VREAD_PTR((uint64_t)info->cfm_loc);
			num_regs &= 0x7f;
		}
		a = (uaddr_t)info->pt + kl_member_offset("pt_regs", "ar_pfs");
		info->pfs_loc = KL_VREAD_PTR((uint64_t)a);
	} else {
		/* Get the size of locals
		 */
		num_regs = KL_VREAD_PTR((uint64_t)info->cfm_loc);
		num_regs = (num_regs >> 7) & 0x7f;
	}
	info->bsp = (uint64_t)
		rse_skip_regs((uint64_t)info->bsp, -num_regs);
	if (info->bsp < info->regstk.limit || info->bsp > info->regstk.top) {
		UNW_DPRINT(0, "unwind.%s: bsp (0x%"FMTPTR"x) "
			"out of range [0x%"FMTPTR"x-0x%"FMTPTR"x]\n", __FUNCTION__,
			info->bsp, info->regstk.limit, info->regstk.top);
		return(-1);
	}

	/* restore the sp:
	 */
	info->sp = info->psp;
	if (info->sp < info->memstk.top || info->sp > info->memstk.limit) {
		UNW_DPRINT(0, "unwind.%s: sp (0x%"FMTPTR"x) out of range "
			"[0x%"FMTPTR"x-0x%"FMTPTR"x]\n", __FUNCTION__,
			info->sp, info->memstk.top, info->memstk.limit);
		return(-1);
	}

	if (info->ip == prev_ip &&
			info->sp == prev_sp && info->bsp == prev_bsp) {
		UNW_DPRINT(0, "unwind.%s: ip, sp, bsp unchanged; "
			"stopping here (ip=0x%"FMTPTR"x)\n", __FUNCTION__, ip);
		return(-1);
	}

	/* as we unwind, the saved ar.unat becomes the primary unat:
	 */
	info->pri_unat_loc = info->unat_loc;

	/* finally, restore the predicates:
	 */
	unw_get_pr(info, &info->pr);

	retval = find_save_locs(info);
	return(retval);
}

/*
 * ia64_init_frame_info()
 */
void
ia64_init_frame_info(
	struct unw_frame_info *info,
	uint64_t task,
	switch_stack_t *s)
{
	return(init_frame_info(info, task, s));
}

/*
 * ia64_unwind()
 */
int
ia64_unwind(struct unw_frame_info *info)
{
	int ret;

	if (!info->ip) {
		return(1);
	}
	ret = unwind(info);
	return(ret);

}

#define REGION_HEADER	0x1000
#define DESCRIPTOR	0x2000
#define PROLOGUE	0x4000
#define BODY		0x8000

#define BAD_DESC	0x0000
#define R1_DESC		0x0001
#define R2_DESC		0x0002
#define R3_DESC		0x0003
#define P1_DESC		0x0004
#define P2_DESC		0x0005
#define P3_DESC		0x0006
#define P4_DESC		0x0007
#define P5_DESC		0x0008
#define P6_DESC		0x0009
#define P7_DESC		0x000a
#define P8_DESC		0x000b
#define P9_DESC		0x000c
#define P10_DESC	0x000d
#define B1_DESC		0x000e
#define B2_DESC		0x000f
#define B3_DESC		0x0010
#define B4_DESC		0x0011
#define X1_DESC		0x0012		
#define X2_DESC		0x0013
#define X3_DESC		0x0014
#define X4_DESC		0x0015

#define FMT_CNT		22

#define FMT_MASK	0x00ff

char *fmt_name[FMT_CNT] = { 
	"BAD", 
	"R1", 
	"R2", 
	"R3", 
	"P1", 
	"P2", 
	"P3", 
	"P4", 
	"P5", 
	"P6", 
	"P7", 
	"P8", 
	"P9", 
	"P10", 
	"B1",
	"B2",
	"B3",
	"B4",
	"X1", 
	"X2", 
	"X3", 
	"X4" 
};

/*
 * get_code_type()
 */
static int
get_code_type(unsigned char code, int in_body)
{
	int code_type = 0;

	switch (code >> 7) {
		case 0: 
			code_type = REGION_HEADER;
			break;
		case 1:
			code_type = DESCRIPTOR;
			break;
	}

	if (code_type == REGION_HEADER) {
		switch (code >> 5) {
			case 0:
				code_type |= (PROLOGUE|R1_DESC);
				break;
			case 1:
				code_type |= (BODY|R1_DESC);
				break;

			case 2:
				code_type |= (PROLOGUE|R2_DESC);
				break;

			case 3:
				code_type |= R3_DESC;
				if (code & 3) {
					code_type |= BODY;
				} else {
					code_type |= PROLOGUE;
				}	
				break;
		}
	} else if (code_type == DESCRIPTOR) {
		if (in_body) {
			code_type |= BODY;
			if ((code >> 6) == 2) {
				code_type |= B1_DESC;
			} else if ((code >> 5) == 6) {
				code_type |= B2_DESC;
			} else if (code == 240) {
				code_type |= B3_DESC;
			} else if (code & 240) {
				code_type |= B4_DESC;
			} else if (code == 249) {
				code_type |= X1_DESC;
			} else if (code == 250) {
				code_type |= X2_DESC;
			} else if (code == 251) {
				code_type |= X3_DESC;
			} else if (code == 252) {
				code_type |= X4_DESC;
			}
		} else {
			code_type |= PROLOGUE;
			if ((code >> 5) == 4) {
				code_type |= P1_DESC;
			} else if ((code >> 4) == 10) {
				code_type |= P2_DESC;
			} else if ((code >> 3) == 22) {
				code_type |= P3_DESC;
			} else if (code == 184) {
				code_type |= P4_DESC;
			} else if (code == 185) {
				code_type |= P5_DESC;
			} else if ((code >> 5) == 6) {
				code_type |= P6_DESC;
			} else if ((code >> 4) == 14) {
				code_type |= P7_DESC;
			} else if (code == 240) {
				code_type |= P8_DESC;
			} else if (code == 241) {
				code_type |= P9_DESC;
			} else if (code == 249) {
				code_type |= X1_DESC;
			} else if (code == 250) {
				code_type |= X2_DESC;
			} else if (code == 251) {
				code_type |= X3_DESC;
			} else if (code == 252) {
				code_type |= X4_DESC;
			} else if (code == 255) {
				code_type |= P10_DESC;
			}
		}
	}
	return(code_type);
}

/*
 * print_code()
 */
void
print_code(FILE *ofp, kaddr_t dp, int in_body) 
{
	int code_type;
        unsigned char code;

        code = (unsigned char)KL_VREAD_UINT8((kaddr_t)dp);
	code_type = get_code_type(code, in_body);
	fprintf(ofp, "dp=0x%"FMTPTR"x, code=0x%x (%s: %s %s)\n", 
		(uint64_t)dp, code, fmt_name[(code_type & FMT_MASK)],
		(code_type & PROLOGUE) ? "PROLOGUE" : "BODY",
		(code_type & REGION_HEADER) ? "HEADER" : "DESCRIPTOR");
}

/*
 * unwind_ip()
 */
int
unwind_ip(FILE *ofp, uint64_t ip)
{
        struct unw_table *t;
        struct unw_table_entry *e;
        char name[128];
        struct unw_state_record sr;
        struct unw_reg_info *r;
        kaddr_t dp, desc_end;
        u64 hdr;

	init_unw();

	if ((t = lookup_unw_table(ip))) {

		GET_BLOCK(t->name, 128, (void*)name);

		fprintf(ofp, "\n");
		fprintf(ofp, "Unwind information for IP: 0x%"FMTPTR"x\n\n", ip);
		fprintf(ofp, "  Unwind Table:\n");
		fprintf(ofp, "    name: %s, gp: 0x%"FMTPTR"x\n", name, t->gp);
		fprintf(ofp, "    array: 0x%"FMTPTR"x, length: %"FMTPTR"u\n",
			(uint64_t)t->array, t->length);
		fprintf(ofp, "    segemnt_base: 0x%"FMTPTR"x\n", t->segment_base);
		fprintf(ofp, "    start: 0x%"FMTPTR"x, end: 0x%"FMTPTR"x\n",
			t->start, t->end);
	}

	if ((e = lookup_unw_table_entry(ip))) {
		fprintf(ofp, "\n");
		fprintf(ofp, "  Table Entry:\n");
		fprintf(ofp, "    start_offset: 0x%"FMTPTR"x, "
			"end_offset: 0x%"FMTPTR"x, info_offset: 0x%"FMTPTR"x\n",
			e->start_offset, e->end_offset, e->info_offset);
	} else {
		fprintf(ofp, "Not able to locate unw_table_entry for "
			" for ip 0x%"FMTPTR"x\n", ip);
		return(1);
	}

	/* build state record
	 */
	memset(&sr, 0, sizeof(sr));
	for (r = sr.curr.reg; r < sr.curr.reg + UNW_NUM_REGS; ++r) {
		r->when = UNW_WHEN_NEVER;
	}
/*
	sr.pr_val = info->pr;
*/

	hdr = KL_VREAD_UINT64((t->segment_base + e->info_offset));
	dp = (kaddr_t)(t->segment_base + e->info_offset + 8);
	desc_end = dp + 8 * UNW_LENGTH(hdr);

	fprintf(ofp, "\n");
	fprintf(ofp, "hdr=0x%"FMTPTR"x, dp=0x%"FMTPTR"x, desc_end=0x%"FMTPTR"x\n",
		hdr, (uint64_t)dp, (uint64_t)desc_end);

	while (!sr.done && dp < desc_end) {
		print_code(ofp, dp, sr.in_body);
		dp = unw_decode(dp, sr.in_body, &sr);
	}
	return(0);
}
