/*
 * $Id: dis_s390.c,v 1.1 2004/12/21 23:26:17 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>
#include <stdarg.h>

/*
 * function declarations
 */
static kaddr_t start_instr_s390(kaddr_t, int);
static kaddr_t do_dis_s390(kaddr_t, int, FILE*);
static kaddr_t print_instr_stream_s390(kaddr_t, int, int, int, FILE*);
static int  print_instr_s390(kaddr_t, FILE*, int);


/*
 * function definitions
 */

/*
 * start_instr_s390() - static function to ensure proper padding of instruction
 */
static kaddr_t
start_instr_s390(kaddr_t pc, int before)
{
	/* instruction on s390 can be up to 6 bytes */
	while (before--) {
		pc -= 6;
	}
	return(pc);
}

/*
 * do_dis_s390() - static function to disassemble i instructions 
 */
static kaddr_t
do_dis_s390(kaddr_t value, int lines, FILE *ofp)
{
	bfd_vma pc;
	int i;

	set_dis_ofp(ofp);

        pc = value;
	for (i = 0; i < lines; i++) {
		dis_printintaddr(pc, &DISINFO, 1);
		pc += print_insn_s390(pc, &DISINFO);
		DISINFO.fprintf_func(ofp, "\n");
	}
	return(pc);
}

/*              
 * print_instr_stream_s390() - static function to disassemble an
 *                              instructions stream
 */
static kaddr_t
print_instr_stream_s390(kaddr_t v, int bcount, int acount, int flags,
			FILE *ofp)
{
	int count;
	kaddr_t pc = v;

	/* Make sure that output goes to the right place
	 */ 
	set_dis_ofp(ofp);

	/* 
	 * XXX - don't know how to find out for s390
	 * whether pc is aligned properly or not
	 */
	count = acount + 1;
	if (bcount) {
		pc = start_instr_s390(pc, bcount);
		count += bcount;	
	}
	pc = do_dis_s390(pc, count, ofp);
	return(pc);
}

/* 
 * print_instr_s390() - static function
 */
static int                     
print_instr_s390(kaddr_t pc, FILE *ofp, int flag)
{               
	return(do_dis_s390(pc, 1, ofp) - pc);
}

/*
 * dis_init_s390() - init arch specific stuff for disassembling
 */
int
dis_init_s390(FILE *ofp, int dumparch)
{
	PRINT_INSTR        = print_instr_s390;
	PRINT_INSTR_STREAM = print_instr_stream_s390;
	USE_OPCODE         = 1;
	
	DISINFO.fprintf_func   = dis_fprintf;
	DISINFO.stream         = ofp;
	DISINFO.flavour        = bfd_target_elf_flavour;
	DISINFO.arch           = bfd_arch_s390;
#if defined(DUMP_ARCH_S390)
	if(dumparch == KL_ARCH_S390){
# if defined(bfd_mach_s390_esa)
		DISINFO.mach = bfd_mach_s390_esa;
# elif defined(bfd_mach_s390_31)
		DISINFO.mach = bfd_mach_s390_31;
# else
#  error "Invalid binutils version. (s390 not supported by bfd.h)"
# endif
	}
#endif

#if defined(DUMP_ARCH_S390X)
	if(dumparch == KL_ARCH_S390X){
# if defined(bfd_mach_s390_esame)
		DISINFO.mach = bfd_mach_s390_esame;
# elif defined(bfd_mach_s390_64)
        	DISINFO.mach = bfd_mach_s390_64;
# else
#  error "Invalid binutils version. (s390x not supported by bfd.h)"
# endif
	}
#endif

        DISINFO.endian         = BFD_ENDIAN_BIG;
	DISINFO.read_memory_func       = getidmem;
	DISINFO.print_address_func     = dis_printaddr;
	DISINFO.symbol_at_address_func = dis_getsym;
	/* XXX check how do we have to set display_endianess here */
        DISINFO.display_endian = BFD_ENDIAN_BIG;

	return(0);
}
