/*
 * $Id: cmd_defcpu_i386.c,v 1.2 2005/02/23 19:31:20 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

int defcpu = -1;

/*
 * deftask_cmd() -- Run the 'deftask' command.
 */
int
defcpu_cmd(command_t *cmd)
{
	kl_dump_header_i386_t dha;
	int cpu;

	if (cmd->nargs == 0) {
		if (defcpu == -1) {
			fprintf(cmd->efp, "No default cpu set\n");
		} else {
			fprintf(cmd->ofp, "Default cpu is %d\n", defcpu);
		}
		return(0);
	} 

	if (CORE_IS_KMEM) {
		fprintf(cmd->efp, "Can't use this command on live system\n");
		return (1);
	}
	if (kl_get_dump_header_i386(&dha))
		return (1);

	cpu = strtol(cmd->args[0], NULL, 10);

	if (cpu >= dha.smp_num_cpus) {
		fprintf(cmd->efp, "Error setting defcpu to %s\n", cmd->args[0]);
		return (1);
	}
	defcpu = cpu;
	fprintf(cmd->ofp, "Default cpu is %d\n", defcpu);

	if (dha.stack[defcpu]) {
		deftask = (kaddr_t)dha.smp_current_task[defcpu];
		fprintf(cmd->ofp, "Default task is 0x%"FMTPTR"x\n", deftask);
	}
	return (0);
}

#define _DEFCPU_USAGE	"[-w outfile] [cpu]"

/*
 * defcpu_usage() -- Print the usage string for the 'defcpu' command.
 */
void
defcpu_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _DEFCPU_USAGE);
}

/*
 * defcpu_help() -- Print the help information for the 'defcpu' command.
 */
void
defcpu_help(command_t *cmd)
{
	CMD_HELP(cmd, _DEFCPU_USAGE,
	"Set the default cpu if one is indicated. Otherwise print the "
	"value of default cpu." 
        "When 'lcrash' is run on a live system, defcpu has no "
        "meaning.\n\n"
	"This command also sets the default task to the task running "
	"on the default cpu at the time the dump is taken. "
	"The rd command will display the registers on the default cpu "
	"at the time the dump is taken. "
        "The trace command will display a trace wrt the task "
        "running on the default cpu at the time the dump is taken. ");
}

/*
 * defcpu_parse() -- Parse the command line arguments for 'defcpu'.
 */
int
defcpu_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_WRITE), 0)) {
		return(1);
	}
	return(0);
}
