/*
 * cmd_defcpu_arm.c
 * 
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Ported from cmd_defcpu_i386.c
 * by Fleming Feng (fleming.feng@intel.com).
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright (C) 2003, Intel Corp. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

int defcpu = -1;

/*
 * deftask_cmd() -- Run the 'deftask' command.
 */
int
defcpu_cmd(command_t *cmd)
{
	kl_dump_header_arm_t* dha = NULL;
	int cpu;

	if (cmd->nargs == 0) {
		if (defcpu == -1) {
			fprintf(cmd->efp, "No default cpu set\n");
		} else {
			fprintf(cmd->ofp, "Default cpu is %d\n", defcpu);
		}
		return(0);
	} 

	if (CORE_IS_KMEM) {
		fprintf(cmd->efp, "Can't use this command on live system\n");
		return (1);
	}

	dha = kl_alloc_block(sizeof(kl_dump_header_arm_t), K_TEMP);
	if (dha == NULL){
		KL_ERROR = KLE_NO_MEMORY;
		return (1);
	}

	if (kl_get_dump_header_arm(dha)){
		kl_free_block(dha);
		return (1);
	}

	cpu = strtol(cmd->args[0], NULL, 10);
	if (cpu < 0){
		fprintf(cmd->efp, "Error negative cpu number input!\n");
		kl_free_block(dha);
		return(1);
	}

	if (cpu >= KL_GET_UINT32(&(dha->smp_num_cpus))) {
		fprintf(cmd->efp, "Error setting defcpu to %s\n", cmd->args[0]);
		kl_free_block(dha);
		return (1);
	}
	defcpu = cpu;
	fprintf(cmd->ofp, "Default cpu is %d\n", defcpu);

	if (KL_GET_UINT32(&(dha->stack[defcpu]))) {
		deftask = (kaddr_t)
	  	(KL_GET_UINT32(&(dha->smp_current_task[defcpu])));
		fprintf(cmd->ofp, "Default task is %#"FMTPTR"x\n", deftask);
	}
	kl_free_block(dha);
	return (0);
}

#define _DEFCPU_USAGE	"[-w outfile] [cpu]"

/*
 * defcpu_usage() -- Print the usage string for the 'defcpu' command.
 */
void
defcpu_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _DEFCPU_USAGE);
}

/*
 * defcpu_help() -- Print the help information for the 'defcpu' command.
 */
void
defcpu_help(command_t *cmd)
{
	CMD_HELP(cmd, _DEFCPU_USAGE,
	"Set the default cpu if one is indicated. Otherwise print the "
	"value of default cpu." 
        "When 'lcrash' is run on a live system, defcpu has no "
        "meaning.\n\n"
	"This command also sets the default task to the task running "
	"on the default cpu at the time the dump is taken. "
	"The rd command will display the registers on the default cpu "
	"at the time the dump is taken. "
        "The trace command will display a trace wrt the task "
        "running on the default cpu at the time the dump is taken. ");
}

/*
 * defcpu_parse() -- Parse the command line arguments for 'defcpu'.
 */
int
defcpu_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_WRITE), 0)) {
		return(1);
	}
	return(0);
}
