/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <qcombobox.h>
#include <qlineedit.h>
#include <qmultilineedit.h>
#include <qcheckbox.h>
#include <kdebug.h>
#include <knuminput.h>
#include <kcolorbutton.h>
#include <kapplication.h>
#include <klocale.h>
#include "kxstitch.h"
#include "patternpropertiesdialog.h"
#include "configuration.h"

PatternPropertiesDialog::PatternPropertiesDialog(QWidget* parent)
  : PatternPropertiesDlg(parent)
{
  reset();
  connect(SizeUnits,SIGNAL(activated(const QString&)),this,SLOT(slotUnitsChanged(const QString&)));
  connect(HelpButton,SIGNAL(clicked()),this,SLOT(slotContextHelp()));
}

void PatternPropertiesDialog::show()
{
  Configuration::setComboBoxItem(SizeUnits,m_sizeUnits);
  m_sizeDisplayUnits = m_sizeUnits;
  calculateSizes(m_sizeWidth,m_sizeHeight,m_clothCount,m_clothCountUnits);
  PatternTitle->setText(m_patternTitle);
  PatternAuthor->setText(m_patternAuthor);
  PatternCopyright->setText(m_patternCopyright);
  PatternFabric->setText(m_patternFabric);
  FabricColor->setColor(m_fabricColor);
  FlossScheme->clear();
  FlossScheme->insertStringList(((KXStitchApplication*)kapp)->schemes());
  Configuration::setComboBoxItem(FlossScheme,m_flossScheme);
  PatternInstructions->setText(m_patternInstructions);
  SetDefault->setChecked(false);
  PatternPropertiesDlg::show();
}

void PatternPropertiesDialog::reset()
{
  m_sizeDisplayUnits = m_sizeUnits = KXSConfig().NewPattern_Units;
  m_sizeWidth = KXSConfig().NewPattern_Width;
  m_sizeHeight = KXSConfig().NewPattern_Height;
  m_clothCount = KXSConfig().NewPattern_ClothCount;
  m_clothCountUnits = KXSConfig().NewPattern_ClothCountUnits;
  m_patternTitle = KXSConfig().NewPattern_Title;
  m_patternAuthor = KXSConfig().NewPattern_Author;
  m_patternCopyright = KXSConfig().NewPattern_Copyright;
  m_patternFabric = KXSConfig().NewPattern_Fabric;
  m_fabricColor = KXSConfig().NewPattern_Color;
  m_flossScheme = KXSConfig().NewPattern_FlossScheme;
  m_patternInstructions = KXSConfig().NewPattern_Instructions;
}

void PatternPropertiesDialog::setUnits(QString sizeUnits)
{
  m_sizeDisplayUnits = m_sizeUnits = sizeUnits;
}

void PatternPropertiesDialog::setPatternWidth(int width)
{
  m_sizeWidth = width;
}

void PatternPropertiesDialog::setPatternHeight(int height)
{
  m_sizeHeight = height;
}

void PatternPropertiesDialog::setPatternSize(int width,int height)
{
  m_sizeWidth = width;
  m_sizeHeight = height;
}

void PatternPropertiesDialog::setClothCount(double clothCount)
{
  m_clothCount = clothCount;
}

void PatternPropertiesDialog::setClothCountUnits(QString clothCountUnits)
{
  m_clothCountUnits = clothCountUnits;
}

void PatternPropertiesDialog::setTitle(QString title)
{
  m_patternTitle = title;
}

void PatternPropertiesDialog::setAuthor(QString author)
{
  m_patternAuthor = author;
}

void PatternPropertiesDialog::setCopyright(QString copyright)
{
  m_patternCopyright = copyright;
}

void PatternPropertiesDialog::setFabric(QString fabric)
{
  m_patternFabric = fabric;
}

void PatternPropertiesDialog::setFabricColor(QColor fabricColor)
{
  m_fabricColor = fabricColor;
}

void PatternPropertiesDialog::setScheme(QString scheme)
{
  m_flossScheme = scheme;
}

void PatternPropertiesDialog::setInstructions(QString instructions)
{
  m_patternInstructions = instructions;
}

QString PatternPropertiesDialog::units()
{
  return m_sizeUnits;
}

int PatternPropertiesDialog::patternWidth()
{
  return m_sizeWidth;
}

int PatternPropertiesDialog::patternHeight()
{
  return m_sizeHeight;
}

QSize PatternPropertiesDialog::patternSize()
{
  return QSize(m_sizeWidth,m_sizeHeight);
}

double PatternPropertiesDialog::clothCount()
{
  return m_clothCount;
}

QString PatternPropertiesDialog::clothCountUnits()
{
  return m_clothCountUnits;
}

QString PatternPropertiesDialog::title()
{
  return m_patternTitle;
}

QString PatternPropertiesDialog::author()
{
  return m_patternAuthor;
}

QString PatternPropertiesDialog::copyright()
{
  return m_patternCopyright;
}

QString PatternPropertiesDialog::fabric()
{
  return m_patternFabric;
}

QColor PatternPropertiesDialog::fabricColor()
{
  return m_fabricColor;
}

QString PatternPropertiesDialog::scheme()
{
  return m_flossScheme;
}

QString PatternPropertiesDialog::instructions()
{
  return m_patternInstructions;
}

void PatternPropertiesDialog::accept()
{
  double w = SizeWidth->value();
  double h = SizeHeight->value();
  double cc = ClothCount->value();
  if (m_sizeDisplayUnits != "Stitches")
  {
    w *= cc;
    h *= cc;
  }
  m_sizeUnits = m_sizeDisplayUnits;
  m_sizeWidth = int(w);
  m_sizeHeight = int(h);
  m_clothCount = cc;
  m_clothCountUnits = ClothCount->suffix();
  m_patternTitle = PatternTitle->text();
  m_patternAuthor = PatternAuthor->text();
  m_patternCopyright = PatternCopyright->text();
  m_patternFabric = PatternFabric->text();
  m_fabricColor = FabricColor->color();
  m_flossScheme = FlossScheme->currentText();
  m_patternInstructions = PatternInstructions->text();
  if (SetDefault->isChecked())
  {
    KXSConfig().NewPattern_Units = m_sizeUnits;
    KXSConfig().NewPattern_Width = m_sizeWidth;
    KXSConfig().NewPattern_Height = m_sizeHeight;
    KXSConfig().NewPattern_ClothCount = m_clothCount;
    KXSConfig().NewPattern_ClothCountUnits = m_clothCountUnits;
    KXSConfig().NewPattern_Title = m_patternTitle;
    KXSConfig().NewPattern_Author = m_patternAuthor;
    KXSConfig().NewPattern_Copyright = m_patternCopyright;
    KXSConfig().NewPattern_Fabric = m_patternFabric;
    KXSConfig().NewPattern_Color = m_fabricColor;
    KXSConfig().NewPattern_FlossScheme = m_flossScheme;
    KXSConfig().NewPattern_Instructions = m_patternInstructions;
    KXSConfig().writeSettings();
  }
  PatternPropertiesDlg::accept();
}

void PatternPropertiesDialog::slotContextHelp()
{
  kapp->invokeHelp("PatternPropertiesDialog");
}

void PatternPropertiesDialog::slotUnitsChanged(const QString& sizeUnits)
{
  double  cc = ClothCount->value();
  QString ccu = ClothCount->suffix();
  double w = SizeWidth->value();
  double h = SizeHeight->value();
  if (m_sizeDisplayUnits != i18n("Stitches"))
  {
    w *= cc;
    h *= cc;
  }
  if (sizeUnits == i18n("CM"))
  {
    if (ccu == "/in")
    {
      ccu = "/cm";
      cc /= 2.54;
    }
  }
  if (sizeUnits == i18n("Inches"))
  {
    if (ccu == "/cm")
    {
      ccu = "/in";
      cc = (double)(int)(cc*2.54);
    }
  }
  m_sizeDisplayUnits = sizeUnits;
  calculateSizes(int(w),int(h),cc,ccu);
}

void PatternPropertiesDialog::calculateSizes(int w,int h,double cc,QString ccu)
{
  double  scale = cc;
  double  sizeStep = 0.01;
  if (SizeUnits->currentText() == i18n("Stitches"))
  {
    scale = 1.0;
    sizeStep = 1.0;
  }
  SizeWidth->setRange(1,100000,sizeStep,false);
  SizeWidth->setValue(w/scale);
  SizeHeight->setRange(1,100000,sizeStep,false);
  SizeHeight->setValue(h/scale);
  ClothCount->setRange(1,55,(ccu == "/cm")?0.01:1.0,false);
  ClothCount->setValue(cc);
  ClothCount->setSuffix(ccu);
}

QDataStream& operator<<(QDataStream& stream,const PatternPropertiesDialog& properties)
{
  stream << properties.m_sizeUnits;
  stream << Q_INT32(properties.m_sizeWidth);
  stream << Q_INT32(properties.m_sizeHeight);
  stream << properties.m_clothCount;
  stream << properties.m_clothCountUnits;
  stream << properties.m_patternTitle;
  stream << properties.m_patternAuthor;
  stream << properties.m_patternCopyright;
  stream << properties.m_patternFabric;
  stream << properties.m_fabricColor;
  stream << properties.m_flossScheme;
  stream << properties.m_patternInstructions;
  return stream;
}

QDataStream& operator>>(QDataStream& stream,PatternPropertiesDialog& properties)
{
  Q_INT32 w,h;
  stream >> properties.m_sizeUnits;
  stream >> w;
  properties.m_sizeWidth = w;
  stream >> h;
  properties.m_sizeHeight = h;
  stream >> properties.m_clothCount;
  stream >> properties.m_clothCountUnits;
  stream >> properties.m_patternTitle;
  stream >> properties.m_patternAuthor;
  stream >> properties.m_patternCopyright;
  stream >> properties.m_patternFabric;
  stream >> properties.m_fabricColor;
  stream >> properties.m_flossScheme;
  stream >> properties.m_patternInstructions;
  return stream;
}
