/*
    This file is part of kio_obex.

    Copyright (c) 2003 Mathias Froehlich <Mathias.Froehlich@web.de>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qstring.h>
#include <qwidget.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qlistbox.h>
#include <qwidgetstack.h>
#include <krestrictedline.h>

#include <klocale.h>

#include "bluetoothdevicewidget.h"
#include "irdadevicewidget.h"
#include "serialdevicewidget.h"
#include "ipdevicewidget.h"

#include "wizard.h"

Wizard::Wizard( QWidget* parent, const char* name, bool modal, WFlags fl )
  : QWizard( parent, name, modal, fl )
{
  int margin = 11;
  int space = 6;

  setCaption( i18n( "Configure New OBEX Device" ) );

  // Prepare page 0 of the wizard.
  mWizardPage0 = new QWidget( this, "mWizardPage0" );
  mWizardPage0Layout = new QHBoxLayout( mWizardPage0, margin, space, "mWizardPage0Layout");

  mAliasInputText = new QLabel( mWizardPage0, "mAliasInputText" );
  mAliasInputText->setText( i18n( "Choose an alias:" ) );
  mWizardPage0Layout->addWidget( mAliasInputText );

  mAliasInput = new KRestrictedLine( mWizardPage0, "mAliasInput" );
  mAliasInput->setValidChars( "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_-" );
  mAliasInput->setFocus();
  mAliasInput->setText( QString::null );
  mWizardPage0Layout->addWidget( mAliasInput );

  addPage( mWizardPage0, i18n( "OBEX Connection Configuration" ) );

  // Prepare page 1 of the wizard.
  mWizardPage1 = new QWidget( this, "mWizardPage1" );
  mWizardPage1Layout = new QHBoxLayout( mWizardPage1, margin, space, "mWizardPage1Layout");

  mTransportInputText = new QLabel( mWizardPage1, "mTransportInputText" );
  mTransportInputText->setText( i18n( "Choose a connection transport:" ) );
  mWizardPage1Layout->addWidget( mTransportInputText );

  mTransportInputBox = new QListBox( mWizardPage1, "mTransportInputBox" );
//   mTransportInputBox->setFocus();
  mTransportInputBox->insertItem( i18n( "Bluetooth" ), 0 );
#if !defined(HAVE_QOBEX_BLUETOOTH)
  mTransportInputBox->item( 0 )->setSelectable( false );
#endif
  mTransportInputBox->insertItem( i18n( "IrDA" ), 1 );
#if !defined(HAVE_QOBEX_IRDA)
  mTransportInputBox->item( 1 )->setSelectable( false );
#endif
  mTransportInputBox->insertItem( i18n( "Serial Line - Ericsson Mobile Protocol" ) );
  mTransportInputBox->insertItem( i18n( "Serial Line - Siemens Mobile Protocol" ) );
  mTransportInputBox->insertItem( i18n( "Serial Line" ) );
  mTransportInputBox->insertItem( i18n( "TCP/IP" ) );
  mWizardPage1Layout->addWidget( mTransportInputBox );

  addPage( mWizardPage1, i18n( "OBEX Transport Selection" ) );


  // Prepare page 2 of the wizard
  mWidgetStack = new QWidgetStack( this, "mWidgetStack" );

  mSerial = new SerialDeviceWidget( mWidgetStack, "SerialTransportPage" );
  mWidgetStack->addWidget( mSerial, SerialConfigurationPage );


  // Prepare page 2 for irda transports
  mIrDA = new IrDADeviceWidget( mWidgetStack, "IrDATransportPage" );
  mWidgetStack->addWidget( mIrDA, IrDAConfigurationPage );

  // Prepare page 2 for bluetooth transports
  mBluetooth = new BluetoothDeviceWidget( mWidgetStack, "BluetoothTransportPage" );
  mWidgetStack->addWidget( mBluetooth, BluetoothConfigurationPage );

  // Prepare page 2 of the wizard for ip connections
  mIp = new IpDeviceWidget( mWidgetStack, "IpTransportPage" );
  mWidgetStack->addWidget( mIp, IpConfigurationPage );

  addPage( mWidgetStack, i18n( "OBEX Connection Parameters" ) );


  // signals and slots connections
  connect( mAliasInput, SIGNAL(textChanged(const QString&)),
	   SLOT(slotAliasInput(const QString&)) );
  connect( mTransportInputBox, SIGNAL(selectionChanged()),
	   SLOT(slotTransportSelectionChanged()) );
  connect( mTransportInputBox, SIGNAL(highlighted(int)),
	   SLOT(slotTransportHighlighted(int)) );

  connect( mSerial, SIGNAL(configurationReady(bool)),
	   SLOT(slotPageThree(bool)) );
  connect( mIrDA, SIGNAL(configurationReady(bool)),
	   SLOT(slotPageThree(bool)) );
  connect( mBluetooth, SIGNAL(configurationReady(bool)),
	   SLOT(slotPageThree(bool)) );
  connect( mIp, SIGNAL(configurationReady(bool)),
	   SLOT(slotPageThree(bool)) );

  // tab order
  // FIXME ...
  setTabOrder( (QWidget*)mAliasInput, (QWidget*)finishButton() );

  setNextEnabled( page( 0 ), false );
  setNextEnabled( page( 1 ), false );
}

Wizard::~Wizard()
{
}

QString Wizard::name() const
{
  return mAliasInput->text();
}

QString Wizard::transport() const
{
  switch ( mTransportInputBox->index( mTransportInputBox->selectedItem() ) ) {
  case 0:
    return "bluetooth";
  case 1:
    return "irda";
  case 2:
    return "serialericsson";
  case 3:
    return "serialsiemens";
  case 4:
    return "serial";
  case 5:
    return "ip";
  }
  return QString::null;
}

QString Wizard::peer() const
{
  if ( transport() == "irda" )
    return mIrDA->address();
  else if ( transport() == "bluetooth" )
    return mBluetooth->address();
  else
    return mIp->peer();
}

QString Wizard::device() const
{
  return mSerial->device();
}

int Wizard::ipPort() const
{
  return mIp->ipPort();
}

int Wizard::speed() const
{
  return mSerial->speed();
}

void Wizard::slotAliasInput( const QString & str )
{
  setNextEnabled( page( 0 ), 0 < str.length() );
}

void Wizard::slotTransportSelectionChanged()
{
  setNextEnabled( page( 1 ), mTransportInputBox->selectedItem() );
}

void Wizard::slotPageThree( bool ok )
{
  setFinishEnabled( page( 2 ), ok );
}

void Wizard::slotTransportHighlighted( int idx )
{
  switch ( idx ) {
  case 0:
    mWidgetStack->raiseWidget( int(BluetoothConfigurationPage) );
    mBluetooth->checkConfiguration();
    break;
  case 1:
    mWidgetStack->raiseWidget( int(IrDAConfigurationPage) );
    mIrDA->checkConfiguration();
    break;
  case 2:
  case 4:
    mWidgetStack->raiseWidget( int(SerialConfigurationPage) );
    mSerial->setAvailableSpeeds( SerialDeviceWidget::AllSpeeds );
    mSerial->checkConfiguration();
    break;
  case 3:
    mWidgetStack->raiseWidget( int(SerialConfigurationPage) );
    mSerial->setAvailableSpeeds( SerialDeviceWidget::BFBSpeeds );
    mSerial->checkConfiguration();
    break;
  case 5:
    mWidgetStack->raiseWidget( int(IpConfigurationPage) );
    mIp->checkConfiguration();
    break;
  }
}

#include "wizard.moc"
