//-*-c++-*-
/***************************************************************************
 *   Copyright (C) 2003 by Fred Schaettgen                                 *
 *   kbluetoothd@schaettgen.de                                             *
 *   Derived from kinetd, (C) 2002 by Tim Jansen <tim@tjansen.de>
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifndef _METASERVER_H_
#define _METASERVER_H_

#include <dcopobject.h>
#include <ksock.h>
#include <kprocess.h>
#include <qstringlist.h>
#include <qstring.h>
#include <qdatetime.h>
#include <qtimer.h>
#include <qguardedptr.h>
#include "portlistener.h"
#include <libkbluetooth/hcisocket.h>
#include <libkbluetooth/deviceaddress.h>
#include <qmap.h>

class KSystemTray;
class Confirmation;

class MetaServer : public QObject, virtual public DCOPObject
{
    Q_OBJECT
    K_DCOP
k_dcop:

    /**
     * Returns a list of all registered services in KBluetoothd.
     * To add a service you need to add a .desktop file with
     * the servicetype "KBluetoothDModule" into the services director
     * (see kinetdmodule.desktop in servicetypes dir).
     * @return a list with the names of all services
     */
    QStringList services();

    /**
     * Returns true if the service exists and is available.
     * @param service name of a service as specified in its .desktop file
     * @return true if a service with the given name exists and is enabled
     */
    bool isEnabled(QString service);

    /**
     * Enables or disables the given service. Ignored if the given service
     * does not exist.
     * @param service name of a service as specified in its .desktop file
     * @param enable true to enable, false to disable.
     */
    void setEnabled(QString service, bool enable);

    /**
     * Enables or disables authentication for the given service.
     * @param service name of a service as specified in its .desktop file
     * @param enable true to enable, false to disable.
     */
    void setAuthentication(QString service, bool enable);
    bool getAuthentication(QString service);
        
    /**
     * Enables or disables encryption for the given service.
     * @param service name of a service as specified in its .desktop file
     * @param enable true to enable, false to disable.
     */
    void setEncryption(QString service, bool enable);
    bool getEncryption(QString service);
    
    /**
     * Returns a list of resource types this service has allocated
     * @param service Service name
     * @return list of service types for use with resources()
     */
    QStringList resourceTypes(QString service);

    /**
     * Returns the value of an allocated resource like an rfcommchannel.
     * @param service name of a service as specified in its .desktop file
     * @param resourceType Type of an allocated resource
     * @return the resource value or QString::null if resource is not allocated
     */
    QString resources(QString service, QString resourceType);

    /**
     * Reloads the metaserver and reparses the service registration files
     */
    void reload();

    /**
     * Tests whether the given service is installed..
     * @param service name of a service as specified in its .desktop file
     * @return true if installed, false otherwise
     */
    bool isInstalled(QString service);

    /**
     * Enables or disables the SLP registration. Ignored if the service does
     * not have a service URL. If the service is disabled the service will
     * registered as soon as it is enabled.
     * @param service name of a service as specified in its .desktop file
     * @param enable true to enable, false to disable.
     */
    void setServiceRegistrationEnabled(QString service, bool enabled);

    /**
     * Returns true if service registration for the given service is enabled.
     * Note that this does not mean that the service is currently registered,
     * because the service may be disabled.
     * @param service name of a service as specified in its .desktop file
     * @return true if service registration is enabled
     */
    bool isServiceRegistrationEnabled(QString service);

    /**
     * Returns an relative path pointing to the documentation of the given service
     * @param service name of a service as specified in its .desktop file
     * @return Documentation path
     */
    QString docPath(QString service);
    
    /**
     * Returns a short description of the service
     * @param service name of a service as specified in its .desktop file
     * @return description
     */
    QString description(QString service);
    
    /**
     * Tells if a given service can be configured
     */
    bool canConfigure(QString service);
    
    /**
     * Configure the given service 
     */
    void configure(QString service);

    /**
     * Returns a list of bluetooth addresses for which some default setting has
     * been changed.
     * @return Device address list
     */
    //QStringList customizedDevices();
    
    int numConnectionRules();
    QString getRuleAddress(int ruleNumber);
    QString getRuleService(int ruleNumber);
    QString getRulePolicy(int ruleNumber);
    bool insertConnectionRule(int ruleNumber, QString addr, QString service, QString policy);
    bool deleteConnectionRule(int ruleNumber);
    QStringList getRulePolicyList();
    
    /**
     * Queries if a device is denied or allowed to connect or if a
     * confirmation dialog will be displayed.
     * @param deviceaddress Device address
     * @param servicename Name of the service as returned by services()
     * @return 0 if access is denied, 1 if access is silently granted,
     * 2 if a confirmation dialog is displayed.
     */
    //int connectPermissionSetting(QString deviceaddress, QString servicename);

    /**
     * Deicde if a device is denied or allowed to connect or if a
     * confirmation dialog will be displayed.
     * @param deviceaddress Device address
     * @param servicename Name of the service as returned by services()
     * @param setting 0 if access is denied, 1 if access is silently granted,
     * 2 if a confirmation dialog is displayed.
     */
    //void setConnectPermissionSetting(QString deviceaddress,
    //    QString servicename, int setting);

private:
    KConfig *m_config;
    QPtrList<PortListener> m_portListeners;
    QTimer m_reregistrationTimer;
    
private slots:

public:
    MetaServer(QObject *parent);
    virtual ~MetaServer();
    void loadServiceList();
    PortListener *getListenerByName(QString name);
    Confirmation *confirmation;
};


#endif
