#ifndef KADU_THEMES_H
#define KADU_THEMES_H

#include <qmap.h>
#include <qobject.h>
#include <qregexp.h>
#include <qstring.h>
#include <qstringlist.h>

/**
	Klasa obsugujca rne zestawy (jak ikony i emotikony)
	\class Themes
	\brief Klasa zarzdzajca zestawami
**/

class Themes : public QObject
{
	Q_OBJECT

	friend class EmoticonsManager;

	QStringList ThemesList; //!< lista z nazwami dostpnych zestaww
	QStringList ThemesPaths; //!< lista cieek w ktrych szukamy zestaww
	QStringList additional;
	QString ConfigName; //!< nazwa pliku z konfiguracj zestawu
	QString Name; //!< typ zestawu (np. "icons", "emoticons")
	QString ActualTheme; //!< nazwa aktualnie wybranego zestawu
	QMap<QString, QString> entries;

	/**
		\fn QStringList getSubDirs(const QString& path, bool validate = true) const
		Wyszukuje podkatalogi w podanej ciece i zwraca list tych zawierajcych
		pliki konfiguracyjne zestawu. Jeli parametr validate ma warto false,
		zwrcona zostanie lista wszystkich znalezionych podkatalogw.
		\param path cieka do katalogu
		\param validate sprawdzenie zawartoci
		\return lista podkatalogw
	**/
	QStringList getSubDirs(const QString& path, bool validate = true) const;

	/**
		\fn bool validateDir(const QString& path) const
		Sprawdza czy w podanym katalogu, bd jego podkatalogach znajduj
		si pliki z konfiguracj zestawu.
		\param path cieka do katalogu
	**/
	bool validateDir(const QString& path) const;

	QString fixFileName(const QString& path, const QString& fn) const;

	public:
		/**
			\fn Themes(const QString& name, const QString& configname, const char *cname = 0)
			Konstruktor tworzcy nowy zestaw
			\param name typ zestawu
			\param configname plik z konfiguracj zestawu
			\param cname nazwa obiektu
		**/
		Themes(const QString& name, const QString& configname, const char *cname = 0);

		/**
			\fn QStringList defaultKaduPathsWithThemes() const
			Tworzy list cieek do zestaww znajdujcych si w lokalizacjach
			domylnych dla typu zestawu.
			\return lista cieek do zestaww
		**/
		QStringList defaultKaduPathsWithThemes() const;

		/**
			\fn const QStringList &themes() const
			Funkcja zwraca list z nazwami dostpnych zestaww
		**/
		const QStringList &themes() const;

		/**
			\fn const QString &theme() const
			Funkcja zwraca nazw aktualnie wybranego zestawu
		**/
		const QString &theme() const;

		/**
			\fn const QStringList &paths() const
			Funkcja zwraca list cieek, w ktrych wyszukiwane s zestawy
		**/
		const QStringList &paths() const;

		const QStringList &additionalPaths() const;

		/**
			\fn const QStringList &paths() const
			Funkcja zwraca ciek do wskazanego zestawu
		**/
		QString themePath(const QString& theme = QString::null) const;

		QString getThemeEntry(const QString& name) const;
		const QMap<QString, QString> & getEntries() { return entries; }

	public slots:
		/**
			\fn void setTheme(const QString& theme)
			Zmienia aktualny zestaw na wskazany o ile ten istnieje
			\param theme nazwa zestawu
		**/
		void setTheme(const QString& theme);

		/**
			\fn void setPaths(const QStringList& paths)
			Ustawia list cieek, w ktrych wyszukiwane s zestawy
			\param paths lista cieek
		**/
		void setPaths(const QStringList& paths);

	signals:
		/**
			\fn void themeChanged(const QString& theme)
		 	Sygna jest emitowany, gdy zmieniono aktualny zestaw
			\param theme nazwa nowego zestawu
		**/
		void themeChanged(const QString& theme);

		/**
			\fn void pathsChanged(const QStringList& list)
		 	Sygna jest emitowany, gdy zmieniono ciek wyszukiwania
			\param list lista nowych cieek
		**/
		void pathsChanged(const QStringList& list);
};

#endif
