#include <config.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <getopt.h>
#include "common/memory.h"
#include "common/error.h"
#include "common/strbuf.h"
#include "common/string-utils.h"
#include "common/intparse.h"
#include "common/io-utils.h"
#include "icotool.h"

#ifndef HAVE_DECL_PROGRAM_INVOCATION_NAME
char *program_invocation_name;
#endif

#define PROGRAM "icotool"

static int32_t image_index = -1;
static int32_t width = -1;
static int32_t height = -1;
static int32_t bitdepth = -1;
static int32_t palettesize = -1;
static int32_t hotspot_x = 0;
static int32_t hotspot_y = 0;
static bool hotspot_x_set = false;
static bool hotspot_y_set = false;
static int32_t alpha_threshold = 127;
static bool icon_only = false;	
static bool cursor_only = false;
static char *output = NULL;

enum {
	VERSION_OPT	= 1000,
	HELP_OPT,
	ICON_OPT,
	CURSOR_OPT,
};

static char *short_opts = "xlco:i:w:h:p:b:X:Y:t:";
static struct option long_opts[] = {
    { "extract",		 no_argument,       NULL, 'x' 			},
    { "list",	    	 no_argument, 	    NULL, 'l' 			},
    { "create",     	 no_argument,       NULL, 'c' 			},
    { "version", 		 no_argument, 	    NULL, VERSION_OPT	},
    { "help", 	    	 no_argument,	    NULL, HELP_OPT		},
    { "output", 		 required_argument, NULL, 'o'			},
    { "index",	    	 required_argument, NULL, 'i'			},
    { "width",	    	 required_argument, NULL, 'w'			},
    { "height", 		 required_argument, NULL, 'h'			},
    { "palette-size",	 required_argument, NULL, 'p'			},
    { "bit-depth",		 required_argument, NULL, 'b'			},
    { "hotspot-x",       required_argument, NULL, 'X'			},
    { "hotspot-y",       required_argument, NULL, 'Y'			},
    { "alpha-threshold", required_argument, NULL, 't'			},
    { "icon",       	 no_argument,       NULL, ICON_OPT		},
    { "cursor",     	 no_argument,       NULL, CURSOR_OPT	},
    { 0, 0, 0, 0 }
};

static bool
filter(int i, int w, int h, int bd, int ps, bool icon, int hx, int hy)
{
	if (image_index != -1 && i != image_index)
		return false;
	if (width != -1 && w != width)
		return false;
	if (height != -1 && h != height)
		return false;
	if (bitdepth != -1 && bd != bitdepth)
		return false;
	if (palettesize != -1 && ps != palettesize)
		return false;
	if ((icon_only && !icon) || (cursor_only && icon))
		return false;
	if (hotspot_x_set && hx != hotspot_x)
		return false;
	if (hotspot_y_set && hy != hotspot_y)	
		return false;
	return true;
}

static FILE *
create_outfile_gen(char **out)
{
	if (output != NULL) {
		*out = xstrdup(output);
		return fopen(output, "wb");
	}
	if (isatty(STDOUT_FILENO))
		die("refusing to write binary data to terminal");
	*out = xstrdup("(standard out)");
	return stdout;
}

static FILE *
extract_outfile_gen(char **outname, int w, int h, int bc, int i)
{
	char *inname = *outname;

	if (output == NULL || is_directory(output)) {
		char *inbase;
	
		*outname = strbuf_new();
		if (output != NULL) {
			strbuf_append(outname, output);
			if (!ends_with(output, "/"))
				strbuf_append(outname, "/");
		}
		inbase = strrchr(inname, '/');
		inbase = (inbase == NULL ? inname : inbase+1);
		if (ends_with_nocase(inbase, ".ico") || ends_with_nocase(inbase, ".cur")) {
			strbuf_append_substring(outname, inbase, 0, strlen(inbase)-4);
		} else {
			strbuf_append(outname, inbase);
		}
		strbuf_appendf(outname, "_%d_%dx%dx%d.png", i, w, h, bc);
		*outname = strbuf_free_to_string(outname);
		return fopen(*outname, "wb");
	}
	else if (strcmp(output, "-") == 0) {
		*outname = xstrdup("(standard out)");
		return stdout;
	}

	*outname = xstrdup(output);
	return fopen(output, "wb");
}

static void
display_version(void)
{
	printf("%s (%s) %s\n\
Written by %s.\n\
\n\
Copyright (C) %s.\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n",
		PROGRAM, PACKAGE, VERSION, AUTHORS, COPYRIGHT);
}

static void
display_help(void)
{
	printf("Usage: %s [OPTION]... [FILE]...\n\
Convert and create Win32 icon (.ico) and cursor (.cur) files.\n\
\n\
Commands:\n\
  -x, --extract                extract images from files\n\
  -l, --list                   print a list of images in files\n\
  -c, --create                 create an icon file from specified files\n\
      --help                   display this help and exit\n\
      --version                output version information and exit\n\
\n\
Options:\n\
  -i, --index=NUMBER           match index of image (first is 1)\n\
  -w, --width=PIXELS           match width of image\n\
  -h, --height=PIXELS          match height of image\n\
  -p, --palette-size=COUNT     match number of colors in palette (or 0)\n\
  -b, --bit-depth=COUNT        match number of bits per pixel\n\
  -X, --hotspot-x=COORD        match or set cursor hotspot x-coordinate\n\
  -Y, --hotspot-y=COORD        match or set cursor hotspot y-coordinate\n\
  -t, --alpha-threshold=LEVEL  highest level in alpha channel indicating\n\
                               transparent image portions (default is 127)\n\
      --icon                   match icons only\n\
      --cursor                 match cursors only\n\
  -o, --output=PATH            where to place extracted files\n\
\n\
Report bugs to <%s>.\n", program_invocation_name, BUG_EMAIL);
}

static bool
open_file_or_stdin(char *name, FILE **outfile, char **outname)
{
	if (strcmp(name, "-") == 0) {
		*outfile = stdin;
		*outname = "(standard in)";
	} else {
		*outfile = fopen(name, "rb");
		*outname = name;
		if (*outfile == NULL) {
			warn("%s: cannot open file", name);
			return false;
		}
	}
	return true;
}

int
main(int argc, char **argv)
{
	int c;
	bool list_mode = false;
	bool extract_mode = false;
	bool create_mode = false;
	FILE *in;
	char *inname;

#ifndef HAVE_DECL_PROGRAM_INVOCATION_NAME
    	program_invocation_name = argv[0];
#endif

	while ((c = getopt_long(argc, argv, short_opts, long_opts, NULL)) != -1) {
		switch (c) {
		case 'x':
			extract_mode = true;
			break;
		case 'l':
			list_mode = true;
			break;
		case 'c':
			create_mode = true;
			break;
		case VERSION_OPT:
			display_version();
			exit(0);
			break;
		case HELP_OPT:
			display_help();
			exit(0);
			break;
		case 'o':
			output = optarg;
			break;
		case 'i':
			if (!parse_int32(optarg, &image_index))
				die("invalid index value: %s", optarg);
			break;
		case 'w':
			if (!parse_int32(optarg, &width))
				die("invalid width value: %s", optarg);
			break;
		case 'h':
			if (!parse_int32(optarg, &height))
				die("invalid height value: %s", optarg);
			break;
		case 'p':
			if (!parse_int32(optarg, &palettesize))
				die("invalid palette-size value: %s", optarg);
			break;
		case 'b':
			if (!parse_int32(optarg, &bitdepth))
				die("invalid bit-depth value: %s", optarg);
			break;
		case 'X':
			if (!parse_int32(optarg, &hotspot_x))
				die("invalid hotspot-x value: %s", optarg);
    	    	    	hotspot_x_set = true;
			break;
		case 'Y':
			if (!parse_int32(optarg, &hotspot_y))
				die("invalid hotspot-y value: %s", optarg);
    	    	    	hotspot_y_set = true;
			break;
		case 't':
			if (!parse_int32(optarg, &alpha_threshold))
				die("invalid alpha-threshold value: %s", optarg);
			break;
		case ICON_OPT:
			icon_only = true;
			break;
		case CURSOR_OPT:
			cursor_only = true;
			break;
		case '?':
			exit(1);
		}
	}

	if (extract_mode + create_mode + list_mode > 1)
		die("multiple commands specified");
	if (extract_mode + create_mode + list_mode == 0)
		die("missing argument\nTry `%s --help' for more information.", argv[0]);
	if (icon_only && cursor_only)
		die("only one of --icon and --cursor may be specified");

	if (list_mode) {
		if (argc-optind <= 0)
			die("missing file argument");
	    for (c = optind ; c < argc ; c++) {
			if (open_file_or_stdin(argv[c], &in, &inname)) {
				if (!extract_icons(in, inname, true, NULL, filter))
					exit(1);
				if (in != stdin)
					fclose(in);
			}
		}
	}

	if (extract_mode) {
		if (argc-optind <= 0)
			die("missing arguments");

	    for (c = optind ; c < argc ; c++) {
			int matched;

			if (open_file_or_stdin(argv[c], &in, &inname)) {
				matched = extract_icons(in, inname, false, extract_outfile_gen, filter);
				if (matched == -1)
					exit(1);
				if (matched == 0)
					fprintf(stderr, "%s: no images matched\n", inname);
				if (in != stdin)
					fclose(in);
			}
		}
	}

	if (create_mode) {
		if (argc-optind <= 0)
			die("missing arguments");
		if (!create_icon(argc-optind, argv+optind, create_outfile_gen, (icon_only ? true : !cursor_only), hotspot_x, hotspot_y, alpha_threshold, bitdepth))
			exit(1);
	}

	exit(0);
}
