#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "dump.h"
#include "types.h"
#include "util.h"
#include "log.h"
#include "irc.h"

extern rl_htable_t *iac_channels;
extern rl_htable_t *iac_users;
extern rl_list_t *iac_server_conns;

extern iac_server_configuration_t iac_cfg;
extern iac_server_status_t iac_status;


void
iac_dump_channel(iac_client_conn_t *src, const char *channame)
{
    iac_channel_t *chan;
    int i;

    if ( !(chan = rl_hashtable_get(iac_channels, channame)) ) {
        iac_irc_notice_user(src, "Channel %s not found\r\n", channame);
        return;
    }

    iac_irc_notice_user(src, "Dumping channel: %s\r\n", (chan->name) ? chan->name : "(null)");
    iac_irc_notice_user(src, "Flags (%x):\r\n", chan->flags);

    iac_irc_notice_user(src, "Owner: %s\r\n",
        (chan->owner) ? chan->owner : "(NULL)");

    for (i=0; i < IAC_TLIST_MAX; i++) {
        iac_irc_notice_user(src, "Topic %02d: (%s:%ld) %s\r\n",
            (i == chan->itopic) ? 99 : i,
            (chan->topic_owner[i]) ? (chan->topic_owner[i]) : "none",
            chan->topic_time[i],
            (chan->topic[i]) ? (chan->topic[i]) : "none");
    }

}

void
iac_dump_user(iac_client_conn_t *src, const char *nick)
{
    iac_client_conn_t *cc;
    iac_channel_link_t *cl;

#ifdef IAC_COUNT_IDLETIME
    char *s;
#endif

    if ( !(cc = rl_hashtable_get(iac_users, nick)) ) {
        iac_irc_notice_user(src, "Nick %s not found\r\n", (nick) ? nick : "(null)");
        return;
    }

    iac_irc_notice_user(src, "Dumping user: %s\r\n", (cc->hostmask) ? cc->hostmask : "(null)");

    iac_irc_notice_user(src, "Nick: %s\r\n",
        (cc->nick) ? cc->nick : "(NULL)");

    iac_irc_notice_user(src, "User: %s\r\n",
        (cc->user) ? cc->user : "(NULL)");

    iac_irc_notice_user(src, "Hostname: %s\r\n",
        (cc->host) ? cc->host : "(NULL)");

    iac_irc_notice_user(src, "Ircname: %s\r\n",
        (cc->ircname) ? cc->ircname : "(NULL)");

    iac_irc_notice_user(src, "IP: %s\r\n",
        (cc->ip) ? cc->ip : "(NULL)");

    iac_irc_notice_user(src, "Port: %d\r\n", cc->port);
    iac_irc_notice_user(src, "Hopcount: %d\r\n", cc->hopcount);

    iac_irc_notice_user(src, "Away Message: %s\r\n",
        (cc->away) ? (cc->away) : "none set");

    iac_irc_notice_user(src, "Server: %s\r\n",
        (cc->server) ? (cc->server) : "(NULL)");

    iac_irc_notice_user(src, "Server info: %s\r\n",
        (cc->server_info) ? (cc->server_info) : "(NULL)");

    iac_irc_notice_user(src, "Quit message: %s\r\n",
        (cc->quitmsg) ? (cc->quitmsg) : "none set");

#ifdef IAC_COUNT_IDLETIME
    s = asctime(localtime(&(cc->signon)));
    if (s[strlen(s) -1] == '\n')
        s[strlen(s) -1] = '\0';

    iac_irc_notice_user(src, "Signon: %ld [%s]\r\n",
        cc->signon, s);

    iac_irc_notice_user(src, "Idle: %ld [%ld seconds]\r\n",
        cc->last_action, (time(0) - cc->last_action));
#endif

    iac_irc_notice_user(src, "Flags: %x\r\n", cc->flags);

    if ( cc->flags & IAC_NICKSENT )
        iac_irc_notice_user(src, " - IAC_NICKSENT\r\n");

    if ( cc->flags & IAC_USERSENT )
        iac_irc_notice_user(src, " - IAC_USERSENT\r\n");

    if ( cc->flags & IAC_OK )
        iac_irc_notice_user(src, " - IAC_OK\r\n");

    if ( cc->flags & IAC_LOCAL )
        iac_irc_notice_user(src, " - IAC_LOCAL\r\n");

    if ( cc->flags & IAC_PONG )
        iac_irc_notice_user(src, " - IAC_PONG\r\n");

    if ( cc->flags & IAC_TOBEREMOVED )
        iac_irc_notice_user(src, " - IAC_TOBEREMOVED\r\n");

    if ( cc->flags & IAC_OPERATOR )
        iac_irc_notice_user(src, " - IAC_OPERATOR\r\n");

    if ( cc->flags & IAC_WALLOPS )
        iac_irc_notice_user(src, " - IAC_WALLOPS\r\n");

    if ( cc->flags & IAC_FORTUNE )
        iac_irc_notice_user(src, " - IAC_FORTUNE\r\n");

    if ( cc->flags & IAC_INVISIBLE )
        iac_irc_notice_user(src, " - IAC_INVISIBLE\r\n");

    iac_irc_notice_user(src, "Joined channels:\r\n");

    cl = cc->channels;
    while (cl) {
        iac_channel_t *chan = cl->link;

        iac_irc_notice_user(src, " - %s\r\n",
            (chan->name) ? (chan->name) : "(NULL)");

        cl = cl->next;
    }
}

void
p_dump_server(iac_client_conn_t *src, iac_server_t *serv)
{
    iac_irc_notice_user(src, "Dumping server: %s\r\n",
        (serv->name) ? serv->name : "(NULL)");

    iac_irc_notice_user(src, "Info: %s\r\n",
        (serv->info) ? serv->info : "(NULL)");

    iac_irc_notice_user(src, "Numeric: %d\r\n", serv->numeric);
    iac_irc_notice_user(src, "Hopcount: %d\r\n", serv->hopcount);

    iac_irc_notice_user(src, "Flags: %x\r\n", serv->flags);

    if (serv->flags & IAC_TOBEREMOVED)
        iac_irc_notice_user(src, " - IAC_TOBEREMOVED\r\n");
}

void
iac_dump_server(iac_client_conn_t *src, const char *servername)
{
    iac_server_t *s;

    s = iac_status.servers;
    while ( s ) {

        if ( servername ) {

            if ( iac_match_mask(servername, s->name) )
                p_dump_server(src, s);
        } else {
            p_dump_server(src, s);
        }

        s = s->next;
    }
}


void
p_dump_server_connection(iac_client_conn_t *src, iac_server_conn_t *sc)
{
    if ( !sc )
        return;

    iac_irc_notice_user(src, "Dumping server connection %s\r\n",
        (sc->hostname) ? sc->hostname : "(NULL)");

    iac_irc_notice_user(src, "Numeric: %d\r\n", sc->numeric);

    iac_irc_notice_user(src, "Info: %s\r\n",
        (sc->info) ? sc->info : "(NULL)");

    iac_irc_notice_user(src, "IP: %s\r\n",
        (sc->ip) ? sc->ip : "(NULL)");

    iac_irc_notice_user(src, "Init time: %s\r\n",
        asctime(localtime(&(sc->time_init))));

    iac_irc_notice_user(src, "Hopcount: %d\r\n", sc->hops);

    iac_irc_notice_user(src, "Ping (seconds): %d\r\n", sc->pingtime);

    iac_irc_notice_user(src, "Port: %d\r\n", sc->port);

    iac_irc_notice_user(src, "Socket: %d\r\n", sc->sock);

    iac_irc_notice_user(src, "Queue length: %u\r\n", sc->queued);
    iac_irc_notice_user(src, "Bytes sent: %u\r\n", sc->bytes_sent);
    iac_irc_notice_user(src, "Messages sent: %u\r\n", sc->msgs_sent);
    iac_irc_notice_user(src, "Bytes received: %u\r\n", sc->bytes_recvd);
    iac_irc_notice_user(src, "Messages received: %u\r\n", sc->msgs_recvd);

    iac_irc_notice_user(src, "Flags: %x\r\n", sc->flags);

    if ( sc->flags & IAC_PASSSENT )
        iac_irc_notice_user(src, " - IAC_PASSSENT\r\n");

    if ( sc->flags & IAC_AUTH )
        iac_irc_notice_user(src, " - IAC_AUTH\r\n");

    if ( sc->flags & IAC_OK )
        iac_irc_notice_user(src, " - IAC_OK\r\n");

    if ( sc->flags & IAC_AUTH_OK )
        iac_irc_notice_user(src, " - IAC_AUTH_OK\r\n");

    if ( sc->flags & IAC_MASTER )
        iac_irc_notice_user(src, " - IAC_MASTER\r\n");

    if ( sc->flags & IAC_PONG )
        iac_irc_notice_user(src, " - IAC_PONG\r\n");

    if ( sc->flags & IAC_TOBEREMOVED )
        iac_irc_notice_user(src, " - IAC_TOBEREMOVED\r\n");

    if ( sc->flags & IAC_USEMSGID )
        iac_irc_notice_user(src, " - IAC_USEMSGID\r\n");

    if ( sc->flags & IAC_COMPRESSED )
        iac_irc_notice_user(src, " - IAC_COMPRESSED\r\n");

    iac_irc_notice_user(src, "Current message being sent: %s\r\n",
        (sc->msg) ? sc->msg : "(NULL)");
}


void
iac_dump_server_connection(iac_client_conn_t *src, const char *name)
{
    rl_list_item_t *li;

    li = iac_server_conns->rl_list_start;
    while ( li ) {
        iac_server_conn_t *sc = (iac_server_conn_t *) li->rli_data;

        if ( name ) {

            if ( iac_match_mask(name, sc->hostname) )
                p_dump_server_connection(src, sc);

        } else  {
            p_dump_server_connection(src, sc);
        }
        
        li = li->rli_next;
    }
}

void
p_dump_block(iac_client_conn_t *src, iac_block_t *b)
{
    iac_irc_notice_user(src, "Dumping block %s\r\n", (b->mask) ? b->mask : "(null)");
    iac_irc_notice_user(src, "Reason: %s\r\n", (b->reason) ? b->reason : "(null)");
    iac_irc_notice_user(src, "#Matches: %u\r\n", b->matches);
    iac_irc_notice_user(src, "Next pointer: %p\r\n",  b->next);
}

#ifdef HAVE_SCRIPTS
void
p_dump_script(iac_client_conn_t *src, iac_script_t *s)
{
    iac_irc_notice_user(src, "Dumping script %s\r\n", (s->nick) ? s->nick : "(null)");
    iac_irc_notice_user(src, "Nick: %s\r\n", (s->nick) ? s->nick : "(null)");
    iac_irc_notice_user(src, "Command: %s\r\n", (s->cmd) ? s->cmd : "(null)");
    iac_irc_notice_user(src, "Char Regex: %s\r\n", (s->char_regex) ? s->char_regex : "(null)");
    iac_irc_notice_user(src, "Next Pointer: %p\r\n", s->next);
}
#endif

void
iac_dump_block(iac_client_conn_t *src, const char *name)
{
    iac_block_t *b;

    b = iac_cfg.blocked;
    while ( b ) {

        if (name) {

            if ( iac_match_mask(name, b->mask) )
                p_dump_block(src, b);

        } else {
            p_dump_block(src, b);
        }

        b = b->next;
    }
}

void
p_dump_link(iac_client_conn_t *src, iac_link_line_t *ll)
{
    iac_irc_notice_user(src, "Dumping link: %d\r\n", ll->numeric);
    iac_irc_notice_user(src, "Host: %s\r\n",
        (ll->host) ? ll->host : "*");
    iac_irc_notice_user(src, "Port: %d\r\n", ll->port);
    iac_irc_notice_user(src, "Password: %s\r\n",
        (ll->password) ? ll->password : "*");
    iac_irc_notice_user(src, "Options: %s\r\n",
        (ll->options) ? ll->options : "*");
    iac_irc_notice_user(src, "Mask: %s\r\n",
        (ll->mask) ? ll->mask : "*");
    iac_irc_notice_user(src, "Flags: %x\r\n", ll->flags);

    if ( ll->flags & IAC_AUTOCONNECT )
        iac_irc_notice_user(src, " - IAC_AUTOCONNECT\r\n");

    if ( ll->flags & IAC_CONNECTED )
        iac_irc_notice_user(src, " - IAC_CONNECTED\r\n");

    if ( ll->flags & IAC_FAILED )
        iac_irc_notice_user(src, " - IAC_FAILED\r\n");

    iac_irc_notice_user(src, "Next pointer: %p\r\n", ll->next);
}

void
iac_dump_link(iac_client_conn_t *src, const char *name)
{
    iac_link_line_t *ll;

    ll = iac_cfg.links;
    while ( ll ) {

        if (name) {
            if (iac_match_mask(name, ll->host))
                p_dump_link(src, ll);
        } else {
            p_dump_link(src, ll);
        }

        ll = ll->next;
    }
}

void
iac_dump_script(iac_client_conn_t *src, const char *name)
{
#ifdef HAVE_SCRIPTS
    iac_script_t *s;

    for (s = iac_cfg.scripts; s; s = s->next )
    {
        if ( name )
        {
            if ( iac_match_mask( name, s->nick ) )
                p_dump_script( src, s );
        }
        else
        {
            p_dump_script( src, s );
        }
    }
#endif
}
