/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* board_info.c
 *
 * Copyright (C) 2001  JP Rosevear
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: JP Rosevear
 */

#include <config.h>
#include <gnome.h>
#include "makros.h"
#include "timer.h"
#include "board_info.h"

struct _BoardInfoPrivate 
{
	gboolean flip;

	GtkWidget *wcont;
	GtkWidget *bcont;
	
	GtkWidget *white_name;
	GtkWidget *black_name;
	GtkWidget *white_time;
	GtkWidget *black_time;
	GtkWidget *result;

	GameResult resultval;
};

enum {
	WHITE_EXPIRED_SIGNAL,
	BLACK_EXPIRED_SIGNAL,
	LAST_SIGNAL
};
static gint signals [LAST_SIGNAL] = { 0 };

static GtkVBoxClass *parent_class = NULL;

static void class_init (BoardInfoClass *klass);
static void init (BoardInfo *info);

static void bi_white_expired_cb (GtkWidget *widget, gpointer data);
static void bi_black_expired_cb (GtkWidget *widget, gpointer data);

GtkType
board_info_get_type (void)
{
  static GtkType type = 0;

  if (type == 0)
    {
      static const GtkTypeInfo info =
      {
        "BoardInfo",
        sizeof (BoardInfo),
        sizeof (BoardInfoClass),
        (GtkClassInitFunc) class_init,
        (GtkObjectInitFunc) init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      type = gtk_type_unique (gtk_vbox_get_type (), &info);
    }

  return type;
}

static void
class_init (BoardInfoClass *klass)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (klass);

	parent_class = gtk_type_class (gtk_vbox_get_type ());

	signals [WHITE_EXPIRED_SIGNAL] =
		gtk_signal_new ("white_expired", 
				GTK_RUN_FIRST, object_class->type,
				GTK_SIGNAL_OFFSET (BoardInfoClass, white_expired),
				gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

	signals [BLACK_EXPIRED_SIGNAL] =
		gtk_signal_new ("black_expired", 
				GTK_RUN_FIRST, object_class->type,
				GTK_SIGNAL_OFFSET (BoardInfoClass, black_expired),
				gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}

static void
init (BoardInfo *info)
{
	BoardInfoPrivate *priv;
	GtkStyle *style;

	priv = g_new0 (BoardInfoPrivate, 1);

	info->priv = priv;

	/* Set defaults */
	gtk_box_set_homogeneous (GTK_BOX (info), TRUE);
	gtk_box_set_spacing (GTK_BOX (info), 2);

	/* Players */
	priv->white_name = gtk_label_new (_("White"));
	gtk_label_set_justify (GTK_LABEL (priv->white_name), GTK_JUSTIFY_LEFT);
	gtk_widget_show (priv->white_name);

	priv->black_name = gtk_label_new (_("Black"));
	gtk_label_set_justify (GTK_LABEL (priv->black_name), GTK_JUSTIFY_LEFT);
	gtk_widget_show (priv->black_name);

	/* The Timers */
	priv->white_time = timer_new (300);
	priv->black_time = timer_new (300);

	gtk_signal_connect (GTK_OBJECT (priv->white_time), "expired",
			    (GtkSignalFunc) bi_white_expired_cb, info);
	gtk_signal_connect (GTK_OBJECT (priv->black_time), "expired",
			    (GtkSignalFunc) bi_black_expired_cb, info);

	gtk_widget_show (priv->white_time);
	gtk_widget_show (priv->black_time);

	/* The Result */
	priv->result = gtk_label_new ("");
	priv->resultval = GAME_NONE;
	gtk_widget_show (priv->result);

	/* Style */
	style = gtk_style_new ();
	style->font = gdk_font_load ("-adobe-courier-medium-r-normal-*-*-180-*-*-m-*-iso8859-1");
	gtk_widget_set_style (priv->white_name, style);	
	gtk_widget_set_style (priv->black_name, style);

	style->font = gdk_font_load ("-adobe-courier-medium-r-normal-*-*-240-*-*-m-*-iso8859-1");
	gtk_widget_set_style (priv->white_time, style);	
	gtk_widget_set_style (priv->black_time, style);
	gtk_style_unref (style);
	
	priv->wcont = gtk_vbox_new (TRUE,  2);
	gtk_box_pack_start (GTK_BOX (priv->wcont), priv->white_name, 
			    TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (priv->wcont), priv->white_time,
			    TRUE, TRUE, 0);
	gtk_widget_show (priv->wcont);
	
	priv->bcont = gtk_vbox_new (TRUE, 2);
	gtk_box_pack_start (GTK_BOX (priv->bcont), priv->black_name,
			    TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (priv->bcont), priv->black_time,
			    TRUE, TRUE, 0);
	gtk_widget_show (priv->bcont);
	
	/* Pack the widgets */
	gtk_box_pack_start (GTK_BOX (info), priv->bcont, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (info), priv->result, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (info), priv->wcont, FALSE, FALSE, 0);
}



GtkWidget *
board_info_new (void)
{
	return gtk_type_new (BOARD_TYPE_INFO);	
}

void
board_info_set_flip (BoardInfo *info, gboolean flip)
{
	BoardInfoPrivate *priv;
	
	priv = info->priv;

	if (priv->flip == flip)
		return;

	gtk_box_reorder_child (GTK_BOX (info), priv->bcont, flip ? 2 : 0);
	gtk_box_reorder_child (GTK_BOX (info), priv->wcont, flip ? 0 : 2);
	
	priv->flip = flip;
}

void
board_info_set_game_info (BoardInfo *info, EngineInfo *gameinfo)
{
	BoardInfoPrivate *priv;
	
	priv = info->priv;

	board_info_set_white_name (info, gameinfo->white);
	board_info_set_black_name (info, gameinfo->black);
	board_info_set_white_time (info, gameinfo->white_time);
	board_info_set_black_time (info, gameinfo->black_time);
	board_info_set_result (info, gameinfo->result);

	if (gameinfo->result == GAME_PROGRESS) {
		if (gameinfo->tomove == WHITE) {
			timer_start (TIMER (priv->white_time));
			timer_stop (TIMER (priv->black_time));
		} else if (gameinfo->tomove == BLACK) {
			timer_start (TIMER (priv->black_time));
			timer_stop (TIMER (priv->white_time));
		}
	} else {
		timer_stop (TIMER (priv->white_time));
		timer_stop (TIMER (priv->black_time));
	}
}

gchar *
board_info_get_white_name (BoardInfo *info) 
{
	BoardInfoPrivate *priv;
	gchar *str;

	priv = info->priv;
	
	gtk_label_get (GTK_LABEL (priv->white_name), &str);

	return str;
}

void
board_info_set_white_name (BoardInfo *info, const gchar *name)
{
	BoardInfoPrivate *priv;

	priv = info->priv;
	
	gtk_label_set_text (GTK_LABEL (priv->white_name), name);
}

gchar *
board_info_get_black_name (BoardInfo *info)
{
	BoardInfoPrivate *priv;
	gchar *str;

	priv = info->priv;

	gtk_label_get (GTK_LABEL (priv->black_name), &str);

	return str;
}

void
board_info_set_black_name (BoardInfo *info, const gchar *name)
{
	BoardInfoPrivate *priv;
	
	priv = info->priv;
	
	gtk_label_set_text (GTK_LABEL (priv->black_name), name);
}

gint
board_info_get_white_time (BoardInfo *info)
{
	BoardInfoPrivate *priv;
	
	priv = info->priv;

	return timer_get (TIMER (priv->white_time));
}

void
board_info_set_white_time (BoardInfo *info, gint secs)
{
	BoardInfoPrivate *priv;
	
	priv = info->priv;

	timer_set (TIMER (priv->white_time), secs);
}

gint
board_info_get_black_time (BoardInfo *info)
{
	BoardInfoPrivate *priv;
	
	priv = info->priv;

	return timer_get (TIMER (priv->black_time));
}

void
board_info_set_black_time (BoardInfo *info, gint secs)
{
	BoardInfoPrivate *priv;
	
	priv = info->priv;

	timer_set (TIMER (priv->black_time), secs);
}

gint
board_info_get_result (BoardInfo *info)
{
	BoardInfoPrivate *priv;
	
	priv = info->priv;

	return priv->resultval;
}

void
board_info_set_result (BoardInfo *info, GameResult result)
{
	BoardInfoPrivate *priv;
	const char *text = NULL;
	
	priv = info->priv;

	switch (result) {
	case GAME_PROGRESS:
		text = _("Playing");
		break;
	case GAME_PAUSED:
		text = _("Paused");
		break;
	case GAME_ABORTED:
		text = _("Aborted");
		break;
	case GAME_WHITE:
		text = _("1-0");
		break;
	case GAME_BLACK:
		text = _("0-1");
		break;
	case GAME_DRAW:
		text = _("1/2-1/2");
		break;
	default:
		text = "";
	}
	gtk_label_set_text (GTK_LABEL (priv->result), text);
		
	priv->resultval = result;
}

static void
bi_white_expired_cb (GtkWidget *widget, gpointer data)
{
	BoardInfo *info = BOARD_INFO (data);
	
	gtk_signal_emit (GTK_OBJECT (info), signals [WHITE_EXPIRED_SIGNAL]);
}

static void
bi_black_expired_cb (GtkWidget *widget, gpointer data)
{
	BoardInfo *info = BOARD_INFO (data);
	
	gtk_signal_emit (GTK_OBJECT (info), signals [BLACK_EXPIRED_SIGNAL]);
}
