/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
|  4/22/2001  Solaris code contributed by Daisuke Yabuki <dxy@acm.org>
|  1/23/2001  Hajimu UMEMOTO added get_bufspace(), improved
|               read_freebsd_meminfo().
| 10/12/2000  NetBSD code contributed by Anthony Mallet
|		<anthony.mallet@useless-ficus.net>
|  5/27/2000  Patch from William Carrel <william.a@carrel.org> and 
|               Hajimu UMEMOTO ume@mahoroba.org to cleanup FreeBSD swap code.
|  2/25/2000  FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"

#include "pixmaps/mem/krell_buffers.xpm"
#include "pixmaps/mem/krell_cache.xpm"

  /* The mem monitor has two extra krells which can be themed.
  |  So, the theme "mem" subdir can have:
  |      bg_panel.png
  |      krell.png
  |      krell_buffers.png		# an extension image, defaults to included xpm
  |      krell_cache.png		# an extension image, defaults to included xpm
  |
  |  The gkrellmrc can have theme extension variables for these extra krells:
  |      set_integer mem_krell_buffers_yoff n
  |      set_integer mem_krell_buffers_depth n
  |      set_integer mem_krell_buffers_x_hot n
  |      set_integer mem_krell_cache_yoff n
  |      set_integer mem_krell_cache_depth n
  |      set_integer mem_krell_cache_x_hot n
  */

#if defined(G_HAVE_GINT64)
typedef guint64	memint;
#else
typedef gulong	memint;
#endif

typedef struct
	{
	Panel		*panel;
	Krell		*krell_used,	/* Meter styled, shows fraction used	*/
				*krell_delta,	/* Chart styled, for page in/out deltas */
				*krell_buffers,
				*krell_cache;
	gchar		*label,
				*data_format;	/* Format string for scrolling text		*/
	gint		style_id;
	gint		x_label;
	Decal		*decal_label;
	gboolean	label_is_data,
				restore_label,	/* Helper to know when to toggle data fmt off*/
				mouse_entered,
				all_krells;
	gint		enabled;
	Launcher	launch;

	/* (*(read_system_meminfo))() fills in this data
	*/
	memint	total,		/* Total memory or swap in system */
			used,		/* Amount of memory (calulated) or swap used  */
			x_used,		/* Raw kernel value, not used by swap monitor */
			free,		/* Not used by swap	monitor */
			shared,		/* Not used by swap	monitor */
			buffers,	/* Not used by swap	monitor */
			cached;		/* Not used by swap	monitor */
	}
	MeminfoMeter;

MeminfoMeter	mem,
				swap;

typedef struct
	{
	GtkWidget	*vbox;
	Chart		*chart;
	ChartData	*in_cd,
				*out_cd;
	ChartConfig	*chart_config;
	gboolean	enabled;
	gboolean	extra_info;

	/* (*(read_system_meminfo))() fills in this data
	*/
	gulong		page_in,
				page_out;
	}
	MeminfoChart;

MeminfoChart	swap_chart;


void	(*read_system_meminfo)();

gint force_meminfo_update();


/* ====== System dependent interface ====================================== */

/* ------- FreeBSD ----------------------------------------------------- */

#if defined(__FreeBSD__)
#include <osreldate.h>
#include <kvm.h>
#include <limits.h>
#include <sys/conf.h>
#if __FreeBSD_version < 400000
#include <sys/rlist.h>
#endif
#include <sys/vmmeter.h>
#include <sys/sysctl.h>
#include <vm/vm_param.h>

static struct nlist nl[] = {
#define N_CNT		0
	{ "_cnt" },
#if __FreeBSD_version < 400000
#define VM_SWAPLIST	1
	{ "_swaplist" },
#define VM_SWDEVT	2
	{ "_swdevt" },
#define VM_NSWAP	3
	{ "_nswap" },
#define VM_NSWDEV	4
	{ "_nswdev" },
#define VM_DMMAX	5
	{ "_dmmax" },
#if  __FreeBSD_version < 300000
#define N_BUFSPACE	6
	{ "_bufspace" },
#endif
#endif
	{ "" }
};

extern	kvm_t	*kvmd;
extern	char	errbuf[];

static int
swapmode(memint *retavail, memint *retfree)
	{
	memint used, avail;
#if  __FreeBSD_version >= 400000
	static int psize = -1;
	struct kvm_swap kvmswap;
#else
	char *header;
	int hlen, nswap, nswdev, dmmax;
	int i, div, nfree, npfree;
	struct swdevt *sw;
	long blocksize, *perdev;
	u_long ptr;
	struct rlist head;
#  if __FreeBSD_version >= 220000
	struct rlisthdr swaplist;
#  else 
	struct rlist *swaplist;
#  endif
	struct rlist *swapptr;
#endif

	/*
	 * Counter for error messages. If we reach the limit,
	 * stop reading information from swap devices and
	 * return zero. This prevent endless 'bad address'
	 * messages.
	 */
	static int warning = 10;

	if (warning <= 0)
		{
		/* a single warning */
		if (!warning)
	    		{
			warning--;
			fprintf(stderr, "Too much errors, stop reading swap devices ...\n");
			}
		return(0);
		}
	warning--;		/* decrease counter, see end of function */

	if (kvmd == NULL)
		return(0);
#if  __FreeBSD_version >= 400000
	if (kvm_getswapinfo(kvmd, &kvmswap, 1, 0) < 0)
		{
		fprintf(stderr, "kvm_getswapinfo failed\n");
		return(0);
		}

	if (psize < 0)
	    psize = getpagesize();
	*retavail = avail = (quad_t)kvmswap.ksw_total * psize;
	used = (quad_t)kvmswap.ksw_used * psize;
	*retfree = avail - used;
#else
	if (kvm_read(kvmd, nl[VM_NSWAP].n_value,
		     &nswap, sizeof(nswap)) != sizeof(nswap))
		return(0);
	if (!nswap)
		{
		fprintf(stderr, "No swap space available\n");
		return(0);
		}

	if (kvm_read(kvmd, nl[VM_NSWDEV].n_value,
		     &nswdev, sizeof(nswdev)) != sizeof(nswdev))
		return(0);
	if (kvm_read(kvmd, nl[VM_DMMAX].n_value,
		     &dmmax, sizeof(dmmax)) != sizeof(dmmax))
		return(0);
	if (kvm_read(kvmd, nl[VM_SWAPLIST].n_value,
		     &swaplist, sizeof(swaplist)) != sizeof(swaplist))
		return(0);

	if ((sw = (struct swdevt *)malloc(nswdev * sizeof(*sw))) == NULL ||
	    (perdev = (long *)malloc(nswdev * sizeof(*perdev))) == NULL)
		{
		perror("malloc");
		exit(1);
		}
	if (kvm_read(kvmd, nl[VM_SWDEVT].n_value,
		     &ptr, sizeof ptr) != sizeof ptr)
		return(0);
	if (kvm_read(kvmd, ptr,
		     sw, nswdev * sizeof(*sw)) != nswdev * sizeof(*sw))
		return(0);

	/* Count up swap space. */
	nfree = 0;
	memset(perdev, 0, nswdev * sizeof(*perdev));
#if  __FreeBSD_version >= 220000
	swapptr = swaplist.rlh_list;
	while (swapptr)
#else
	while (swaplist)
#endif
		{
		int	top, bottom, next_block;
#if  __FreeBSD_version >= 220000
		if (kvm_read(kvmd, (u_long)swapptr, &head,
			     sizeof(struct rlist)) != sizeof(struct rlist))
			return (0);
#else
		if (kvm_read(kvmd, (u_long)swaplist, &head,
			     sizeof(struct rlist)) != sizeof(struct rlist))
			return (0);
#endif

		top = head.rl_end;
		bottom = head.rl_start;

		nfree += top - bottom + 1;

		/*
		 * Swap space is split up among the configured disks.
		 *
		 * For interleaved swap devices, the first dmmax blocks
		 * of swap space some from the first disk, the next dmmax
		 * blocks from the next, and so on up to nswap blocks.
		 *
		 * The list of free space joins adjacent free blocks,
		 * ignoring device boundries.  If we want to keep track
		 * of this information per device, we'll just have to
		 * extract it ourselves.
		 */

		while (top / dmmax != bottom / dmmax)
			{
			next_block = ((bottom + dmmax) / dmmax);
			perdev[(bottom / dmmax) % nswdev] +=
				next_block * dmmax - bottom;
			bottom = next_block * dmmax;
			}
		perdev[(bottom / dmmax) % nswdev] +=
			top - bottom + 1;

#if  __FreeBSD_version >= 220000
		swapptr = head.rl_next;
#else
		swaplist = head.rl_next;
#endif
		}

	header = getbsize(&hlen, &blocksize);
	div = blocksize / 512;
	avail = npfree = 0;
	for (i = 0; i < nswdev; i++)
		{
		int xsize, xfree;

		/*
		 * Don't report statistics for partitions which have not
		 * yet been activated via swapon(8).
		 */

		xsize = sw[i].sw_nblks;
		xfree = perdev[i];
		used = xsize - xfree;
		npfree++;
		avail += xsize;
		}

	/* 
	 * If only one partition has been set up via swapon(8), we don't
	 * need to bother with totals.
	 */
	*retavail = avail << 9;
	*retfree = nfree << 9;
	used = avail - nfree;
	free(sw); free(perdev);
#endif /* __FreeBSD_version >= 400000 */

	/* increase counter, no errors occurs */
	warning++; 

	return  (int)(((double)used / (double)avail * 100.0) + 0.5);
	}

static int
get_bufspace(memint *bufspacep)
	{
#if  __FreeBSD_version < 300000
	u_int	bufspace;

	if (kvm_read(kvmd, nl[N_BUFSPACE].n_value, (char *)&bufspace,
		     sizeof(bufspace)) != sizeof(bufspace))
		return 0;
#else
	static char	*name = "vfs.bufspace";
	static int	oid_name2oid[2] = { 0, 3 };
	static int	oid_bufspace[CTL_MAXNAME + 2];
	static size_t	oid_bufspace_len = sizeof(oid_bufspace);
	static gint	initialized = 0;
	u_int		bufspace;
	size_t		bufspace_len = sizeof(bufspace);

	if (!initialized)
		{
		if (sysctl(oid_name2oid, 2, oid_bufspace,
			   &oid_bufspace_len, (void *)name, strlen(name)) < 0)
			return 0;
		oid_bufspace_len /= sizeof(int);
		++initialized;
		}

	if (sysctl(oid_bufspace, oid_bufspace_len,
		   &bufspace, &bufspace_len, 0, 0) < 0)
		return 0;
#endif
	*bufspacep = bufspace;
	return 1;	
	}

#if __FreeBSD_version >= 410000
struct mibtab {
    char	*name;
    int		oid[CTL_MAXNAME + 2];
    size_t	oid_len;
    u_int	value;
    size_t	value_len;
};

static struct mibtab mibs[] = {
#define MIB_V_PAGE_COUNT	0
    { "vm.stats.vm.v_page_count" },
#define MIB_V_FREE_COUNT	1
    { "vm.stats.vm.v_free_count" },
#define MIB_V_WIRE_COUNT	2
    { "vm.stats.vm.v_wire_count" },
#define MIB_V_ACTIVE_COUNT	3
    { "vm.stats.vm.v_active_count" },
#define MIB_V_INACTIVE_COUNT	4
    { "vm.stats.vm.v_inactive_count" },
#define MIB_V_CACHE_COUNT	5
    { "vm.stats.vm.v_cache_count" },
#define MIB_V_SWAPPGSIN		6
    { "vm.stats.vm.v_swappgsin" },
#define MIB_V_SWAPPGSOUT	7
    { "vm.stats.vm.v_swappgsout" },
    { NULL }
};

#define	PROC_MEMINFO_FILE	"/compat/linux/proc/meminfo"
#endif

static void
read_freebsd_meminfo()
	{
	static gint	psize, pshift = 0;
	static gint	first_time_done = 0;
	static gint	swappgsin = -1;
	static gint	swappgsout = -1;
	gint		dpagein, dpageout;
	struct vmmeter	sum;
	unsigned long	buffers;
	struct vmtotal	vmt;
	size_t		length_vmt = sizeof(vmt);
	static int	oid_vmt[] = { CTL_VM, VM_METER };
#if __FreeBSD_version >= 410000
	static int	oid_name2oid[2] = { 0, 3 };
	gint		i;
	FILE		*f;
	gchar		buf[160];
#endif

	/* Collecting meminfo data is expensive under FreeBSD, so
	|  take extra precautions to minimize reading it.
	*/
	if (!GK.ten_second_tick && !force_meminfo_update())
		return;

	if (pshift == 0)
		{
		for (psize = getpagesize(); psize > 1; psize >>= 1)
			pshift++;
		}

	if (kvmd == NULL)
		{
#if __FreeBSD_version >= 410000
		if (!first_time_done)
			{
			for (i = 0; mibs[i].name; ++i)
				{
				mibs[i].oid_len = sizeof(mibs[i].oid);
				if (sysctl(oid_name2oid, 2, mibs[i].oid,
					   &mibs[i].oid_len,
					   (void *)mibs[i].name,
					   strlen(mibs[i].name)) < 0)
				 	return;
				mibs[i].oid_len /= sizeof(int);
				mibs[i].value_len = sizeof(mibs[i].value);
				}
			++first_time_done;
			}
		for (i = 0; mibs[i].name; ++i)
			if (sysctl(mibs[i].oid, mibs[i].oid_len,
				   &mibs[i].value,
				   &mibs[i].value_len, 0, 0) < 0)
				return;
		mem.total = (mibs[MIB_V_PAGE_COUNT].value -
			     mibs[MIB_V_WIRE_COUNT].value) << pshift;
		mem.x_used = (mibs[MIB_V_ACTIVE_COUNT].value +
			      mibs[MIB_V_INACTIVE_COUNT].value) << pshift;
		mem.free = mibs[MIB_V_FREE_COUNT].value << pshift;
		if (sysctl(oid_vmt, 2, &vmt, &length_vmt, NULL, 0) == 0)
			mem.shared = vmt.t_rmshr << pshift;
		get_bufspace(&mem.buffers);
		mem.cached = mibs[MIB_V_CACHE_COUNT].value << pshift;
		mem.used = mem.x_used - mem.buffers - mem.cached;

		swap_chart.page_in = mibs[MIB_V_SWAPPGSIN].value;
		swap_chart.page_out = mibs[MIB_V_SWAPPGSOUT].value;

		/* Try linprocfs for swap info */
		if ((f = fopen(PROC_MEMINFO_FILE, "r")) == NULL)
			return;
		/* total: used: free: shared: buffers: cached: */
		while ((fgets(buf, sizeof(buf), f)) != NULL)
			{
			if (strncmp(buf, "Swap:", 5) == 0)
				{
				sscanf(buf, "Swap: %lu %lu",
				       &swap.total, &swap.used);
				break;
				}
			}
		fclose(f);
#endif
		return;
		}

	if (nl[0].n_type == 0)
		if (kvm_nlist(kvmd, nl) < 0 || nl[0].n_type == 0)
			return;
	if (kvm_read(kvmd, nl[N_CNT].n_value, (char *)&sum,
		     sizeof(sum)) != sizeof(sum))
		return;

	mem.total = (sum.v_page_count - sum.v_wire_count) << pshift;
	mem.x_used = (sum.v_active_count + sum.v_inactive_count) << pshift;
	mem.free = sum.v_free_count << pshift;
	if (sysctl(oid_vmt, 2, &vmt, &length_vmt, NULL, 0) == 0)
		mem.shared = vmt.t_rmshr << pshift;
	get_bufspace(&mem.buffers);
	mem.cached = sum.v_cache_count << pshift;
	if (swappgsin < 0)
		{
		dpagein = 0;
		dpageout = 0;
		}
	else
		{
		dpagein = (sum.v_swappgsin - swappgsin) << (pshift - 10);
		dpageout = (sum.v_swappgsout - swappgsout) << (pshift - 10);
		}
	swappgsin = sum.v_swappgsin;
	swappgsout = sum.v_swappgsout;

	if (dpagein > 0 || dpageout > 0 || first_time_done == 0)
		{
		swapmode(&swap.total, &swap.used);
		swap.used = swap.total - swap.used;
		}
	first_time_done = 1;
	mem.used = mem.x_used - mem.buffers - mem.cached;
	swap_chart.page_in = swappgsin;
	swap_chart.page_out = swappgsout;
	}
#endif


/* ------- Linux ----------------------------------------------------- */
#if defined(__linux__)

#define	PROC_MEMINFO_FILE	"/proc/meminfo"

  /* Kernels >= 2.5.x have tagged formats only in kb units.
  */
static void
tagged_format_meminfo(gchar *buf, memint *mint)
	{
#if defined(G_HAVE_GINT64)
	sscanf(buf,"%*s %Lu", mint);
#else
	sscanf(buf,"%*s %lu", mint);
#endif
	*mint *= 1024;
	}

static void
read_proc_meminfo(void)
	{
	FILE		*f;
	gchar		buf[160];
	gboolean	using_tagged = FALSE;

	/* Reading /proc/meminfo is expensive under Linux kernel 2.2, so
	|  take extra precautions to minimize reading it.  If kernel 2.4
	|  improves this situation, I will later speed this up for it.
	*/
	if (GK.ten_second_tick || force_meminfo_update())
		{
		if ((f = fopen(PROC_MEMINFO_FILE, "r")) != NULL)
			{		/* total:    used:    free:  shared: buffers:  cached: */
			while ((fgets(buf, sizeof(buf), f)) != NULL)
				{
				if (buf[0] == 'M')
					{
					if (!strncmp(buf, "Mem:", 4))
#if defined(G_HAVE_GINT64)
						sscanf(buf,"Mem: %Lu %Lu %Lu %Lu %Lu %Lu",
							&mem.total, &mem.x_used, &mem.free,
							&mem.shared, &mem.buffers, &mem.cached);
#else
						sscanf(buf,"Mem: %lu %lu %lu %lu %lu %lu",
							&mem.total, &mem.x_used, &mem.free,
							&mem.shared, &mem.buffers, &mem.cached);
#endif
					else if (!strncmp(buf, "MemTotal:", 9))
						{
						tagged_format_meminfo(buf, &mem.total);
						using_tagged = TRUE;
						}
					else if (!strncmp(buf, "MemFree:", 8))
						tagged_format_meminfo(buf, &mem.free);
					else if (!strncmp(buf, "MemShared:", 10))
						tagged_format_meminfo(buf, &mem.shared);
					}
				else if (buf[0] == 'S')
					{
					if (!strncmp(buf, "Swap:", 5))
						{
#if defined(G_HAVE_GINT64)
						sscanf(buf,"Swap: %Lu %Lu", &swap.total, &swap.used);
#else
						sscanf(buf,"Swap: %lu %lu", &swap.total, &swap.used);
#endif
						break;
						}
					else if (!strncmp(buf, "SwapTotal:", 10))
						tagged_format_meminfo(buf, &swap.total);
					else if (!strncmp(buf, "SwapFree:", 9))
						tagged_format_meminfo(buf, &swap.used);
					}
				else if (buf[0] == 'B' && !strncmp(buf, "Buffers:", 8))
					tagged_format_meminfo(buf, &mem.buffers);
				else if (buf[0] == 'C' && !strncmp(buf, "Cached:", 7))
					tagged_format_meminfo(buf, &mem.cached);
				}
			if (using_tagged)
				{
				mem.x_used = mem.total - mem.free;
				swap.used = swap.total - swap.used;
				}
			mem.used = mem.x_used - mem.buffers - mem.cached;
			fclose(f);
			}
		}
	/* swap page in/out data is in /proc/stat and this is read in cpu.c
	|  at full speed where all /proc/stat data is collected.
	|  Just go get what is there.
	*/
	read_stat_swap(&swap_chart.page_in, &swap_chart.page_out);
	}
#endif	/* __linux__ */


/* --------- NetBSD ----------------------------------------------------- */

#if defined(__NetBSD__) || defined(__OpenBSD__)

#include <sys/vmmeter.h>
#include <sys/sysctl.h>
#include <uvm/uvm_extern.h>
#include <kvm.h>

static struct nlist nl[] = {
#define X_UVM_EXP    0
   { "_uvmexp" },
   { NULL }
};

extern	kvm_t	*kvmd;

void
read_netbsd_meminfo()
{
   static int mib[] = { CTL_VM, VM_METER };
   static int pgout, pgin;
   struct vmtotal vmt;
   struct uvmexp uvmexp;
   int len;

   if (kvmd == NULL) return;

   /* get the name list if it's not done yet */
   if (nl[0].n_type == 0) kvm_nlist(kvmd, nl);

   if (nl[0].n_type != 0)
      if (kvm_read(kvmd, nl[X_UVM_EXP].n_value,
		   &uvmexp, sizeof(uvmexp)) != sizeof(uvmexp))
	 memset(&uvmexp, 0, sizeof(uvmexp));

   if (sysctl(mib, 2, &vmt, &len, NULL, 0) < 0)
      memset(&vmt, 0, sizeof(vmt));

   mem.total = (uvmexp.npages - uvmexp.wired) << uvmexp.pageshift;

   /* not sure of what must be computed */
   mem.x_used = (uvmexp.active + uvmexp.inactive) << uvmexp.pageshift;
   mem.free = uvmexp.free << uvmexp.pageshift;
   mem.shared = vmt.t_rmshr << uvmexp.pageshift;

   /* want to see only this in the chat. this could be changed */
   mem.used = uvmexp.active << uvmexp.pageshift;

   /* don't know how to get those values */
   mem.buffers = 0;
   mem.cached = 0;

   /* show only the pages located on the disk and not in memory */
   swap.total = uvmexp.swpages << uvmexp.pageshift;
   swap.used = uvmexp.swpgonly << uvmexp.pageshift;

   /* For page in/out operations, uvmexp struct doesn't seem to be reliable */

   /* if the number of swapped pages that are in memory (inuse - only) is
    * greater that the previous value (pgin), we count this a "page in" */
   if (uvmexp.swpginuse - uvmexp.swpgonly > pgin)
      swap_chart.page_in += uvmexp.swpginuse - uvmexp.swpgonly - pgin;
   pgin = uvmexp.swpginuse - uvmexp.swpgonly;

   /* same for page out */
   if (uvmexp.swpgonly > pgout)
      swap_chart.page_out += uvmexp.swpgonly - pgout;
   pgout = uvmexp.swpgonly;
}

#endif /* __NetBSD__ || __OpenBSD__ */

/* --------- Solaris -------------------------------------------------- */

#if defined(__solaris__)

#include <unistd.h>
#include <kstat.h>
#include <sys/stat.h>
#include <sys/swap.h>

static void
read_solaris_meminfo() {

    gulong pagesize;
    static gulong pageshift = 0, physpages = 0;
    extern kstat_ctl_t *kc;
    kstat_t *ksp;
    kstat_named_t *knp;  

    struct anoninfo ai;

    if (!GK.second_tick)
        return;

    if (pageshift == 0) {
        for (pagesize = sysconf(_SC_PAGESIZE); pagesize > 1; pagesize >>= 1)
            pageshift++;
    }
    if (physpages == 0) {
        physpages = sysconf(_SC_PHYS_PAGES);
    }

    mem.total = physpages << pageshift;

    ksp = kstat_lookup(kc, "unix", -1, "system_pages");
    if (ksp && kstat_read(kc, ksp, NULL) >= 0) {
        knp = (kstat_named_t *)kstat_data_lookup(ksp, "pagesfree");
        if (knp) {
            mem.free = knp->value.ui32 << pageshift;
            mem.used = mem.total - mem.free;
        }
    }

    if (swapctl(SC_AINFO, &ai) == -1) {
        perror("swapctl");
    }
    swap.total = ai.ani_max << pageshift;
    swap.used  = ai.ani_resv << pageshift;

    /* UNIMPLEMENTED */
    swap_chart.page_in = swap_chart.page_out = 0; 

    /* NEED TO BE COMPLETED
     * mem.total, mem.used, (they are used in memory meter)
     * mem.x_used, mem.free, mem.shared, mem.buffers, mem.cached 
     * swap.total, swap.used (they are used in swap meter) 
     * swap_chart.page_in, swap_chart.page_out (for swap pages in/out chart) 
     */ 

}

#endif /* __solaris__ */


/* --------- Others --------------------------------------------------- */
#if defined(USE_LIBGTOP)

#include <glibtop/mem.h>
#include <glibtop/swap.h>

static void
read_glibtop_meminfo(void)
	{
	glibtop_mem		glt_mem;
	glibtop_swap	glt_swap;

	/* I don't know if collecting meminfo data is expensive for other
	|  systems as it is on Linux, so slow down to once/sec just in case.
    */
	if (! GK.second_tick)
		return;

	glibtop_get_mem (&glt_mem);
	mem.total   = (memint) glt_mem.total;
	mem.x_used    = (memint) glt_mem.used;
	mem.free    = (memint) glt_mem.free;
	mem.shared  = (memint) glt_mem.shared;
	mem.buffers = (memint) glt_mem.buffer;
	mem.cached  = (memint) glt_mem.cached;

	/* Not sure why, but glibtop has a used memory calculation:
	|  glt_mem.used = mem.total - mem.free - mem.shared
	|			- mem.buffers;
	|  while the free command calculates a "used" the way I have here:
	*/
	mem.used = mem.x_used - mem.buffers - mem.cached;

	glibtop_get_swap (&glt_swap);
	swap.total = (memint) glt_swap.total;
	swap.used  = (memint) glt_swap.used;
	if (glt_swap.flags
				& ((1 << GLIBTOP_SWAP_PAGEIN) + (1 << GLIBTOP_SWAP_PAGEOUT)))
		{
		swap_chart.page_in = (gulong) glt_swap.pagein;
		swap_chart.page_out = (gulong) glt_swap.pageout;
		}
	else
		{
		swap_chart.page_in = 0;
		swap_chart.page_out = 0;
		}
	}
#endif


/* ----- Pick a system interface ----------------------------------------- */
static gint
setup_meminfo_interface(void)
	{
#if defined(__FreeBSD__)
	read_system_meminfo = read_freebsd_meminfo;
#elif defined(__linux__)
	read_system_meminfo = read_proc_meminfo;
#elif defined(__NetBSD__) || defined(__OpenBSD__)
	read_system_meminfo = read_netbsd_meminfo;
#elif defined(__solaris__)
	read_system_meminfo = read_solaris_meminfo;
#else
	read_system_meminfo = read_glibtop_meminfo;
#endif
	return TRUE;
	}

/* ======================================================================== */

#define	MIN_GRID_RES		20
#define	MAX_GRID_RES		100000
#define DEFAULT_GRID_RES	1000

#define DEFAULT_SWAP_CHART_HEIGHT 20

#define	DEFAULT_FORMAT	_("$t - $f free")

static Monitor		*mon_mem,
					*mon_swap;

static gint			x_scroll,
					x_mon_motion,
					x_moved,
					ascent;

static MeminfoMeter		*mon_in_motion;

#define MEG(x)	((gulong)(((x) + (1 << 19)) >> 20))


  /* Updating the /proc/meminfo meters can be expensive in Linux 2.2,
  |  so I do some dynamic adjustments on how often I do the updates.
  |  I increase the update rate if system activity is detected.
  |  This is an effort to get good meter response and to
  |  not contribute to cpu chart activity during quiet times, ie maintain
  |  a good S/N where I'm the noise.
  */
#define PIPE_SIZE	3
static gint		mem_pipe[PIPE_SIZE],
				swap_pipe[PIPE_SIZE];
static gint		force_update	= TRUE;

gboolean
force_meminfo_update(void)
	{
	gint	i, force;

	force = force_update ? TRUE : FALSE;
	force_update = FALSE;
	if (GK.second_tick)
		{
		for (i = 1; i < PIPE_SIZE; ++i)
			if (mem_pipe[i] || swap_pipe[i])
				force = TRUE;
		if (GK.cpu_sys_activity > 3)
			force = TRUE;
		}
	return force;
	}

static gint
format_meminfo_data(MeminfoMeter *mm, gchar *buf, gint size)
	{
	gulong	t, u, f;
	gchar	*s, *format, *label;
	gint	len;

	--size;
	*buf = '\0';
	label = mm->label;
	t = MEG(mm->total);
	u = MEG(mm->used);
	f = t - u;
	format = mm->data_format;

	for (s = format; *s != '\0' && size > 0; ++s)
		{
		len = 1;
		if (*s == '$' && *(s + 1) != '\0')
			{
			switch(*(s + 1))
				{
				case 'l':
					len = snprintf(buf, size, "%s", label);
					break;
				case 't':
					len = snprintf(buf, size, "%ldM", t);
					break;
				case 'u':
					len = snprintf(buf, size, "%ldM", u);
					break;
				case 'U':
					if (t > 0)
						len = snprintf(buf, size, "%ld%%", 100 * u / t);
					break;
				case 'f':
					len = snprintf(buf, size, "%ldM", f);
					break;
				case 'F':
					if (t > 0)
						len = snprintf(buf, size, "%ld%%", 100 * f / t);
					break;
				case 's':
					if (mm == &mem)
						len = snprintf(buf, size, "%ldM", MEG(mem.shared));
					break;
				case 'b':
					if (mm == &mem)
						len = snprintf(buf, size, "%ldM",MEG(mem.buffers));
					break;
				case 'c':
					if (mm == &mem)
						len = snprintf(buf, size, "%ldM", MEG(mem.cached));
					break;
				default:
					*buf = *s;
					if (size > 1)
						{
						*(buf + 1) = *(s + 1);
						++len;
						}
					break;
				}
			++s;
			}
		else
			*buf = *s;
		size -= len;
		buf += len;
		}
	*buf = '\0';
	return u + 1;		/* A way to know if decal text changed. */
	}

static gint
draw_decal_label(MeminfoMeter *mm, gint draw_to_screen)
	{
	Decal		*d;
	TextStyle	ts_save;
	gchar		buf[128];
	gint		changed;
	gint		w	= 0;

	d = mm->decal_label;
	if (! mm->label_is_data)
		{
		d->x_off = mm->x_label;
		gkrellm_draw_decal_text(mm->panel, d, mm->label, 0);
		}
    else
        {
		ts_save = d->text_style;
		d->text_style = *gkrellm_meter_alt_textstyle(mm->style_id);

		changed = format_meminfo_data(mm, buf, sizeof(buf));
		w = gdk_string_width(d->text_style.font, buf);

		if (w  > d->w)
			{
			d->x_off = d->w / 3 - x_scroll;
			changed = x_scroll + 1;
			}
		else
			d->x_off = 0;

		/* Draw the decal, actual draws occur only if "changed" is different
		|  from last call.  If scrolling, will be so each time, if not
		|  will be so as format_meminfo_data() return value changes.
		*/
		gkrellm_draw_decal_text(mm->panel, d, buf, changed);
		d->text_style = ts_save;
		}
	if (draw_to_screen)
		gkrellm_draw_panel_layers(mm->panel);
	return w;
	}


static void
record_activity(gint *pipe, gint modified)
	{
	gint	i;

	for (i = PIPE_SIZE - 1; i > 0; --i)
		pipe[i] = pipe[i-1];
	pipe[0] = modified;
	}


static size_abbrev_table	swap_blocks_abbrev[] =
	{
	{ KB_SIZE(1),		1,				"%.0f" },
	{ KB_SIZE(20),		KB_SIZE(1),		"%.1fK" },
	{ MB_SIZE(1),		KB_SIZE(1),		"%.0fK" },
	{ MB_SIZE(20),		MB_SIZE(1),		"%.1fM" }
	};


static gchar    *text_format;

static void
format_chart_text(MeminfoChart *mc, gchar *buf, gint size)
	{
	Chart	*cp;
	gchar	c, *s;
	size_t	tbl_size;
	gint	len, in_blocks, out_blocks, blocks;

	--size;
	*buf = '\0';
	cp = mc->chart;
	in_blocks = gkrellm_get_current_chartdata(mc->in_cd);
	out_blocks = gkrellm_get_current_chartdata(mc->out_cd);
	tbl_size = sizeof(swap_blocks_abbrev) / sizeof(size_abbrev_table);
	for (s = text_format; *s != '\0' && size > 0; ++s)
		{
		len = 1;
		if (*s == '$' && *(s + 1) != '\0')
			{
			blocks = -1;
			if ((c = *(s + 1)) == 'T')
				blocks = in_blocks + out_blocks;
			else if (c == 'M')
				blocks = gkrellm_get_chart_scalemax(cp);
			else if (c == 'i')
				blocks = in_blocks;
			else if (c == 'o')
				blocks = out_blocks;
			else
				{
				*buf = *s;
				if (size > 1)
					{
					*(buf + 1) = *(s + 1);
					++len;
					}
				}
			if (blocks >= 0)
				len = format_size_abbrev(buf, size, (gfloat) blocks,
						&swap_blocks_abbrev[0], tbl_size);
			++s;
			}
		else
			*buf = *s;
		size -= len;
		buf += len;
		}
	*buf = '\0';	
	}

static void
draw_extra(MeminfoChart *mc)
    {
    gchar       buf[128];

	if (!mc->extra_info)
		return;
	format_chart_text(mc, buf, sizeof(buf));
	gkrellm_draw_chart_text(mc->chart, DEFAULT_STYLE_ID, buf);	/* XXX */
	}

static void
refresh_chart(MeminfoChart *mc)
	{
	if (mc->chart)
		{
		gkrellm_draw_chartdata(mc->chart);
		draw_extra(mc);
		gkrellm_draw_chart_to_screen(mc->chart);
		}
	}

static gint
cb_extra(GtkWidget *widget, GdkEventButton *ev, gpointer data)
	{
	MeminfoChart	*mc	= (MeminfoChart *) data;

	if (ev->button == 1 && ev->type == GDK_BUTTON_PRESS)
		{
		mc->extra_info = !mc->extra_info;
		gkrellm_config_modified();
		refresh_chart(mc);
		}
	else if (   ev->button == 3
			 || (ev->button == 1 && ev->type == GDK_2BUTTON_PRESS)
			)
		gkrellm_chartconfig_window_create(mc->chart);
	return TRUE;
	}

static void
update_meminfo(void)
	{
	Chart	*cp;
	gulong	u, b, c;
	gint	w_scroll, w;

	if (! (mem.enabled || swap.enabled || swap_chart.enabled))
		return;
	(*read_system_meminfo)();

	if (GK.second_tick)
		{
		MeminfoChart	*mc	= &swap_chart;

		if ((cp = mc->chart) != NULL && GK.second_tick)
			{
			gkrellm_store_chartdata(cp, 0, mc->page_out, mc->page_in);
			refresh_chart(mc);
			}
		}
	if (mem.enabled)
		{
		mem.krell_used->full_scale = (gint) (mem.total >> 12);
		mem.krell_buffers->full_scale = mem.krell_used->full_scale;
		mem.krell_cache->full_scale = mem.krell_used->full_scale;
		u = (gulong) (mem.used >> 12);
		b = u + (gulong)(mem.buffers >> 12);
		c = b + (gulong)(mem.cached >> 12);
		if (   (mem.label_is_data && mon_in_motion && x_moved)
			|| mem.mouse_entered
		   )
			u = b = c = 0;
		gkrellm_update_krell(mem.panel, mem.krell_used, u);
		gkrellm_update_krell(mem.panel, mem.krell_buffers, b);
		gkrellm_update_krell(mem.panel, mem.krell_cache, c);
		record_activity(mem_pipe, mem.krell_used->modified);
		}
	if (swap.enabled)
		{
		swap.krell_used->full_scale = (gint) (swap.total >> 12);
		u = (gulong)(swap.used >> 12);
		if (   (swap.label_is_data && mon_in_motion && x_moved)
			|| swap.mouse_entered
		   )
			u = 0;
		gkrellm_update_krell(swap.panel, swap.krell_used, u);
		record_activity(swap_pipe, swap.krell_used->modified);
		}
	if (swap.krell_delta && swap.enabled)
		gkrellm_update_krell(swap.panel, swap.krell_delta,
				swap_chart.page_in + swap_chart.page_out);

	w = w_scroll = 0;
	if (mem.label_is_data && mon_in_motion != &mem && mem.enabled)
		w_scroll = draw_decal_label(&mem, 1);
	if (swap.label_is_data && mon_in_motion != &swap && swap.enabled)
		{
		if ((w = draw_decal_label(&swap, 1)) > w_scroll)
			w_scroll = w;
		}
	if (!mon_in_motion)
		{
		if (w_scroll > mem.decal_label->w)
			x_scroll = (x_scroll + ((gkrellm_update_HZ() < 7) ? 2 : 1))
						% (w_scroll - mem.decal_label->w / 3);
		else
			x_scroll = 0;
		}
	gkrellm_draw_panel_layers(mem.panel);
	gkrellm_draw_panel_layers(swap.panel);
	}

static gint
meminfo_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	GdkPixmap	*pixmap	= NULL;

	if (widget == mem.panel->drawing_area)
		pixmap = mem.panel->pixmap;
	else if (widget == swap.panel->drawing_area)
		pixmap = swap.panel->pixmap;
	else if (swap_chart.chart && widget == swap_chart.chart->drawing_area)
		pixmap = swap_chart.chart->pixmap;
	if (pixmap)
		gdk_draw_pixmap(widget->window, GK.draw1_GC, pixmap,
				ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				ev->area.width, ev->area.height);
	return FALSE;
	}

static gint
cb_panel_enter(GtkWidget *w, GdkEventButton *ev, MeminfoMeter *mm)
	{
	if (mm->label_is_data)
		mm->mouse_entered = TRUE;
	return TRUE;
	}

static gint
cb_panel_leave(GtkWidget *w, GdkEventButton *ev, MeminfoMeter *mm)
	{
	mm->mouse_entered = FALSE;
	return TRUE;
	}

static gint
cb_panel_release(GtkWidget *widget, GdkEventButton *ev)
	{
    if (ev->button != 2)
		return TRUE;
	if (mon_in_motion)
		{
		if (mon_in_motion->restore_label)
			{
			if (mon_in_motion->label_is_data)
				gkrellm_config_modified();
			mon_in_motion->label_is_data = FALSE;
			draw_decal_label(mon_in_motion, 1);
			}
		mon_in_motion->restore_label = TRUE;
		}
	mon_in_motion = NULL;
	x_moved = FALSE;
	return TRUE;
	}

static gint
cb_panel_press(GtkWidget *widget, GdkEventButton *ev)
    {
    if (ev->button != 2)
		return TRUE;
	if (widget == mem.panel->drawing_area)
		mon_in_motion = &mem;
	else if (widget == swap.panel->drawing_area)
		mon_in_motion = &swap;
	else
		return TRUE;
	if (! mon_in_motion->label_is_data)
		{
		mon_in_motion->label_is_data = TRUE;
		mon_in_motion->restore_label = FALSE;
		mon_in_motion->mouse_entered = TRUE;
		gkrellm_config_modified();
		}
	x_mon_motion = ev->x;
	draw_decal_label(mon_in_motion, 1);
	x_moved = FALSE;
	return TRUE;
	}

static gint
cb_panel_motion(GtkWidget *widget, GdkEventButton *ev)
	{
	GdkModifierType	state;
	Decal			*d;
	GdkFont			*font;
	gchar			buf[128];
	gint			w, x_delta;

	state = ev->state;
	if (   ! mon_in_motion
		|| ! (state & (GDK_BUTTON2_MASK | GDK_BUTTON3_MASK))
		|| ! mon_in_motion->label_is_data
	   )
		{
		mon_in_motion = NULL;
		return TRUE;
		}
	d = mon_in_motion->decal_label;
	font = gkrellm_meter_alt_textstyle(mon_in_motion->style_id)->font;

	format_meminfo_data(mon_in_motion, buf, sizeof(buf));
	w = gdk_string_width(font, buf);
	if (w > d->w)
		{
		x_delta = ev->x - x_mon_motion;
		x_mon_motion = ev->x;
		d->x_off += x_delta;
		if (d->x_off < -w)
			d->x_off = -w;
		if (d->x_off > d->w)
			d->x_off = d->w;
		x_scroll = d->w / 3 - d->x_off;
		if (mem.label_is_data)
			draw_decal_label(&mem, 1);
		if (swap.label_is_data)
			draw_decal_label(&swap, 1);
		mon_in_motion->restore_label = FALSE;
		}
	x_moved = TRUE;
	return TRUE;
	}

static void
setup_scaling(ChartConfig *cf, MeminfoChart *mc)
	{
	Chart	*cp	  = mc->chart;
	gint	res   = DEFAULT_GRID_RES,
			grids = FULL_SCALE_GRIDS;

	if (cp)
		{
		grids = gkrellm_get_chartconfig_fixed_grids(cp->config);
		res = gkrellm_get_chartconfig_grid_resolution(cp->config);
		}
	if (grids == 0)
		grids = FULL_SCALE_GRIDS;

	if (swap.krell_delta)
		swap.krell_delta->full_scale = res * grids / gkrellm_update_HZ();
	}

static void
destroy_chart(MeminfoChart *mc)
	{
	if (! mc->chart)
		return;
	gkrellm_chart_destroy(mc->chart);
	mc->chart = NULL;
	mc->enabled = FALSE;
	}

static void
create_chart(MeminfoChart *mc, gint first_create)
	{
	Chart		*cp;

	if (first_create)
		mc->chart = gkrellm_chart_new0();
	cp = mc->chart;

	gkrellm_set_chart_height_default(cp, DEFAULT_SWAP_CHART_HEIGHT);
	gkrellm_chart_create(mc->vbox, mon_swap, cp, &mc->chart_config);
	mc->out_cd = gkrellm_add_default_chartdata(cp, _("Swap Out"));
	mc->in_cd = gkrellm_add_default_chartdata(cp, _("Swap In"));
	gkrellm_set_draw_chart_function(cp, refresh_chart, mc);
    gkrellm_chartconfig_fixed_grids_connect(cp->config,
                setup_scaling, mc);
    gkrellm_chartconfig_grid_resolution_connect(cp->config,
                setup_scaling, mc);
    gkrellm_chartconfig_grid_resolution_adjustment(cp->config, TRUE,
                0, (gfloat) MIN_GRID_RES, (gfloat) MAX_GRID_RES, 0, 0, 0, 70);
    gkrellm_chartconfig_grid_resolution_label(cp->config,
                _("Swap in/out pages per sec"));
    if (gkrellm_get_chartconfig_grid_resolution(cp->config) < MIN_GRID_RES)
        gkrellm_set_chartconfig_grid_resolution(cp->config, DEFAULT_GRID_RES);

	gkrellm_alloc_chartdata(cp);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area), "expose_event",
				(GtkSignalFunc) meminfo_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area),
				"button_press_event", (GtkSignalFunc) cb_extra, mc);
		gtk_widget_show(mc->vbox);
		}
	else
		refresh_chart(mc);	/* Avoid second lag at theme/size switches */
	mc->enabled = TRUE;
	ascent = 0;
	}

static gint
cb_panel_config(GtkWidget *widget, GdkEventButton *ev)
	{
	if (ev->button == 3)
		gkrellm_open_config_window(mon_mem);
	return TRUE;
	}

static void
connect_panel_signals(Panel *p, MeminfoMeter *mm)
		{
		gtk_signal_connect(GTK_OBJECT (p->drawing_area),
				"expose_event", (GtkSignalFunc) meminfo_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
			"button_press_event", (GtkSignalFunc) cb_panel_press, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
			"button_release_event", (GtkSignalFunc) cb_panel_release, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"motion_notify_event", (GtkSignalFunc) cb_panel_motion, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"button_press_event", (GtkSignalFunc) cb_panel_config, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"enter_notify_event", (GtkSignalFunc) cb_panel_enter, mm);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"leave_notify_event", (GtkSignalFunc) cb_panel_leave, mm);
		}

static void
create_mem_panel(GtkWidget *vbox, gint first_create)
	{
	GdkImlibImage	*im = NULL;
	MeminfoMeter	*mm;
	Panel			*p;
	TextStyle		*ts;
	Style			*s;
	gchar			*expand;

	mm = &mem;
	if (first_create)
		mm->panel = gkrellm_panel_new0();
	if (!mem.all_krells)	/* Krells are not in panel krell list where */
		{					/* they would be automatically destroyed. */
		gkrellm_destroy_krell(mem.krell_buffers);
		gkrellm_destroy_krell(mem.krell_cache);
		}
	p = mm->panel;

	/* I've got two extra krells for buffers and cache.  Use the #included
	|  images unless themer has customized.
	*/
	s = gkrellm_copy_style(gkrellm_meter_style(mm->style_id));
	s->krell_yoff = 0;
	s->krell_depth = 1;
	s->krell_x_hot = -1;
	gkrellm_get_gkrellmrc_integer("mem_krell_buffers_depth", &s->krell_depth);
	gkrellm_get_gkrellmrc_integer("mem_krell_buffers_x_hot", &s->krell_x_hot);
	gkrellm_get_gkrellmrc_integer("mem_krell_buffers_yoff", &s->krell_yoff);
	expand = gkrellm_get_gkrellmrc_string("mem_krell_buffers_expand");
	gkrellm_set_krell_expand(s, expand);
	g_free(expand);
	gkrellm_load_image("krell_buffers", krell_buffers_xpm, &im,MEM_STYLE_NAME);
	mm->krell_buffers = gkrellm_create_krell(p, im, s);
	gkrellm_monotonic_krell_values(mm->krell_buffers, FALSE);

	s->krell_yoff = 0;
	s->krell_depth = 1;
	s->krell_x_hot = -1;
	gkrellm_get_gkrellmrc_integer("mem_krell_cache_depth", &s->krell_depth);
	gkrellm_get_gkrellmrc_integer("mem_krell_cache_x_hot", &s->krell_x_hot);
	gkrellm_get_gkrellmrc_integer("mem_krell_cache_yoff", &s->krell_yoff);
	expand = gkrellm_get_gkrellmrc_string("mem_krell_cache_expand");
	gkrellm_set_krell_expand(s, expand);
	g_free(expand);
	gkrellm_load_image("krell_cache", krell_cache_xpm, &im, MEM_STYLE_NAME);
	mm->krell_cache = gkrellm_create_krell(p, im, s);
	gkrellm_monotonic_krell_values(mm->krell_cache, FALSE);

	/* Unlike the style pointer passed to gkrellm_panel_configure(), the krells
	|  don't need the style to persist.
	*/
	g_free(s);
	if (im)
		gdk_imlib_kill_image(im);

	s = gkrellm_meter_style(mm->style_id);
	mm->krell_used = gkrellm_create_krell(p,
						gkrellm_krell_meter_image(mm->style_id), s);
	gkrellm_monotonic_krell_values(mm->krell_used, FALSE);

	mm->decal_label = gkrellm_create_decal_text(p, "M8",
				gkrellm_meter_textstyle(mm->style_id), s, -1, -1, -1);
	gkrellm_panel_configure(p, NULL, s);
	gkrellm_panel_create(vbox, mon_mem, p);

	if (s->label_position == GKRELLM_LABEL_NONE)
		mem.label = "";
	else
		mem.label = _("Mem");
	ts = &mm->decal_label->text_style;
	mm->x_label = x_label_position(p->label->position, mm->decal_label->w,
				gdk_string_width(ts->font, mm->label), 0);
	draw_decal_label(mm, 0);

	if (first_create)
		connect_panel_signals(p, mm);
	gkrellm_setup_launcher(p, &mm->launch, METER_PANEL_TYPE, 4);

	if (!mm->enabled)
		gkrellm_panel_hide(p);
	if (!mem.all_krells)
		{
		gkrellm_remove_krell(mem.panel, mem.krell_buffers);
		gkrellm_remove_krell(mem.panel, mem.krell_cache);
		}
	}

static void
create_swap_panel(GtkWidget *vbox, gint first_create)
	{
	MeminfoMeter	*mm;
	Panel			*p;
	Style			*style, *panel_style;
	TextStyle		*ts;

	mm = &swap;
	if (first_create)
		mm->panel = gkrellm_panel_new0();
	p = mm->panel;

	style = gkrellm_meter_style(mm->style_id);

	/* Need a chart styled krell on the swap meter panel, but want it to track
	|  the meter krell margins.
	*/
	panel_style = gkrellm_copy_style(gkrellm_panel_style(DEFAULT_STYLE_ID));
	panel_style->krell_left_margin = style->krell_left_margin;
	panel_style->krell_right_margin = style->krell_right_margin;
	mm->krell_delta = gkrellm_create_krell(p,
			gkrellm_krell_panel_image(DEFAULT_STYLE_ID), panel_style);
	g_free(panel_style);	/* unlike panels, krell styles need not persist */

	mm->krell_used = gkrellm_create_krell(p,
				gkrellm_krell_meter_image(mm->style_id), style);
	gkrellm_monotonic_krell_values(mm->krell_used, FALSE);

	mm->decal_label = gkrellm_create_decal_text(p, "Sp8",
				gkrellm_meter_textstyle(mm->style_id), style, -1, -1, -1);

	gkrellm_panel_configure(p, NULL, style);
	gkrellm_panel_create(vbox, mon_swap, p);

	if (style->label_position == GKRELLM_LABEL_NONE)
		swap.label = "";
	else
		swap.label = _("Swap");
	ts = &mm->decal_label->text_style;
	mm->x_label = x_label_position(p->label->position, mm->decal_label->w,
				gdk_string_width(ts->font, mm->label), 0);
	draw_decal_label(mm, 0);

	if (first_create)
		connect_panel_signals(p, mm);

	gkrellm_setup_launcher(p, &mm->launch, METER_PANEL_TYPE, 4);

	if (!mm->enabled)
		gkrellm_panel_hide(p);
	}

  /* No separate swap monitor create function.  Use create_mem() to create
  |  swap chart, mem meter, and swap meter so they will all be a unit in
  |  the same vbox.
  */
static void
create_mem(GtkWidget *vbox, gint first_create)
	{
	if (first_create)
		{
		swap_chart.vbox = gtk_vbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox), swap_chart.vbox, FALSE, FALSE, 0);

		(*read_system_meminfo)();
		}
	if (swap_chart.enabled)
		create_chart(&swap_chart, first_create);
	create_mem_panel(vbox, first_create);
	create_swap_panel(vbox, first_create);
	setup_scaling(NULL, &swap_chart);
	}


#define	MEM_CONFIG_KEYWORD	"meminfo"

static void
save_meminfo_config(FILE *f)
	{
	fprintf(f, "%s mem_meter %d %d %d\n", MEM_CONFIG_KEYWORD,
				mem.enabled, mem.label_is_data, mem.all_krells);
	fprintf(f, "%s swap_meter %d %d\n", MEM_CONFIG_KEYWORD,
				swap.enabled, swap.label_is_data);
	fprintf(f, "%s swap_chart %d %d\n", MEM_CONFIG_KEYWORD,
				swap_chart.enabled, swap_chart.extra_info);
	gkrellm_save_chartconfig(f, swap_chart.chart_config,
				MEM_CONFIG_KEYWORD, NULL);

	fprintf(f, "%s mem_launch %s\n", MEM_CONFIG_KEYWORD,
				mem.launch.command);
	fprintf(f, "%s mem_tooltip %s\n", MEM_CONFIG_KEYWORD,
				mem.launch.tooltip_comment);
	fprintf(f, "%s mem_data_format %s\n", MEM_CONFIG_KEYWORD,mem.data_format);

	fprintf(f, "%s swap_launch %s\n", MEM_CONFIG_KEYWORD,
				swap.launch.command);
	fprintf(f, "%s swap_tooltip %s\n", MEM_CONFIG_KEYWORD,
				swap.launch.tooltip_comment);
	fprintf(f, "%s swap_data_format %s\n", MEM_CONFIG_KEYWORD,
				swap.data_format);

	fprintf(f, "%s text_format %s\n", MEM_CONFIG_KEYWORD, text_format);
	}

static void
load_meminfo_config(gchar *arg)
	{
	gchar	config[32], item[CFG_BUFSIZE];
	gint	n;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n == 2)
		{
		if (strcmp(config, "mem_meter") == 0)
			{
			sscanf(item, "%d %d %d", &mem.enabled,
					&mem.label_is_data, &mem.all_krells);
			if (mem.label_is_data)
				mem.restore_label = TRUE;
			}
		else if (strcmp(config, "swap_meter") == 0)
			{
			sscanf(item, "%d %d", &swap.enabled, &swap.label_is_data);
			if (swap.label_is_data)
				swap.restore_label = TRUE;
			}
		else if (strcmp(config, "swap_chart") == 0)
			sscanf(item, "%d %d", &swap_chart.enabled, &swap_chart.extra_info);
		else if (!strcmp(config, GKRELLM_CHARTCONFIG_KEYWORD))
			gkrellm_load_chartconfig(&swap_chart.chart_config, item, 2);
		else if (!strcmp(config, "mem_launch"))
			mem.launch.command = g_strdup(item);
		else if (!strcmp(config, "mem_tooltip"))
			mem.launch.tooltip_comment = g_strdup(item);
		else if (!strcmp(config, "mem_data_format"))
			gkrellm_dup_string(&mem.data_format, item);
		else if (!strcmp(config, "swap_launch"))
			swap.launch.command = g_strdup(item);
		else if (!strcmp(config, "swap_tooltip"))
			swap.launch.tooltip_comment = g_strdup(item);
		else if (!strcmp(config, "swap_data_format"))
			gkrellm_dup_string(&swap.data_format, item);
		else if (!strcmp(config, "text_format"))
			gkrellm_dup_string(&text_format, item);
		}
	}

/* --------------------------------------------------------------------- */
static GtkWidget	*mem_meter_enable_button,
					*swap_meter_enable_button,
					*swap_chart_enable_button,
					*all_krells_enable_button;

static GtkWidget	*mem_launch_entry,
					*mem_tooltip_entry,
					*swap_launch_entry,
					*swap_tooltip_entry;

static GtkWidget	*mem_format_combo,
					*swap_format_combo;

static GtkWidget	*text_format_combo;

static void
apply_meminfo_config(void)
	{
	gchar	*s;
	gint	new_enabled;

	s = gkrellm_entry_get_text(&(GTK_COMBO(mem_format_combo)->entry));
	if (gkrellm_dup_string(&mem.data_format, s))
		mem.decal_label->value = -1;	/* Force redraw */
	s = gkrellm_entry_get_text(&(GTK_COMBO(swap_format_combo)->entry));
	if (gkrellm_dup_string(&swap.data_format, s))
		swap.decal_label->value = -1;

	s = gkrellm_entry_get_text(&(GTK_COMBO(text_format_combo)->entry));
	gkrellm_dup_string(&text_format, s);

	gkrellm_apply_launcher(&mem_launch_entry, &mem_tooltip_entry, mem.panel,
				&mem.launch, gkrellm_launch_button_cb);
	gkrellm_apply_launcher(&swap_launch_entry, &swap_tooltip_entry, swap.panel,
				&swap.launch, gkrellm_launch_button_cb);

	new_enabled = GTK_TOGGLE_BUTTON(mem_meter_enable_button)->active;
	gkrellm_panel_enable_visibility(mem.panel, new_enabled, &mem.enabled);
	new_enabled = GTK_TOGGLE_BUTTON(swap_meter_enable_button)->active;
	gkrellm_panel_enable_visibility(swap.panel, new_enabled, &swap.enabled);
	if (swap.enabled)
		show_spacers(mon_swap);
	else
		hide_spacers(mon_swap);
	if (mem.enabled)
		show_spacers(mon_mem);
	else
		hide_spacers(mon_mem);

	new_enabled = GTK_TOGGLE_BUTTON(swap_chart_enable_button)->active;
	if (new_enabled && ! swap_chart.enabled)
		create_chart(&swap_chart, TRUE);
	else if (! new_enabled && swap_chart.enabled)
		destroy_chart(&swap_chart);

	new_enabled = GTK_TOGGLE_BUTTON(all_krells_enable_button)->active;
	if (new_enabled && !mem.all_krells)
		{
		gkrellm_insert_krell(mem.panel, mem.krell_buffers, FALSE);
		gkrellm_insert_krell(mem.panel, mem.krell_cache, FALSE);
		}
	else if (!new_enabled && mem.all_krells)
		{
		gkrellm_remove_krell(mem.panel, mem.krell_buffers);
		gkrellm_remove_krell(mem.panel, mem.krell_cache);
		}
	mem.all_krells = new_enabled;
	setup_scaling(NULL, &swap_chart);
	}

#define	DEFAULT_TEXT_FORMAT	"$T"

static gchar	*mem_info_text[] =
{
N_("<b>Used and Free\n"),
N_("The used and free memory here are calculated from the kernel reported\n"
"used and free by subtracting or adding the buffers and cache memory.  See\n"
"the README and compare to the \"-/+ buffers/cache:\" line from the free\n"
"command.  If you show three memory krells, the kernel \"raw free\" is\n"
"the space after the rightmost krell.\n"),
"\n",
N_("<b>Chart Labels\n"),
N_("Substitution variables for the format string for chart labels:\n"),
N_("\t$M    maximum chart value\n"),
N_("\t$T    total swap in blocks + swap out blocks\n"),
N_("\t$i    swap in blocks\n"),
N_("\t$o    swap out blocks\n"),
"\n",
N_("<b>Panel Labels\n"),
N_("Substition variables for the format string for the Mem and Swap\n"
"panels (a MiB is a binary megabyte - 2^20):\n"),

N_("For memory and swap:\n"),
N_("\t$t    total MiB\n"),
N_("\t$u    used MiB\n"),
N_("\t$f    free MiB\n"),
N_("\t$U    used %\n"),
N_("\t$F    free %\n"),
N_("\t$l    the panel label"),
"\n",
N_("For memory only:\n"),
N_("\t$s    shared MiB\n"),
N_("\t$b    buffered MiB\n"),
N_("\t$c    cached MiB\n"),
"\n",
N_("<b>Mouse Button Actions:\n"),
N_("<b>\tMiddle "),
N_("click on a panel to scroll a programmable display of\n"
"\t\tof memory or swap usage.\n")


};

static void
create_meminfo_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*vbox, *vbox1;
	GtkWidget		*table;
	GtkWidget		*hbox;
	GtkWidget		*text, *label;
	GList			*list;
	gint			i;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* --Options Tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Options"));

	vbox1 = gkrellm_framed_vbox(vbox, _("Swap"), 4, FALSE, 0, 2);
    gkrellm_check_button(vbox1, &swap_chart_enable_button, swap_chart.enabled,
				 FALSE, 0, _("Enable swap pages in/out chart"));
    gkrellm_check_button(vbox1, &swap_meter_enable_button, swap.enabled,
				FALSE, 0, _("Enable swap meter"));

	vbox1 = gkrellm_framed_vbox(vbox, _("Memory"), 4, FALSE, 0, 2);
    gkrellm_check_button(vbox1, &mem_meter_enable_button, mem.enabled,
				FALSE, 0, _("Enable memory meter"));
    gkrellm_check_button(vbox1, &all_krells_enable_button, mem.all_krells,
				FALSE, 0,
		_("Show three memory krells:   [used | buffers | cache | raw free]"));

/* -- Setup tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Setup"));

	vbox1 = gkrellm_framed_vbox(vbox, _("Format String for Chart Labels"),
			 4, FALSE, 0, 2);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, TRUE, 5);
	text_format_combo = gtk_combo_new();
	gtk_widget_set_usize (GTK_WIDGET(text_format_combo), 300, 0);
	gtk_box_pack_start(GTK_BOX(hbox), text_format_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, DEFAULT_TEXT_FORMAT);
	list = g_list_append(list, "$T\\C\\f$M");
	list = g_list_append(list, "\\c\\f$M\\b$T");
	list = g_list_append(list,
				"\\ww\\C\\f$M\\D2\\f\\ai\\.$i\\D1\\f\\ao\\.$o");
	list = g_list_append(list,
				"\\ww\\C\\f$M\\D3\\f\\ai\\.$i\\D0\\f\\ao\\.$o");
	gtk_combo_set_popdown_strings(GTK_COMBO(text_format_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(text_format_combo)->entry),
			text_format);

	vbox1 = gkrellm_framed_vbox(vbox, _("Format String for Panel Labels"),
			4, FALSE, 0, 2);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);
	mem_format_combo = gtk_combo_new();
	gtk_box_pack_start(GTK_BOX(hbox), mem_format_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, DEFAULT_FORMAT);
	list = g_list_append(list, _("$t - $u used"));
	list = g_list_append(list, _("$t - $U"));
	list = g_list_append(list, _("$t - $u used  $s sh  $b bf  $c ca"));
	gtk_combo_set_popdown_strings(GTK_COMBO(mem_format_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(mem_format_combo)->entry),
			mem.data_format);
	label = gtk_label_new(_("Mem"));
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 2);

	swap_format_combo = gtk_combo_new();
	gtk_box_pack_start(GTK_BOX(hbox), swap_format_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, DEFAULT_FORMAT);
	list = g_list_append(list, _("$t - $u used"));
	list = g_list_append(list, _("$t - $U"));
	gtk_combo_set_popdown_strings(GTK_COMBO(swap_format_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(swap_format_combo)->entry),
			swap.data_format);
	label = gtk_label_new(_("Swap"));
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 2);

	vbox1 = gkrellm_framed_vbox_end(vbox, _("Launch Commands"), 4, FALSE, 0, 2);
	table = gkrellm_launcher_table_new(vbox1, 2);
	gkrellm_config_launcher(table, 0,  &mem_launch_entry, &mem_tooltip_entry,
		_("Mem"), &(mem.launch));
	gkrellm_config_launcher(table, 1,  &swap_launch_entry, &swap_tooltip_entry,
		_("Swap"), &(swap.launch));

/* --Info tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Info"));
	text = gkrellm_scrolled_text(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(mem_info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(mem_info_text[i]));
	}


  /* The meminfo monitor is a bit of a hybrid.  To provide for easy theming,
  |  the mem, swap, and swap_chart monitors are created as separate monitors,
  |  but they all have several common routines (update, config, ...).  Where
  |  a common routine is used, it is entered in only one of the Monitor
  |  structures, and NULL is entered in the others.
  */
static Monitor	monitor_mem =
	{
	N_("Memory"),			/* Name, for config tab.	*/
	MON_MEM,					/* Id,  0 if a plugin		*/
	create_mem,			/* The create function		*/
	update_meminfo,		/* The update function		*/
	create_meminfo_tab, /* The config tab create function	*/
	apply_meminfo_config, /* Apply the config function		*/

	save_meminfo_config, /* Save user conifg			*/
	load_meminfo_config, /* Load user config			*/
	MEM_CONFIG_KEYWORD, /* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_mem_monitor(void)
	{
	monitor_mem.name = _(monitor_mem.name);
	mon_mem = &monitor_mem;
	mem.style_id = gkrellm_add_meter_style(mon_mem, MEM_STYLE_NAME);

	mem.data_format = g_strdup(DEFAULT_FORMAT);
	mem.enabled = TRUE;

	if (setup_meminfo_interface())
		return &monitor_mem;
	return NULL;
	}


static Monitor	monitor_swap =
	{
	NULL,			/* Name, for config tab. Done in mon_mem*/
	MON_SWAP,		/* Id,  0 if a plugin		*/
	NULL,			/* The create function		*/
	NULL,			/* The update function		*/
	NULL,			/* The config tab create function	*/
	NULL,			/* Apply the config function		*/
	NULL,			/* Save user conifg			*/
	NULL,			/* Load user config			*/
	NULL,			/* config keyword			*/
	NULL,			/* Undef 2	*/
	NULL,			/* Undef 1	*/
	NULL,			/* Undef 0	*/
	0,				/* insert_before_id - place plugin before this mon */
	NULL,			/* Handle if a plugin, filled in by GKrellM		*/
	NULL			/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_swap_monitor(void)
	{
	mon_swap = &monitor_swap;
	swap.style_id = gkrellm_add_meter_style(mon_swap, SWAP_STYLE_NAME);

	swap.data_format = g_strdup(DEFAULT_FORMAT);
	swap.enabled = TRUE;

	swap_chart.enabled = FALSE;
	swap_chart.extra_info = TRUE;
	text_format = g_strdup(DEFAULT_TEXT_FORMAT);

	if (setup_meminfo_interface())	/* XXX */
		return &monitor_swap;
	return NULL;
	}
