-----------------------------------------------------------------------------
-- $Id: MArray.hs,v 1.18.2.1 2001/10/09 16:00:08 sewardj Exp $
--
-- (c) The GHC Team 2000
--

module MArray ( 
    module Ix,

    -- Class of mutable array types
    MArray,       -- :: (* -> * -> *) -> * -> (* -> *) -> class
    -- Class of array types with immutable bounds
    HasBounds,    -- :: (* -> * -> *) -> class

    -- Non-strict polymorphic arrays
    STArray,      -- :: * -> * -> *
    IOArray,      -- :: * -> * -> *
    -- Strict unboxed arrays
    STUArray,     -- :: * -> * -> *
    IOUArray,     -- :: * -> * -> *

    newArray,     -- :: (MArray a e m, Ix i) => (i,i) -> e -> m (a i e)
    newArray_,    -- :: (MArray a e m, Ix i) => (i,i) -> m (a i e)
    newListArray, -- :: (MArray a e m, Ix i) => (i,i) -> [e] -> m (a i e)
    readArray,    -- :: (MArray a e m, Ix i) => a i e -> i -> m e
    writeArray,   -- :: (MArray a e m, Ix i) => a i e -> i -> e -> m ()
    bounds,       -- :: (HasBounds a, Ix i) => a i e -> (i,i)
    indices,      -- :: (HasBounds a, Ix i) => a i e -> [i]
    getElems,     -- :: (MArray a e m, Ix i) => a i e -> m [e]
    getAssocs,    -- :: (MArray a e m, Ix i) => a i e -> m [(i, e)]
    mapArray,     -- :: (MArray a e' m, MArray a e m, Ix i) => (e' -> e) -> a i e' -> m (a i e)
    mapIndices,   -- :: (MArray a e m, Ix i, Ix j) => (i,i) -> (i -> j) -> a j e -> m (a i e)

    freeze,       -- :: (Ix i, MArray a e m, IArray b e) => a i e -> m (b i e)
    unsafeFreeze, -- :: (Ix i, MArray a e m, IArray b e) => a i e -> m (b i e)
    thaw,         -- :: (Ix i, IArray a e, MArray b e m) => a i e -> m (b i e)
    unsafeThaw,   -- :: (Ix i, IArray a e, MArray b e m) => a i e -> m (b i e)

    castSTUArray, -- :: STUArray s i a -> ST s (STUArray s i b)
    castIOUArray, -- :: IOUArray i a -> IO (IOUArray i b)

    -- Legacy non-overloaded byte array interface
    -- (CharArrays here work on bytes, not Chars!)
    newCharArray,       -- :: Ix i => (i,i) -> ST s (STUArray s i Char)
    newIntArray,        -- :: Ix i => (i,i) -> ST s (STUArray s i Int)
    newWordArray,       -- :: Ix i => (i,i) -> ST s (STUArray s i Word)
    newAddrArray,       -- :: Ix i => (i,i) -> ST s (STUArray s i Addr)
    newFloatArray,      -- :: Ix i => (i,i) -> ST s (STUArray s i Float)
    newDoubleArray,     -- :: Ix i => (i,i) -> ST s (STUArray s i Double)
    newStablePtrArray,  -- :: Ix i => (i,i) -> ST s (STUArray s i (StablePtr a))
    newInt8Array,       -- :: Ix i => (i,i) -> ST s (STUArray s i Int8)
    newInt16Array,      -- :: Ix i => (i,i) -> ST s (STUArray s i Int16)
    newInt32Array,      -- :: Ix i => (i,i) -> ST s (STUArray s i Int32)
    newInt64Array,      -- :: Ix i => (i,i) -> ST s (STUArray s i Int64)
    newWord8Array,      -- :: Ix i => (i,i) -> ST s (STUArray s i Word8)
    newWord16Array,     -- :: Ix i => (i,i) -> ST s (STUArray s i Word16)
    newWord32Array,     -- :: Ix i => (i,i) -> ST s (STUArray s i Word32)
    newWord64Array,     -- :: Ix i => (i,i) -> ST s (STUArray s i Word64)

    readCharArray,       -- :: Ix i => STUArray s i Char          -> i -> ST s Char
    readIntArray,        -- :: Ix i => STUArray s i Int           -> i -> ST s Int
    readWordArray,       -- :: Ix i => STUArray s i Word          -> i -> ST s Word
    readAddrArray,       -- :: Ix i => STUArray s i Addr          -> i -> ST s Addr
    readFloatArray,      -- :: Ix i => STUArray s i Float         -> i -> ST s Float
    readDoubleArray,     -- :: Ix i => STUArray s i Double        -> i -> ST s Double
    readStablePtrArray,  -- :: Ix i => STUArray s i (StablePtr a) -> i -> ST s (StablePtr a)
    readInt8Array,       -- :: Ix i => STUArray s i Int8          -> i -> ST s Int8
    readInt16Array,      -- :: Ix i => STUArray s i Int16         -> i -> ST s Int16
    readInt32Array,      -- :: Ix i => STUArray s i Int32         -> i -> ST s Int32
    readInt64Array,      -- :: Ix i => STUArray s i Int64         -> i -> ST s Int64
    readWord8Array,      -- :: Ix i => STUArray s i Word8         -> i -> ST s Word8
    readWord16Array,     -- :: Ix i => STUArray s i Word16        -> i -> ST s Word16
    readWord32Array,     -- :: Ix i => STUArray s i Word32        -> i -> ST s Word32
    readWord64Array,     -- :: Ix i => STUArray s i Word64        -> i -> ST s Word64

    writeCharArray,      -- :: Ix i => STUArray s i Char          -> i -> Char        -> ST s ()
    writeIntArray,       -- :: Ix i => STUArray s i Int           -> i -> Int         -> ST s ()
    writeWordArray,      -- :: Ix i => STUArray s i Word          -> i -> Word        -> ST s ()
    writeAddrArray,      -- :: Ix i => STUArray s i Addr          -> i -> Addr        -> ST s ()
    writeFloatArray,     -- :: Ix i => STUArray s i Float         -> i -> Float       -> ST s ()
    writeDoubleArray,    -- :: Ix i => STUArray s i Double        -> i -> Double      -> ST s ()
    writeStablePtrArray, -- :: Ix i => STUArray s i (StablePtr a) -> i -> StablePtr a -> ST s ()
    writeInt8Array,      -- :: Ix i => STUArray s i Int8          -> i -> Int8        -> ST s ()
    writeInt16Array,     -- :: Ix i => STUArray s i Int16         -> i -> Int16       -> ST s ()
    writeInt32Array,     -- :: Ix i => STUArray s i Int32         -> i -> Int32       -> ST s ()
    writeInt64Array,     -- :: Ix i => STUArray s i Int64         -> i -> Int64       -> ST s ()
    writeWord8Array,     -- :: Ix i => STUArray s i Word8         -> i -> Word8       -> ST s ()
    writeWord16Array,    -- :: Ix i => STUArray s i Word16        -> i -> Word16      -> ST s ()
    writeWord32Array,    -- :: Ix i => STUArray s i Word32        -> i -> Word32      -> ST s ()
    writeWord64Array)    -- :: Ix i => STUArray s i Word64        -> i -> Word64      -> ST s ()
    where

import Ix
import PrelArr    (STArray,
                   freezeSTArray, unsafeFreezeSTArray,
                   thawSTArray, unsafeThawSTArray)
import ArrayBase
import Array      (Array)
import PrelST     (ST(..))
import PrelIOBase (stToIO)

import PrelBase
import PrelWord   (Word)
import Addr       (Addr)
import PrelPtr    (Ptr, FunPtr)
import PrelStable (StablePtr)
import PrelInt    (Int8,  Int16,  Int32,  Int64)
import PrelWord   (Word8, Word16, Word32, Word64)

-----------------------------------------------------------------------------
-- Polymorphic non-strict mutable arrays (IO monad)

newtype IOArray i e = IOArray (STArray RealWorld i e) deriving Eq

instance HasBounds IOArray where
    {-# INLINE bounds #-}
    bounds (IOArray marr) = bounds marr

instance MArray IOArray e IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOArray marr) i e = stToIO (unsafeWrite marr i e)

-----------------------------------------------------------------------------
-- Flat unboxed mutable arrays (IO monad)

newtype IOUArray i e = IOUArray (STUArray RealWorld i e) deriving Eq

instance HasBounds IOUArray where
    {-# INLINE bounds #-}
    bounds (IOUArray marr) = bounds marr

instance MArray IOUArray Bool IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Char IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Int IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Word IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Addr IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray (Ptr a) IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray (FunPtr a) IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Float IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Double IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray (StablePtr a) IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Int8 IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Int16 IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Int32 IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Int64 IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Word8 IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Word16 IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Word32 IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

instance MArray IOUArray Word64 IO where
    {-# INLINE newArray #-}
    newArray lu init = stToIO $ do
        marr <- newArray lu init; return (IOUArray marr)
    {-# INLINE newArray_ #-}
    newArray_ lu = stToIO $ do
        marr <- newArray_ lu; return (IOUArray marr)
    {-# INLINE unsafeRead #-}
    unsafeRead (IOUArray marr) i = stToIO (unsafeRead marr i)
    {-# INLINE unsafeWrite #-}
    unsafeWrite (IOUArray marr) i e = stToIO (unsafeWrite marr i e)

-----------------------------------------------------------------------------
-- Freezing

freezeIOArray :: Ix ix => IOArray ix e -> IO (Array ix e)
freezeIOArray (IOArray marr) = stToIO (freezeSTArray marr)

freezeIOUArray :: Ix ix => IOUArray ix e -> IO (UArray ix e)
freezeIOUArray (IOUArray marr) = stToIO (freezeSTUArray marr)

{-# RULES
"freeze/IOArray"  freeze = freezeIOArray
"freeze/IOUArray" freeze = freezeIOUArray
    #-}

{-# INLINE unsafeFreezeIOArray #-}
unsafeFreezeIOArray :: Ix ix => IOArray ix e -> IO (Array ix e)
unsafeFreezeIOArray (IOArray marr) = stToIO (unsafeFreezeSTArray marr)

{-# INLINE unsafeFreezeIOUArray #-}
unsafeFreezeIOUArray :: Ix ix => IOUArray ix e -> IO (UArray ix e)
unsafeFreezeIOUArray (IOUArray marr) = stToIO (unsafeFreezeSTUArray marr)

{-# RULES
"unsafeFreeze/IOArray"  unsafeFreeze = unsafeFreezeIOArray
"unsafeFreeze/IOUArray" unsafeFreeze = unsafeFreezeIOUArray
    #-}

-----------------------------------------------------------------------------
-- Thawing

thawIOArray :: Ix ix => Array ix e -> IO (IOArray ix e)
thawIOArray arr = stToIO $ do
    marr <- thawSTArray arr
    return (IOArray marr)

thawIOUArray :: Ix ix => UArray ix e -> IO (IOUArray ix e)
thawIOUArray arr = stToIO $ do
    marr <- thawSTUArray arr
    return (IOUArray marr)

{-# RULES
"thaw/IOArray"  thaw = thawIOArray
"thaw/IOUArray" thaw = thawIOUArray
    #-}

{-# INLINE unsafeThawIOArray #-}
unsafeThawIOArray :: Ix ix => Array ix e -> IO (IOArray ix e)
unsafeThawIOArray arr = stToIO $ do
    marr <- unsafeThawSTArray arr
    return (IOArray marr)

{-# INLINE unsafeThawIOUArray #-}
unsafeThawIOUArray :: Ix ix => UArray ix e -> IO (IOUArray ix e)
unsafeThawIOUArray arr = stToIO $ do
    marr <- unsafeThawSTUArray arr
    return (IOUArray marr)

{-# RULES
"unsafeThaw/IOArray"  unsafeThaw = unsafeThawIOArray
"unsafeThaw/IOUArray" unsafeThaw = unsafeThawIOUArray
    #-}

castSTUArray :: STUArray s ix a -> ST s (STUArray s ix b)
castSTUArray (STUArray l u marr#) = return (STUArray l u marr#)

castIOUArray :: IOUArray ix a -> IO (IOUArray ix b)
castIOUArray (IOUArray marr) = stToIO $ do
    marr' <- castSTUArray marr
    return (IOUArray marr')

-----------------------------------------------------------------------------
-- Legacy non-overloaded byte array interface

newCharArray      :: Ix i => (i,i) -> ST s (STUArray s i Char)
newIntArray       :: Ix i => (i,i) -> ST s (STUArray s i Int)
newWordArray      :: Ix i => (i,i) -> ST s (STUArray s i Word)
newAddrArray      :: Ix i => (i,i) -> ST s (STUArray s i Addr)
newFloatArray     :: Ix i => (i,i) -> ST s (STUArray s i Float)
newDoubleArray    :: Ix i => (i,i) -> ST s (STUArray s i Double)
newStablePtrArray :: Ix i => (i,i) -> ST s (STUArray s i (StablePtr a))
newInt8Array      :: Ix i => (i,i) -> ST s (STUArray s i Int8)
newInt16Array     :: Ix i => (i,i) -> ST s (STUArray s i Int16)
newInt32Array     :: Ix i => (i,i) -> ST s (STUArray s i Int32)
newInt64Array     :: Ix i => (i,i) -> ST s (STUArray s i Int64)
newWord8Array     :: Ix i => (i,i) -> ST s (STUArray s i Word8)
newWord16Array    :: Ix i => (i,i) -> ST s (STUArray s i Word16)
newWord32Array    :: Ix i => (i,i) -> ST s (STUArray s i Word32)
newWord64Array    :: Ix i => (i,i) -> ST s (STUArray s i Word64)

newCharArray (l,u) = ST $ \s1# ->
    case rangeSize (l,u)                of { I# n# ->
    case newByteArray# n# s1#           of { (# s2#, marr# #) ->
    (# s2#, STUArray l u marr# #) }}
newIntArray       = newArray_
newWordArray      = newArray_
newAddrArray      = newArray_
newFloatArray     = newArray_
newDoubleArray    = newArray_
newStablePtrArray = newArray_
newInt8Array      = newArray_
newInt16Array     = newArray_
newInt32Array     = newArray_
newInt64Array     = newArray_
newWord8Array     = newArray_
newWord16Array    = newArray_
newWord32Array    = newArray_
newWord64Array    = newArray_

readCharArray      :: Ix i => STUArray s i Char          -> i -> ST s Char
readIntArray       :: Ix i => STUArray s i Int           -> i -> ST s Int
readWordArray      :: Ix i => STUArray s i Word          -> i -> ST s Word
readAddrArray      :: Ix i => STUArray s i Addr          -> i -> ST s Addr
readFloatArray     :: Ix i => STUArray s i Float         -> i -> ST s Float
readDoubleArray    :: Ix i => STUArray s i Double        -> i -> ST s Double
readStablePtrArray :: Ix i => STUArray s i (StablePtr a) -> i -> ST s (StablePtr a)
readInt8Array      :: Ix i => STUArray s i Int8          -> i -> ST s Int8
readInt16Array     :: Ix i => STUArray s i Int16         -> i -> ST s Int16
readInt32Array     :: Ix i => STUArray s i Int32         -> i -> ST s Int32
readInt64Array     :: Ix i => STUArray s i Int64         -> i -> ST s Int64
readWord8Array     :: Ix i => STUArray s i Word8         -> i -> ST s Word8
readWord16Array    :: Ix i => STUArray s i Word16        -> i -> ST s Word16
readWord32Array    :: Ix i => STUArray s i Word32        -> i -> ST s Word32
readWord64Array    :: Ix i => STUArray s i Word64        -> i -> ST s Word64

readCharArray (STUArray l u marr#) i = ST $ \s1# ->
    case index (l,u) i                  of { I# i# ->
    case readCharArray# marr# i# s1#    of { (# s2#, e# #) ->
    (# s2#, C# e# #) }}
readIntArray       = readArray
readWordArray      = readArray
readAddrArray      = readArray
readFloatArray     = readArray
readDoubleArray    = readArray
readStablePtrArray = readArray
readInt8Array      = readArray
readInt16Array     = readArray
readInt32Array     = readArray
readInt64Array     = readArray
readWord8Array     = readArray
readWord16Array    = readArray
readWord32Array    = readArray
readWord64Array    = readArray

writeCharArray      :: Ix i => STUArray s i Char          -> i -> Char        -> ST s ()
writeIntArray       :: Ix i => STUArray s i Int           -> i -> Int         -> ST s ()
writeWordArray      :: Ix i => STUArray s i Word          -> i -> Word        -> ST s ()
writeAddrArray      :: Ix i => STUArray s i Addr          -> i -> Addr        -> ST s ()
writeFloatArray     :: Ix i => STUArray s i Float         -> i -> Float       -> ST s ()
writeDoubleArray    :: Ix i => STUArray s i Double        -> i -> Double      -> ST s ()
writeStablePtrArray :: Ix i => STUArray s i (StablePtr a) -> i -> StablePtr a -> ST s ()
writeInt8Array      :: Ix i => STUArray s i Int8          -> i -> Int8        -> ST s ()
writeInt16Array     :: Ix i => STUArray s i Int16         -> i -> Int16       -> ST s ()
writeInt32Array     :: Ix i => STUArray s i Int32         -> i -> Int32       -> ST s ()
writeInt64Array     :: Ix i => STUArray s i Int64         -> i -> Int64       -> ST s ()
writeWord8Array     :: Ix i => STUArray s i Word8         -> i -> Word8       -> ST s ()
writeWord16Array    :: Ix i => STUArray s i Word16        -> i -> Word16      -> ST s ()
writeWord32Array    :: Ix i => STUArray s i Word32        -> i -> Word32      -> ST s ()
writeWord64Array    :: Ix i => STUArray s i Word64        -> i -> Word64      -> ST s ()

writeCharArray (STUArray l u marr#) i (C# e#) = ST $ \s1# ->
    case index (l,u) i                  of { I# i# ->
    case writeCharArray# marr# i# e# s1# of { s2# ->
    (# s2#, () #) }}
writeIntArray       = writeArray
writeWordArray      = writeArray
writeAddrArray      = writeArray
writeFloatArray     = writeArray
writeDoubleArray    = writeArray
writeStablePtrArray = writeArray
writeInt8Array      = writeArray
writeInt16Array     = writeArray
writeInt32Array     = writeArray
writeInt64Array     = writeArray
writeWord8Array     = writeArray
writeWord16Array    = writeArray
writeWord32Array    = writeArray
writeWord64Array    = writeArray
