{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2001  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module GL_QueryUtils (
   getGLboolean, getGLint, getGLushort, getGLenum, getGLuint, getFace,
   getViewport, getPolygonModes, getGLfloat, getGLmatrixf, getVertex4f,
   getNormal3f, getColor4f, getRasterPos4f, getGLdouble, getGLmatrixd,
   getDepthRange, getPointer, getClipPlaned, getMaterialColor4f, getMaterialf,
   getMaterialColorIndex3i, getLightColor4f, getLightPosition4f, getLightf,
   getLightNormal3f, getPolygonStipple, getTexLevelParameter
) where

import Foreign          ( Ptr, Storable(..), alloca, allocaArray, peekArray )
import Monad            ( liftM )

import GL_BasicTypes    ( GLboolean_, GLboolean, unmarshalGLboolean, GLubyte,
                          GLint, GLuint, GLushort, GLfloat, GLdouble, GLclampd,
                          GLenum, GLsizei, Viewport )
import GL_Colors        ( Face, unmarshalFace )
import GL_CoordTrans    ( GLmatrix, Plane )
import GL_Polygons      ( StipplePattern, stipplePattern, polygonStippleSize,
                          PolygonMode, unmarshalPolygonMode )
import GL_RasterPos     ( RasterPos4 )
import GL_VertexSpec    ( Vertex2(..), Vertex3(..), Vertex4, Normal3, Color4,
                          ColorIndex(..) )

---------------------------------------------------------------------------

peekFromBuf :: Storable a => (Ptr a -> IO ()) -> IO a
peekFromBuf fill = alloca $ \buf -> fill buf >> peek buf

---------------------------------------------------------------------------

getGLboolean :: GLenum -> IO GLboolean
getGLboolean = liftM unmarshalGLboolean . peekFromBuf . glGetBooleanv

foreign import unsafe glGetBooleanv :: GLenum -> Ptr GLboolean_ -> IO ()

---------------------------------------------------------------------------

getGLint :: GLenum -> IO GLint
getGLint = peekFromBuf . glGetIntegerv

foreign import unsafe glGetIntegerv :: GLenum -> Ptr GLint -> IO ()

---------------------------------------------------------------------------

getGLushort :: GLenum -> IO GLushort
getGLushort = liftM fromIntegral . getGLint

---------------------------------------------------------------------------

getGLenum :: GLenum -> IO GLenum
getGLenum = liftM fromIntegral . getGLint

---------------------------------------------------------------------------

getGLuint :: GLenum -> IO GLuint
getGLuint = liftM fromIntegral . getGLint

---------------------------------------------------------------------------

getFace :: GLenum -> IO Face
getFace = liftM unmarshalFace . getGLenum

---------------------------------------------------------------------------

getViewport :: GLenum -> IO Viewport
getViewport = peekFromBuf . glGetViewport

foreign import "glGetIntegerv" unsafe glGetViewport :: GLenum -> Ptr Viewport -> IO ()

---------------------------------------------------------------------------

-- little hack: abuse of Vertex2 as a Storable pair
getPolygonModes :: GLenum -> IO (PolygonMode, PolygonMode)
getPolygonModes = liftM (\(Vertex2 f b) -> (i2m f, i2m b)) . peekFromBuf . glGetPolygonModes
   where i2m = unmarshalPolygonMode . fromIntegral

foreign import "glGetIntegerv" unsafe glGetPolygonModes :: GLenum ->  Ptr (Vertex2 GLint) -> IO ()

---------------------------------------------------------------------------

getGLfloat :: GLenum -> IO GLfloat
getGLfloat = peekFromBuf . glGetFloatv

foreign import unsafe glGetFloatv :: GLenum -> Ptr GLfloat -> IO ()

---------------------------------------------------------------------------

getVertex4f :: GLenum -> IO (Vertex4 GLfloat)
getVertex4f = peekFromBuf . glGetVertex4fv

foreign import "glGetFloatv" unsafe glGetVertex4fv :: GLenum -> Ptr (Vertex4 GLfloat) -> IO ()

---------------------------------------------------------------------------

getNormal3f :: GLenum -> IO (Normal3 GLfloat)
getNormal3f = peekFromBuf . glGetNormal3fv

foreign import "glGetFloatv" unsafe glGetNormal3fv :: GLenum -> Ptr (Normal3 GLfloat) -> IO ()

---------------------------------------------------------------------------

getColor4f :: GLenum -> IO (Color4 GLfloat)
getColor4f = peekFromBuf . glGetColor4fv

foreign import "glGetFloatv" unsafe glGetColor4fv :: GLenum -> Ptr (Color4 GLfloat) -> IO ()

---------------------------------------------------------------------------

getRasterPos4f :: GLenum -> IO (RasterPos4 GLfloat)
getRasterPos4f = peekFromBuf . glGetRasterPos4fv

foreign import "glGetFloatv" unsafe glGetRasterPos4fv :: GLenum -> Ptr (RasterPos4 GLfloat) -> IO ()

---------------------------------------------------------------------------

getGLmatrixf :: GLenum -> IO (GLmatrix GLfloat)
getGLmatrixf = peekFromBuf . glGetMatrixfv

foreign import "glGetFloatv" unsafe glGetMatrixfv :: GLenum -> Ptr (GLmatrix GLfloat) -> IO ()

---------------------------------------------------------------------------

getGLdouble :: GLenum -> IO GLdouble
getGLdouble = peekFromBuf . glGetDoublev

foreign import unsafe glGetDoublev :: GLenum -> Ptr GLdouble -> IO ()

---------------------------------------------------------------------------

getGLmatrixd :: GLenum -> IO (GLmatrix GLdouble)
getGLmatrixd = peekFromBuf . glGetMatrixdv

foreign import "glGetDoublev" unsafe glGetMatrixdv :: GLenum -> Ptr (GLmatrix GLdouble) -> IO ()

---------------------------------------------------------------------------

-- little hack: abuse of Vertex2 as a Storable pair
getDepthRange :: GLenum -> IO (GLclampd, GLclampd)
getDepthRange = liftM (\(Vertex2 near far) -> (near, far)) . peekFromBuf . glGetDepthRange

foreign import "glGetDoublev" unsafe glGetDepthRange :: GLenum ->  Ptr (Vertex2 GLclampd) -> IO ()

---------------------------------------------------------------------------

getPointer :: GLenum -> IO (Ptr a)
getPointer = peekFromBuf . glGetPointerv

foreign import unsafe glGetPointerv :: GLenum -> Ptr (Ptr a) -> IO ()

---------------------------------------------------------------------------

getClipPlaned :: GLenum -> IO (Plane GLdouble)
getClipPlaned = peekFromBuf . glGetClipPlane

foreign import unsafe glGetClipPlane :: GLenum ->  Ptr (Plane GLdouble) -> IO ()

---------------------------------------------------------------------------

getMaterialColor4f :: GLenum -> GLenum -> IO (Color4 GLfloat)
getMaterialColor4f face lightType = peekFromBuf $ glGetMaterialColor4fv face lightType

foreign import "glGetMaterialfv" unsafe glGetMaterialColor4fv :: GLenum -> GLenum -> Ptr (Color4 GLfloat) -> IO ()

---------------------------------------------------------------------------

getMaterialf :: GLenum -> GLenum -> IO GLfloat
getMaterialf face param = peekFromBuf $ glGetMaterialfv face param

foreign import unsafe glGetMaterialfv :: GLenum -> GLenum -> Ptr GLfloat -> IO ()

---------------------------------------------------------------------------

-- little hack: abuse of Vertex3 as a Storable triple
getMaterialColorIndex3i :: GLenum -> GLenum -> IO (ColorIndex GLint, ColorIndex GLint, ColorIndex GLint)
getMaterialColorIndex3i face param =
   liftM (\ (Vertex3 x y z) -> (ColorIndex x, ColorIndex y, ColorIndex z)) .
   peekFromBuf $ glGetMaterialColorIndex3i face param

foreign import "glGetMaterialfv" unsafe glGetMaterialColorIndex3i :: GLenum -> GLenum -> Ptr (Vertex3 GLint) -> IO ()

---------------------------------------------------------------------------

getLightColor4f :: GLenum -> GLenum -> IO (Color4 GLfloat)
getLightColor4f light param = peekFromBuf $ glGetLightColor4fv light param

foreign import "glGetLightfv" unsafe glGetLightColor4fv :: GLenum -> GLenum -> Ptr (Color4 GLfloat) -> IO ()

---------------------------------------------------------------------------

getLightPosition4f :: GLenum -> GLenum -> IO (Vertex4 GLfloat)
getLightPosition4f light param = peekFromBuf $ glGetLightPosition4fv light param

foreign import "glGetLightfv" unsafe glGetLightPosition4fv :: GLenum -> GLenum -> Ptr (Vertex4 GLfloat) -> IO ()

---------------------------------------------------------------------------

getLightf :: GLenum -> GLenum -> IO GLfloat
getLightf light param = peekFromBuf $ glGetLightfv light param

foreign import unsafe glGetLightfv :: GLenum -> GLenum -> Ptr GLfloat -> IO ()

---------------------------------------------------------------------------

getLightNormal3f :: GLenum -> GLenum -> IO (Normal3 GLfloat)
getLightNormal3f light param = peekFromBuf $ glGetLightNormal3f light param

foreign import "glGetLightfv" unsafe glGetLightNormal3f :: GLenum -> GLenum -> Ptr (Normal3 GLfloat) -> IO ()

---------------------------------------------------------------------------

getPolygonStipple :: IO StipplePattern
getPolygonStipple = allocaArray polygonStippleSize $ \buf -> do
                    glGetPolygonStipple buf
                    liftM stipplePattern $ peekArray polygonStippleSize buf

foreign import unsafe glGetPolygonStipple :: Ptr GLubyte -> IO ()

---------------------------------------------------------------------------

getTexLevelParameter :: GLenum -> GLint -> GLenum -> IO GLsizei
getTexLevelParameter target lod param = peekFromBuf $ glGetTexLevelParameteriv target lod param

foreign import unsafe glGetTexLevelParameteriv :: GLenum -> GLint -> GLenum -> Ptr GLsizei -> IO ()
