{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2001  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to table 6.12 (Texture Objects) of the OpenGL 1.2.1 specs.
-}

module GL_QueryTexObj1 (
   VarTextureNames(..),
   VarTexture(..),
   VarTextureSizes(..),
   VarTextureInternalFormat()
) where

import Foreign          ( Ptr )
import Monad            ( liftM )

import GL_Constants     
import GL_BasicTypes    ( GLenum, GLint, GLsizei, Gettable(..),)
import GL_PixelRect     ( PixelFormat, marshalPixelFormat )
import GL_QueryUtils    ( getGLuint, getTexLevelParameter )
import GL_Texturing     ( TextureName(..), TextureTarget, marshalTextureTarget,
                          LOD(..), InternalFormat, unmarshalInternalFormat )
import GL_VertexArray   ( Type, marshalType )

---------------------------------------------------------------------------

data VarTextureNames =
     VarTextureBinding1d
   | VarTextureBinding2d
   | VarTextureBinding3d   -- @GL_1_2@
   deriving (Eq,Ord)

instance Gettable VarTextureNames TextureName where
   get VarTextureBinding1d = liftM TextureName $ getGLuint gl_TEXTURE_BINDING_1D
   get VarTextureBinding2d = liftM TextureName $ getGLuint gl_TEXTURE_BINDING_2D
   get VarTextureBinding3d = liftM TextureName $ getGLuint gl_TEXTURE_BINDING_3D   -- @GL_1_2@

data VarTexture = VarTexture deriving (Eq,Ord)

instance Gettable (TextureTarget, LOD, PixelFormat, Type, Ptr a, VarTexture) () where -- Hmmm...
   get (target, LOD lod, format, typ, buf, VarTexture) =
      glGetTexImage (marshalTextureTarget target) lod (marshalPixelFormat format)
                    (marshalType typ) buf

foreign import unsafe glGetTexImage :: GLenum -> GLint -> GLenum -> GLenum -> Ptr a -> IO ()

data VarTextureSizes =
     VarTextureWidth
   | VarTextureHeight
   | VarTextureDepth   -- @GL_1_2@
   | VarTextureBorder
   | VarTextureRedSize
   | VarTextureGreenSize
   | VarTextureBlueSize
   | VarTextureAlphaSize
   | VarTextureLuminanceSize
   | VarTextureIntensitySize
   deriving (Eq,Ord)

instance Gettable (TextureTarget, LOD, VarTextureSizes) GLsizei where
   get (target, LOD l, VarTextureWidth        ) = getTexLevelParameter (marshalTextureTarget target) l gl_TEXTURE_WIDTH
   get (target, LOD l, VarTextureHeight       ) = getTexLevelParameter (marshalTextureTarget target) l gl_TEXTURE_HEIGHT
   get (target, LOD l, VarTextureDepth        ) = getTexLevelParameter (marshalTextureTarget target) l gl_TEXTURE_DEPTH   -- @GL_1_2@
   get (target, LOD l, VarTextureBorder       ) = getTexLevelParameter (marshalTextureTarget target) l gl_TEXTURE_BORDER
   get (target, LOD l, VarTextureRedSize      ) = getTexLevelParameter (marshalTextureTarget target) l gl_TEXTURE_RED_SIZE
   get (target, LOD l, VarTextureGreenSize    ) = getTexLevelParameter (marshalTextureTarget target) l gl_TEXTURE_GREEN_SIZE
   get (target, LOD l, VarTextureBlueSize     ) = getTexLevelParameter (marshalTextureTarget target) l gl_TEXTURE_BLUE_SIZE
   get (target, LOD l, VarTextureAlphaSize    ) = getTexLevelParameter (marshalTextureTarget target) l gl_TEXTURE_ALPHA_SIZE
   get (target, LOD l, VarTextureLuminanceSize) = getTexLevelParameter (marshalTextureTarget target) l gl_TEXTURE_LUMINANCE_SIZE
   get (target, LOD l, VarTextureIntensitySize) = getTexLevelParameter (marshalTextureTarget target) l gl_TEXTURE_INTENSITY_SIZE

data VarTextureInternalFormat = VarTextureInternalFormat deriving (Eq,Ord)

instance Gettable (TextureTarget, LOD, VarTextureInternalFormat) InternalFormat where
   get (target, LOD l, VarTextureInternalFormat) =
      liftM (unmarshalInternalFormat . fromIntegral) $
      getTexLevelParameter (marshalTextureTarget target) l gl_TEXTURE_INTERNAL_FORMAT
