{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2001  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to the GetString part of section 6.1.11 (Pointer and
String Queries) of the OpenGL 1.2.1 specs. 
-}

module GL_QueryString (
   VersionInfo(..),
   parseVersionString,   -- internal use only
   ExtensionsInfo(..),
   VarVendorRenderer(..),
   VarGLVersion(..),
   VarGLExtensions(..)
) where

import Monad            ( liftM )
import Numeric          ( readDec )
import CString          ( CString, peekCString )

import GL_Constants     ( gl_VENDOR, gl_RENDERER, gl_VERSION, gl_EXTENSIONS )
import GL_BasicTypes    ( GLenum, Gettable(..) )

---------------------------------------------------------------------------

data VersionInfo =
   VersionInfo Int             -- major
               Int             -- minor
               (Maybe Int)     -- release
               (Maybe String)  -- vendor
   deriving (Eq,Ord)

readChar :: ReadS Char
readChar ""     = []
readChar (x:xs) = [(x,xs)]

readMajorMinor :: ReadS (Int, Int)
readMajorMinor s = [ ((major, minor), v) | (major, t) <- readDec  s,
                                           ('.',   u) <- readChar t,
                                           (minor, v) <- readDec  u ]

readRelease :: ReadS (Maybe Int)
readRelease s = take 1 $
   [ (Just release, u) | ('.',     t) <- readChar s,
                         (release, u) <- readDec  t ] ++
   [ (Nothing,      s) ]

readVendorInfo :: ReadS (Maybe String)
readVendorInfo s = take 1 $
   [ (Just vendorInfo, "") | (' ', vendorInfo) <- readChar s ] ++
   [ (Nothing,         s ) ]

-- Sigh... Once again Microsoft did what they always do to standards, i.e.
-- they ignore them. The GL/GLU specs dictate that the version number should
-- either have the form major.minor or major.minor.release, where major,
-- minor, and release consist of digits only. But alas, M$'s GLU on Win98
-- returns 1.2.2.0, which is invalid. We simply ignore the trailing cruft...

readVersionInfo :: ReadS VersionInfo
readVersionInfo s =
   [ (VersionInfo major minor release vendorInfo, w) |
     ((major, minor), t) <- readMajorMinor s,
     (release,        u) <- readRelease    t,
     (_,              v) <- readRelease    u,
     (vendorInfo,     w) <- readVendorInfo v ]

parseVersionString :: String -> String -> VersionInfo
parseVersionString kind versionString =
   case readVersionInfo versionString of
      [(versionInfo, "")] -> versionInfo
      _                   -> error ("ill-formed " ++ kind ++ " version '" ++ versionString ++ "'")

---------------------------------------------------------------------------
-- We split the extensions string into words, so there is no need
-- for gluCheckExtension anymore.

data ExtensionsInfo = ExtensionsInfo [String]
   deriving (Eq,Ord)

---------------------------------------------------------------------------

data VarVendorRenderer =
     VarVendor
   | VarRenderer
   deriving (Eq,Ord)

instance Gettable VarVendorRenderer String where
   get VarVendor   = getString gl_VENDOR
   get VarRenderer = getString gl_RENDERER

data VarGLVersion = VarVersion deriving (Eq,Ord)

instance Gettable VarGLVersion VersionInfo where
   get VarVersion = liftM (parseVersionString "GL") $ getString gl_VERSION

data VarGLExtensions = VarExtensions deriving (Eq,Ord)

instance Gettable VarGLExtensions ExtensionsInfo  where
   get VarExtensions = liftM (ExtensionsInfo . words) $ getString gl_EXTENSIONS

---------------------------------------------------------------------------

getString :: GLenum -> IO String
getString v = glGetString v >>= peekCString

foreign import unsafe glGetString :: GLenum -> IO CString
